import json
import logging
import math
import mimetypes
import os
import pathlib
import shutil
import subprocess

from django.conf import settings

import requests


parent_logger = logging.getLogger(__name__)

TMP_DIR = pathlib.Path("/tmp").resolve()
UA_STRING = settings.OUTBOUND_USER_AGENT_TEMPLATE.format(purpose="Waveform")


def ext_from_url(url):
    """
    Get the file extension from the given URL.

    Looks at the last part of the URL path, and returns the string after the last dot.

    :param url: the URL to the file whose extension is being determined
    :returns: the file extension or ``None``
    """

    file_name = url.split("/")[-1]
    if "." in file_name:
        ext = file_name.split(".")[-1]
        return f".{ext}"
    else:
        return None


def download_audio(url, identifier):
    """
    Download the audio from the given URL to a location on the disk.

    :param url: the URL to the file being downloaded
    :param identifier: the identifier of the media object to name the file
    :returns: the name of the file on the disk
    """

    logger = parent_logger.getChild("download_audio")
    logger.info(f"downloading file url={url}")

    headers = {"User-Agent": UA_STRING}
    with requests.get(url, stream=True, headers=headers) as res:
        logger.debug(f"res.status_code={res.status_code}")
        mimetype = res.headers["content-type"]
        logger.debug(f"mimetype={mimetype}")
        ext = ext_from_url(url) or mimetypes.guess_extension(mimetype)
        if ext is None:
            raise ValueError("Could not identify media extension")
        file_name = f"audio-{identifier}{ext}"
        logger.debug(f"file name={file_name}")
        with open(TMP_DIR.joinpath(file_name), "wb") as file:
            shutil.copyfileobj(res.raw, file)
    return file_name


def generate_waveform(file_name, duration):
    """
    Generate the waveform for the file by invoking the ``audiowaveform`` binary.

    The Python module ``subprocess`` is used to execute the binary and get the
    results that it emits to STDOUT.

    :param file_name: the name of the downloaded audio file
    :param duration: the duration of the audio to determine pixels per second
    """

    logger = parent_logger.getChild("generate_waveform")
    logger.info("Invoking audiowaveform")

    pps = math.ceil(1e6 / duration)  # approx 1000 points in total
    args = [
        "audiowaveform",
        "--input-filename",
        file_name,
        "--output-format",
        "json",
        "--pixels-per-second",
        str(pps),
    ]
    logger.debug(f'executing subprocess command={" ".join(args)}')
    proc = subprocess.run(args, cwd=TMP_DIR, check=True, capture_output=True)
    logger.debug(f"finished subprocess proc.returncode={proc.returncode}")
    return proc.stdout


def process_waveform_output(json_out):
    """
    Parse the waveform output generated by the ``audiowaveform`` binary.

    The output consists of alternating positive and negative values, that are almost
    equal in amplitude. We discard the negative values. We also scale down the
    amplitudes by the largest value so that they lie in the range [0, 1].

    :param json_out: the JSON output generated by ``audiowaveform``
    :returns: the list of peaks
    """

    logger = parent_logger.getChild("process_waveform_output")
    logger.info("Transforming points")

    output = json.loads(json_out)
    data = output["data"]
    logger.debug(f"initial points len(data)={len(data)}")

    transformed_data = []
    max_val = 0
    for idx, val in enumerate(data):
        if idx % 2 == 0:
            continue
        if val < 0:  # Any other odd values are negligible and can be ignored
            val = 0
        transformed_data.append(val)
        if val > max_val:
            max_val = val
    transformed_data = [round(val / max_val, 5) for val in transformed_data]
    logger.debug(
        f"finished transformation len(transformed_data)={len(transformed_data)}"
    )
    return transformed_data


def cleanup(file_name):
    """
    Delete the audio file after it has been processed.

    :param file_name: the name of the file to delete
    """

    logger = parent_logger.getChild("cleanup")

    file_path = TMP_DIR.joinpath(file_name)
    logger.debug(f"file_path={file_path}")
    if file_path.exists():
        logger.debug("deleting file")
        os.remove(file_path)
        logger.debug("file deleted")
    else:
        logger.debug("file not found, nothing deleted")


def generate_peaks(audio) -> list[float]:
    file_name = None
    try:
        file_name = download_audio(audio.url, audio.identifier)
        awf_out = generate_waveform(file_name, audio.duration)
        return process_waveform_output(awf_out)
    finally:
        if file_name is not None:
            cleanup(file_name)
