﻿// Copyright (c) 2011 AlphaSierraPapa for the SharpDevelop Team
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy of this
// software and associated documentation files (the "Software"), to deal in the Software
// without restriction, including without limitation the rights to use, copy, modify, merge,
// publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons
// to whom the Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all copies or
// substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

//using ICSharpCode.ILSpy.Options;
//using ICSharpCode.ILSpy.XmlDoc;
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using ICSharpCode.Decompiler;
using ICSharpCode.Decompiler.Ast;
using ICSharpCode.Decompiler.Ast.Transforms;
using ICSharpCode.NRefactory.CSharp;
using Mono.Cecil;

namespace ICSharpCode.ILSpy
{
    /// <summary>
    ///     Decompiler logic for C#.
    /// </summary>
    [Export(typeof (Language))]
    public class CSharpLanguage : Language
    {
        private string name = "C#";
        private bool showAllMembers;
        private Predicate<IAstTransform> transformAbortCondition;

#if DEBUG
        internal static IEnumerable<CSharpLanguage> GetDebugLanguages()
        {
            var context = new DecompilerContext(ModuleDefinition.CreateModule("dummy", ModuleKind.Dll));
            string lastTransformName = "no transforms";
            foreach (
                Type _transformType in
                    TransformationPipeline.CreatePipeline(context).Select(v => v.GetType()).Distinct())
            {
                Type transformType = _transformType; // copy for lambda
                yield return new CSharpLanguage
                {
                    transformAbortCondition = v => transformType.IsInstanceOfType(v),
                    name = "C# - " + lastTransformName,
                    showAllMembers = true
                };
                lastTransformName = "after " + transformType.Name;
            }
            yield return new CSharpLanguage
            {
                name = "C# - " + lastTransformName,
                showAllMembers = true
            };
        }
#endif

        public override string Name
        {
            get { return name; }
        }

        public override string FileExtension
        {
            get { return ".cs"; }
        }

        public override string ProjectFileExtension
        {
            get { return ".csproj"; }
        }

        public override void DecompileMethod(MethodDefinition method, ITextOutput output, DecompilationOptions options)
        {
            WriteCommentLine(output, TypeToString(method.DeclaringType, true));
            AstBuilder codeDomBuilder = CreateAstBuilder(options, currentType: method.DeclaringType,
                isSingleMember: true);
            if (method.IsConstructor && !method.IsStatic && !method.DeclaringType.IsValueType)
            {
                // also fields and other ctors so that the field initializers can be shown as such
                AddFieldsAndCtors(codeDomBuilder, method.DeclaringType, method.IsStatic);
                RunTransformsAndGenerateCode(codeDomBuilder, output, options, new SelectCtorTransform(method));
            }
            else
            {
                codeDomBuilder.AddMethod(method);
                RunTransformsAndGenerateCode(codeDomBuilder, output, options);
            }
        }

        private class SelectCtorTransform : IAstTransform
        {
            private readonly MethodDefinition ctorDef;

            public SelectCtorTransform(MethodDefinition ctorDef)
            {
                this.ctorDef = ctorDef;
            }

            public void Run(AstNode compilationUnit)
            {
                ConstructorDeclaration ctorDecl = null;
                foreach (AstNode node in compilationUnit.Children)
                {
                    var ctor = node as ConstructorDeclaration;
                    if (ctor != null)
                    {
                        if (ctor.Annotation<MethodDefinition>() == ctorDef)
                        {
                            ctorDecl = ctor;
                        }
                        else
                        {
                            // remove other ctors
                            ctor.Remove();
                        }
                    }
                    // Remove any fields without initializers
                    var fd = node as FieldDeclaration;
                    if (fd != null && fd.Variables.All(v => v.Initializer.IsNull))
                        fd.Remove();
                }
                if (ctorDecl.Initializer.ConstructorInitializerType == ConstructorInitializerType.This)
                {
                    // remove all fields
                    foreach (AstNode node in compilationUnit.Children)
                        if (node is FieldDeclaration)
                            node.Remove();
                }
            }
        }

        public override void DecompileProperty(PropertyDefinition property, ITextOutput output,
            DecompilationOptions options)
        {
            WriteCommentLine(output, TypeToString(property.DeclaringType, true));
            AstBuilder codeDomBuilder = CreateAstBuilder(options, currentType: property.DeclaringType,
                isSingleMember: true);
            codeDomBuilder.AddProperty(property);
            RunTransformsAndGenerateCode(codeDomBuilder, output, options);
        }

        public override void DecompileField(FieldDefinition field, ITextOutput output, DecompilationOptions options)
        {
            WriteCommentLine(output, TypeToString(field.DeclaringType, true));
            AstBuilder codeDomBuilder = CreateAstBuilder(options, currentType: field.DeclaringType, isSingleMember: true);
            if (field.IsLiteral)
            {
                codeDomBuilder.AddField(field);
            }
            else
            {
                // also decompile ctors so that the field initializer can be shown
                AddFieldsAndCtors(codeDomBuilder, field.DeclaringType, field.IsStatic);
            }
            RunTransformsAndGenerateCode(codeDomBuilder, output, options, new SelectFieldTransform(field));
        }

        /// <summary>
        ///     Removes all top-level members except for the specified fields.
        /// </summary>
        private sealed class SelectFieldTransform : IAstTransform
        {
            private readonly FieldDefinition field;

            public SelectFieldTransform(FieldDefinition field)
            {
                this.field = field;
            }

            public void Run(AstNode compilationUnit)
            {
                foreach (AstNode child in compilationUnit.Children)
                {
                    if (child is EntityDeclaration)
                    {
                        if (child.Annotation<FieldDefinition>() != field)
                            child.Remove();
                    }
                }
            }
        }

        private void AddFieldsAndCtors(AstBuilder codeDomBuilder, TypeDefinition declaringType, bool isStatic)
        {
            foreach (FieldDefinition field in declaringType.Fields)
            {
                if (field.IsStatic == isStatic)
                    codeDomBuilder.AddField(field);
            }
            foreach (MethodDefinition ctor in declaringType.Methods)
            {
                if (ctor.IsConstructor && ctor.IsStatic == isStatic)
                    codeDomBuilder.AddMethod(ctor);
            }
        }

        public override void DecompileEvent(EventDefinition ev, ITextOutput output, DecompilationOptions options)
        {
            WriteCommentLine(output, TypeToString(ev.DeclaringType, true));
            AstBuilder codeDomBuilder = CreateAstBuilder(options, currentType: ev.DeclaringType, isSingleMember: true);
            codeDomBuilder.AddEvent(ev);
            RunTransformsAndGenerateCode(codeDomBuilder, output, options);
        }

        public override void DecompileType(TypeDefinition type, ITextOutput output, DecompilationOptions options)
        {
            AstBuilder codeDomBuilder = CreateAstBuilder(options, currentType: type);
            codeDomBuilder.AddType(type);
            RunTransformsAndGenerateCode(codeDomBuilder, output, options);
        }

        private void RunTransformsAndGenerateCode(AstBuilder astBuilder, ITextOutput output,
            DecompilationOptions options, IAstTransform additionalTransform = null)
        {
            astBuilder.RunTransformations(transformAbortCondition);
            if (additionalTransform != null)
            {
                additionalTransform.Run(astBuilder.SyntaxTree);
            }
            //if (options.DecompilerSettings.ShowXmlDocumentation) {
            //    AddXmlDocTransform.Run(astBuilder.CompilationUnit);
            //}
            astBuilder.GenerateCode(output);
        }


        private AstBuilder CreateAstBuilder(DecompilationOptions options, ModuleDefinition currentModule = null,
            TypeDefinition currentType = null, bool isSingleMember = false)
        {
            if (currentModule == null)
                currentModule = currentType.Module;
            DecompilerSettings settings = options.DecompilerSettings;
            if (isSingleMember)
            {
                settings = settings.Clone();
                settings.UsingDeclarations = false;
            }
            return new AstBuilder(
                new DecompilerContext(currentModule)
                {
                    CancellationToken = options.CancellationToken,
                    CurrentType = currentType,
                    Settings = settings
                });
        }

        public override string TypeToString(TypeReference type, bool includeNamespace,
            ICustomAttributeProvider typeAttributes = null)
        {
            var options = ConvertTypeOptions.IncludeTypeParameterDefinitions;
            if (includeNamespace)
                options |= ConvertTypeOptions.IncludeNamespace;

            return TypeToString(options, type, typeAttributes);
        }

        private string TypeToString(ConvertTypeOptions options, TypeReference type,
            ICustomAttributeProvider typeAttributes = null)
        {
            AstType astType = AstBuilder.ConvertType(type, typeAttributes, options);

            var w = new StringWriter();
            if (type.IsByReference)
            {
                var pd = typeAttributes as ParameterDefinition;
                if (pd != null && (!pd.IsIn && pd.IsOut))
                    w.Write("out ");
                else
                    w.Write("ref ");

                if (astType is ComposedType && ((ComposedType) astType).PointerRank > 0)
                    ((ComposedType) astType).PointerRank--;
            }

            astType.AcceptVisitor(new CSharpOutputVisitor(w, FormattingOptionsFactory.CreateAllman()));
            return w.ToString();
        }

        public override string FormatPropertyName(PropertyDefinition property, bool? isIndexer)
        {
            if (property == null)
                throw new ArgumentNullException("property");

            if (!isIndexer.HasValue)
            {
                isIndexer = property.IsIndexer();
            }
            if (isIndexer.Value)
            {
                var buffer = new StringBuilder();
                MethodDefinition accessor = property.GetMethod ?? property.SetMethod;
                if (accessor.HasOverrides)
                {
                    TypeReference declaringType = accessor.Overrides.First().DeclaringType;
                    buffer.Append(TypeToString(declaringType, true));
                    buffer.Append(@".");
                }
                buffer.Append(@"this[");
                bool addSeparator = false;
                foreach (ParameterDefinition p in property.Parameters)
                {
                    if (addSeparator)
                        buffer.Append(@", ");
                    else
                        addSeparator = true;
                    buffer.Append(TypeToString(p.ParameterType, true));
                }
                buffer.Append(@"]");
                return buffer.ToString();
            }
            return property.Name;
        }

        public override string FormatTypeName(TypeDefinition type)
        {
            if (type == null)
                throw new ArgumentNullException("type");

            return
                TypeToString(
                    ConvertTypeOptions.DoNotUsePrimitiveTypeNames | ConvertTypeOptions.IncludeTypeParameterDefinitions,
                    type);
        }

        public override bool ShowMember(MemberReference member)
        {
            return showAllMembers || !AstBuilder.MemberIsHidden(member, new DecompilationOptions().DecompilerSettings);
        }

        public override string GetTooltip(MemberReference member)
        {
            var md = member as MethodDefinition;
            var pd = member as PropertyDefinition;
            var ed = member as EventDefinition;
            var fd = member as FieldDefinition;
            if (md != null || pd != null || ed != null || fd != null)
            {
                var b =
                    new AstBuilder(new DecompilerContext(member.Module)
                    {
                        Settings = new DecompilerSettings {UsingDeclarations = false}
                    });
                b.DecompileMethodBodies = false;
                if (md != null)
                    b.AddMethod(md);
                else if (pd != null)
                    b.AddProperty(pd);
                else if (ed != null)
                    b.AddEvent(ed);
                else
                    b.AddField(fd);
                b.RunTransformations();
                foreach (AttributeSection attribute in b.SyntaxTree.Descendants.OfType<AttributeSection>())
                    attribute.Remove();

                var w = new StringWriter();
                b.GenerateCode(new PlainTextOutput(w));
                return Regex.Replace(w.ToString(), @"\s+", " ").TrimEnd();
            }

            return base.GetTooltip(member);
        }
    }
}