#pragma once

#include <string>
#include "binaryninjaapi.h"
extern "C" {
    #include "xed-interface.h"
}


// The difference between DF_INTEL and DF_BN_INTEL is
//   DF_INTEL is more verbose (states size more often, and some more subtle things)
//   DF_INTEL uses some different opcodes (jz and jnz vs je and jne)
enum DISASSEMBLY_FLAVOR_ENUM
{
    DF_INTEL,
    DF_BN_INTEL,
    DF_ATT,
    DF_XED
};

struct DISASSEMBLY_OPTIONS
{
    DISASSEMBLY_FLAVOR_ENUM df;

    bool lowerCase;
    std::string separator;

    DISASSEMBLY_OPTIONS(DISASSEMBLY_FLAVOR_ENUM df = DF_BN_INTEL, bool lowerCase = true, std::string separator = ", ")
     : df(df), lowerCase(lowerCase), separator(separator) { };
};

#define IL_FLAG_C 0
#define IL_FLAG_P 2
#define IL_FLAG_A 4
#define IL_FLAG_Z 6
#define IL_FLAG_S 7
#define IL_FLAG_D 10
#define IL_FLAG_O 11
#define IL_FLAG_C0 32
#define IL_FLAG_C1 33
#define IL_FLAG_C2 34
#define IL_FLAG_C3 35

#define IL_FLAGWRITE_ALL     1
#define IL_FLAGWRITE_NOCARRY 2
#define IL_FLAGWRITE_CO      3
#define IL_FLAGWRITE_X87COM  4
#define IL_FLAGWRITE_X87COMI 5
#define IL_FLAGWRITE_X87C1Z  6
#define IL_FLAGWRITE_X87RND  7
#define IL_FLAGWRITE_VCOMI   8

#define IL_FLAG_CLASS_INT     0 // Default
#define IL_FLAG_CLASS_X87COM  1
#define IL_FLAG_CLASS_X87COMI 2
#define IL_FLAG_CLASS_VCOMI   3

#define IL_FLAG_GROUP_E  0
#define IL_FLAG_GROUP_NE 1
#define IL_FLAG_GROUP_LT 2
#define IL_FLAG_GROUP_LE 3
#define IL_FLAG_GROUP_GE 4
#define IL_FLAG_GROUP_GT 5
#define IL_FLAG_GROUP_PE 6
#define IL_FLAG_GROUP_PO 7

#define TRAP_DIV       0
#define TRAP_ICEBP     1
#define TRAP_NMI       2
#define TRAP_BP        3
#define TRAP_OVERFLOW  4
#define TRAP_BOUND     5
#define TRAP_ILL       6
#define TRAP_NOT_AVAIL 7
#define TRAP_DOUBLE    8
#define TRAP_TSS       10
#define TRAP_NO_SEG    11
#define TRAP_STACK     12
#define TRAP_GPF       13
#define TRAP_PAGE      14
#define TRAP_FPU       16
#define TRAP_ALIGN     17
#define TRAP_MCE       18
#define TRAP_SIMD      19
// interrupts [20, 31] aren't mentioned in Intel SDM
// interrupts [32, 255] are user defined

#define REG_STACK_X87   0x999
#define REG_X87_r(n)   (0x1000 + (n))             // The fake "real" (physical) register
#define REG_X87_TOP     0x1008

enum X86_INTRINSIC
{
    INTRINSIC_F2XM1,
    INTRINSIC_FBLD,
    INTRINSIC_FBST,
    INTRINSIC_FSIN,
    INTRINSIC_FCOS,
    INTRINSIC_FSINCOS,
    INTRINSIC_FPATAN,
    INTRINSIC_FPREM,
    INTRINSIC_FPREM1,
    INTRINSIC_FPTAN,
    INTRINSIC_FSCALE,
    INTRINSIC_FXAM,
    INTRINSIC_FXTRACT,
    INTRINSIC_FYL2X,
    INTRINSIC_FYL2XP1,
    // below are for vector instrinsics
    // copied from public/arch/x86/xedInc/xed-iform-enum.h
    INTRINSIC_XED_IFORM_INVALID = 1000,
    INTRINSIC_XED_IFORM_AAA,
    INTRINSIC_XED_IFORM_AAD_IMMb,
    INTRINSIC_XED_IFORM_AADD_MEM32_GPR32,
    INTRINSIC_XED_IFORM_AADD_MEM64_GPR64,
    INTRINSIC_XED_IFORM_AADD_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_AADD_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_AAM_IMMb,
    INTRINSIC_XED_IFORM_AAND_MEM32_GPR32,
    INTRINSIC_XED_IFORM_AAND_MEM64_GPR64,
    INTRINSIC_XED_IFORM_AAND_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_AAND_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_AAS,
    INTRINSIC_XED_IFORM_ADC_AL_IMMb,
    INTRINSIC_XED_IFORM_ADC_GPR8_GPR8_10,
    INTRINSIC_XED_IFORM_ADC_GPR8_GPR8_12,
    INTRINSIC_XED_IFORM_ADC_GPR8_IMMb_80r2,
    INTRINSIC_XED_IFORM_ADC_GPR8_IMMb_82r2,
    INTRINSIC_XED_IFORM_ADC_GPR8_MEMb,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADC_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_11,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_13,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_IMMb,
    INTRINSIC_XED_IFORM_ADC_GPRv_IMMz,
    INTRINSIC_XED_IFORM_ADC_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_MEMv,
    INTRINSIC_XED_IFORM_ADC_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADC_MEMb_GPR8,
    INTRINSIC_XED_IFORM_ADC_MEMb_IMMb_80r2,
    INTRINSIC_XED_IFORM_ADC_MEMb_IMMb_82r2,
    INTRINSIC_XED_IFORM_ADC_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADC_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_MEMv_GPRv,
    INTRINSIC_XED_IFORM_ADC_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADC_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADC_MEMv_IMMb,
    INTRINSIC_XED_IFORM_ADC_MEMv_IMMz,
    INTRINSIC_XED_IFORM_ADC_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADC_OrAX_IMMz,
    INTRINSIC_XED_IFORM_ADCX_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_ADCX_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_ADCX_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR32i32_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR64i64_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_ADCX_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_ADCX_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_ADC_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_ADC_LOCK_MEMb_IMMb_80r2,
    INTRINSIC_XED_IFORM_ADC_LOCK_MEMb_IMMb_82r2,
    INTRINSIC_XED_IFORM_ADC_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_ADC_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_ADC_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_ADD_AL_IMMb,
    INTRINSIC_XED_IFORM_ADD_GPR8_GPR8_00,
    INTRINSIC_XED_IFORM_ADD_GPR8_GPR8_02,
    INTRINSIC_XED_IFORM_ADD_GPR8_IMMb_80r0,
    INTRINSIC_XED_IFORM_ADD_GPR8_IMMb_82r0,
    INTRINSIC_XED_IFORM_ADD_GPR8_MEMb,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADD_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_01,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_03,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_IMMb,
    INTRINSIC_XED_IFORM_ADD_GPRv_IMMz,
    INTRINSIC_XED_IFORM_ADD_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_MEMv,
    INTRINSIC_XED_IFORM_ADD_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADD_MEMb_GPR8,
    INTRINSIC_XED_IFORM_ADD_MEMb_IMMb_80r0,
    INTRINSIC_XED_IFORM_ADD_MEMb_IMMb_82r0,
    INTRINSIC_XED_IFORM_ADD_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_ADD_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_MEMv_GPRv,
    INTRINSIC_XED_IFORM_ADD_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_ADD_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ADD_MEMv_IMMb,
    INTRINSIC_XED_IFORM_ADD_MEMv_IMMz,
    INTRINSIC_XED_IFORM_ADD_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_ADD_OrAX_IMMz,
    INTRINSIC_XED_IFORM_ADDPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_ADDPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_ADDPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_ADDPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_ADDSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_ADDSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_ADDSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_ADDSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_ADDSUBPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_ADDSUBPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_ADDSUBPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_ADDSUBPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_ADD_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_ADD_LOCK_MEMb_IMMb_80r0,
    INTRINSIC_XED_IFORM_ADD_LOCK_MEMb_IMMb_82r0,
    INTRINSIC_XED_IFORM_ADD_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_ADD_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_ADD_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_ADOX_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_ADOX_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_ADOX_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR32i32_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR64i64_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_ADOX_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_ADOX_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_AESDEC_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_AESDEC_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_AESDEC128KL_XMMu8_MEMu8,
    INTRINSIC_XED_IFORM_AESDEC256KL_XMMu8_MEMu8,
    INTRINSIC_XED_IFORM_AESDECLAST_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_AESDECLAST_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_AESDECWIDE128KL_MEMu8,
    INTRINSIC_XED_IFORM_AESDECWIDE256KL_MEMu8,
    INTRINSIC_XED_IFORM_AESENC_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_AESENC_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_AESENC128KL_XMMu8_MEMu8,
    INTRINSIC_XED_IFORM_AESENC256KL_XMMu8_MEMu8,
    INTRINSIC_XED_IFORM_AESENCLAST_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_AESENCLAST_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_AESENCWIDE128KL_MEMu8,
    INTRINSIC_XED_IFORM_AESENCWIDE256KL_MEMu8,
    INTRINSIC_XED_IFORM_AESIMC_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_AESIMC_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_AESKEYGENASSIST_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_AESKEYGENASSIST_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_AND_AL_IMMb,
    INTRINSIC_XED_IFORM_AND_GPR8_GPR8_20,
    INTRINSIC_XED_IFORM_AND_GPR8_GPR8_22,
    INTRINSIC_XED_IFORM_AND_GPR8_IMMb_80r4,
    INTRINSIC_XED_IFORM_AND_GPR8_IMMb_82r4,
    INTRINSIC_XED_IFORM_AND_GPR8_MEMb,
    INTRINSIC_XED_IFORM_AND_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_AND_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_21,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_23,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_IMMb,
    INTRINSIC_XED_IFORM_AND_GPRv_IMMz,
    INTRINSIC_XED_IFORM_AND_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_MEMv,
    INTRINSIC_XED_IFORM_AND_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_AND_MEMb_GPR8,
    INTRINSIC_XED_IFORM_AND_MEMb_IMMb_80r4,
    INTRINSIC_XED_IFORM_AND_MEMb_IMMb_82r4,
    INTRINSIC_XED_IFORM_AND_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_AND_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_MEMv_GPRv,
    INTRINSIC_XED_IFORM_AND_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_AND_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_AND_MEMv_IMMb,
    INTRINSIC_XED_IFORM_AND_MEMv_IMMz,
    INTRINSIC_XED_IFORM_AND_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_AND_OrAX_IMMz,
    INTRINSIC_XED_IFORM_ANDN_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_ANDN_GPR32d_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_ANDN_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_ANDN_GPR32i32_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_ANDN_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_ANDN_GPR64i64_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_ANDN_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_ANDN_GPR64q_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_ANDNPD_XMMxuq_MEMxuq,
    INTRINSIC_XED_IFORM_ANDNPD_XMMxuq_XMMxuq,
    INTRINSIC_XED_IFORM_ANDNPS_XMMxud_MEMxud,
    INTRINSIC_XED_IFORM_ANDNPS_XMMxud_XMMxud,
    INTRINSIC_XED_IFORM_ANDPD_XMMxuq_MEMxuq,
    INTRINSIC_XED_IFORM_ANDPD_XMMxuq_XMMxuq,
    INTRINSIC_XED_IFORM_ANDPS_XMMxud_MEMxud,
    INTRINSIC_XED_IFORM_ANDPS_XMMxud_XMMxud,
    INTRINSIC_XED_IFORM_AND_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_AND_LOCK_MEMb_IMMb_80r4,
    INTRINSIC_XED_IFORM_AND_LOCK_MEMb_IMMb_82r4,
    INTRINSIC_XED_IFORM_AND_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_AND_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_AND_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_AOR_MEM32_GPR32,
    INTRINSIC_XED_IFORM_AOR_MEM64_GPR64,
    INTRINSIC_XED_IFORM_AOR_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_AOR_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_ARPL_GPR16_GPR16,
    INTRINSIC_XED_IFORM_ARPL_MEMw_GPR16,
    INTRINSIC_XED_IFORM_AXOR_MEM32_GPR32,
    INTRINSIC_XED_IFORM_AXOR_MEM64_GPR64,
    INTRINSIC_XED_IFORM_AXOR_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_AXOR_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BEXTR_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BEXTR_GPR32d_MEMd_GPR32d,
    INTRINSIC_XED_IFORM_BEXTR_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BEXTR_GPR32i32_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BEXTR_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BEXTR_GPR64i64_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BEXTR_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_BEXTR_GPR64q_MEMq_GPR64q,
    INTRINSIC_XED_IFORM_BEXTR_XOP_GPR32d_GPR32d_IMMd,
    INTRINSIC_XED_IFORM_BEXTR_XOP_GPR32d_MEMd_IMMd,
    INTRINSIC_XED_IFORM_BEXTR_XOP_GPRyy_GPRyy_IMMd,
    INTRINSIC_XED_IFORM_BEXTR_XOP_GPRyy_MEMy_IMMd,
    INTRINSIC_XED_IFORM_BLCFILL_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLCFILL_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLCFILL_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLCFILL_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLCI_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLCI_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLCI_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLCI_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLCIC_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLCIC_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLCIC_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLCIC_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLCMSK_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLCMSK_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLCMSK_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLCMSK_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLCS_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLCS_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLCS_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLCS_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLENDPD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_BLENDPD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_BLENDPS_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_BLENDPS_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_BLENDVPD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_BLENDVPD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_BLENDVPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_BLENDVPS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_BLSFILL_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLSFILL_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLSFILL_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLSFILL_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLSI_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLSI_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLSI_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BLSI_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_BLSI_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BLSI_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_BLSI_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_BLSI_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_BLSIC_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLSIC_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLSIC_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_BLSIC_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_BLSMSK_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLSMSK_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLSMSK_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BLSMSK_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_BLSMSK_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BLSMSK_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_BLSMSK_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_BLSMSK_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_BLSR_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BLSR_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_BLSR_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BLSR_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_BLSR_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BLSR_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_BLSR_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_BLSR_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_BNDCL_BND_AGEN,
    INTRINSIC_XED_IFORM_BNDCL_BND_GPR32,
    INTRINSIC_XED_IFORM_BNDCL_BND_GPR64,
    INTRINSIC_XED_IFORM_BNDCN_BND_AGEN,
    INTRINSIC_XED_IFORM_BNDCN_BND_GPR32,
    INTRINSIC_XED_IFORM_BNDCN_BND_GPR64,
    INTRINSIC_XED_IFORM_BNDCU_BND_AGEN,
    INTRINSIC_XED_IFORM_BNDCU_BND_GPR32,
    INTRINSIC_XED_IFORM_BNDCU_BND_GPR64,
    INTRINSIC_XED_IFORM_BNDLDX_BND_MEMbnd32,
    INTRINSIC_XED_IFORM_BNDLDX_BND_MEMbnd64,
    INTRINSIC_XED_IFORM_BNDMK_BND_AGEN,
    INTRINSIC_XED_IFORM_BNDMOV_BND_BND,
    INTRINSIC_XED_IFORM_BNDMOV_BND_MEMdq,
    INTRINSIC_XED_IFORM_BNDMOV_BND_MEMq,
    INTRINSIC_XED_IFORM_BNDMOV_MEMdq_BND,
    INTRINSIC_XED_IFORM_BNDMOV_MEMq_BND,
    INTRINSIC_XED_IFORM_BNDSTX_MEMbnd32_BND,
    INTRINSIC_XED_IFORM_BNDSTX_MEMbnd64_BND,
    INTRINSIC_XED_IFORM_BOUND_GPRv_MEMa16,
    INTRINSIC_XED_IFORM_BOUND_GPRv_MEMa32,
    INTRINSIC_XED_IFORM_BSF_GPRv_GPRv,
    INTRINSIC_XED_IFORM_BSF_GPRv_MEMv,
    INTRINSIC_XED_IFORM_BSR_GPRv_GPRv,
    INTRINSIC_XED_IFORM_BSR_GPRv_MEMv,
    INTRINSIC_XED_IFORM_BSWAP_GPRv,
    INTRINSIC_XED_IFORM_BT_GPRv_GPRv,
    INTRINSIC_XED_IFORM_BT_GPRv_IMMb,
    INTRINSIC_XED_IFORM_BT_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BT_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BTC_GPRv_GPRv,
    INTRINSIC_XED_IFORM_BTC_GPRv_IMMb,
    INTRINSIC_XED_IFORM_BTC_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BTC_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BTC_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BTC_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BTR_GPRv_GPRv,
    INTRINSIC_XED_IFORM_BTR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_BTR_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BTR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BTR_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BTR_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BTS_GPRv_GPRv,
    INTRINSIC_XED_IFORM_BTS_GPRv_IMMb,
    INTRINSIC_XED_IFORM_BTS_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BTS_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BTS_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_BTS_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_BZHI_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_BZHI_GPR32d_MEMd_GPR32d,
    INTRINSIC_XED_IFORM_BZHI_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BZHI_GPR32i32_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_BZHI_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BZHI_GPR64i64_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_BZHI_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_BZHI_GPR64q_MEMq_GPR64q,
    INTRINSIC_XED_IFORM_CALL_FAR_MEMp2,
    INTRINSIC_XED_IFORM_CALL_FAR_PTRp_IMMw,
    INTRINSIC_XED_IFORM_CALL_NEAR_GPRv,
    INTRINSIC_XED_IFORM_CALL_NEAR_MEMv,
    INTRINSIC_XED_IFORM_CALL_NEAR_RELBRd,
    INTRINSIC_XED_IFORM_CALL_NEAR_RELBRz,
    INTRINSIC_XED_IFORM_CBW,
    INTRINSIC_XED_IFORM_CCMPB_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPB_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPBE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPF_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPL_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPLE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNB_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNBE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNL_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNLE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNO_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNS_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPNZ_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPO_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPS_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPT_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPR8i8_MEMi8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPRv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_GPRv_MEMv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_MEMv_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CCMPZ_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CDQ,
    INTRINSIC_XED_IFORM_CDQE,
    INTRINSIC_XED_IFORM_CFCMOVB_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVB_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVB_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVB_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVB_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVBE_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVBE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVBE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVBE_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVBE_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVL_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVL_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVL_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVL_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVL_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVLE_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVLE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVLE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVLE_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVLE_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNB_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNB_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNB_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNB_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNB_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNBE_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNBE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNBE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNBE_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNBE_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNL_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNL_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNL_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNL_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNL_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNLE_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNLE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNLE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNLE_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNLE_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNO_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNO_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNO_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNO_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNO_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNP_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNP_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNP_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNP_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNP_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNS_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNS_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNS_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNS_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNS_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNZ_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNZ_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNZ_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNZ_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVNZ_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVO_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVO_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVO_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVO_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVO_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVP_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVP_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVP_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVP_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVP_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVS_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVS_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVS_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVS_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVS_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVZ_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVZ_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CFCMOVZ_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVZ_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CFCMOVZ_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_CLAC,
    INTRINSIC_XED_IFORM_CLC,
    INTRINSIC_XED_IFORM_CLD,
    INTRINSIC_XED_IFORM_CLDEMOTE_MEMu8,
    INTRINSIC_XED_IFORM_CLFLUSH_MEMmprefetch,
    INTRINSIC_XED_IFORM_CLFLUSHOPT_MEMmprefetch,
    INTRINSIC_XED_IFORM_CLGI,
    INTRINSIC_XED_IFORM_CLI,
    INTRINSIC_XED_IFORM_CLRSSBSY_MEMu64,
    INTRINSIC_XED_IFORM_CLTS,
    INTRINSIC_XED_IFORM_CLUI,
    INTRINSIC_XED_IFORM_CLWB_MEMmprefetch,
    INTRINSIC_XED_IFORM_CLZERO,
    INTRINSIC_XED_IFORM_CMC,
    INTRINSIC_XED_IFORM_CMOVB_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVB_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVB_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVB_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVBE_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVBE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVBE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVBE_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVL_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVL_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVL_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVL_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVLE_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVLE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVLE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVLE_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNB_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNB_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNB_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNB_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNBE_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNBE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNBE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNBE_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNL_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNL_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNL_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNL_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNLE_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNLE_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNLE_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNLE_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNO_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNO_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNO_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNO_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNP_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNP_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNP_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNP_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNS_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNS_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNS_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNS_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVNZ_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVNZ_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVNZ_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVNZ_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVO_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVO_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVO_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVO_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVP_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVP_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVP_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVP_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVS_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVS_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVS_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVS_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMOVZ_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMOVZ_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_CMOVZ_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_CMOVZ_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMP_AL_IMMb,
    INTRINSIC_XED_IFORM_CMP_GPR8_GPR8_38,
    INTRINSIC_XED_IFORM_CMP_GPR8_GPR8_3A,
    INTRINSIC_XED_IFORM_CMP_GPR8_IMMb_80r7,
    INTRINSIC_XED_IFORM_CMP_GPR8_IMMb_82r7,
    INTRINSIC_XED_IFORM_CMP_GPR8_MEMb,
    INTRINSIC_XED_IFORM_CMP_GPRv_GPRv_39,
    INTRINSIC_XED_IFORM_CMP_GPRv_GPRv_3B,
    INTRINSIC_XED_IFORM_CMP_GPRv_IMMb,
    INTRINSIC_XED_IFORM_CMP_GPRv_IMMz,
    INTRINSIC_XED_IFORM_CMP_GPRv_MEMv,
    INTRINSIC_XED_IFORM_CMP_MEMb_GPR8,
    INTRINSIC_XED_IFORM_CMP_MEMb_IMMb_80r7,
    INTRINSIC_XED_IFORM_CMP_MEMb_IMMb_82r7,
    INTRINSIC_XED_IFORM_CMP_MEMv_GPRv,
    INTRINSIC_XED_IFORM_CMP_MEMv_IMMb,
    INTRINSIC_XED_IFORM_CMP_MEMv_IMMz,
    INTRINSIC_XED_IFORM_CMP_OrAX_IMMz,
    INTRINSIC_XED_IFORM_CMPBEXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPBEXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPBEXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPBEXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPBXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPBXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPBXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPBXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPLEXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPLEXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPLEXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPLEXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPLXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPLXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPLXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPLXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNBEXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNBEXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNBEXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNBEXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNBXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNBXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNBXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNBXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNLEXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNLEXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNLEXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNLEXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNLXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNLXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNLXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNLXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNOXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNOXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNOXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNOXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNPXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNPXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNPXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNPXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNSXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNSXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNSXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNSXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPNZXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPNZXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPNZXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPNZXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPOXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPOXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPOXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPOXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPPD_XMMpd_MEMpd_IMMb,
    INTRINSIC_XED_IFORM_CMPPD_XMMpd_XMMpd_IMMb,
    INTRINSIC_XED_IFORM_CMPPS_XMMps_MEMps_IMMb,
    INTRINSIC_XED_IFORM_CMPPS_XMMps_XMMps_IMMb,
    INTRINSIC_XED_IFORM_CMPPXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPPXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPPXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPPXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPSB,
    INTRINSIC_XED_IFORM_CMPSD,
    INTRINSIC_XED_IFORM_CMPSD_XMM_XMMsd_MEMsd_IMMb,
    INTRINSIC_XED_IFORM_CMPSD_XMM_XMMsd_XMMsd_IMMb,
    INTRINSIC_XED_IFORM_CMPSQ,
    INTRINSIC_XED_IFORM_CMPSS_XMMss_MEMss_IMMb,
    INTRINSIC_XED_IFORM_CMPSS_XMMss_XMMss_IMMb,
    INTRINSIC_XED_IFORM_CMPSW,
    INTRINSIC_XED_IFORM_CMPSXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPSXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPSXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPSXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_CMPXCHG_GPR8_GPR8,
    INTRINSIC_XED_IFORM_CMPXCHG_GPRv_GPRv,
    INTRINSIC_XED_IFORM_CMPXCHG_MEMb_GPR8,
    INTRINSIC_XED_IFORM_CMPXCHG_MEMv_GPRv,
    INTRINSIC_XED_IFORM_CMPXCHG16B_MEMdq,
    INTRINSIC_XED_IFORM_CMPXCHG16B_LOCK_MEMdq,
    INTRINSIC_XED_IFORM_CMPXCHG8B_MEMq,
    INTRINSIC_XED_IFORM_CMPXCHG8B_LOCK_MEMq,
    INTRINSIC_XED_IFORM_CMPXCHG_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_CMPXCHG_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_CMPZXADD_MEMu32_GPR32u32_GPR32u32,
    INTRINSIC_XED_IFORM_CMPZXADD_MEMu32_GPR32u32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_CMPZXADD_MEMu64_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_CMPZXADD_MEMu64_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_COMISD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_COMISD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_COMISS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_COMISS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_CPUID,
    INTRINSIC_XED_IFORM_CQO,
    INTRINSIC_XED_IFORM_CRC32_GPRy_GPR8i8_APX,
    INTRINSIC_XED_IFORM_CRC32_GPRy_GPRv_APX,
    INTRINSIC_XED_IFORM_CRC32_GPRy_MEMi8_APX,
    INTRINSIC_XED_IFORM_CRC32_GPRy_MEMv_APX,
    INTRINSIC_XED_IFORM_CRC32_GPRyy_GPR8b,
    INTRINSIC_XED_IFORM_CRC32_GPRyy_GPRv,
    INTRINSIC_XED_IFORM_CRC32_GPRyy_MEMb,
    INTRINSIC_XED_IFORM_CRC32_GPRyy_MEMv,
    INTRINSIC_XED_IFORM_CTESTB_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTB_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTBE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTF_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTL_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTLE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNB_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNBE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNL_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNLE_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNO_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNS_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTNZ_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTO_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTS_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTT_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_GPR8i8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_GPR8i8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_GPRv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_GPRv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_MEMi8_GPR8i8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_MEMi8_IMM8_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_MEMv_GPRv_DFV_APX,
    INTRINSIC_XED_IFORM_CTESTZ_MEMv_IMMz_DFV_APX,
    INTRINSIC_XED_IFORM_CVTDQ2PD_XMMpd_MEMq,
    INTRINSIC_XED_IFORM_CVTDQ2PD_XMMpd_XMMq,
    INTRINSIC_XED_IFORM_CVTDQ2PS_XMMps_MEMdq,
    INTRINSIC_XED_IFORM_CVTDQ2PS_XMMps_XMMdq,
    INTRINSIC_XED_IFORM_CVTPD2DQ_XMMdq_MEMpd,
    INTRINSIC_XED_IFORM_CVTPD2DQ_XMMdq_XMMpd,
    INTRINSIC_XED_IFORM_CVTPD2PI_MMXq_MEMpd,
    INTRINSIC_XED_IFORM_CVTPD2PI_MMXq_XMMpd,
    INTRINSIC_XED_IFORM_CVTPD2PS_XMMps_MEMpd,
    INTRINSIC_XED_IFORM_CVTPD2PS_XMMps_XMMpd,
    INTRINSIC_XED_IFORM_CVTPI2PD_XMMpd_MEMq,
    INTRINSIC_XED_IFORM_CVTPI2PD_XMMpd_MMXq,
    INTRINSIC_XED_IFORM_CVTPI2PS_XMMq_MEMq,
    INTRINSIC_XED_IFORM_CVTPI2PS_XMMq_MMXq,
    INTRINSIC_XED_IFORM_CVTPS2DQ_XMMdq_MEMps,
    INTRINSIC_XED_IFORM_CVTPS2DQ_XMMdq_XMMps,
    INTRINSIC_XED_IFORM_CVTPS2PD_XMMpd_MEMq,
    INTRINSIC_XED_IFORM_CVTPS2PD_XMMpd_XMMq,
    INTRINSIC_XED_IFORM_CVTPS2PI_MMXq_MEMq,
    INTRINSIC_XED_IFORM_CVTPS2PI_MMXq_XMMq,
    INTRINSIC_XED_IFORM_CVTSD2SI_GPR32d_MEMsd,
    INTRINSIC_XED_IFORM_CVTSD2SI_GPR32d_XMMsd,
    INTRINSIC_XED_IFORM_CVTSD2SI_GPR64q_MEMsd,
    INTRINSIC_XED_IFORM_CVTSD2SI_GPR64q_XMMsd,
    INTRINSIC_XED_IFORM_CVTSD2SS_XMMss_MEMsd,
    INTRINSIC_XED_IFORM_CVTSD2SS_XMMss_XMMsd,
    INTRINSIC_XED_IFORM_CVTSI2SD_XMMsd_GPR32d,
    INTRINSIC_XED_IFORM_CVTSI2SD_XMMsd_GPR64q,
    INTRINSIC_XED_IFORM_CVTSI2SD_XMMsd_MEMd,
    INTRINSIC_XED_IFORM_CVTSI2SD_XMMsd_MEMq,
    INTRINSIC_XED_IFORM_CVTSI2SS_XMMss_GPR32d,
    INTRINSIC_XED_IFORM_CVTSI2SS_XMMss_GPR64q,
    INTRINSIC_XED_IFORM_CVTSI2SS_XMMss_MEMd,
    INTRINSIC_XED_IFORM_CVTSI2SS_XMMss_MEMq,
    INTRINSIC_XED_IFORM_CVTSS2SD_XMMsd_MEMss,
    INTRINSIC_XED_IFORM_CVTSS2SD_XMMsd_XMMss,
    INTRINSIC_XED_IFORM_CVTSS2SI_GPR32d_MEMss,
    INTRINSIC_XED_IFORM_CVTSS2SI_GPR32d_XMMss,
    INTRINSIC_XED_IFORM_CVTSS2SI_GPR64q_MEMss,
    INTRINSIC_XED_IFORM_CVTSS2SI_GPR64q_XMMss,
    INTRINSIC_XED_IFORM_CVTTPD2DQ_XMMdq_MEMpd,
    INTRINSIC_XED_IFORM_CVTTPD2DQ_XMMdq_XMMpd,
    INTRINSIC_XED_IFORM_CVTTPD2PI_MMXq_MEMpd,
    INTRINSIC_XED_IFORM_CVTTPD2PI_MMXq_XMMpd,
    INTRINSIC_XED_IFORM_CVTTPS2DQ_XMMdq_MEMps,
    INTRINSIC_XED_IFORM_CVTTPS2DQ_XMMdq_XMMps,
    INTRINSIC_XED_IFORM_CVTTPS2PI_MMXq_MEMq,
    INTRINSIC_XED_IFORM_CVTTPS2PI_MMXq_XMMq,
    INTRINSIC_XED_IFORM_CVTTSD2SI_GPR32d_MEMsd,
    INTRINSIC_XED_IFORM_CVTTSD2SI_GPR32d_XMMsd,
    INTRINSIC_XED_IFORM_CVTTSD2SI_GPR64q_MEMsd,
    INTRINSIC_XED_IFORM_CVTTSD2SI_GPR64q_XMMsd,
    INTRINSIC_XED_IFORM_CVTTSS2SI_GPR32d_MEMss,
    INTRINSIC_XED_IFORM_CVTTSS2SI_GPR32d_XMMss,
    INTRINSIC_XED_IFORM_CVTTSS2SI_GPR64q_MEMss,
    INTRINSIC_XED_IFORM_CVTTSS2SI_GPR64q_XMMss,
    INTRINSIC_XED_IFORM_CWD,
    INTRINSIC_XED_IFORM_CWDE,
    INTRINSIC_XED_IFORM_DAA,
    INTRINSIC_XED_IFORM_DAS,
    INTRINSIC_XED_IFORM_DEC_GPR8,
    INTRINSIC_XED_IFORM_DEC_GPR8i8_APX,
    INTRINSIC_XED_IFORM_DEC_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_DEC_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_DEC_GPRv_48,
    INTRINSIC_XED_IFORM_DEC_GPRv_APX,
    INTRINSIC_XED_IFORM_DEC_GPRv_FFr1,
    INTRINSIC_XED_IFORM_DEC_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_DEC_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_DEC_MEMb,
    INTRINSIC_XED_IFORM_DEC_MEMi8_APX,
    INTRINSIC_XED_IFORM_DEC_MEMv,
    INTRINSIC_XED_IFORM_DEC_MEMv_APX,
    INTRINSIC_XED_IFORM_DEC_LOCK_MEMb,
    INTRINSIC_XED_IFORM_DEC_LOCK_MEMv,
    INTRINSIC_XED_IFORM_DIV_GPR8,
    INTRINSIC_XED_IFORM_DIV_GPR8i8_APX,
    INTRINSIC_XED_IFORM_DIV_GPRv,
    INTRINSIC_XED_IFORM_DIV_GPRv_APX,
    INTRINSIC_XED_IFORM_DIV_MEMb,
    INTRINSIC_XED_IFORM_DIV_MEMi8_APX,
    INTRINSIC_XED_IFORM_DIV_MEMv,
    INTRINSIC_XED_IFORM_DIV_MEMv_APX,
    INTRINSIC_XED_IFORM_DIVPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_DIVPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_DIVPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_DIVPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_DIVSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_DIVSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_DIVSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_DIVSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_DPPD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_DPPD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_DPPS_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_DPPS_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_EMMS,
    INTRINSIC_XED_IFORM_ENCLS,
    INTRINSIC_XED_IFORM_ENCLU,
    INTRINSIC_XED_IFORM_ENCLV,
    INTRINSIC_XED_IFORM_ENCODEKEY128_GPR32u8_GPR32u8,
    INTRINSIC_XED_IFORM_ENCODEKEY256_GPR32u8_GPR32u8,
    INTRINSIC_XED_IFORM_ENDBR32,
    INTRINSIC_XED_IFORM_ENDBR64,
    INTRINSIC_XED_IFORM_ENQCMD_GPRa_MEMu32,
    INTRINSIC_XED_IFORM_ENQCMD_GPRav_MEMu32_APX,
    INTRINSIC_XED_IFORM_ENQCMDS_GPRa_MEMu32,
    INTRINSIC_XED_IFORM_ENQCMDS_GPRav_MEMu32_APX,
    INTRINSIC_XED_IFORM_ENTER_IMMw_IMMb,
    INTRINSIC_XED_IFORM_ERETS,
    INTRINSIC_XED_IFORM_ERETU,
    INTRINSIC_XED_IFORM_EXTRACTPS_GPR32d_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_EXTRACTPS_MEMd_XMMps_IMMb,
    INTRINSIC_XED_IFORM_EXTRQ_XMMq_IMMb_IMMb,
    INTRINSIC_XED_IFORM_EXTRQ_XMMq_XMMdq,
    INTRINSIC_XED_IFORM_F2XM1,
    INTRINSIC_XED_IFORM_FABS,
    INTRINSIC_XED_IFORM_FADD_MEMm64real,
    INTRINSIC_XED_IFORM_FADD_MEMmem32real,
    INTRINSIC_XED_IFORM_FADD_ST0_X87,
    INTRINSIC_XED_IFORM_FADD_X87_ST0,
    INTRINSIC_XED_IFORM_FADDP_X87_ST0,
    INTRINSIC_XED_IFORM_FBLD_ST0_MEMmem80dec,
    INTRINSIC_XED_IFORM_FBSTP_MEMmem80dec_ST0,
    INTRINSIC_XED_IFORM_FCHS,
    INTRINSIC_XED_IFORM_FCMOVB_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVBE_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVE_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVNB_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVNBE_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVNE_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVNU_ST0_X87,
    INTRINSIC_XED_IFORM_FCMOVU_ST0_X87,
    INTRINSIC_XED_IFORM_FCOM_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FCOM_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FCOM_ST0_X87,
    INTRINSIC_XED_IFORM_FCOM_ST0_X87_DCD0,
    INTRINSIC_XED_IFORM_FCOMI_ST0_X87,
    INTRINSIC_XED_IFORM_FCOMIP_ST0_X87,
    INTRINSIC_XED_IFORM_FCOMP_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FCOMP_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FCOMP_ST0_X87,
    INTRINSIC_XED_IFORM_FCOMP_ST0_X87_DCD1,
    INTRINSIC_XED_IFORM_FCOMP_ST0_X87_DED0,
    INTRINSIC_XED_IFORM_FCOMPP,
    INTRINSIC_XED_IFORM_FCOS,
    INTRINSIC_XED_IFORM_FDECSTP,
    INTRINSIC_XED_IFORM_FDISI8087_NOP,
    INTRINSIC_XED_IFORM_FDIV_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FDIV_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FDIV_ST0_X87,
    INTRINSIC_XED_IFORM_FDIV_X87_ST0,
    INTRINSIC_XED_IFORM_FDIVP_X87_ST0,
    INTRINSIC_XED_IFORM_FDIVR_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FDIVR_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FDIVR_ST0_X87,
    INTRINSIC_XED_IFORM_FDIVR_X87_ST0,
    INTRINSIC_XED_IFORM_FDIVRP_X87_ST0,
    INTRINSIC_XED_IFORM_FEMMS,
    INTRINSIC_XED_IFORM_FENI8087_NOP,
    INTRINSIC_XED_IFORM_FFREE_X87,
    INTRINSIC_XED_IFORM_FFREEP_X87,
    INTRINSIC_XED_IFORM_FIADD_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FIADD_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FICOM_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FICOM_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FICOMP_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FICOMP_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FIDIV_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FIDIV_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FIDIVR_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FIDIVR_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FILD_ST0_MEMm64int,
    INTRINSIC_XED_IFORM_FILD_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FILD_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FIMUL_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FIMUL_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FINCSTP,
    INTRINSIC_XED_IFORM_FIST_MEMmem16int_ST0,
    INTRINSIC_XED_IFORM_FIST_MEMmem32int_ST0,
    INTRINSIC_XED_IFORM_FISTP_MEMm64int_ST0,
    INTRINSIC_XED_IFORM_FISTP_MEMmem16int_ST0,
    INTRINSIC_XED_IFORM_FISTP_MEMmem32int_ST0,
    INTRINSIC_XED_IFORM_FISTTP_MEMm64int_ST0,
    INTRINSIC_XED_IFORM_FISTTP_MEMmem16int_ST0,
    INTRINSIC_XED_IFORM_FISTTP_MEMmem32int_ST0,
    INTRINSIC_XED_IFORM_FISUB_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FISUB_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FISUBR_ST0_MEMmem16int,
    INTRINSIC_XED_IFORM_FISUBR_ST0_MEMmem32int,
    INTRINSIC_XED_IFORM_FLD_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FLD_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FLD_ST0_MEMmem80real,
    INTRINSIC_XED_IFORM_FLD_ST0_X87,
    INTRINSIC_XED_IFORM_FLD1,
    INTRINSIC_XED_IFORM_FLDCW_MEMmem16,
    INTRINSIC_XED_IFORM_FLDENV_MEMmem14,
    INTRINSIC_XED_IFORM_FLDENV_MEMmem28,
    INTRINSIC_XED_IFORM_FLDL2E,
    INTRINSIC_XED_IFORM_FLDL2T,
    INTRINSIC_XED_IFORM_FLDLG2,
    INTRINSIC_XED_IFORM_FLDLN2,
    INTRINSIC_XED_IFORM_FLDPI,
    INTRINSIC_XED_IFORM_FLDZ,
    INTRINSIC_XED_IFORM_FMUL_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FMUL_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FMUL_ST0_X87,
    INTRINSIC_XED_IFORM_FMUL_X87_ST0,
    INTRINSIC_XED_IFORM_FMULP_X87_ST0,
    INTRINSIC_XED_IFORM_FNCLEX,
    INTRINSIC_XED_IFORM_FNINIT,
    INTRINSIC_XED_IFORM_FNOP,
    INTRINSIC_XED_IFORM_FNSAVE_MEMmem108,
    INTRINSIC_XED_IFORM_FNSAVE_MEMmem94,
    INTRINSIC_XED_IFORM_FNSTCW_MEMmem16,
    INTRINSIC_XED_IFORM_FNSTENV_MEMmem14,
    INTRINSIC_XED_IFORM_FNSTENV_MEMmem28,
    INTRINSIC_XED_IFORM_FNSTSW_AX,
    INTRINSIC_XED_IFORM_FNSTSW_MEMmem16,
    INTRINSIC_XED_IFORM_FPATAN,
    INTRINSIC_XED_IFORM_FPREM,
    INTRINSIC_XED_IFORM_FPREM1,
    INTRINSIC_XED_IFORM_FPTAN,
    INTRINSIC_XED_IFORM_FRNDINT,
    INTRINSIC_XED_IFORM_FRSTOR_MEMmem108,
    INTRINSIC_XED_IFORM_FRSTOR_MEMmem94,
    INTRINSIC_XED_IFORM_FSCALE,
    INTRINSIC_XED_IFORM_FSETPM287_NOP,
    INTRINSIC_XED_IFORM_FSIN,
    INTRINSIC_XED_IFORM_FSINCOS,
    INTRINSIC_XED_IFORM_FSQRT,
    INTRINSIC_XED_IFORM_FST_MEMm64real_ST0,
    INTRINSIC_XED_IFORM_FST_MEMmem32real_ST0,
    INTRINSIC_XED_IFORM_FST_X87_ST0,
    INTRINSIC_XED_IFORM_FSTP_MEMm64real_ST0,
    INTRINSIC_XED_IFORM_FSTP_MEMmem32real_ST0,
    INTRINSIC_XED_IFORM_FSTP_MEMmem80real_ST0,
    INTRINSIC_XED_IFORM_FSTP_X87_ST0,
    INTRINSIC_XED_IFORM_FSTP_X87_ST0_DFD0,
    INTRINSIC_XED_IFORM_FSTP_X87_ST0_DFD1,
    INTRINSIC_XED_IFORM_FSTPNCE_X87_ST0,
    INTRINSIC_XED_IFORM_FSUB_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FSUB_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FSUB_ST0_X87,
    INTRINSIC_XED_IFORM_FSUB_X87_ST0,
    INTRINSIC_XED_IFORM_FSUBP_X87_ST0,
    INTRINSIC_XED_IFORM_FSUBR_ST0_MEMm64real,
    INTRINSIC_XED_IFORM_FSUBR_ST0_MEMmem32real,
    INTRINSIC_XED_IFORM_FSUBR_ST0_X87,
    INTRINSIC_XED_IFORM_FSUBR_X87_ST0,
    INTRINSIC_XED_IFORM_FSUBRP_X87_ST0,
    INTRINSIC_XED_IFORM_FTST,
    INTRINSIC_XED_IFORM_FUCOM_ST0_X87,
    INTRINSIC_XED_IFORM_FUCOMI_ST0_X87,
    INTRINSIC_XED_IFORM_FUCOMIP_ST0_X87,
    INTRINSIC_XED_IFORM_FUCOMP_ST0_X87,
    INTRINSIC_XED_IFORM_FUCOMPP,
    INTRINSIC_XED_IFORM_FWAIT,
    INTRINSIC_XED_IFORM_FXAM,
    INTRINSIC_XED_IFORM_FXCH_ST0_X87,
    INTRINSIC_XED_IFORM_FXCH_ST0_X87_DDC1,
    INTRINSIC_XED_IFORM_FXCH_ST0_X87_DFC1,
    INTRINSIC_XED_IFORM_FXRSTOR_MEMmfpxenv,
    INTRINSIC_XED_IFORM_FXRSTOR64_MEMmfpxenv,
    INTRINSIC_XED_IFORM_FXSAVE_MEMmfpxenv,
    INTRINSIC_XED_IFORM_FXSAVE64_MEMmfpxenv,
    INTRINSIC_XED_IFORM_FXTRACT,
    INTRINSIC_XED_IFORM_FYL2X,
    INTRINSIC_XED_IFORM_FYL2XP1,
    INTRINSIC_XED_IFORM_GETSEC,
    INTRINSIC_XED_IFORM_GF2P8AFFINEINVQB_XMMu8_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_GF2P8AFFINEINVQB_XMMu8_XMMu64_IMM8,
    INTRINSIC_XED_IFORM_GF2P8AFFINEQB_XMMu8_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_GF2P8AFFINEQB_XMMu8_XMMu64_IMM8,
    INTRINSIC_XED_IFORM_GF2P8MULB_XMMu8_MEMu8,
    INTRINSIC_XED_IFORM_GF2P8MULB_XMMu8_XMMu8,
    INTRINSIC_XED_IFORM_HADDPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_HADDPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_HADDPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_HADDPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_HLT,
    INTRINSIC_XED_IFORM_HRESET_IMM8,
    INTRINSIC_XED_IFORM_HSUBPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_HSUBPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_HSUBPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_HSUBPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_IDIV_GPR8,
    INTRINSIC_XED_IFORM_IDIV_GPR8i8_APX,
    INTRINSIC_XED_IFORM_IDIV_GPRv,
    INTRINSIC_XED_IFORM_IDIV_GPRv_APX,
    INTRINSIC_XED_IFORM_IDIV_MEMb,
    INTRINSIC_XED_IFORM_IDIV_MEMi8_APX,
    INTRINSIC_XED_IFORM_IDIV_MEMv,
    INTRINSIC_XED_IFORM_IDIV_MEMv_APX,
    INTRINSIC_XED_IFORM_IMUL_GPR8,
    INTRINSIC_XED_IFORM_IMUL_GPR8i8_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv,
    INTRINSIC_XED_IFORM_IMUL_GPRv_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_IMMb,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_IMMz,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_MEMv,
    INTRINSIC_XED_IFORM_IMUL_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_IMUL_GPRv_MEMv_IMMb,
    INTRINSIC_XED_IFORM_IMUL_GPRv_MEMv_IMMz,
    INTRINSIC_XED_IFORM_IMUL_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_IMUL_MEMb,
    INTRINSIC_XED_IFORM_IMUL_MEMi8_APX,
    INTRINSIC_XED_IFORM_IMUL_MEMv,
    INTRINSIC_XED_IFORM_IMUL_MEMv_APX,
    INTRINSIC_XED_IFORM_IN_AL_DX,
    INTRINSIC_XED_IFORM_IN_AL_IMMb,
    INTRINSIC_XED_IFORM_IN_OeAX_DX,
    INTRINSIC_XED_IFORM_IN_OeAX_IMMb,
    INTRINSIC_XED_IFORM_INC_GPR8,
    INTRINSIC_XED_IFORM_INC_GPR8i8_APX,
    INTRINSIC_XED_IFORM_INC_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_INC_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_INC_GPRv_40,
    INTRINSIC_XED_IFORM_INC_GPRv_APX,
    INTRINSIC_XED_IFORM_INC_GPRv_FFr0,
    INTRINSIC_XED_IFORM_INC_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_INC_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_INC_MEMb,
    INTRINSIC_XED_IFORM_INC_MEMi8_APX,
    INTRINSIC_XED_IFORM_INC_MEMv,
    INTRINSIC_XED_IFORM_INC_MEMv_APX,
    INTRINSIC_XED_IFORM_INCSSPD_GPR32u8,
    INTRINSIC_XED_IFORM_INCSSPQ_GPR64u8,
    INTRINSIC_XED_IFORM_INC_LOCK_MEMb,
    INTRINSIC_XED_IFORM_INC_LOCK_MEMv,
    INTRINSIC_XED_IFORM_INSB,
    INTRINSIC_XED_IFORM_INSD,
    INTRINSIC_XED_IFORM_INSERTPS_XMMps_MEMd_IMMb,
    INTRINSIC_XED_IFORM_INSERTPS_XMMps_XMMps_IMMb,
    INTRINSIC_XED_IFORM_INSERTQ_XMMq_XMMdq,
    INTRINSIC_XED_IFORM_INSERTQ_XMMq_XMMq_IMMb_IMMb,
    INTRINSIC_XED_IFORM_INSW,
    INTRINSIC_XED_IFORM_INT_IMMb,
    INTRINSIC_XED_IFORM_INT1,
    INTRINSIC_XED_IFORM_INT3,
    INTRINSIC_XED_IFORM_INTO,
    INTRINSIC_XED_IFORM_INVD,
    INTRINSIC_XED_IFORM_INVEPT_GPR32_MEMdq,
    INTRINSIC_XED_IFORM_INVEPT_GPR64_MEMdq,
    INTRINSIC_XED_IFORM_INVEPT_GPR64i64_MEMi128_APX,
    INTRINSIC_XED_IFORM_INVLPG_MEMb,
    INTRINSIC_XED_IFORM_INVLPGA_ArAX_ECX,
    INTRINSIC_XED_IFORM_INVLPGB_EAX_EDX_ECX,
    INTRINSIC_XED_IFORM_INVLPGB_RAX_EDX_ECX,
    INTRINSIC_XED_IFORM_INVPCID_GPR32_MEMdq,
    INTRINSIC_XED_IFORM_INVPCID_GPR64_MEMdq,
    INTRINSIC_XED_IFORM_INVPCID_GPR64i64_MEMi128_APX,
    INTRINSIC_XED_IFORM_INVVPID_GPR32_MEMdq,
    INTRINSIC_XED_IFORM_INVVPID_GPR64_MEMdq,
    INTRINSIC_XED_IFORM_INVVPID_GPR64i64_MEMi128_APX,
    INTRINSIC_XED_IFORM_IRET,
    INTRINSIC_XED_IFORM_IRETD,
    INTRINSIC_XED_IFORM_IRETQ,
    INTRINSIC_XED_IFORM_JB_RELBRb,
    INTRINSIC_XED_IFORM_JB_RELBRd,
    INTRINSIC_XED_IFORM_JB_RELBRz,
    INTRINSIC_XED_IFORM_JBE_RELBRb,
    INTRINSIC_XED_IFORM_JBE_RELBRd,
    INTRINSIC_XED_IFORM_JBE_RELBRz,
    INTRINSIC_XED_IFORM_JCXZ_RELBRb,
    INTRINSIC_XED_IFORM_JECXZ_RELBRb,
    INTRINSIC_XED_IFORM_JL_RELBRb,
    INTRINSIC_XED_IFORM_JL_RELBRd,
    INTRINSIC_XED_IFORM_JL_RELBRz,
    INTRINSIC_XED_IFORM_JLE_RELBRb,
    INTRINSIC_XED_IFORM_JLE_RELBRd,
    INTRINSIC_XED_IFORM_JLE_RELBRz,
    INTRINSIC_XED_IFORM_JMP_GPRv,
    INTRINSIC_XED_IFORM_JMP_MEMv,
    INTRINSIC_XED_IFORM_JMP_RELBRb,
    INTRINSIC_XED_IFORM_JMP_RELBRd,
    INTRINSIC_XED_IFORM_JMP_RELBRz,
    INTRINSIC_XED_IFORM_JMPABS_ABSBRu64_APX,
    INTRINSIC_XED_IFORM_JMP_FAR_MEMp2,
    INTRINSIC_XED_IFORM_JMP_FAR_PTRp_IMMw,
    INTRINSIC_XED_IFORM_JNB_RELBRb,
    INTRINSIC_XED_IFORM_JNB_RELBRd,
    INTRINSIC_XED_IFORM_JNB_RELBRz,
    INTRINSIC_XED_IFORM_JNBE_RELBRb,
    INTRINSIC_XED_IFORM_JNBE_RELBRd,
    INTRINSIC_XED_IFORM_JNBE_RELBRz,
    INTRINSIC_XED_IFORM_JNL_RELBRb,
    INTRINSIC_XED_IFORM_JNL_RELBRd,
    INTRINSIC_XED_IFORM_JNL_RELBRz,
    INTRINSIC_XED_IFORM_JNLE_RELBRb,
    INTRINSIC_XED_IFORM_JNLE_RELBRd,
    INTRINSIC_XED_IFORM_JNLE_RELBRz,
    INTRINSIC_XED_IFORM_JNO_RELBRb,
    INTRINSIC_XED_IFORM_JNO_RELBRd,
    INTRINSIC_XED_IFORM_JNO_RELBRz,
    INTRINSIC_XED_IFORM_JNP_RELBRb,
    INTRINSIC_XED_IFORM_JNP_RELBRd,
    INTRINSIC_XED_IFORM_JNP_RELBRz,
    INTRINSIC_XED_IFORM_JNS_RELBRb,
    INTRINSIC_XED_IFORM_JNS_RELBRd,
    INTRINSIC_XED_IFORM_JNS_RELBRz,
    INTRINSIC_XED_IFORM_JNZ_RELBRb,
    INTRINSIC_XED_IFORM_JNZ_RELBRd,
    INTRINSIC_XED_IFORM_JNZ_RELBRz,
    INTRINSIC_XED_IFORM_JO_RELBRb,
    INTRINSIC_XED_IFORM_JO_RELBRd,
    INTRINSIC_XED_IFORM_JO_RELBRz,
    INTRINSIC_XED_IFORM_JP_RELBRb,
    INTRINSIC_XED_IFORM_JP_RELBRd,
    INTRINSIC_XED_IFORM_JP_RELBRz,
    INTRINSIC_XED_IFORM_JRCXZ_RELBRb,
    INTRINSIC_XED_IFORM_JS_RELBRb,
    INTRINSIC_XED_IFORM_JS_RELBRd,
    INTRINSIC_XED_IFORM_JS_RELBRz,
    INTRINSIC_XED_IFORM_JZ_RELBRb,
    INTRINSIC_XED_IFORM_JZ_RELBRd,
    INTRINSIC_XED_IFORM_JZ_RELBRz,
    INTRINSIC_XED_IFORM_KADDB_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KADDD_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KADDQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KADDW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDB_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDD_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDNB_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDND_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDNQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDNW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KANDW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVB_GPR32u32_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVB_GPR32u32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVB_MASKmskw_GPR32u32_APX,
    INTRINSIC_XED_IFORM_KMOVB_MASKmskw_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_KMOVB_MASKmskw_MASKu8_APX,
    INTRINSIC_XED_IFORM_KMOVB_MASKmskw_MASKu8_AVX512,
    INTRINSIC_XED_IFORM_KMOVB_MASKmskw_MEMu8_APX,
    INTRINSIC_XED_IFORM_KMOVB_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_KMOVB_MEMu8_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVB_MEMu8_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVD_GPR32u32_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVD_GPR32u32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVD_MASKmskw_GPR32u32_APX,
    INTRINSIC_XED_IFORM_KMOVD_MASKmskw_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_KMOVD_MASKmskw_MASKu32_APX,
    INTRINSIC_XED_IFORM_KMOVD_MASKmskw_MASKu32_AVX512,
    INTRINSIC_XED_IFORM_KMOVD_MASKmskw_MEMu32_APX,
    INTRINSIC_XED_IFORM_KMOVD_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_KMOVD_MEMu32_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVD_MEMu32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVQ_GPR64u64_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVQ_GPR64u64_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVQ_MASKmskw_GPR64u64_APX,
    INTRINSIC_XED_IFORM_KMOVQ_MASKmskw_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_KMOVQ_MASKmskw_MASKu64_APX,
    INTRINSIC_XED_IFORM_KMOVQ_MASKmskw_MASKu64_AVX512,
    INTRINSIC_XED_IFORM_KMOVQ_MASKmskw_MEMu64_APX,
    INTRINSIC_XED_IFORM_KMOVQ_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_KMOVQ_MEMu64_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVQ_MEMu64_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVW_GPR32u32_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVW_GPR32u32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KMOVW_MASKmskw_GPR32u32_APX,
    INTRINSIC_XED_IFORM_KMOVW_MASKmskw_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_KMOVW_MASKmskw_MASKu16_APX,
    INTRINSIC_XED_IFORM_KMOVW_MASKmskw_MASKu16_AVX512,
    INTRINSIC_XED_IFORM_KMOVW_MASKmskw_MEMu16_APX,
    INTRINSIC_XED_IFORM_KMOVW_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_KMOVW_MEMu16_MASKmskw_APX,
    INTRINSIC_XED_IFORM_KMOVW_MEMu16_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KNOTB_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KNOTD_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KNOTQ_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KNOTW_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORB_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORD_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORTESTB_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORTESTD_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORTESTQ_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORTESTW_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KORW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTLB_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTLD_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTLQ_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTLW_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTRB_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTRD_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTRQ_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KSHIFTRW_MASKmskw_MASKmskw_IMM8_AVX512,
    INTRINSIC_XED_IFORM_KTESTB_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KTESTD_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KTESTQ_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KTESTW_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KUNPCKBW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KUNPCKDQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KUNPCKWD_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXNORB_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXNORD_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXNORQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXNORW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXORB_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXORD_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXORQ_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_KXORW_MASKmskw_MASKmskw_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_LAHF,
    INTRINSIC_XED_IFORM_LAR_GPRv_GPRv,
    INTRINSIC_XED_IFORM_LAR_GPRv_MEMw,
    INTRINSIC_XED_IFORM_LDDQU_XMMpd_MEMdq,
    INTRINSIC_XED_IFORM_LDMXCSR_MEMd,
    INTRINSIC_XED_IFORM_LDS_GPRz_MEMp,
    INTRINSIC_XED_IFORM_LDTILECFG_MEM,
    INTRINSIC_XED_IFORM_LDTILECFG_MEM_APX,
    INTRINSIC_XED_IFORM_LEA_GPRv_AGEN,
    INTRINSIC_XED_IFORM_LEAVE,
    INTRINSIC_XED_IFORM_LES_GPRz_MEMp,
    INTRINSIC_XED_IFORM_LFENCE,
    INTRINSIC_XED_IFORM_LFS_GPRv_MEMp2,
    INTRINSIC_XED_IFORM_LGDT_MEMs,
    INTRINSIC_XED_IFORM_LGDT_MEMs64,
    INTRINSIC_XED_IFORM_LGS_GPRv_MEMp2,
    INTRINSIC_XED_IFORM_LIDT_MEMs,
    INTRINSIC_XED_IFORM_LIDT_MEMs64,
    INTRINSIC_XED_IFORM_LKGS_GPR16u16,
    INTRINSIC_XED_IFORM_LKGS_MEMu16,
    INTRINSIC_XED_IFORM_LLDT_GPR16,
    INTRINSIC_XED_IFORM_LLDT_MEMw,
    INTRINSIC_XED_IFORM_LLWPCB_GPRyy,
    INTRINSIC_XED_IFORM_LMSW_GPR16,
    INTRINSIC_XED_IFORM_LMSW_MEMw,
    INTRINSIC_XED_IFORM_LOADIWKEY_XMMu8_XMMu8,
    INTRINSIC_XED_IFORM_LODSB,
    INTRINSIC_XED_IFORM_LODSD,
    INTRINSIC_XED_IFORM_LODSQ,
    INTRINSIC_XED_IFORM_LODSW,
    INTRINSIC_XED_IFORM_LOOP_RELBRb,
    INTRINSIC_XED_IFORM_LOOPE_RELBRb,
    INTRINSIC_XED_IFORM_LOOPNE_RELBRb,
    INTRINSIC_XED_IFORM_LSL_GPRv_GPRz,
    INTRINSIC_XED_IFORM_LSL_GPRv_MEMw,
    INTRINSIC_XED_IFORM_LSS_GPRv_MEMp2,
    INTRINSIC_XED_IFORM_LTR_GPR16,
    INTRINSIC_XED_IFORM_LTR_MEMw,
    INTRINSIC_XED_IFORM_LWPINS_GPRyy_GPR32d_IMMd,
    INTRINSIC_XED_IFORM_LWPINS_GPRyy_MEMd_IMMd,
    INTRINSIC_XED_IFORM_LWPVAL_GPRyy_GPR32d_IMMd,
    INTRINSIC_XED_IFORM_LWPVAL_GPRyy_MEMd_IMMd,
    INTRINSIC_XED_IFORM_LZCNT_GPRv_GPRv,
    INTRINSIC_XED_IFORM_LZCNT_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_LZCNT_GPRv_MEMv,
    INTRINSIC_XED_IFORM_LZCNT_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_MASKMOVDQU_XMMxub_XMMxub,
    INTRINSIC_XED_IFORM_MASKMOVQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_MAXPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_MAXPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_MAXPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MAXPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_MAXSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_MAXSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_MAXSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_MAXSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_MCOMMIT,
    INTRINSIC_XED_IFORM_MFENCE,
    INTRINSIC_XED_IFORM_MINPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_MINPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_MINPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MINPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_MINSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_MINSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_MINSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_MINSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_MONITOR,
    INTRINSIC_XED_IFORM_MONITORX,
    INTRINSIC_XED_IFORM_MOV_AL_MEMb,
    INTRINSIC_XED_IFORM_MOV_GPR8_GPR8_88,
    INTRINSIC_XED_IFORM_MOV_GPR8_GPR8_8A,
    INTRINSIC_XED_IFORM_MOV_GPR8_IMMb_B0,
    INTRINSIC_XED_IFORM_MOV_GPR8_IMMb_C6r0,
    INTRINSIC_XED_IFORM_MOV_GPR8_MEMb,
    INTRINSIC_XED_IFORM_MOV_GPRv_GPRv_89,
    INTRINSIC_XED_IFORM_MOV_GPRv_GPRv_8B,
    INTRINSIC_XED_IFORM_MOV_GPRv_IMMv,
    INTRINSIC_XED_IFORM_MOV_GPRv_IMMz,
    INTRINSIC_XED_IFORM_MOV_GPRv_MEMv,
    INTRINSIC_XED_IFORM_MOV_GPRv_SEG,
    INTRINSIC_XED_IFORM_MOV_MEMb_AL,
    INTRINSIC_XED_IFORM_MOV_MEMb_GPR8,
    INTRINSIC_XED_IFORM_MOV_MEMb_IMMb,
    INTRINSIC_XED_IFORM_MOV_MEMv_GPRv,
    INTRINSIC_XED_IFORM_MOV_MEMv_IMMz,
    INTRINSIC_XED_IFORM_MOV_MEMv_OrAX,
    INTRINSIC_XED_IFORM_MOV_MEMw_SEG,
    INTRINSIC_XED_IFORM_MOV_OrAX_MEMv,
    INTRINSIC_XED_IFORM_MOV_SEG_GPR16,
    INTRINSIC_XED_IFORM_MOV_SEG_MEMw,
    INTRINSIC_XED_IFORM_MOVAPD_MEMpd_XMMpd,
    INTRINSIC_XED_IFORM_MOVAPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_MOVAPD_XMMpd_XMMpd_0F28,
    INTRINSIC_XED_IFORM_MOVAPD_XMMpd_XMMpd_0F29,
    INTRINSIC_XED_IFORM_MOVAPS_MEMps_XMMps,
    INTRINSIC_XED_IFORM_MOVAPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MOVAPS_XMMps_XMMps_0F28,
    INTRINSIC_XED_IFORM_MOVAPS_XMMps_XMMps_0F29,
    INTRINSIC_XED_IFORM_MOVBE_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_MOVBE_GPRv_MEMv,
    INTRINSIC_XED_IFORM_MOVBE_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_MOVBE_MEMv_GPRv,
    INTRINSIC_XED_IFORM_MOVBE_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_MOVD_GPR32_MMXd,
    INTRINSIC_XED_IFORM_MOVD_GPR32_XMMd,
    INTRINSIC_XED_IFORM_MOVD_MEMd_MMXd,
    INTRINSIC_XED_IFORM_MOVD_MEMd_XMMd,
    INTRINSIC_XED_IFORM_MOVD_MMXq_GPR32,
    INTRINSIC_XED_IFORM_MOVD_MMXq_MEMd,
    INTRINSIC_XED_IFORM_MOVD_XMMdq_GPR32,
    INTRINSIC_XED_IFORM_MOVD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_MOVDDUP_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_MOVDDUP_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_MOVDIR64B_GPRa_MEM,
    INTRINSIC_XED_IFORM_MOVDIR64B_GPRav_MEMu32_APX,
    INTRINSIC_XED_IFORM_MOVDIRI_MEMu32_GPR32u32,
    INTRINSIC_XED_IFORM_MOVDIRI_MEMu64_GPR64u64,
    INTRINSIC_XED_IFORM_MOVDIRI_MEMyu_GPRyu_APX,
    INTRINSIC_XED_IFORM_MOVDQ2Q_MMXq_XMMq,
    INTRINSIC_XED_IFORM_MOVDQA_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_MOVDQA_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_MOVDQA_XMMdq_XMMdq_0F6F,
    INTRINSIC_XED_IFORM_MOVDQA_XMMdq_XMMdq_0F7F,
    INTRINSIC_XED_IFORM_MOVDQU_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_MOVDQU_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_MOVDQU_XMMdq_XMMdq_0F6F,
    INTRINSIC_XED_IFORM_MOVDQU_XMMdq_XMMdq_0F7F,
    INTRINSIC_XED_IFORM_MOVHLPS_XMMq_XMMq,
    INTRINSIC_XED_IFORM_MOVHPD_MEMq_XMMsd,
    INTRINSIC_XED_IFORM_MOVHPD_XMMsd_MEMq,
    INTRINSIC_XED_IFORM_MOVHPS_MEMq_XMMps,
    INTRINSIC_XED_IFORM_MOVHPS_XMMq_MEMq,
    INTRINSIC_XED_IFORM_MOVLHPS_XMMq_XMMq,
    INTRINSIC_XED_IFORM_MOVLPD_MEMq_XMMsd,
    INTRINSIC_XED_IFORM_MOVLPD_XMMsd_MEMq,
    INTRINSIC_XED_IFORM_MOVLPS_MEMq_XMMq,
    INTRINSIC_XED_IFORM_MOVLPS_XMMq_MEMq,
    INTRINSIC_XED_IFORM_MOVMSKPD_GPR32_XMMpd,
    INTRINSIC_XED_IFORM_MOVMSKPS_GPR32_XMMps,
    INTRINSIC_XED_IFORM_MOVNTDQ_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_MOVNTDQA_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_MOVNTI_MEMd_GPR32,
    INTRINSIC_XED_IFORM_MOVNTI_MEMq_GPR64,
    INTRINSIC_XED_IFORM_MOVNTPD_MEMdq_XMMpd,
    INTRINSIC_XED_IFORM_MOVNTPS_MEMdq_XMMps,
    INTRINSIC_XED_IFORM_MOVNTQ_MEMq_MMXq,
    INTRINSIC_XED_IFORM_MOVNTSD_MEMq_XMMq,
    INTRINSIC_XED_IFORM_MOVNTSS_MEMd_XMMd,
    INTRINSIC_XED_IFORM_MOVQ_GPR64_MMXq,
    INTRINSIC_XED_IFORM_MOVQ_GPR64_XMMq,
    INTRINSIC_XED_IFORM_MOVQ_MEMq_MMXq_0F7E,
    INTRINSIC_XED_IFORM_MOVQ_MEMq_MMXq_0F7F,
    INTRINSIC_XED_IFORM_MOVQ_MEMq_XMMq_0F7E,
    INTRINSIC_XED_IFORM_MOVQ_MEMq_XMMq_0FD6,
    INTRINSIC_XED_IFORM_MOVQ_MMXq_GPR64,
    INTRINSIC_XED_IFORM_MOVQ_MMXq_MEMq_0F6E,
    INTRINSIC_XED_IFORM_MOVQ_MMXq_MEMq_0F6F,
    INTRINSIC_XED_IFORM_MOVQ_MMXq_MMXq_0F6F,
    INTRINSIC_XED_IFORM_MOVQ_MMXq_MMXq_0F7F,
    INTRINSIC_XED_IFORM_MOVQ_XMMdq_GPR64,
    INTRINSIC_XED_IFORM_MOVQ_XMMdq_MEMq_0F6E,
    INTRINSIC_XED_IFORM_MOVQ_XMMdq_MEMq_0F7E,
    INTRINSIC_XED_IFORM_MOVQ_XMMdq_XMMq_0F7E,
    INTRINSIC_XED_IFORM_MOVQ_XMMdq_XMMq_0FD6,
    INTRINSIC_XED_IFORM_MOVQ2DQ_XMMdq_MMXq,
    INTRINSIC_XED_IFORM_MOVSB,
    INTRINSIC_XED_IFORM_MOVSD,
    INTRINSIC_XED_IFORM_MOVSD_XMM_MEMsd_XMMsd,
    INTRINSIC_XED_IFORM_MOVSD_XMM_XMMdq_MEMsd,
    INTRINSIC_XED_IFORM_MOVSD_XMM_XMMsd_XMMsd_0F10,
    INTRINSIC_XED_IFORM_MOVSD_XMM_XMMsd_XMMsd_0F11,
    INTRINSIC_XED_IFORM_MOVSHDUP_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MOVSHDUP_XMMps_XMMps,
    INTRINSIC_XED_IFORM_MOVSLDUP_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MOVSLDUP_XMMps_XMMps,
    INTRINSIC_XED_IFORM_MOVSQ,
    INTRINSIC_XED_IFORM_MOVSS_MEMss_XMMss,
    INTRINSIC_XED_IFORM_MOVSS_XMMdq_MEMss,
    INTRINSIC_XED_IFORM_MOVSS_XMMss_XMMss_0F10,
    INTRINSIC_XED_IFORM_MOVSS_XMMss_XMMss_0F11,
    INTRINSIC_XED_IFORM_MOVSW,
    INTRINSIC_XED_IFORM_MOVSX_GPRv_GPR16,
    INTRINSIC_XED_IFORM_MOVSX_GPRv_GPR8,
    INTRINSIC_XED_IFORM_MOVSX_GPRv_MEMb,
    INTRINSIC_XED_IFORM_MOVSX_GPRv_MEMw,
    INTRINSIC_XED_IFORM_MOVSXD_GPRv_GPRz,
    INTRINSIC_XED_IFORM_MOVSXD_GPRv_MEMz,
    INTRINSIC_XED_IFORM_MOVUPD_MEMpd_XMMpd,
    INTRINSIC_XED_IFORM_MOVUPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_MOVUPD_XMMpd_XMMpd_0F10,
    INTRINSIC_XED_IFORM_MOVUPD_XMMpd_XMMpd_0F11,
    INTRINSIC_XED_IFORM_MOVUPS_MEMps_XMMps,
    INTRINSIC_XED_IFORM_MOVUPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MOVUPS_XMMps_XMMps_0F10,
    INTRINSIC_XED_IFORM_MOVUPS_XMMps_XMMps_0F11,
    INTRINSIC_XED_IFORM_MOVZX_GPRv_GPR16,
    INTRINSIC_XED_IFORM_MOVZX_GPRv_GPR8,
    INTRINSIC_XED_IFORM_MOVZX_GPRv_MEMb,
    INTRINSIC_XED_IFORM_MOVZX_GPRv_MEMw,
    INTRINSIC_XED_IFORM_MOV_CR_CR_GPR32,
    INTRINSIC_XED_IFORM_MOV_CR_CR_GPR64,
    INTRINSIC_XED_IFORM_MOV_CR_GPR32_CR,
    INTRINSIC_XED_IFORM_MOV_CR_GPR64_CR,
    INTRINSIC_XED_IFORM_MOV_DR_DR_GPR32,
    INTRINSIC_XED_IFORM_MOV_DR_DR_GPR64,
    INTRINSIC_XED_IFORM_MOV_DR_GPR32_DR,
    INTRINSIC_XED_IFORM_MOV_DR_GPR64_DR,
    INTRINSIC_XED_IFORM_MPSADBW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_MPSADBW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_MUL_GPR8,
    INTRINSIC_XED_IFORM_MUL_GPR8i8_APX,
    INTRINSIC_XED_IFORM_MUL_GPRv,
    INTRINSIC_XED_IFORM_MUL_GPRv_APX,
    INTRINSIC_XED_IFORM_MUL_MEMb,
    INTRINSIC_XED_IFORM_MUL_MEMi8_APX,
    INTRINSIC_XED_IFORM_MUL_MEMv,
    INTRINSIC_XED_IFORM_MUL_MEMv_APX,
    INTRINSIC_XED_IFORM_MULPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_MULPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_MULPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_MULPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_MULSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_MULSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_MULSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_MULSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_MULX_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_MULX_GPR32d_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_MULX_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_MULX_GPR32i32_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_MULX_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_MULX_GPR64i64_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_MULX_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_MULX_GPR64q_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_MWAIT,
    INTRINSIC_XED_IFORM_MWAITX,
    INTRINSIC_XED_IFORM_NEG_GPR8,
    INTRINSIC_XED_IFORM_NEG_GPR8i8_APX,
    INTRINSIC_XED_IFORM_NEG_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_NEG_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_NEG_GPRv,
    INTRINSIC_XED_IFORM_NEG_GPRv_APX,
    INTRINSIC_XED_IFORM_NEG_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_NEG_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_NEG_MEMb,
    INTRINSIC_XED_IFORM_NEG_MEMi8_APX,
    INTRINSIC_XED_IFORM_NEG_MEMv,
    INTRINSIC_XED_IFORM_NEG_MEMv_APX,
    INTRINSIC_XED_IFORM_NEG_LOCK_MEMb,
    INTRINSIC_XED_IFORM_NEG_LOCK_MEMv,
    INTRINSIC_XED_IFORM_NOP_90,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r0,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r1,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r2,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r3,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r4,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r5,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r6,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F18r7,
    INTRINSIC_XED_IFORM_NOP_GPRv_0F1F,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F0D,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F19,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F1A,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F1B,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F1C,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F1D,
    INTRINSIC_XED_IFORM_NOP_GPRv_GPRv_0F1E,
    INTRINSIC_XED_IFORM_NOP_GPRv_MEM_0F1B,
    INTRINSIC_XED_IFORM_NOP_GPRv_MEMv_0F1A,
    INTRINSIC_XED_IFORM_NOP_MEMv_0F18r4,
    INTRINSIC_XED_IFORM_NOP_MEMv_0F18r5,
    INTRINSIC_XED_IFORM_NOP_MEMv_0F1F,
    INTRINSIC_XED_IFORM_NOP_MEMv_GPRv_0F19,
    INTRINSIC_XED_IFORM_NOP_MEMv_GPRv_0F1C,
    INTRINSIC_XED_IFORM_NOP_MEMv_GPRv_0F1D,
    INTRINSIC_XED_IFORM_NOP_MEMv_GPRv_0F1E,
    INTRINSIC_XED_IFORM_NOT_GPR8,
    INTRINSIC_XED_IFORM_NOT_GPR8i8_APX,
    INTRINSIC_XED_IFORM_NOT_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_NOT_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_NOT_GPRv,
    INTRINSIC_XED_IFORM_NOT_GPRv_APX,
    INTRINSIC_XED_IFORM_NOT_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_NOT_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_NOT_MEMb,
    INTRINSIC_XED_IFORM_NOT_MEMi8_APX,
    INTRINSIC_XED_IFORM_NOT_MEMv,
    INTRINSIC_XED_IFORM_NOT_MEMv_APX,
    INTRINSIC_XED_IFORM_NOT_LOCK_MEMb,
    INTRINSIC_XED_IFORM_NOT_LOCK_MEMv,
    INTRINSIC_XED_IFORM_OR_AL_IMMb,
    INTRINSIC_XED_IFORM_OR_GPR8_GPR8_08,
    INTRINSIC_XED_IFORM_OR_GPR8_GPR8_0A,
    INTRINSIC_XED_IFORM_OR_GPR8_IMMb_80r1,
    INTRINSIC_XED_IFORM_OR_GPR8_IMMb_82r1,
    INTRINSIC_XED_IFORM_OR_GPR8_MEMb,
    INTRINSIC_XED_IFORM_OR_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_OR_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_09,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_0B,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_OR_GPRv_IMMz,
    INTRINSIC_XED_IFORM_OR_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_MEMv,
    INTRINSIC_XED_IFORM_OR_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_OR_MEMb_GPR8,
    INTRINSIC_XED_IFORM_OR_MEMb_IMMb_80r1,
    INTRINSIC_XED_IFORM_OR_MEMb_IMMb_82r1,
    INTRINSIC_XED_IFORM_OR_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_OR_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_MEMv_GPRv,
    INTRINSIC_XED_IFORM_OR_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_OR_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_OR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_OR_MEMv_IMMz,
    INTRINSIC_XED_IFORM_OR_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_OR_OrAX_IMMz,
    INTRINSIC_XED_IFORM_ORPD_XMMxuq_MEMxuq,
    INTRINSIC_XED_IFORM_ORPD_XMMxuq_XMMxuq,
    INTRINSIC_XED_IFORM_ORPS_XMMxud_MEMxud,
    INTRINSIC_XED_IFORM_ORPS_XMMxud_XMMxud,
    INTRINSIC_XED_IFORM_OR_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_OR_LOCK_MEMb_IMMb_80r1,
    INTRINSIC_XED_IFORM_OR_LOCK_MEMb_IMMb_82r1,
    INTRINSIC_XED_IFORM_OR_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_OR_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_OR_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_OUT_DX_AL,
    INTRINSIC_XED_IFORM_OUT_DX_OeAX,
    INTRINSIC_XED_IFORM_OUT_IMMb_AL,
    INTRINSIC_XED_IFORM_OUT_IMMb_OeAX,
    INTRINSIC_XED_IFORM_OUTSB,
    INTRINSIC_XED_IFORM_OUTSD,
    INTRINSIC_XED_IFORM_OUTSW,
    INTRINSIC_XED_IFORM_PABSB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PABSB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PABSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PABSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PABSD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PABSD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PABSD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PABSD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PABSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PABSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PABSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PABSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PACKSSDW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PACKSSDW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PACKSSDW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PACKSSDW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PACKSSWB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PACKSSWB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PACKSSWB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PACKSSWB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PACKUSDW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PACKUSDW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PACKUSWB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PACKUSWB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PACKUSWB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PACKUSWB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDSB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDSB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDUSB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDUSB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDUSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDUSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDUSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDUSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDUSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDUSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PADDW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PADDW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PADDW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PADDW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PALIGNR_MMXq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_PALIGNR_MMXq_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PALIGNR_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PALIGNR_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PAND_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PAND_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PAND_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PAND_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PANDN_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PANDN_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PANDN_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PANDN_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PAUSE,
    INTRINSIC_XED_IFORM_PAVGB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PAVGB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PAVGB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PAVGB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PAVGUSB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PAVGUSB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PAVGW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PAVGW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PAVGW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PAVGW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PBLENDVB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PBLENDVB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PBLENDW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PBLENDW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PBNDKB,
    INTRINSIC_XED_IFORM_PCLMULQDQ_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCLMULQDQ_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPEQB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PCMPEQB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PCMPEQB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPEQB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPEQD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PCMPEQD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PCMPEQD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPEQD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPEQQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPEQQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPEQW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PCMPEQW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PCMPEQW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPEQW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPESTRI_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRI_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRI64_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRI64_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRM_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRM_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRM64_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPESTRM64_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPGTB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PCMPGTB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PCMPGTB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPGTB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPGTD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PCMPGTD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PCMPGTD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPGTD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPGTQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPGTQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPGTW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PCMPGTW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PCMPGTW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PCMPGTW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PCMPISTRI_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPISTRI_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPISTRI64_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPISTRI64_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPISTRM_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PCMPISTRM_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PCONFIG,
    INTRINSIC_XED_IFORM_PCONFIG64,
    INTRINSIC_XED_IFORM_PDEP_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_PDEP_GPR32d_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_PDEP_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_PDEP_GPR32i32_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_PDEP_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_PDEP_GPR64i64_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_PDEP_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_PDEP_GPR64q_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_PEXT_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_PEXT_GPR32d_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_PEXT_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_PEXT_GPR32i32_GPR32i32_MEMi32_APX,
    INTRINSIC_XED_IFORM_PEXT_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_PEXT_GPR64i64_GPR64i64_MEMi64_APX,
    INTRINSIC_XED_IFORM_PEXT_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_PEXT_GPR64q_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_PEXTRB_GPR32d_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRB_MEMb_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRD_GPR32d_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRD_MEMd_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRQ_GPR64q_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRQ_MEMq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRW_GPR32_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRW_GPR32_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRW_SSE4_GPR32_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PEXTRW_SSE4_MEMw_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PF2ID_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PF2ID_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PF2IW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PF2IW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFACC_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFACC_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFADD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFADD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFCMPEQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFCMPEQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFCMPGE_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFCMPGE_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFCMPGT_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFCMPGT_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFMAX_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFMAX_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFMIN_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFMIN_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFMUL_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFMUL_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFNACC_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFNACC_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFPNACC_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFPNACC_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFRCP_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFRCP_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFRCPIT1_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFRCPIT1_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFRCPIT2_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFRCPIT2_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFRSQIT1_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFRSQIT1_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFRSQRT_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFRSQRT_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFSUB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFSUB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PFSUBR_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PFSUBR_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHADDD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PHADDD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHADDD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHADDD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PHADDSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PHADDSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHADDSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHADDSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PHADDW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PHADDW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHADDW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHADDW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PHMINPOSUW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHMINPOSUW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PHSUBD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PHSUBD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHSUBD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHSUBD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PHSUBSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PHSUBSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHSUBSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHSUBSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PHSUBW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PHSUBW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PHSUBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PHSUBW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PI2FD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PI2FD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PI2FW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PI2FW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PINSRB_XMMdq_GPR32d_IMMb,
    INTRINSIC_XED_IFORM_PINSRB_XMMdq_MEMb_IMMb,
    INTRINSIC_XED_IFORM_PINSRD_XMMdq_GPR32d_IMMb,
    INTRINSIC_XED_IFORM_PINSRD_XMMdq_MEMd_IMMb,
    INTRINSIC_XED_IFORM_PINSRQ_XMMdq_GPR64q_IMMb,
    INTRINSIC_XED_IFORM_PINSRQ_XMMdq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_PINSRW_MMXq_GPR32_IMMb,
    INTRINSIC_XED_IFORM_PINSRW_MMXq_MEMw_IMMb,
    INTRINSIC_XED_IFORM_PINSRW_XMMdq_GPR32_IMMb,
    INTRINSIC_XED_IFORM_PINSRW_XMMdq_MEMw_IMMb,
    INTRINSIC_XED_IFORM_PMADDUBSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMADDUBSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMADDUBSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMADDUBSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMADDWD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMADDWD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMADDWD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMADDWD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMAXSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMAXSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMAXSD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMAXSD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMAXSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMAXSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMAXSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMAXSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMAXUB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMAXUB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMAXUB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMAXUB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMAXUD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMAXUD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMAXUW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMAXUW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMINSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMINSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMINSD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMINSD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMINSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMINSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMINSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMINSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMINUB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMINUB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMINUB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMINUB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMINUD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMINUD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMINUW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMINUW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMOVMSKB_GPR32_MMXq,
    INTRINSIC_XED_IFORM_PMOVMSKB_GPR32_XMMdq,
    INTRINSIC_XED_IFORM_PMOVSXBD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_PMOVSXBD_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_PMOVSXBQ_XMMdq_MEMw,
    INTRINSIC_XED_IFORM_PMOVSXBQ_XMMdq_XMMw,
    INTRINSIC_XED_IFORM_PMOVSXBW_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_PMOVSXBW_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PMOVSXDQ_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_PMOVSXDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PMOVSXWD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_PMOVSXWD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PMOVSXWQ_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_PMOVSXWQ_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_PMOVZXBD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_PMOVZXBD_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_PMOVZXBQ_XMMdq_MEMw,
    INTRINSIC_XED_IFORM_PMOVZXBQ_XMMdq_XMMw,
    INTRINSIC_XED_IFORM_PMOVZXBW_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_PMOVZXBW_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PMOVZXDQ_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_PMOVZXDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PMOVZXWD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_PMOVZXWD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PMOVZXWQ_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_PMOVZXWQ_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_PMULDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULDQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMULHRSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMULHRSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMULHRSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULHRSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMULHRW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMULHRW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMULHUW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMULHUW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMULHUW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULHUW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMULHW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMULHW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMULHW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULHW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMULLD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULLD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMULLW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMULLW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMULLW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULLW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PMULUDQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PMULUDQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PMULUDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PMULUDQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_POP_DS,
    INTRINSIC_XED_IFORM_POP_ES,
    INTRINSIC_XED_IFORM_POP_FS,
    INTRINSIC_XED_IFORM_POP_GPRv_58,
    INTRINSIC_XED_IFORM_POP_GPRv_8F,
    INTRINSIC_XED_IFORM_POP_GS,
    INTRINSIC_XED_IFORM_POP_MEMv,
    INTRINSIC_XED_IFORM_POP_SS,
    INTRINSIC_XED_IFORM_POP2_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_POP2P_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_POPA,
    INTRINSIC_XED_IFORM_POPAD,
    INTRINSIC_XED_IFORM_POPCNT_GPRv_GPRv,
    INTRINSIC_XED_IFORM_POPCNT_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_POPCNT_GPRv_MEMv,
    INTRINSIC_XED_IFORM_POPCNT_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_POPF,
    INTRINSIC_XED_IFORM_POPFD,
    INTRINSIC_XED_IFORM_POPFQ,
    INTRINSIC_XED_IFORM_POPP_GPR64,
    INTRINSIC_XED_IFORM_POR_MMXq_MEMq,
    INTRINSIC_XED_IFORM_POR_MMXq_MMXq,
    INTRINSIC_XED_IFORM_POR_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_POR_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PREFETCHIT0_MEMu8,
    INTRINSIC_XED_IFORM_PREFETCHIT1_MEMu8,
    INTRINSIC_XED_IFORM_PREFETCHNTA_MEMmprefetch,
    INTRINSIC_XED_IFORM_PREFETCHT0_MEMmprefetch,
    INTRINSIC_XED_IFORM_PREFETCHT1_MEMmprefetch,
    INTRINSIC_XED_IFORM_PREFETCHT2_MEMmprefetch,
    INTRINSIC_XED_IFORM_PREFETCHW_0F0Dr1,
    INTRINSIC_XED_IFORM_PREFETCHW_0F0Dr3,
    INTRINSIC_XED_IFORM_PREFETCHWT1_MEMu8,
    INTRINSIC_XED_IFORM_PREFETCH_EXCLUSIVE_MEMmprefetch,
    INTRINSIC_XED_IFORM_PREFETCH_RESERVED_0F0Dr4,
    INTRINSIC_XED_IFORM_PREFETCH_RESERVED_0F0Dr5,
    INTRINSIC_XED_IFORM_PREFETCH_RESERVED_0F0Dr6,
    INTRINSIC_XED_IFORM_PREFETCH_RESERVED_0F0Dr7,
    INTRINSIC_XED_IFORM_PSADBW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSADBW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSADBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSADBW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSHUFB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSHUFB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSHUFB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSHUFB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSHUFD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFHW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFHW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFLW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFLW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFW_MMXq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_PSHUFW_MMXq_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSIGNB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSIGNB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSIGNB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSIGNB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSIGND_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSIGND_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSIGND_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSIGND_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSIGNW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSIGNW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSIGNW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSIGNW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSLLD_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSLLD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSLLD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSLLD_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSLLD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSLLD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSLLDQ_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSLLQ_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSLLQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSLLQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSLLQ_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSLLQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSLLQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSLLW_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSLLW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSLLW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSLLW_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSLLW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSLLW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSMASH_RAX,
    INTRINSIC_XED_IFORM_PSRAD_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSRAD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSRAD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSRAD_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSRAD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSRAD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSRAW_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSRAW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSRAW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSRAW_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSRAW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSRAW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSRLD_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSRLD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSRLD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSRLD_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSRLD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSRLD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSRLDQ_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSRLQ_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSRLQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSRLQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSRLQ_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSRLQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSRLQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSRLW_MMXq_IMMb,
    INTRINSIC_XED_IFORM_PSRLW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSRLW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSRLW_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_PSRLW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSRLW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBQ_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBSB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBSB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBUSB_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBUSB_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBUSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBUSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBUSW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBUSW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBUSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBUSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSUBW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSUBW_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PSUBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PSUBW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PSWAPD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PSWAPD_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PTEST_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PTEST_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_PTWRITE_GPRy,
    INTRINSIC_XED_IFORM_PTWRITE_MEMy,
    INTRINSIC_XED_IFORM_PUNPCKHBW_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PUNPCKHBW_MMXq_MMXd,
    INTRINSIC_XED_IFORM_PUNPCKHBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKHBW_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKHDQ_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PUNPCKHDQ_MMXq_MMXd,
    INTRINSIC_XED_IFORM_PUNPCKHDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKHDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKHQDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKHQDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKHWD_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PUNPCKHWD_MMXq_MMXd,
    INTRINSIC_XED_IFORM_PUNPCKHWD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKHWD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKLBW_MMXq_MEMd,
    INTRINSIC_XED_IFORM_PUNPCKLBW_MMXq_MMXd,
    INTRINSIC_XED_IFORM_PUNPCKLBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKLBW_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKLDQ_MMXq_MEMd,
    INTRINSIC_XED_IFORM_PUNPCKLDQ_MMXq_MMXd,
    INTRINSIC_XED_IFORM_PUNPCKLDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKLDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKLQDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKLQDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUNPCKLWD_MMXq_MEMd,
    INTRINSIC_XED_IFORM_PUNPCKLWD_MMXq_MMXd,
    INTRINSIC_XED_IFORM_PUNPCKLWD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PUNPCKLWD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_PUSH_CS,
    INTRINSIC_XED_IFORM_PUSH_DS,
    INTRINSIC_XED_IFORM_PUSH_ES,
    INTRINSIC_XED_IFORM_PUSH_FS,
    INTRINSIC_XED_IFORM_PUSH_GPRv_50,
    INTRINSIC_XED_IFORM_PUSH_GPRv_FFr6,
    INTRINSIC_XED_IFORM_PUSH_GS,
    INTRINSIC_XED_IFORM_PUSH_IMMb,
    INTRINSIC_XED_IFORM_PUSH_IMMz,
    INTRINSIC_XED_IFORM_PUSH_MEMv,
    INTRINSIC_XED_IFORM_PUSH_SS,
    INTRINSIC_XED_IFORM_PUSH2_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_PUSH2P_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_PUSHA,
    INTRINSIC_XED_IFORM_PUSHAD,
    INTRINSIC_XED_IFORM_PUSHF,
    INTRINSIC_XED_IFORM_PUSHFD,
    INTRINSIC_XED_IFORM_PUSHFQ,
    INTRINSIC_XED_IFORM_PUSHP_GPR64,
    INTRINSIC_XED_IFORM_PVALIDATE_RAX_ECX_EDX,
    INTRINSIC_XED_IFORM_PXOR_MMXq_MEMq,
    INTRINSIC_XED_IFORM_PXOR_MMXq_MMXq,
    INTRINSIC_XED_IFORM_PXOR_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_PXOR_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_RCL_GPR8_CL,
    INTRINSIC_XED_IFORM_RCL_GPR8_IMMb,
    INTRINSIC_XED_IFORM_RCL_GPR8_ONE,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_CL,
    INTRINSIC_XED_IFORM_RCL_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_IMMb,
    INTRINSIC_XED_IFORM_RCL_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_GPRv_ONE,
    INTRINSIC_XED_IFORM_RCL_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_MEMb_CL,
    INTRINSIC_XED_IFORM_RCL_MEMb_IMMb,
    INTRINSIC_XED_IFORM_RCL_MEMb_ONE,
    INTRINSIC_XED_IFORM_RCL_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_RCL_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_RCL_MEMv_CL,
    INTRINSIC_XED_IFORM_RCL_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_RCL_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCL_MEMv_IMMb,
    INTRINSIC_XED_IFORM_RCL_MEMv_ONE,
    INTRINSIC_XED_IFORM_RCL_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_RCPPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_RCPPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_RCPSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_RCPSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_RCR_GPR8_CL,
    INTRINSIC_XED_IFORM_RCR_GPR8_IMMb,
    INTRINSIC_XED_IFORM_RCR_GPR8_ONE,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_CL,
    INTRINSIC_XED_IFORM_RCR_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_RCR_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_GPRv_ONE,
    INTRINSIC_XED_IFORM_RCR_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_MEMb_CL,
    INTRINSIC_XED_IFORM_RCR_MEMb_IMMb,
    INTRINSIC_XED_IFORM_RCR_MEMb_ONE,
    INTRINSIC_XED_IFORM_RCR_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_RCR_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_RCR_MEMv_CL,
    INTRINSIC_XED_IFORM_RCR_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_RCR_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_RCR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_RCR_MEMv_ONE,
    INTRINSIC_XED_IFORM_RCR_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_RDFSBASE_GPRy,
    INTRINSIC_XED_IFORM_RDGSBASE_GPRy,
    INTRINSIC_XED_IFORM_RDMSR,
    INTRINSIC_XED_IFORM_RDMSRLIST,
    INTRINSIC_XED_IFORM_RDPID_GPR32u32,
    INTRINSIC_XED_IFORM_RDPID_GPR64u64,
    INTRINSIC_XED_IFORM_RDPKRU,
    INTRINSIC_XED_IFORM_RDPMC,
    INTRINSIC_XED_IFORM_RDPRU,
    INTRINSIC_XED_IFORM_RDRAND_GPRv,
    INTRINSIC_XED_IFORM_RDSEED_GPRv,
    INTRINSIC_XED_IFORM_RDSSPD_GPR32u32,
    INTRINSIC_XED_IFORM_RDSSPQ_GPR64u64,
    INTRINSIC_XED_IFORM_RDTSC,
    INTRINSIC_XED_IFORM_RDTSCP,
    INTRINSIC_XED_IFORM_REPE_CMPSB,
    INTRINSIC_XED_IFORM_REPE_CMPSD,
    INTRINSIC_XED_IFORM_REPE_CMPSQ,
    INTRINSIC_XED_IFORM_REPE_CMPSW,
    INTRINSIC_XED_IFORM_REPE_SCASB,
    INTRINSIC_XED_IFORM_REPE_SCASD,
    INTRINSIC_XED_IFORM_REPE_SCASQ,
    INTRINSIC_XED_IFORM_REPE_SCASW,
    INTRINSIC_XED_IFORM_REPNE_CMPSB,
    INTRINSIC_XED_IFORM_REPNE_CMPSD,
    INTRINSIC_XED_IFORM_REPNE_CMPSQ,
    INTRINSIC_XED_IFORM_REPNE_CMPSW,
    INTRINSIC_XED_IFORM_REPNE_SCASB,
    INTRINSIC_XED_IFORM_REPNE_SCASD,
    INTRINSIC_XED_IFORM_REPNE_SCASQ,
    INTRINSIC_XED_IFORM_REPNE_SCASW,
    INTRINSIC_XED_IFORM_REP_INSB,
    INTRINSIC_XED_IFORM_REP_INSD,
    INTRINSIC_XED_IFORM_REP_INSW,
    INTRINSIC_XED_IFORM_REP_LODSB,
    INTRINSIC_XED_IFORM_REP_LODSD,
    INTRINSIC_XED_IFORM_REP_LODSQ,
    INTRINSIC_XED_IFORM_REP_LODSW,
    INTRINSIC_XED_IFORM_REP_MONTMUL,
    INTRINSIC_XED_IFORM_REP_MOVSB,
    INTRINSIC_XED_IFORM_REP_MOVSD,
    INTRINSIC_XED_IFORM_REP_MOVSQ,
    INTRINSIC_XED_IFORM_REP_MOVSW,
    INTRINSIC_XED_IFORM_REP_OUTSB,
    INTRINSIC_XED_IFORM_REP_OUTSD,
    INTRINSIC_XED_IFORM_REP_OUTSW,
    INTRINSIC_XED_IFORM_REP_STOSB,
    INTRINSIC_XED_IFORM_REP_STOSD,
    INTRINSIC_XED_IFORM_REP_STOSQ,
    INTRINSIC_XED_IFORM_REP_STOSW,
    INTRINSIC_XED_IFORM_REP_XCRYPTCBC,
    INTRINSIC_XED_IFORM_REP_XCRYPTCFB,
    INTRINSIC_XED_IFORM_REP_XCRYPTCTR,
    INTRINSIC_XED_IFORM_REP_XCRYPTECB,
    INTRINSIC_XED_IFORM_REP_XCRYPTOFB,
    INTRINSIC_XED_IFORM_REP_XSHA1,
    INTRINSIC_XED_IFORM_REP_XSHA256,
    INTRINSIC_XED_IFORM_REP_XSTORE,
    INTRINSIC_XED_IFORM_RET_FAR,
    INTRINSIC_XED_IFORM_RET_FAR_IMMw,
    INTRINSIC_XED_IFORM_RET_NEAR,
    INTRINSIC_XED_IFORM_RET_NEAR_IMMw,
    INTRINSIC_XED_IFORM_RMPADJUST_RAX_RCX_RDX,
    INTRINSIC_XED_IFORM_RMPUPDATE_RAX_RCX,
    INTRINSIC_XED_IFORM_ROL_GPR8_CL,
    INTRINSIC_XED_IFORM_ROL_GPR8_IMMb,
    INTRINSIC_XED_IFORM_ROL_GPR8_ONE,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_CL,
    INTRINSIC_XED_IFORM_ROL_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_IMMb,
    INTRINSIC_XED_IFORM_ROL_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_GPRv_ONE,
    INTRINSIC_XED_IFORM_ROL_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_MEMb_CL,
    INTRINSIC_XED_IFORM_ROL_MEMb_IMMb,
    INTRINSIC_XED_IFORM_ROL_MEMb_ONE,
    INTRINSIC_XED_IFORM_ROL_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_ROL_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_ROL_MEMv_CL,
    INTRINSIC_XED_IFORM_ROL_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_ROL_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROL_MEMv_IMMb,
    INTRINSIC_XED_IFORM_ROL_MEMv_ONE,
    INTRINSIC_XED_IFORM_ROL_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8_CL,
    INTRINSIC_XED_IFORM_ROR_GPR8_IMMb,
    INTRINSIC_XED_IFORM_ROR_GPR8_ONE,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_CL,
    INTRINSIC_XED_IFORM_ROR_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_ROR_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_GPRv_ONE,
    INTRINSIC_XED_IFORM_ROR_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_MEMb_CL,
    INTRINSIC_XED_IFORM_ROR_MEMb_IMMb,
    INTRINSIC_XED_IFORM_ROR_MEMb_ONE,
    INTRINSIC_XED_IFORM_ROR_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_ROR_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_ROR_MEMv_CL,
    INTRINSIC_XED_IFORM_ROR_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_ROR_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_ROR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_ROR_MEMv_ONE,
    INTRINSIC_XED_IFORM_ROR_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_RORX_GPR32d_GPR32d_IMMb,
    INTRINSIC_XED_IFORM_RORX_GPR32d_MEMd_IMMb,
    INTRINSIC_XED_IFORM_RORX_GPR32i32_GPR32i32_IMM8_APX,
    INTRINSIC_XED_IFORM_RORX_GPR32i32_MEMi32_IMM8_APX,
    INTRINSIC_XED_IFORM_RORX_GPR64i64_GPR64i64_IMM8_APX,
    INTRINSIC_XED_IFORM_RORX_GPR64i64_MEMi64_IMM8_APX,
    INTRINSIC_XED_IFORM_RORX_GPR64q_GPR64q_IMMb,
    INTRINSIC_XED_IFORM_RORX_GPR64q_MEMq_IMMb,
    INTRINSIC_XED_IFORM_ROUNDPD_XMMpd_MEMpd_IMMb,
    INTRINSIC_XED_IFORM_ROUNDPD_XMMpd_XMMpd_IMMb,
    INTRINSIC_XED_IFORM_ROUNDPS_XMMps_MEMps_IMMb,
    INTRINSIC_XED_IFORM_ROUNDPS_XMMps_XMMps_IMMb,
    INTRINSIC_XED_IFORM_ROUNDSD_XMMq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_ROUNDSD_XMMq_XMMq_IMMb,
    INTRINSIC_XED_IFORM_ROUNDSS_XMMd_MEMd_IMMb,
    INTRINSIC_XED_IFORM_ROUNDSS_XMMd_XMMd_IMMb,
    INTRINSIC_XED_IFORM_RSM,
    INTRINSIC_XED_IFORM_RSQRTPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_RSQRTPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_RSQRTSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_RSQRTSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_RSTORSSP_MEMu64,
    INTRINSIC_XED_IFORM_SAHF,
    INTRINSIC_XED_IFORM_SALC,
    INTRINSIC_XED_IFORM_SAR_GPR8_CL,
    INTRINSIC_XED_IFORM_SAR_GPR8_IMMb,
    INTRINSIC_XED_IFORM_SAR_GPR8_ONE,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_CL,
    INTRINSIC_XED_IFORM_SAR_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SAR_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_GPRv_ONE,
    INTRINSIC_XED_IFORM_SAR_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_MEMb_CL,
    INTRINSIC_XED_IFORM_SAR_MEMb_IMMb,
    INTRINSIC_XED_IFORM_SAR_MEMb_ONE,
    INTRINSIC_XED_IFORM_SAR_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_SAR_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_SAR_MEMv_CL,
    INTRINSIC_XED_IFORM_SAR_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_SAR_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SAR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_SAR_MEMv_ONE,
    INTRINSIC_XED_IFORM_SAR_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_SARX_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_SARX_GPR32d_MEMd_GPR32d,
    INTRINSIC_XED_IFORM_SARX_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_SARX_GPR32i32_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_SARX_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_SARX_GPR64i64_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_SARX_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_SARX_GPR64q_MEMq_GPR64q,
    INTRINSIC_XED_IFORM_SAVEPREVSSP,
    INTRINSIC_XED_IFORM_SBB_AL_IMMb,
    INTRINSIC_XED_IFORM_SBB_GPR8_GPR8_18,
    INTRINSIC_XED_IFORM_SBB_GPR8_GPR8_1A,
    INTRINSIC_XED_IFORM_SBB_GPR8_IMMb_80r3,
    INTRINSIC_XED_IFORM_SBB_GPR8_IMMb_82r3,
    INTRINSIC_XED_IFORM_SBB_GPR8_MEMb,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SBB_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_19,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_1B,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SBB_GPRv_IMMz,
    INTRINSIC_XED_IFORM_SBB_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_MEMv,
    INTRINSIC_XED_IFORM_SBB_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_SBB_MEMb_GPR8,
    INTRINSIC_XED_IFORM_SBB_MEMb_IMMb_80r3,
    INTRINSIC_XED_IFORM_SBB_MEMb_IMMb_82r3,
    INTRINSIC_XED_IFORM_SBB_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SBB_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_MEMv_GPRv,
    INTRINSIC_XED_IFORM_SBB_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_SBB_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SBB_MEMv_IMMb,
    INTRINSIC_XED_IFORM_SBB_MEMv_IMMz,
    INTRINSIC_XED_IFORM_SBB_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_SBB_OrAX_IMMz,
    INTRINSIC_XED_IFORM_SBB_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_SBB_LOCK_MEMb_IMMb_80r3,
    INTRINSIC_XED_IFORM_SBB_LOCK_MEMb_IMMb_82r3,
    INTRINSIC_XED_IFORM_SBB_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_SBB_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_SBB_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_SCASB,
    INTRINSIC_XED_IFORM_SCASD,
    INTRINSIC_XED_IFORM_SCASQ,
    INTRINSIC_XED_IFORM_SCASW,
    INTRINSIC_XED_IFORM_SEAMCALL,
    INTRINSIC_XED_IFORM_SEAMOPS,
    INTRINSIC_XED_IFORM_SEAMRET,
    INTRINSIC_XED_IFORM_SENDUIPI_GPR64u32,
    INTRINSIC_XED_IFORM_SERIALIZE,
    INTRINSIC_XED_IFORM_SETB_GPR8,
    INTRINSIC_XED_IFORM_SETB_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETB_MEMb,
    INTRINSIC_XED_IFORM_SETB_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETBE_GPR8,
    INTRINSIC_XED_IFORM_SETBE_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETBE_MEMb,
    INTRINSIC_XED_IFORM_SETBE_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETL_GPR8,
    INTRINSIC_XED_IFORM_SETL_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETL_MEMb,
    INTRINSIC_XED_IFORM_SETL_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETLE_GPR8,
    INTRINSIC_XED_IFORM_SETLE_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETLE_MEMb,
    INTRINSIC_XED_IFORM_SETLE_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNB_GPR8,
    INTRINSIC_XED_IFORM_SETNB_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNB_MEMb,
    INTRINSIC_XED_IFORM_SETNB_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNBE_GPR8,
    INTRINSIC_XED_IFORM_SETNBE_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNBE_MEMb,
    INTRINSIC_XED_IFORM_SETNBE_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNL_GPR8,
    INTRINSIC_XED_IFORM_SETNL_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNL_MEMb,
    INTRINSIC_XED_IFORM_SETNL_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNLE_GPR8,
    INTRINSIC_XED_IFORM_SETNLE_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNLE_MEMb,
    INTRINSIC_XED_IFORM_SETNLE_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNO_GPR8,
    INTRINSIC_XED_IFORM_SETNO_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNO_MEMb,
    INTRINSIC_XED_IFORM_SETNO_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNP_GPR8,
    INTRINSIC_XED_IFORM_SETNP_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNP_MEMb,
    INTRINSIC_XED_IFORM_SETNP_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNS_GPR8,
    INTRINSIC_XED_IFORM_SETNS_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNS_MEMb,
    INTRINSIC_XED_IFORM_SETNS_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETNZ_GPR8,
    INTRINSIC_XED_IFORM_SETNZ_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETNZ_MEMb,
    INTRINSIC_XED_IFORM_SETNZ_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETO_GPR8,
    INTRINSIC_XED_IFORM_SETO_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETO_MEMb,
    INTRINSIC_XED_IFORM_SETO_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETP_GPR8,
    INTRINSIC_XED_IFORM_SETP_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETP_MEMb,
    INTRINSIC_XED_IFORM_SETP_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETS_GPR8,
    INTRINSIC_XED_IFORM_SETS_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETS_MEMb,
    INTRINSIC_XED_IFORM_SETS_MEMi8_APX,
    INTRINSIC_XED_IFORM_SETSSBSY,
    INTRINSIC_XED_IFORM_SETZ_GPR8,
    INTRINSIC_XED_IFORM_SETZ_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SETZ_MEMb,
    INTRINSIC_XED_IFORM_SETZ_MEMi8_APX,
    INTRINSIC_XED_IFORM_SFENCE,
    INTRINSIC_XED_IFORM_SGDT_MEMs,
    INTRINSIC_XED_IFORM_SGDT_MEMs64,
    INTRINSIC_XED_IFORM_SHA1MSG1_XMMi32_MEMi32_SHA,
    INTRINSIC_XED_IFORM_SHA1MSG1_XMMi32_XMMi32_SHA,
    INTRINSIC_XED_IFORM_SHA1MSG2_XMMi32_MEMi32_SHA,
    INTRINSIC_XED_IFORM_SHA1MSG2_XMMi32_XMMi32_SHA,
    INTRINSIC_XED_IFORM_SHA1NEXTE_XMMi32_MEMi32_SHA,
    INTRINSIC_XED_IFORM_SHA1NEXTE_XMMi32_XMMi32_SHA,
    INTRINSIC_XED_IFORM_SHA1RNDS4_XMMi32_MEMi32_IMM8_SHA,
    INTRINSIC_XED_IFORM_SHA1RNDS4_XMMi32_XMMi32_IMM8_SHA,
    INTRINSIC_XED_IFORM_SHA256MSG1_XMMi32_MEMi32_SHA,
    INTRINSIC_XED_IFORM_SHA256MSG1_XMMi32_XMMi32_SHA,
    INTRINSIC_XED_IFORM_SHA256MSG2_XMMi32_MEMi32_SHA,
    INTRINSIC_XED_IFORM_SHA256MSG2_XMMi32_XMMi32_SHA,
    INTRINSIC_XED_IFORM_SHA256RNDS2_XMMi32_MEMi32_SHA,
    INTRINSIC_XED_IFORM_SHA256RNDS2_XMMi32_XMMi32_SHA,
    INTRINSIC_XED_IFORM_SHL_GPR8_CL_D2r4,
    INTRINSIC_XED_IFORM_SHL_GPR8_CL_D2r6,
    INTRINSIC_XED_IFORM_SHL_GPR8_IMMb_C0r4,
    INTRINSIC_XED_IFORM_SHL_GPR8_IMMb_C0r6,
    INTRINSIC_XED_IFORM_SHL_GPR8_ONE_D0r4,
    INTRINSIC_XED_IFORM_SHL_GPR8_ONE_D0r6,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_CL_D3r4,
    INTRINSIC_XED_IFORM_SHL_GPRv_CL_D3r6,
    INTRINSIC_XED_IFORM_SHL_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_IMMb_C1r4,
    INTRINSIC_XED_IFORM_SHL_GPRv_IMMb_C1r6,
    INTRINSIC_XED_IFORM_SHL_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_GPRv_ONE_D1r4,
    INTRINSIC_XED_IFORM_SHL_GPRv_ONE_D1r6,
    INTRINSIC_XED_IFORM_SHL_MEMb_CL_D2r4,
    INTRINSIC_XED_IFORM_SHL_MEMb_CL_D2r6,
    INTRINSIC_XED_IFORM_SHL_MEMb_IMMb_C0r4,
    INTRINSIC_XED_IFORM_SHL_MEMb_IMMb_C0r6,
    INTRINSIC_XED_IFORM_SHL_MEMb_ONE_D0r4,
    INTRINSIC_XED_IFORM_SHL_MEMb_ONE_D0r6,
    INTRINSIC_XED_IFORM_SHL_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_SHL_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_SHL_MEMv_CL_D3r4,
    INTRINSIC_XED_IFORM_SHL_MEMv_CL_D3r6,
    INTRINSIC_XED_IFORM_SHL_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHL_MEMv_IMMb_C1r4,
    INTRINSIC_XED_IFORM_SHL_MEMv_IMMb_C1r6,
    INTRINSIC_XED_IFORM_SHL_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_SHL_MEMv_ONE_D1r4,
    INTRINSIC_XED_IFORM_SHL_MEMv_ONE_D1r6,
    INTRINSIC_XED_IFORM_SHLD_GPRv_GPRv_CL,
    INTRINSIC_XED_IFORM_SHLD_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHLD_GPRv_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHLD_GPRv_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHLD_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHLD_GPRv_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SHLD_GPRv_MEMv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHLD_GPRv_MEMv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHLD_MEMv_GPRv_CL,
    INTRINSIC_XED_IFORM_SHLD_MEMv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHLD_MEMv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHLD_MEMv_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SHLX_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_SHLX_GPR32d_MEMd_GPR32d,
    INTRINSIC_XED_IFORM_SHLX_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_SHLX_GPR32i32_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_SHLX_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_SHLX_GPR64i64_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_SHLX_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_SHLX_GPR64q_MEMq_GPR64q,
    INTRINSIC_XED_IFORM_SHR_GPR8_CL,
    INTRINSIC_XED_IFORM_SHR_GPR8_IMMb,
    INTRINSIC_XED_IFORM_SHR_GPR8_ONE,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_GPR8i8_CL_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_GPR8i8_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_CL,
    INTRINSIC_XED_IFORM_SHR_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SHR_GPRv_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_GPRv_ONE,
    INTRINSIC_XED_IFORM_SHR_GPRv_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_MEMb_CL,
    INTRINSIC_XED_IFORM_SHR_MEMb_IMMb,
    INTRINSIC_XED_IFORM_SHR_MEMb_ONE,
    INTRINSIC_XED_IFORM_SHR_MEMi8_CL_APX,
    INTRINSIC_XED_IFORM_SHR_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_MEMi8_ONE_APX,
    INTRINSIC_XED_IFORM_SHR_MEMv_CL,
    INTRINSIC_XED_IFORM_SHR_MEMv_CL_APX,
    INTRINSIC_XED_IFORM_SHR_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_SHR_MEMv_ONE,
    INTRINSIC_XED_IFORM_SHR_MEMv_ONE_APX,
    INTRINSIC_XED_IFORM_SHRD_GPRv_GPRv_CL,
    INTRINSIC_XED_IFORM_SHRD_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHRD_GPRv_GPRv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHRD_GPRv_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHRD_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHRD_GPRv_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SHRD_GPRv_MEMv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHRD_GPRv_MEMv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHRD_MEMv_GPRv_CL,
    INTRINSIC_XED_IFORM_SHRD_MEMv_GPRv_CL_APX,
    INTRINSIC_XED_IFORM_SHRD_MEMv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SHRD_MEMv_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SHRX_GPR32d_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_SHRX_GPR32d_MEMd_GPR32d,
    INTRINSIC_XED_IFORM_SHRX_GPR32i32_GPR32i32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_SHRX_GPR32i32_MEMi32_GPR32i32_APX,
    INTRINSIC_XED_IFORM_SHRX_GPR64i64_GPR64i64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_SHRX_GPR64i64_MEMi64_GPR64i64_APX,
    INTRINSIC_XED_IFORM_SHRX_GPR64q_GPR64q_GPR64q,
    INTRINSIC_XED_IFORM_SHRX_GPR64q_MEMq_GPR64q,
    INTRINSIC_XED_IFORM_SHUFPD_XMMpd_MEMpd_IMMb,
    INTRINSIC_XED_IFORM_SHUFPD_XMMpd_XMMpd_IMMb,
    INTRINSIC_XED_IFORM_SHUFPS_XMMps_MEMps_IMMb,
    INTRINSIC_XED_IFORM_SHUFPS_XMMps_XMMps_IMMb,
    INTRINSIC_XED_IFORM_SIDT_MEMs,
    INTRINSIC_XED_IFORM_SIDT_MEMs64,
    INTRINSIC_XED_IFORM_SKINIT_EAX,
    INTRINSIC_XED_IFORM_SLDT_GPRv,
    INTRINSIC_XED_IFORM_SLDT_MEMw,
    INTRINSIC_XED_IFORM_SLWPCB_GPRyy,
    INTRINSIC_XED_IFORM_SMSW_GPRv,
    INTRINSIC_XED_IFORM_SMSW_MEMw,
    INTRINSIC_XED_IFORM_SQRTPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_SQRTPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_SQRTPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_SQRTPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_SQRTSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_SQRTSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_SQRTSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_SQRTSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_STAC,
    INTRINSIC_XED_IFORM_STC,
    INTRINSIC_XED_IFORM_STD,
    INTRINSIC_XED_IFORM_STGI,
    INTRINSIC_XED_IFORM_STI,
    INTRINSIC_XED_IFORM_STMXCSR_MEMd,
    INTRINSIC_XED_IFORM_STOSB,
    INTRINSIC_XED_IFORM_STOSD,
    INTRINSIC_XED_IFORM_STOSQ,
    INTRINSIC_XED_IFORM_STOSW,
    INTRINSIC_XED_IFORM_STR_GPRv,
    INTRINSIC_XED_IFORM_STR_MEMw,
    INTRINSIC_XED_IFORM_STTILECFG_MEM,
    INTRINSIC_XED_IFORM_STTILECFG_MEM_APX,
    INTRINSIC_XED_IFORM_STUI,
    INTRINSIC_XED_IFORM_SUB_AL_IMMb,
    INTRINSIC_XED_IFORM_SUB_GPR8_GPR8_28,
    INTRINSIC_XED_IFORM_SUB_GPR8_GPR8_2A,
    INTRINSIC_XED_IFORM_SUB_GPR8_IMMb_80r5,
    INTRINSIC_XED_IFORM_SUB_GPR8_IMMb_82r5,
    INTRINSIC_XED_IFORM_SUB_GPR8_MEMb,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SUB_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_29,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_2B,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_IMMb,
    INTRINSIC_XED_IFORM_SUB_GPRv_IMMz,
    INTRINSIC_XED_IFORM_SUB_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_MEMv,
    INTRINSIC_XED_IFORM_SUB_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_SUB_MEMb_GPR8,
    INTRINSIC_XED_IFORM_SUB_MEMb_IMMb_80r5,
    INTRINSIC_XED_IFORM_SUB_MEMb_IMMb_82r5,
    INTRINSIC_XED_IFORM_SUB_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_SUB_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_MEMv_GPRv,
    INTRINSIC_XED_IFORM_SUB_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_SUB_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_SUB_MEMv_IMMb,
    INTRINSIC_XED_IFORM_SUB_MEMv_IMMz,
    INTRINSIC_XED_IFORM_SUB_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_SUB_OrAX_IMMz,
    INTRINSIC_XED_IFORM_SUBPD_XMMpd_MEMpd,
    INTRINSIC_XED_IFORM_SUBPD_XMMpd_XMMpd,
    INTRINSIC_XED_IFORM_SUBPS_XMMps_MEMps,
    INTRINSIC_XED_IFORM_SUBPS_XMMps_XMMps,
    INTRINSIC_XED_IFORM_SUBSD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_SUBSD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_SUBSS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_SUBSS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_SUB_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_SUB_LOCK_MEMb_IMMb_80r5,
    INTRINSIC_XED_IFORM_SUB_LOCK_MEMb_IMMb_82r5,
    INTRINSIC_XED_IFORM_SUB_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_SUB_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_SUB_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_SWAPGS,
    INTRINSIC_XED_IFORM_SYSCALL,
    INTRINSIC_XED_IFORM_SYSCALL_32,
    INTRINSIC_XED_IFORM_SYSENTER,
    INTRINSIC_XED_IFORM_SYSEXIT,
    INTRINSIC_XED_IFORM_SYSRET,
    INTRINSIC_XED_IFORM_SYSRET64,
    INTRINSIC_XED_IFORM_SYSRET_AMD,
    INTRINSIC_XED_IFORM_T1MSKC_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_T1MSKC_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_T1MSKC_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_T1MSKC_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_TCMMIMFP16PS_TMMf32_TMM2f16_TMM2f16,
    INTRINSIC_XED_IFORM_TCMMRLFP16PS_TMMf32_TMM2f16_TMM2f16,
    INTRINSIC_XED_IFORM_TDCALL,
    INTRINSIC_XED_IFORM_TDPBF16PS_TMMf32_TMM2bf16_TMM2bf16,
    INTRINSIC_XED_IFORM_TDPBSSD_TMMi32_TMM4i8_TMM4i8,
    INTRINSIC_XED_IFORM_TDPBSUD_TMMi32_TMM4i8_TMM4u8,
    INTRINSIC_XED_IFORM_TDPBUSD_TMMi32_TMM4u8_TMM4i8,
    INTRINSIC_XED_IFORM_TDPBUUD_TMMu32_TMM4u8_TMM4u8,
    INTRINSIC_XED_IFORM_TDPFP16PS_TMMf32_TMM2f16_TMM2f16,
    INTRINSIC_XED_IFORM_TEST_AL_IMMb,
    INTRINSIC_XED_IFORM_TEST_GPR8_GPR8,
    INTRINSIC_XED_IFORM_TEST_GPR8_IMMb_F6r0,
    INTRINSIC_XED_IFORM_TEST_GPR8_IMMb_F6r1,
    INTRINSIC_XED_IFORM_TEST_GPRv_GPRv,
    INTRINSIC_XED_IFORM_TEST_GPRv_IMMz_F7r0,
    INTRINSIC_XED_IFORM_TEST_GPRv_IMMz_F7r1,
    INTRINSIC_XED_IFORM_TEST_MEMb_GPR8,
    INTRINSIC_XED_IFORM_TEST_MEMb_IMMb_F6r0,
    INTRINSIC_XED_IFORM_TEST_MEMb_IMMb_F6r1,
    INTRINSIC_XED_IFORM_TEST_MEMv_GPRv,
    INTRINSIC_XED_IFORM_TEST_MEMv_IMMz_F7r0,
    INTRINSIC_XED_IFORM_TEST_MEMv_IMMz_F7r1,
    INTRINSIC_XED_IFORM_TEST_OrAX_IMMz,
    INTRINSIC_XED_IFORM_TESTUI,
    INTRINSIC_XED_IFORM_TILELOADD_TMMu32_MEMu32,
    INTRINSIC_XED_IFORM_TILELOADD_TMMu32_MEMu32_APX,
    INTRINSIC_XED_IFORM_TILELOADDT1_TMMu32_MEMu32,
    INTRINSIC_XED_IFORM_TILELOADDT1_TMMu32_MEMu32_APX,
    INTRINSIC_XED_IFORM_TILERELEASE,
    INTRINSIC_XED_IFORM_TILESTORED_MEMu32_TMMu32,
    INTRINSIC_XED_IFORM_TILESTORED_MEMu32_TMMu32_APX,
    INTRINSIC_XED_IFORM_TILEZERO_TMMu32,
    INTRINSIC_XED_IFORM_TLBSYNC,
    INTRINSIC_XED_IFORM_TPAUSE_GPR32u32,
    INTRINSIC_XED_IFORM_TZCNT_GPRv_GPRv,
    INTRINSIC_XED_IFORM_TZCNT_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_TZCNT_GPRv_MEMv,
    INTRINSIC_XED_IFORM_TZCNT_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_TZMSK_GPR32d_GPR32d,
    INTRINSIC_XED_IFORM_TZMSK_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_TZMSK_GPRyy_GPRyy,
    INTRINSIC_XED_IFORM_TZMSK_GPRyy_MEMy,
    INTRINSIC_XED_IFORM_UCOMISD_XMMsd_MEMsd,
    INTRINSIC_XED_IFORM_UCOMISD_XMMsd_XMMsd,
    INTRINSIC_XED_IFORM_UCOMISS_XMMss_MEMss,
    INTRINSIC_XED_IFORM_UCOMISS_XMMss_XMMss,
    INTRINSIC_XED_IFORM_UD0,
    INTRINSIC_XED_IFORM_UD0_GPR32_GPR32,
    INTRINSIC_XED_IFORM_UD0_GPR32_MEMd,
    INTRINSIC_XED_IFORM_UD1_GPR32_GPR32,
    INTRINSIC_XED_IFORM_UD1_GPR32_MEMd,
    INTRINSIC_XED_IFORM_UD2,
    INTRINSIC_XED_IFORM_UIRET,
    INTRINSIC_XED_IFORM_UMONITOR_GPRa,
    INTRINSIC_XED_IFORM_UMWAIT_GPR32,
    INTRINSIC_XED_IFORM_UNPCKHPD_XMMpd_MEMdq,
    INTRINSIC_XED_IFORM_UNPCKHPD_XMMpd_XMMq,
    INTRINSIC_XED_IFORM_UNPCKHPS_XMMps_MEMdq,
    INTRINSIC_XED_IFORM_UNPCKHPS_XMMps_XMMdq,
    INTRINSIC_XED_IFORM_UNPCKLPD_XMMpd_MEMdq,
    INTRINSIC_XED_IFORM_UNPCKLPD_XMMpd_XMMq,
    INTRINSIC_XED_IFORM_UNPCKLPS_XMMps_MEMdq,
    INTRINSIC_XED_IFORM_UNPCKLPS_XMMps_XMMq,
    INTRINSIC_XED_IFORM_URDMSR_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_URDMSR_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_URDMSR_GPR64u64_IMM32,
    INTRINSIC_XED_IFORM_URDMSR_GPR64u64_IMM32_APX,
    INTRINSIC_XED_IFORM_UWRMSR_GPR64u64_GPR64u64,
    INTRINSIC_XED_IFORM_UWRMSR_GPR64u64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_UWRMSR_IMM32_GPR64u64,
    INTRINSIC_XED_IFORM_UWRMSR_IMM32_GPR64u64_APX,
    INTRINSIC_XED_IFORM_V4FMADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_V4FMADDSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_V4FNMADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_V4FNMADDSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VADDPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VADDPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VADDPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VADDPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VADDPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VADDPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VADDPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VADDPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VADDSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VADDSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VADDSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VADDSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VADDSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VADDSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VADDSUBPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VADDSUBPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VADDSUBPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VADDSUBPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VADDSUBPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VADDSUBPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VADDSUBPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VADDSUBPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VAESDEC_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VAESDEC_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VAESDEC_XMMu128_XMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDEC_XMMu128_XMMu128_XMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDEC_YMMu128_YMMu128_MEMu128,
    INTRINSIC_XED_IFORM_VAESDEC_YMMu128_YMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDEC_YMMu128_YMMu128_YMMu128,
    INTRINSIC_XED_IFORM_VAESDEC_YMMu128_YMMu128_YMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDEC_ZMMu128_ZMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDEC_ZMMu128_ZMMu128_ZMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDECLAST_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VAESDECLAST_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VAESDECLAST_XMMu128_XMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDECLAST_XMMu128_XMMu128_XMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDECLAST_YMMu128_YMMu128_MEMu128,
    INTRINSIC_XED_IFORM_VAESDECLAST_YMMu128_YMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDECLAST_YMMu128_YMMu128_YMMu128,
    INTRINSIC_XED_IFORM_VAESDECLAST_YMMu128_YMMu128_YMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDECLAST_ZMMu128_ZMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESDECLAST_ZMMu128_ZMMu128_ZMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENC_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VAESENC_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VAESENC_XMMu128_XMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENC_XMMu128_XMMu128_XMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENC_YMMu128_YMMu128_MEMu128,
    INTRINSIC_XED_IFORM_VAESENC_YMMu128_YMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENC_YMMu128_YMMu128_YMMu128,
    INTRINSIC_XED_IFORM_VAESENC_YMMu128_YMMu128_YMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENC_ZMMu128_ZMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENC_ZMMu128_ZMMu128_ZMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENCLAST_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VAESENCLAST_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VAESENCLAST_XMMu128_XMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENCLAST_XMMu128_XMMu128_XMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENCLAST_YMMu128_YMMu128_MEMu128,
    INTRINSIC_XED_IFORM_VAESENCLAST_YMMu128_YMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENCLAST_YMMu128_YMMu128_YMMu128,
    INTRINSIC_XED_IFORM_VAESENCLAST_YMMu128_YMMu128_YMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENCLAST_ZMMu128_ZMMu128_MEMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESENCLAST_ZMMu128_ZMMu128_ZMMu128_AVX512,
    INTRINSIC_XED_IFORM_VAESIMC_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VAESIMC_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VAESKEYGENASSIST_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VAESKEYGENASSIST_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VALIGND_XMMu32_MASKmskw_XMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGND_XMMu32_MASKmskw_XMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGND_YMMu32_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGND_YMMu32_MASKmskw_YMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGND_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGND_ZMMu32_MASKmskw_ZMMu32_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGNQ_XMMu64_MASKmskw_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGNQ_XMMu64_MASKmskw_XMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGNQ_YMMu64_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGNQ_YMMu64_MASKmskw_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VALIGNQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VANDNPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VANDNPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VANDNPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDNPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDNPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VANDNPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VANDNPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDNPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDNPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDNPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDNPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VANDNPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VANDNPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDNPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDNPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VANDNPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VANDNPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDNPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDNPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDNPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VANDPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VANDPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VANDPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VANDPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VANDPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VANDPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VANDPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VANDPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VANDPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VANDPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VBCSTNEBF162PS_XMMf32_MEMbf16,
    INTRINSIC_XED_IFORM_VBCSTNEBF162PS_YMMf32_MEMbf16,
    INTRINSIC_XED_IFORM_VBCSTNESH2PS_XMMf32_MEMf16,
    INTRINSIC_XED_IFORM_VBCSTNESH2PS_YMMf32_MEMf16,
    INTRINSIC_XED_IFORM_VBLENDMPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBLENDMPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBLENDPD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPD_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPD_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPS_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPS_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPS_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDPS_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VBLENDVPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VBLENDVPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VBLENDVPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VBLENDVPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VBLENDVPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VBLENDVPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VBLENDVPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VBLENDVPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VBROADCASTF128_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VBROADCASTF32X2_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF32X2_YMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF32X2_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF32X2_ZMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF32X4_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF32X4_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF32X8_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF64X2_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF64X2_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTF64X4_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI128_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VBROADCASTI32X2_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X2_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X2_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X2_YMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X2_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X2_ZMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X4_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X4_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI32X8_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI64X2_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI64X2_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTI64X4_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSD_YMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSD_YMMqq_MEMq,
    INTRINSIC_XED_IFORM_VBROADCASTSD_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VBROADCASTSD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSD_ZMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSS_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VBROADCASTSS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VBROADCASTSS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSS_YMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSS_YMMqq_MEMd,
    INTRINSIC_XED_IFORM_VBROADCASTSS_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VBROADCASTSS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VBROADCASTSS_ZMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_MASKmskw_MASKmskw_YMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_MASKmskw_MASKmskw_YMMf64_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_MASKmskw_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_MASKmskw_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPD_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPD_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPH_MASKmskw_MASKmskw_XMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPH_MASKmskw_MASKmskw_XMMf16_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPH_MASKmskw_MASKmskw_YMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPH_MASKmskw_MASKmskw_YMMf16_YMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPH_MASKmskw_MASKmskw_ZMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPH_MASKmskw_MASKmskw_ZMMf16_ZMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_MASKmskw_MASKmskw_YMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_MASKmskw_MASKmskw_YMMf32_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_MASKmskw_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_MASKmskw_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPPS_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPS_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPS_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VCMPPS_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VCMPSD_MASKmskw_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPSD_MASKmskw_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPSD_XMMdq_XMMdq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_VCMPSD_XMMdq_XMMdq_XMMq_IMMb,
    INTRINSIC_XED_IFORM_VCMPSH_MASKmskw_MASKmskw_XMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPSH_MASKmskw_MASKmskw_XMMf16_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPSS_MASKmskw_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPSS_MASKmskw_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCMPSS_XMMdq_XMMdq_MEMd_IMMb,
    INTRINSIC_XED_IFORM_VCMPSS_XMMdq_XMMdq_XMMd_IMMb,
    INTRINSIC_XED_IFORM_VCOMISD_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMISD_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMISD_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VCOMISD_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VCOMISH_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCOMISH_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCOMISS_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VCOMISS_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VCOMISS_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMISS_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPD_MEMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPD_MEMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPD_MEMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPS_MEMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPS_MEMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPS_MEMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCOMPRESSPS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_XMMf64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_XMMf64_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_YMMf64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_YMMf64_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_ZMMf64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PD_ZMMf64_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PH_XMMf16_MASKmskw_MEMi32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTDQ2PH_XMMf16_MASKmskw_MEMi32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTDQ2PH_XMMf16_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PH_XMMf16_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PH_YMMf16_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PH_YMMf16_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_XMMf32_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_XMMf32_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_YMMf32_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_YMMf32_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_ZMMf32_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTDQ2PS_ZMMf32_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTNE2PS2BF16_XMMbf16_MASKmskw_XMMf32_MEMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTNE2PS2BF16_XMMbf16_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTNE2PS2BF16_YMMbf16_MASKmskw_YMMf32_MEMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTNE2PS2BF16_YMMbf16_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTNE2PS2BF16_ZMMbf16_MASKmskw_ZMMf32_MEMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTNE2PS2BF16_ZMMbf16_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTNEEBF162PS_XMMf32_MEM2bf16,
    INTRINSIC_XED_IFORM_VCVTNEEBF162PS_YMMf32_MEM2bf16,
    INTRINSIC_XED_IFORM_VCVTNEEPH2PS_XMMf32_MEM2f16,
    INTRINSIC_XED_IFORM_VCVTNEEPH2PS_YMMf32_MEM2f16,
    INTRINSIC_XED_IFORM_VCVTNEOBF162PS_XMMf32_MEM2bf16,
    INTRINSIC_XED_IFORM_VCVTNEOBF162PS_YMMf32_MEM2bf16,
    INTRINSIC_XED_IFORM_VCVTNEOPH2PS_XMMf32_MEM2f16,
    INTRINSIC_XED_IFORM_VCVTNEOPH2PS_YMMf32_MEM2f16,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_MASKmskw_MEMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_MASKmskw_MEMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_MEMf32_VL128,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_MEMf32_VL256,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_XMMf32,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_XMMbf16_YMMf32,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_YMMbf16_MASKmskw_MEMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTNEPS2BF16_YMMbf16_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMdq_MEMqq,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMdq_YMMqq,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2PH_XMMf16_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2PH_XMMf16_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2PH_XMMf16_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2PH_XMMf16_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2PH_XMMf16_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2PH_XMMf16_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMdq_MEMqq,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMdq_YMMqq,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMf32_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMf32_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2PS_XMMf32_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2PS_YMMf32_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2PS_YMMf32_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2DQ_XMMi32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2DQ_XMMi32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2DQ_YMMi32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2DQ_YMMi32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2DQ_ZMMi32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2DQ_ZMMi32_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PD_XMMf64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PD_XMMf64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PD_YMMf64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PD_YMMf64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PD_ZMMf64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PD_ZMMf64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PS_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VCVTPH2PS_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VCVTPH2PS_XMMf32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PS_XMMf32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PS_YMMf32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PS_YMMf32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PS_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTPH2PS_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTPH2PS_ZMMf32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PS_ZMMf32_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PSX_XMMf32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PSX_XMMf32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PSX_YMMf32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PSX_YMMf32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PSX_ZMMf32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2PSX_ZMMf32_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2QQ_XMMi64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2QQ_XMMi64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2QQ_YMMi64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2QQ_YMMi64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2QQ_ZMMi64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2QQ_ZMMi64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UDQ_XMMu32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UDQ_XMMu32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UDQ_YMMu32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UDQ_YMMu32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UDQ_ZMMu32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UDQ_ZMMu32_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UQQ_XMMu64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UQQ_XMMu64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UQQ_YMMu64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UQQ_YMMu64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UQQ_ZMMu64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UQQ_ZMMu64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UW_XMMu16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UW_XMMu16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UW_YMMu16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UW_YMMu16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UW_ZMMu16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2UW_ZMMu16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2W_XMMi16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2W_XMMi16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2W_YMMi16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2W_YMMi16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2W_ZMMi16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPH2W_ZMMi16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VCVTPS2PD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VCVTPS2PD_XMMf64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PD_XMMf64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PD_YMMf64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PD_YMMf64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PD_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTPS2PD_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTPS2PD_ZMMf64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PD_ZMMf64_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PH_MEMdq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VCVTPS2PH_MEMf16_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PH_MEMf16_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PH_MEMf16_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PH_MEMq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VCVTPS2PH_XMMdq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VCVTPS2PH_XMMf16_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PH_XMMf16_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PH_XMMq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VCVTPS2PH_YMMf16_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PHX_XMMf16_MASKmskw_MEMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTPS2PHX_XMMf16_MASKmskw_MEMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTPS2PHX_XMMf16_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PHX_XMMf16_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2PHX_YMMf16_MASKmskw_MEMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTPS2PHX_YMMf16_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PD_XMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PD_XMMi64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PD_YMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PD_YMMi64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PD_ZMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PD_ZMMi64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PH_XMMf16_MASKmskw_MEMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTQQ2PH_XMMf16_MASKmskw_MEMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTQQ2PH_XMMf16_MASKmskw_MEMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTQQ2PH_XMMf16_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PH_XMMf16_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PH_XMMf16_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTSD2SH_XMMf16_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SH_XMMf16_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR32d_MEMq,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR32d_XMMq,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR32i32_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR32i32_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR64i64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR64i64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_VCVTSD2SI_GPR64q_XMMq,
    INTRINSIC_XED_IFORM_VCVTSD2SS_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VCVTSD2SS_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VCVTSD2SS_XMMf32_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2SS_XMMf32_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2USI_GPR32u32_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2USI_GPR32u32_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2USI_GPR64u64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSD2USI_GPR64u64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SD_XMMf64_MASKmskw_XMMf64_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SD_XMMf64_MASKmskw_XMMf64_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SI_GPR32i32_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SI_GPR32i32_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SI_GPR64i64_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SI_GPR64i64_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SS_XMMf32_MASKmskw_XMMf32_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2SS_XMMf32_MASKmskw_XMMf32_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2USI_GPR32u32_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2USI_GPR32u32_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2USI_GPR64u64_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSH2USI_GPR64u64_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMdq_XMMdq_GPR32d,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMdq_XMMdq_GPR64q,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMf64_XMMf64_GPR32i32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMf64_XMMf64_GPR64i64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMf64_XMMf64_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SD_XMMf64_XMMf64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SH_XMMf16_XMMf16_GPR32i32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SH_XMMf16_XMMf16_GPR64i64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SH_XMMf16_XMMf16_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SH_XMMf16_XMMf16_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMdq_XMMdq_GPR32d,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMdq_XMMdq_GPR64q,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMf32_XMMf32_GPR32i32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMf32_XMMf32_GPR64i64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMf32_XMMf32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSI2SS_XMMf32_XMMf32_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SD_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VCVTSS2SD_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VCVTSS2SD_XMMf64_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SD_XMMf64_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SH_XMMf16_MASKmskw_XMMf16_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SH_XMMf16_MASKmskw_XMMf16_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR32d_XMMd,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR32i32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR32i32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR64i64_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR64i64_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR64q_MEMd,
    INTRINSIC_XED_IFORM_VCVTSS2SI_GPR64q_XMMd,
    INTRINSIC_XED_IFORM_VCVTSS2USI_GPR32u32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2USI_GPR32u32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2USI_GPR64u64_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTSS2USI_GPR64u64_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMdq_MEMqq,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMdq_YMMqq,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMi32_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMi32_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_XMMi32_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_YMMi32_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTTPD2DQ_YMMi32_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTTPD2QQ_XMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2QQ_XMMi64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2QQ_YMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2QQ_YMMi64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2QQ_ZMMi64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2QQ_ZMMi64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTTPD2UDQ_XMMu32_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTTPD2UDQ_XMMu32_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTTPD2UDQ_XMMu32_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTTPD2UDQ_YMMu32_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTTPD2UDQ_YMMu32_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTTPD2UQQ_XMMu64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2UQQ_XMMu64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2UQQ_YMMu64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2UQQ_YMMu64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2UQQ_ZMMu64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPD2UQQ_ZMMu64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2DQ_XMMi32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2DQ_XMMi32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2DQ_YMMi32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2DQ_YMMi32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2DQ_ZMMi32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2DQ_ZMMi32_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2QQ_XMMi64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2QQ_XMMi64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2QQ_YMMi64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2QQ_YMMi64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2QQ_ZMMi64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2QQ_ZMMi64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UDQ_XMMu32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UDQ_XMMu32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UDQ_YMMu32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UDQ_YMMu32_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UDQ_ZMMu32_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UDQ_ZMMu32_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UQQ_XMMu64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UQQ_XMMu64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UQQ_YMMu64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UQQ_YMMu64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UQQ_ZMMu64_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UQQ_ZMMu64_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UW_XMMu16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UW_XMMu16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UW_YMMu16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UW_YMMu16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UW_ZMMu16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2UW_ZMMu16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2W_XMMi16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2W_XMMi16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2W_YMMi16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2W_YMMi16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2W_ZMMi16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPH2W_ZMMi16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_XMMi32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_XMMi32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_YMMi32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_YMMi32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_ZMMi32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2DQ_ZMMi32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2QQ_XMMi64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2QQ_XMMi64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2QQ_YMMi64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2QQ_YMMi64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2QQ_ZMMi64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2QQ_ZMMi64_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UDQ_XMMu32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UDQ_XMMu32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UDQ_YMMu32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UDQ_YMMu32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UDQ_ZMMu32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UDQ_ZMMu32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UQQ_XMMu64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UQQ_XMMu64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UQQ_YMMu64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UQQ_YMMu64_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UQQ_ZMMu64_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTPS2UQQ_ZMMu64_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR32d_MEMq,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR32d_XMMq,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR32i32_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR32i32_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR64i64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR64i64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR64q_MEMq,
    INTRINSIC_XED_IFORM_VCVTTSD2SI_GPR64q_XMMq,
    INTRINSIC_XED_IFORM_VCVTTSD2USI_GPR32u32_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2USI_GPR32u32_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2USI_GPR64u64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSD2USI_GPR64u64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2SI_GPR32i32_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2SI_GPR32i32_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2SI_GPR64i64_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2SI_GPR64i64_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2USI_GPR32u32_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2USI_GPR32u32_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2USI_GPR64u64_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSH2USI_GPR64u64_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR32d_MEMd,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR32d_XMMd,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR32i32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR32i32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR64i64_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR64i64_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR64q_MEMd,
    INTRINSIC_XED_IFORM_VCVTTSS2SI_GPR64q_XMMd,
    INTRINSIC_XED_IFORM_VCVTTSS2USI_GPR32u32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2USI_GPR32u32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2USI_GPR64u64_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTTSS2USI_GPR64u64_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PD_XMMf64_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PD_XMMf64_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PD_YMMf64_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PD_YMMf64_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PD_ZMMf64_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PD_ZMMf64_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PH_XMMf16_MASKmskw_MEMu32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTUDQ2PH_XMMf16_MASKmskw_MEMu32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTUDQ2PH_XMMf16_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PH_XMMf16_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PH_YMMf16_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PH_YMMf16_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PS_XMMf32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PS_XMMf32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PS_YMMf32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PS_YMMf32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PS_ZMMf32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUDQ2PS_ZMMf32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PD_XMMf64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PD_XMMf64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PD_YMMf64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PD_YMMf64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PD_ZMMf64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PD_ZMMf64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PH_XMMf16_MASKmskw_MEMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTUQQ2PH_XMMf16_MASKmskw_MEMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTUQQ2PH_XMMf16_MASKmskw_MEMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PH_XMMf16_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PH_XMMf16_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PH_XMMf16_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTUQQ2PS_XMMf32_MASKmskw_MEMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTUQQ2PS_XMMf32_MASKmskw_XMMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VCVTUQQ2PS_XMMf32_MASKmskw_YMMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VCVTUQQ2PS_YMMf32_MASKmskw_MEMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTUQQ2PS_YMMf32_MASKmskw_ZMMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VCVTUSI2SD_XMMf64_XMMf64_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SD_XMMf64_XMMf64_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SD_XMMf64_XMMf64_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SD_XMMf64_XMMf64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SH_XMMf16_XMMf16_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SH_XMMf16_XMMf16_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SH_XMMf16_XMMf16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SH_XMMf16_XMMf16_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SS_XMMf32_XMMf32_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SS_XMMf32_XMMf32_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SS_XMMf32_XMMf32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VCVTUSI2SS_XMMf32_XMMf32_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VCVTUW2PH_XMMf16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VCVTUW2PH_XMMf16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VCVTUW2PH_YMMf16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VCVTUW2PH_YMMf16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VCVTUW2PH_ZMMf16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VCVTUW2PH_ZMMf16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VCVTW2PH_XMMf16_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VCVTW2PH_XMMf16_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VCVTW2PH_YMMf16_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VCVTW2PH_YMMf16_MASKmskw_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VCVTW2PH_ZMMf16_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VCVTW2PH_ZMMf16_MASKmskw_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VDBPSADBW_XMMu16_MASKmskw_XMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VDBPSADBW_XMMu16_MASKmskw_XMMu8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VDBPSADBW_YMMu16_MASKmskw_YMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VDBPSADBW_YMMu16_MASKmskw_YMMu8_YMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VDBPSADBW_ZMMu16_MASKmskw_ZMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VDBPSADBW_ZMMu16_MASKmskw_ZMMu8_ZMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VDIVPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VDIVPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VDIVPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VDIVPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VDIVPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VDIVPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VDIVPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VDIVPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VDIVPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VDIVSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VDIVSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VDIVSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VDIVSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VDIVSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VDIVSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VDIVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VDPBF16PS_XMMf32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VDPBF16PS_XMMf32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VDPBF16PS_YMMf32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VDPBF16PS_YMMf32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VDPBF16PS_ZMMf32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VDPBF16PS_ZMMf32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VDPPD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VDPPD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VDPPS_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VDPPS_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VDPPS_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VDPPS_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VERR_GPR16,
    INTRINSIC_XED_IFORM_VERR_MEMw,
    INTRINSIC_XED_IFORM_VERW_GPR16,
    INTRINSIC_XED_IFORM_VERW_MEMw,
    INTRINSIC_XED_IFORM_VEXP2PD_ZMMf64_MASKmskw_MEMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VEXP2PD_ZMMf64_MASKmskw_ZMMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VEXP2PS_ZMMf32_MASKmskw_MEMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VEXP2PS_ZMMf32_MASKmskw_ZMMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VEXPANDPD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VEXPANDPS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF128_MEMdq_YMMdq_IMMb,
    INTRINSIC_XED_IFORM_VEXTRACTF128_XMMdq_YMMdq_IMMb,
    INTRINSIC_XED_IFORM_VEXTRACTF32X4_MEMf32_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF32X4_MEMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF32X4_XMMf32_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF32X4_XMMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF32X8_MEMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF32X8_YMMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF64X2_MEMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF64X2_MEMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF64X2_XMMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF64X2_XMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF64X4_MEMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTF64X4_YMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI128_MEMdq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VEXTRACTI128_XMMdq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VEXTRACTI32X4_MEMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI32X4_MEMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI32X4_XMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI32X4_XMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI32X8_MEMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI32X8_YMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI64X2_MEMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI64X2_MEMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI64X2_XMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI64X2_XMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI64X4_MEMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTI64X4_YMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTPS_GPR32_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VEXTRACTPS_GPR32f32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VEXTRACTPS_MEMd_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VEXTRACTPS_MEMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCPH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCPH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCPH_YMM2f16_MASKmskw_YMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCPH_YMM2f16_MASKmskw_YMM2f16_YMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCPH_ZMM2f16_MASKmskw_ZMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCPH_ZMM2f16_MASKmskw_ZMM2f16_ZMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCSH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMADDCSH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCPH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCPH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCPH_YMM2f16_MASKmskw_YMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCPH_YMM2f16_MASKmskw_YMM2f16_YMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCPH_ZMM2f16_MASKmskw_ZMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCPH_ZMM2f16_MASKmskw_ZMM2f16_ZMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCSH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFCMULCSH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPD_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPS_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMSD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMSD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMSS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFIXUPIMMSS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADD132PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADD132PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADD132PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADD132PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADD132PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADD132PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADD132PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADD132PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMADD132SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMADD132SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMADD132SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMADD132SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD132SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADD213PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADD213PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADD213PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADD213PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADD213PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADD213PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADD213PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADD213PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMADD213SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMADD213SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMADD213SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMADD213SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD213SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADD231PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADD231PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADD231PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADD231PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADD231PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADD231PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADD231PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADD231PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMADD231SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMADD231SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMADD231SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMADD231SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADD231SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCPH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCPH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCPH_YMM2f16_MASKmskw_YMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCPH_YMM2f16_MASKmskw_YMM2f16_YMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCPH_ZMM2f16_MASKmskw_ZMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCPH_ZMM2f16_MASKmskw_ZMM2f16_ZMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCSH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDCSH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDPD_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDPD_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDPS_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDPS_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSD_XMMdq_XMMq_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VFMADDSD_XMMdq_XMMq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMADDSD_XMMdq_XMMq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMADDSS_XMMdq_XMMd_MEMd_XMMd,
    INTRINSIC_XED_IFORM_VFMADDSS_XMMdq_XMMd_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMADDSS_XMMdq_XMMd_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB132PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB213PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUB231PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMADDSUBPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUBPD_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUBPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUBPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUBPD_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUBPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUBPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUBPS_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMADDSUBPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMADDSUBPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMADDSUBPS_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMADDSUBPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB132PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUB132PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUB132PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUB132PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB132PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUB132PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUB132PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUB132PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB132PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMSUB132SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMSUB132SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMSUB132SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMSUB132SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB132SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUB213PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUB213PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUB213PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB213PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUB213PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUB213PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUB213PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB213PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMSUB213SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMSUB213SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMSUB213SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMSUB213SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB213SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUB231PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUB231PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUB231PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB231PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUB231PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUB231PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUB231PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUB231PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMSUB231SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMSUB231SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMSUB231SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMSUB231SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUB231SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD132PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD213PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADD231PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFMSUBADDPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADDPD_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADDPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADDPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADDPD_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADDPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADDPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADDPS_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBADDPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBADDPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBADDPS_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBADDPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBPD_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBPD_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBPS_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFMSUBPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFMSUBPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBPS_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFMSUBPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFMSUBSD_XMMdq_XMMq_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VFMSUBSD_XMMdq_XMMq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFMSUBSD_XMMdq_XMMq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFMSUBSS_XMMdq_XMMd_MEMd_XMMd,
    INTRINSIC_XED_IFORM_VFMSUBSS_XMMdq_XMMd_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFMSUBSS_XMMdq_XMMd_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFMULCPH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCPH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCPH_YMM2f16_MASKmskw_YMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCPH_YMM2f16_MASKmskw_YMM2f16_YMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCPH_ZMM2f16_MASKmskw_ZMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCPH_ZMM2f16_MASKmskw_ZMM2f16_ZMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCSH_XMM2f16_MASKmskw_XMM2f16_MEM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFMULCSH_XMM2f16_MASKmskw_XMM2f16_XMM2f16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADD132PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADD132PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADD132PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADD132PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADD132PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADD132PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADD132PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADD132PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMADD132SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMADD132SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMADD132SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMADD132SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD132SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADD213PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADD213PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADD213PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADD213PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADD213PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADD213PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADD213PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADD213PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMADD213SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMADD213SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMADD213SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMADD213SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD213SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADD231PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADD231PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADD231PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADD231PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADD231PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADD231PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADD231PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADD231PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMADD231SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMADD231SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMADD231SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMADD231SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADD231SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMADDPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADDPD_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADDPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADDPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADDPD_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADDPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADDPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADDPS_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMADDPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMADDPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADDPS_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMADDPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMADDSD_XMMdq_XMMq_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMADDSD_XMMdq_XMMq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMADDSD_XMMdq_XMMq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMADDSS_XMMdq_XMMd_MEMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMADDSS_XMMdq_XMMd_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMADDSS_XMMdq_XMMd_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMSUB132SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMSUB132SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMSUB132SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMSUB132SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB132SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMSUB213SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMSUB213SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMSUB213SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMSUB213SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB213SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231SD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMSUB231SD_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMSUB231SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231SH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231SH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231SS_XMMdq_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMSUB231SS_XMMdq_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMSUB231SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUB231SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VFNMSUBPD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUBPD_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUBPD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUBPD_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUBPD_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUBPD_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUBPS_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUBPS_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFNMSUBPS_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFNMSUBPS_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUBPS_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFNMSUBPS_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFNMSUBSD_XMMdq_XMMq_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMSUBSD_XMMdq_XMMq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VFNMSUBSD_XMMdq_XMMq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VFNMSUBSS_XMMdq_XMMd_MEMd_XMMd,
    INTRINSIC_XED_IFORM_VFNMSUBSS_XMMdq_XMMd_XMMd_MEMd,
    INTRINSIC_XED_IFORM_VFNMSUBSS_XMMdq_XMMd_XMMd_XMMd,
    INTRINSIC_XED_IFORM_VFPCLASSPD_MASKmskw_MASKmskw_MEMf64_IMM8_AVX512_VL128,
    INTRINSIC_XED_IFORM_VFPCLASSPD_MASKmskw_MASKmskw_MEMf64_IMM8_AVX512_VL256,
    INTRINSIC_XED_IFORM_VFPCLASSPD_MASKmskw_MASKmskw_MEMf64_IMM8_AVX512_VL512,
    INTRINSIC_XED_IFORM_VFPCLASSPD_MASKmskw_MASKmskw_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPD_MASKmskw_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPD_MASKmskw_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPH_MASKmskw_MASKmskw_MEMf16_IMM8_AVX512_VL128,
    INTRINSIC_XED_IFORM_VFPCLASSPH_MASKmskw_MASKmskw_MEMf16_IMM8_AVX512_VL256,
    INTRINSIC_XED_IFORM_VFPCLASSPH_MASKmskw_MASKmskw_MEMf16_IMM8_AVX512_VL512,
    INTRINSIC_XED_IFORM_VFPCLASSPH_MASKmskw_MASKmskw_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPH_MASKmskw_MASKmskw_YMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPH_MASKmskw_MASKmskw_ZMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPS_MASKmskw_MASKmskw_MEMf32_IMM8_AVX512_VL128,
    INTRINSIC_XED_IFORM_VFPCLASSPS_MASKmskw_MASKmskw_MEMf32_IMM8_AVX512_VL256,
    INTRINSIC_XED_IFORM_VFPCLASSPS_MASKmskw_MASKmskw_MEMf32_IMM8_AVX512_VL512,
    INTRINSIC_XED_IFORM_VFPCLASSPS_MASKmskw_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPS_MASKmskw_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSPS_MASKmskw_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSSD_MASKmskw_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSSD_MASKmskw_MASKmskw_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSSH_MASKmskw_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSSH_MASKmskw_MASKmskw_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSSS_MASKmskw_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFPCLASSSS_MASKmskw_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VFRCZPD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFRCZPD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFRCZPD_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFRCZPD_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFRCZPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VFRCZPS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VFRCZPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VFRCZPS_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VFRCZSD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VFRCZSD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VFRCZSS_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VFRCZSS_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VGATHERDPD_XMMf64_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VGATHERDPD_XMMf64_MEMf64_XMMi64_VL128,
    INTRINSIC_XED_IFORM_VGATHERDPD_YMMf64_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VGATHERDPD_YMMf64_MEMf64_YMMi64_VL256,
    INTRINSIC_XED_IFORM_VGATHERDPD_ZMMf64_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VGATHERDPS_XMMf32_MASKmskw_MEMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VGATHERDPS_XMMf32_MEMf32_XMMi32_VL128,
    INTRINSIC_XED_IFORM_VGATHERDPS_YMMf32_MASKmskw_MEMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VGATHERDPS_YMMf32_MEMf32_YMMi32_VL256,
    INTRINSIC_XED_IFORM_VGATHERDPS_ZMMf32_MASKmskw_MEMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF0DPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF0DPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF0QPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF0QPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF1DPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF1DPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF1QPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERPF1QPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VGATHERQPD_XMMf64_MASKmskw_MEMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VGATHERQPD_XMMf64_MEMf64_XMMi64_VL128,
    INTRINSIC_XED_IFORM_VGATHERQPD_YMMf64_MASKmskw_MEMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VGATHERQPD_YMMf64_MEMf64_YMMi64_VL256,
    INTRINSIC_XED_IFORM_VGATHERQPD_ZMMf64_MASKmskw_MEMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VGATHERQPS_XMMf32_MASKmskw_MEMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VGATHERQPS_XMMf32_MASKmskw_MEMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VGATHERQPS_XMMf32_MEMf32_XMMi32_VL128,
    INTRINSIC_XED_IFORM_VGATHERQPS_XMMf32_MEMf32_XMMi32_VL256,
    INTRINSIC_XED_IFORM_VGATHERQPS_YMMf32_MASKmskw_MEMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VGETEXPPD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPH_XMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPH_XMMf16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPH_YMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPH_YMMf16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPH_ZMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPH_ZMMf16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPPS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETEXPSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPD_XMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPD_XMMf64_MASKmskw_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPD_YMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPD_YMMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPD_ZMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPD_ZMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPH_XMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPH_XMMf16_MASKmskw_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPH_YMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPH_YMMf16_MASKmskw_YMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPH_ZMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPH_ZMMf16_MASKmskw_ZMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPS_XMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPS_XMMf32_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPS_YMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPS_YMMf32_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPS_ZMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTPS_ZMMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTSD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTSD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTSH_XMMf16_MASKmskw_XMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTSH_XMMf16_MASKmskw_XMMf16_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTSS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGETMANTSS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_XMMu8_MASKmskw_XMMu8_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_XMMu8_MASKmskw_XMMu8_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_XMMu8_XMMu8_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_XMMu8_XMMu8_XMMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_YMMu8_MASKmskw_YMMu8_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_YMMu8_MASKmskw_YMMu8_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_YMMu8_YMMu8_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_YMMu8_YMMu8_YMMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_ZMMu8_MASKmskw_ZMMu8_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEINVQB_ZMMu8_MASKmskw_ZMMu8_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_XMMu8_MASKmskw_XMMu8_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_XMMu8_MASKmskw_XMMu8_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_XMMu8_XMMu8_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_XMMu8_XMMu8_XMMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_YMMu8_MASKmskw_YMMu8_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_YMMu8_MASKmskw_YMMu8_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_YMMu8_YMMu8_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_YMMu8_YMMu8_YMMu64_IMM8,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_ZMMu8_MASKmskw_ZMMu8_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8AFFINEQB_ZMMu8_MASKmskw_ZMMu8_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8MULB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8MULB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8MULB_XMMu8_XMMu8_MEMu8,
    INTRINSIC_XED_IFORM_VGF2P8MULB_XMMu8_XMMu8_XMMu8,
    INTRINSIC_XED_IFORM_VGF2P8MULB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8MULB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8MULB_YMMu8_YMMu8_MEMu8,
    INTRINSIC_XED_IFORM_VGF2P8MULB_YMMu8_YMMu8_YMMu8,
    INTRINSIC_XED_IFORM_VGF2P8MULB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VGF2P8MULB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VHADDPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VHADDPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VHADDPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VHADDPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VHADDPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VHADDPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VHADDPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VHADDPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VHSUBPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VHSUBPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VHSUBPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VHSUBPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VHSUBPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VHSUBPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VHSUBPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VHSUBPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VINSERTF128_YMMqq_YMMqq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VINSERTF128_YMMqq_YMMqq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VINSERTF32X4_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF32X4_YMMf32_MASKmskw_YMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF32X4_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF32X4_ZMMf32_MASKmskw_ZMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF32X8_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF32X8_ZMMf32_MASKmskw_ZMMf32_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF64X2_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF64X2_YMMf64_MASKmskw_YMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF64X2_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF64X2_ZMMf64_MASKmskw_ZMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF64X4_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTF64X4_ZMMf64_MASKmskw_ZMMf64_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI128_YMMqq_YMMqq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VINSERTI128_YMMqq_YMMqq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VINSERTI32X4_YMMu32_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI32X4_YMMu32_MASKmskw_YMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI32X4_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI32X4_ZMMu32_MASKmskw_ZMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI32X8_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI32X8_ZMMu32_MASKmskw_ZMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI64X2_YMMu64_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI64X2_YMMu64_MASKmskw_YMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI64X2_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI64X2_ZMMu64_MASKmskw_ZMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI64X4_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTI64X4_ZMMu64_MASKmskw_ZMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTPS_XMMdq_XMMdq_MEMd_IMMb,
    INTRINSIC_XED_IFORM_VINSERTPS_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VINSERTPS_XMMf32_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VINSERTPS_XMMf32_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VLDDQU_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VLDDQU_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VLDMXCSR_MEMd,
    INTRINSIC_XED_IFORM_VMASKMOVDQU_XMMxub_XMMxub,
    INTRINSIC_XED_IFORM_VMASKMOVPD_MEMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMASKMOVPD_MEMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMASKMOVPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMASKMOVPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMASKMOVPS_MEMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMASKMOVPS_MEMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMASKMOVPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMASKMOVPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMAXPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMAXPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMAXPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMAXPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMAXPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMAXPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMAXPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMAXPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMAXPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMAXSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VMAXSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMAXSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMAXSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VMAXSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VMAXSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMAXSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMCALL,
    INTRINSIC_XED_IFORM_VMCLEAR_MEMq,
    INTRINSIC_XED_IFORM_VMFUNC,
    INTRINSIC_XED_IFORM_VMINPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMINPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMINPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMINPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMINPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMINPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMINPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMINPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMINPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMINSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VMINSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMINSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMINSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VMINSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VMINSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMINSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMLAUNCH,
    INTRINSIC_XED_IFORM_VMLOAD_ArAX,
    INTRINSIC_XED_IFORM_VMMCALL,
    INTRINSIC_XED_IFORM_VMOVAPD_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVAPD_MEMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_MEMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_MEMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVAPD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVAPD_XMMdq_XMMdq_28,
    INTRINSIC_XED_IFORM_VMOVAPD_XMMdq_XMMdq_29,
    INTRINSIC_XED_IFORM_VMOVAPD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVAPD_YMMqq_YMMqq_28,
    INTRINSIC_XED_IFORM_VMOVAPD_YMMqq_YMMqq_29,
    INTRINSIC_XED_IFORM_VMOVAPD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVAPS_MEMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_MEMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_MEMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVAPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVAPS_XMMdq_XMMdq_28,
    INTRINSIC_XED_IFORM_VMOVAPS_XMMdq_XMMdq_29,
    INTRINSIC_XED_IFORM_VMOVAPS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVAPS_YMMqq_YMMqq_28,
    INTRINSIC_XED_IFORM_VMOVAPS_YMMqq_YMMqq_29,
    INTRINSIC_XED_IFORM_VMOVAPS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVAPS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVD_GPR32d_XMMd,
    INTRINSIC_XED_IFORM_VMOVD_GPR32u32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVD_MEMd_XMMd,
    INTRINSIC_XED_IFORM_VMOVD_MEMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVD_XMMdq_GPR32d,
    INTRINSIC_XED_IFORM_VMOVD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VMOVD_XMMu32_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VMOVD_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDDUP_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMOVDDUP_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VMOVDDUP_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDDUP_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDDUP_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDDUP_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDDUP_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVDDUP_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVDDUP_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDDUP_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVDQA_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVDQA_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVDQA_XMMdq_XMMdq_6F,
    INTRINSIC_XED_IFORM_VMOVDQA_XMMdq_XMMdq_7F,
    INTRINSIC_XED_IFORM_VMOVDQA_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVDQA_YMMqq_YMMqq_6F,
    INTRINSIC_XED_IFORM_VMOVDQA_YMMqq_YMMqq_7F,
    INTRINSIC_XED_IFORM_VMOVDQA32_MEMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_MEMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_MEMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA32_ZMMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_MEMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_MEMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_MEMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQA64_ZMMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVDQU_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVDQU_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVDQU_XMMdq_XMMdq_6F,
    INTRINSIC_XED_IFORM_VMOVDQU_XMMdq_XMMdq_7F,
    INTRINSIC_XED_IFORM_VMOVDQU_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVDQU_YMMqq_YMMqq_6F,
    INTRINSIC_XED_IFORM_VMOVDQU_YMMqq_YMMqq_7F,
    INTRINSIC_XED_IFORM_VMOVDQU16_MEMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_MEMu16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_MEMu16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_XMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_XMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_YMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_YMMu16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_ZMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU16_ZMMu16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_MEMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_MEMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_MEMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU32_ZMMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_MEMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_MEMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_MEMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU64_ZMMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_MEMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_MEMu8_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_MEMu8_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_XMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_XMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_YMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_YMMu8_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_ZMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVDQU8_ZMMu8_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VMOVHLPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVHLPS_XMMf32_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVHPD_MEMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVHPD_MEMq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVHPD_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VMOVHPD_XMMf64_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVHPS_MEMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVHPS_MEMq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVHPS_XMMdq_XMMq_MEMq,
    INTRINSIC_XED_IFORM_VMOVHPS_XMMf32_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVLHPS_XMMdq_XMMq_XMMq,
    INTRINSIC_XED_IFORM_VMOVLHPS_XMMf32_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVLPD_MEMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVLPD_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VMOVLPD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMOVLPD_XMMf64_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVLPS_MEMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVLPS_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VMOVLPS_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMOVLPS_XMMf32_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVMSKPD_GPR32d_XMMdq,
    INTRINSIC_XED_IFORM_VMOVMSKPD_GPR32d_YMMqq,
    INTRINSIC_XED_IFORM_VMOVMSKPS_GPR32d_XMMdq,
    INTRINSIC_XED_IFORM_VMOVMSKPS_GPR32d_YMMqq,
    INTRINSIC_XED_IFORM_VMOVNTDQ_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVNTDQ_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVNTDQ_MEMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTDQ_MEMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTDQ_MEMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTDQA_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVNTDQA_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTDQA_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVNTDQA_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTDQA_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPD_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVNTPD_MEMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPD_MEMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPD_MEMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPD_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVNTPS_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVNTPS_MEMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPS_MEMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPS_MEMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVNTPS_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVQ_GPR64q_XMMq,
    INTRINSIC_XED_IFORM_VMOVQ_GPR64u64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVQ_MEMq_XMMq_7E,
    INTRINSIC_XED_IFORM_VMOVQ_MEMq_XMMq_D6,
    INTRINSIC_XED_IFORM_VMOVQ_MEMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVQ_XMMdq_GPR64q,
    INTRINSIC_XED_IFORM_VMOVQ_XMMdq_MEMq_6E,
    INTRINSIC_XED_IFORM_VMOVQ_XMMdq_MEMq_7E,
    INTRINSIC_XED_IFORM_VMOVQ_XMMdq_XMMq_7E,
    INTRINSIC_XED_IFORM_VMOVQ_XMMdq_XMMq_D6,
    INTRINSIC_XED_IFORM_VMOVQ_XMMu64_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VMOVQ_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVQ_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VMOVSD_MEMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVSD_MEMq_XMMq,
    INTRINSIC_XED_IFORM_VMOVSD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMOVSD_XMMdq_XMMdq_XMMq_10,
    INTRINSIC_XED_IFORM_VMOVSD_XMMdq_XMMdq_XMMq_11,
    INTRINSIC_XED_IFORM_VMOVSD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVSH_MEMf16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMOVSH_XMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMOVSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMOVSHDUP_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVSHDUP_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVSHDUP_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSHDUP_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSHDUP_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSHDUP_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSHDUP_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVSHDUP_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVSHDUP_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSHDUP_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSLDUP_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVSLDUP_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVSLDUP_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSLDUP_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSLDUP_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSLDUP_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSLDUP_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVSLDUP_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVSLDUP_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSLDUP_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSS_MEMd_XMMd,
    INTRINSIC_XED_IFORM_VMOVSS_MEMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSS_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VMOVSS_XMMdq_XMMdq_XMMd_10,
    INTRINSIC_XED_IFORM_VMOVSS_XMMdq_XMMdq_XMMd_11,
    INTRINSIC_XED_IFORM_VMOVSS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVUPD_MEMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_MEMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_MEMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVUPD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVUPD_XMMdq_XMMdq_10,
    INTRINSIC_XED_IFORM_VMOVUPD_XMMdq_XMMdq_11,
    INTRINSIC_XED_IFORM_VMOVUPD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVUPD_YMMqq_YMMqq_10,
    INTRINSIC_XED_IFORM_VMOVUPD_YMMqq_YMMqq_11,
    INTRINSIC_XED_IFORM_VMOVUPD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMOVUPS_MEMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_MEMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_MEMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMOVUPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMOVUPS_XMMdq_XMMdq_10,
    INTRINSIC_XED_IFORM_VMOVUPS_XMMdq_XMMdq_11,
    INTRINSIC_XED_IFORM_VMOVUPS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMOVUPS_YMMqq_YMMqq_10,
    INTRINSIC_XED_IFORM_VMOVUPS_YMMqq_YMMqq_11,
    INTRINSIC_XED_IFORM_VMOVUPS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVUPS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMOVW_GPR32f16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMOVW_MEMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMOVW_XMMf16_GPR32f16_AVX512,
    INTRINSIC_XED_IFORM_VMOVW_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMPSADBW_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VMPSADBW_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VMPSADBW_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VMPSADBW_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VMPTRLD_MEMq,
    INTRINSIC_XED_IFORM_VMPTRST_MEMq,
    INTRINSIC_XED_IFORM_VMREAD_GPR32_GPR32,
    INTRINSIC_XED_IFORM_VMREAD_GPR64_GPR64,
    INTRINSIC_XED_IFORM_VMREAD_MEMd_GPR32,
    INTRINSIC_XED_IFORM_VMREAD_MEMq_GPR64,
    INTRINSIC_XED_IFORM_VMRESUME,
    INTRINSIC_XED_IFORM_VMRUN_ArAX,
    INTRINSIC_XED_IFORM_VMSAVE,
    INTRINSIC_XED_IFORM_VMULPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMULPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMULPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMULPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMULPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VMULPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VMULPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VMULPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VMULPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VMULSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VMULSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VMULSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VMULSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VMULSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VMULSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VMULSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VMWRITE_GPR32_GPR32,
    INTRINSIC_XED_IFORM_VMWRITE_GPR32_MEMd,
    INTRINSIC_XED_IFORM_VMWRITE_GPR64_GPR64,
    INTRINSIC_XED_IFORM_VMWRITE_GPR64_MEMq,
    INTRINSIC_XED_IFORM_VMXOFF,
    INTRINSIC_XED_IFORM_VMXON_MEMq,
    INTRINSIC_XED_IFORM_VORPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VORPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VORPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VORPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VORPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VORPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VORPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VORPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTD_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTD_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTD_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTD_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTD_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTD_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTQ_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTQ_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTQ_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTQ_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTQ_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VP2INTERSECTQ_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VP4DPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VP4DPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPABSB_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPABSB_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPABSB_XMMi8_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPABSB_XMMi8_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPABSB_YMMi8_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPABSB_YMMi8_MASKmskw_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPABSB_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPABSB_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPABSB_ZMMi8_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPABSB_ZMMi8_MASKmskw_ZMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPABSD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPABSD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPABSD_XMMi32_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPABSD_XMMi32_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPABSD_YMMi32_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPABSD_YMMi32_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPABSD_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPABSD_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPABSD_ZMMi32_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPABSD_ZMMi32_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPABSQ_XMMi64_MASKmskw_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPABSQ_XMMi64_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPABSQ_YMMi64_MASKmskw_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPABSQ_YMMi64_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPABSQ_ZMMi64_MASKmskw_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPABSQ_ZMMi64_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPABSW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPABSW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPABSW_XMMi16_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPABSW_XMMi16_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPABSW_YMMi16_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPABSW_YMMi16_MASKmskw_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPABSW_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPABSW_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPABSW_ZMMi16_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPABSW_ZMMi16_MASKmskw_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSDW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPACKSSDW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPACKSSDW_XMMi16_MASKmskw_XMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSDW_XMMi16_MASKmskw_XMMi32_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSDW_YMMi16_MASKmskw_YMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSDW_YMMi16_MASKmskw_YMMi32_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSDW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPACKSSDW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSDW_ZMMi16_MASKmskw_ZMMi32_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSWB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPACKSSWB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPACKSSWB_XMMi8_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSWB_XMMi8_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSWB_YMMi8_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSWB_YMMi8_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSWB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPACKSSWB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKSSWB_ZMMi8_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSDW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPACKUSDW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPACKUSDW_XMMu16_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSDW_XMMu16_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSDW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPACKUSDW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPACKUSDW_YMMu16_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSDW_YMMu16_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSDW_ZMMu16_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSWB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPACKUSWB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPACKUSWB_XMMu8_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSWB_XMMu8_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSWB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPACKUSWB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPACKUSWB_YMMu8_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSWB_YMMu8_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPACKUSWB_ZMMu8_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPADDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPADDD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPADDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPADDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPADDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPADDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPADDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPADDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPADDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPADDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPADDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPADDSB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDSB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPADDSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPADDSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPADDSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPADDSB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDSB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPADDSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPADDSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPADDSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPADDSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPADDSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPADDSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPADDSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDUSB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDUSB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDUSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDUSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPADDW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPADDW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPADDW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPADDW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPADDW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPALIGNR_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPALIGNR_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPALIGNR_XMMu8_MASKmskw_XMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPALIGNR_XMMu8_MASKmskw_XMMu8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPALIGNR_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPALIGNR_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPALIGNR_YMMu8_MASKmskw_YMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPALIGNR_YMMu8_MASKmskw_YMMu8_YMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPALIGNR_ZMMu8_MASKmskw_ZMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPALIGNR_ZMMu8_MASKmskw_ZMMu8_ZMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPAND_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPAND_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPAND_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPAND_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPANDD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDN_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPANDN_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPANDN_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPANDN_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPANDND_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDND_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDND_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDND_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDND_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDND_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPANDNQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDNQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDNQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDNQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDNQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDNQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPANDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPAVGB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPAVGB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPAVGB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPAVGB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPAVGB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPAVGB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPAVGB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPAVGB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPAVGB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPAVGB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPAVGW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPAVGW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPAVGW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPAVGW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPAVGW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPAVGW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPAVGW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPAVGW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPAVGW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPAVGW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDD_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDD_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDMB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDMW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBLENDVB_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPBLENDVB_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPBLENDVB_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPBLENDVB_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPBLENDW_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDW_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDW_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPBLENDW_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPBROADCASTB_XMMdq_MEMb,
    INTRINSIC_XED_IFORM_VPBROADCASTB_XMMdq_XMMb,
    INTRINSIC_XED_IFORM_VPBROADCASTB_XMMu8_MASKmskw_GPR32u8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_XMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_XMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_YMMqq_MEMb,
    INTRINSIC_XED_IFORM_VPBROADCASTB_YMMqq_XMMb,
    INTRINSIC_XED_IFORM_VPBROADCASTB_YMMu8_MASKmskw_GPR32u8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_YMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_YMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_ZMMu8_MASKmskw_GPR32u8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_ZMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTB_ZMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VPBROADCASTD_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VPBROADCASTD_XMMu32_MASKmskw_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_YMMqq_MEMd,
    INTRINSIC_XED_IFORM_VPBROADCASTD_YMMqq_XMMd,
    INTRINSIC_XED_IFORM_VPBROADCASTD_YMMu32_MASKmskw_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_YMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_ZMMu32_MASKmskw_GPR32u32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTD_ZMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTMB2Q_XMMu64_MASKu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTMB2Q_YMMu64_MASKu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTMB2Q_ZMMu64_MASKu64_AVX512CD,
    INTRINSIC_XED_IFORM_VPBROADCASTMW2D_XMMu32_MASKu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTMW2D_YMMu32_MASKu32_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTMW2D_ZMMu32_MASKu32_AVX512CD,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_XMMu64_MASKmskw_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_YMMqq_MEMq,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_YMMu64_MASKmskw_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_YMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_ZMMu64_MASKmskw_GPR64u64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTQ_ZMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_XMMdq_MEMw,
    INTRINSIC_XED_IFORM_VPBROADCASTW_XMMdq_XMMw,
    INTRINSIC_XED_IFORM_VPBROADCASTW_XMMu16_MASKmskw_GPR32u16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_XMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_XMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_YMMqq_MEMw,
    INTRINSIC_XED_IFORM_VPBROADCASTW_YMMqq_XMMw,
    INTRINSIC_XED_IFORM_VPBROADCASTW_YMMu16_MASKmskw_GPR32u16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_YMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_YMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_ZMMu16_MASKmskw_GPR32u16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_ZMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPBROADCASTW_ZMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_XMMu128_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_XMMu128_XMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_YMMu128_YMMu64_MEMu64_IMM8,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_YMMu128_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_YMMu128_YMMu64_YMMu64_IMM8,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_YMMu128_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_ZMMu128_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCLMULQDQ_ZMMu128_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMOV_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMOV_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMOV_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMOV_YMMqq_YMMqq_MEMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMOV_YMMqq_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMOV_YMMqq_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPB_MASKmskw_MASKmskw_XMMi8_MEMi8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPB_MASKmskw_MASKmskw_XMMi8_XMMi8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPB_MASKmskw_MASKmskw_YMMi8_MEMi8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPB_MASKmskw_MASKmskw_YMMi8_YMMi8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPB_MASKmskw_MASKmskw_ZMMi8_MEMi8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPB_MASKmskw_MASKmskw_ZMMi8_ZMMi8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPD_MASKmskw_MASKmskw_XMMi32_MEMi32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPD_MASKmskw_MASKmskw_XMMi32_XMMi32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPD_MASKmskw_MASKmskw_YMMi32_MEMi32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPD_MASKmskw_MASKmskw_YMMi32_YMMi32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPD_MASKmskw_MASKmskw_ZMMi32_MEMi32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPD_MASKmskw_MASKmskw_ZMMi32_ZMMi32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPEQB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPEQB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPEQB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPEQD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPEQD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPEQD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPEQD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPEQQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPEQQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPEQQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPEQW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPEQW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPEQW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPEQW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPEQW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPESTRI_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRI_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRI64_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRI64_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRM_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRM_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRM64_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPESTRM64_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPGTB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPGTB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPGTB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPGTB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPGTD_MASKmskw_MASKmskw_XMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTD_MASKmskw_MASKmskw_XMMi32_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTD_MASKmskw_MASKmskw_YMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTD_MASKmskw_MASKmskw_YMMi32_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTD_MASKmskw_MASKmskw_ZMMi32_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPGTD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPGTD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPGTD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTQ_MASKmskw_MASKmskw_XMMi64_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTQ_MASKmskw_MASKmskw_YMMi64_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTQ_MASKmskw_MASKmskw_ZMMi64_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPGTQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPGTQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPGTQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPGTW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCMPGTW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPCMPGTW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPCMPGTW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPCMPGTW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPCMPISTRI_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPISTRI_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPISTRI64_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPISTRI64_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPISTRM_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPISTRM_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCMPQ_MASKmskw_MASKmskw_XMMi64_MEMi64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPQ_MASKmskw_MASKmskw_XMMi64_XMMi64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPQ_MASKmskw_MASKmskw_YMMi64_MEMi64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPQ_MASKmskw_MASKmskw_YMMi64_YMMi64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPQ_MASKmskw_MASKmskw_ZMMi64_MEMi64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPQ_MASKmskw_MASKmskw_ZMMi64_ZMMi64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUB_MASKmskw_MASKmskw_XMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUB_MASKmskw_MASKmskw_XMMu8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUB_MASKmskw_MASKmskw_YMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUB_MASKmskw_MASKmskw_YMMu8_YMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUB_MASKmskw_MASKmskw_ZMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUD_MASKmskw_MASKmskw_XMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUD_MASKmskw_MASKmskw_XMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUD_MASKmskw_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUD_MASKmskw_MASKmskw_YMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUD_MASKmskw_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUQ_MASKmskw_MASKmskw_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUQ_MASKmskw_MASKmskw_XMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUQ_MASKmskw_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUQ_MASKmskw_MASKmskw_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUW_MASKmskw_MASKmskw_XMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUW_MASKmskw_MASKmskw_XMMu16_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUW_MASKmskw_MASKmskw_YMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUW_MASKmskw_MASKmskw_YMMu16_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUW_MASKmskw_MASKmskw_ZMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPUW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPW_MASKmskw_MASKmskw_XMMi16_MEMi16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPW_MASKmskw_MASKmskw_XMMi16_XMMi16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPW_MASKmskw_MASKmskw_YMMi16_MEMi16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPW_MASKmskw_MASKmskw_YMMi16_YMMi16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPW_MASKmskw_MASKmskw_ZMMi16_MEMi16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCMPW_MASKmskw_MASKmskw_ZMMi16_ZMMi16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMB_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMB_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMPRESSB_MEMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSB_MEMu8_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSB_MEMu8_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSB_XMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSB_YMMu8_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSB_ZMMu8_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSD_MEMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSD_MEMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSD_MEMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSD_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSD_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSD_ZMMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSQ_MEMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSQ_MEMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSQ_MEMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSQ_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSQ_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSQ_ZMMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSW_MEMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSW_MEMu16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSW_MEMu16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSW_XMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSW_YMMu16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCOMPRESSW_ZMMu16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPCOMQ_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMQ_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUB_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUB_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUQ_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUQ_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUW_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMUW_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMW_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCOMW_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPCONFLICTD_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTD_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTD_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTD_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTD_ZMMu32_MASKmskw_MEMu32_AVX512CD,
    INTRINSIC_XED_IFORM_VPCONFLICTD_ZMMu32_MASKmskw_ZMMu32_AVX512CD,
    INTRINSIC_XED_IFORM_VPCONFLICTQ_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTQ_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTQ_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTQ_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPCONFLICTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD,
    INTRINSIC_XED_IFORM_VPCONFLICTQ_ZMMu64_MASKmskw_ZMMu64_AVX512CD,
    INTRINSIC_XED_IFORM_VPDPBSSD_XMMi32_XMM4i8_MEM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSD_XMMi32_XMM4i8_XMM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSD_YMMi32_YMM4i8_MEM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSD_YMMi32_YMM4i8_YMM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSDS_XMMi32_XMM4i8_MEM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSDS_XMMi32_XMM4i8_XMM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSDS_YMMi32_YMM4i8_MEM4i8,
    INTRINSIC_XED_IFORM_VPDPBSSDS_YMMi32_YMM4i8_YMM4i8,
    INTRINSIC_XED_IFORM_VPDPBSUD_XMMi32_XMM4i8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUD_XMMi32_XMM4i8_XMM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUD_YMMi32_YMM4i8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUD_YMMi32_YMM4i8_YMM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUDS_XMMi32_XMM4i8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUDS_XMMi32_XMM4i8_XMM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUDS_YMMi32_YMM4i8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBSUDS_YMMi32_YMM4i8_YMM4u8,
    INTRINSIC_XED_IFORM_VPDPBUSD_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSD_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSD_XMMi32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPBUSD_XMMi32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VPDPBUSD_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSD_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSD_YMMi32_YMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPBUSD_YMMi32_YMMu32_YMMu32,
    INTRINSIC_XED_IFORM_VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSD_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSDS_XMMi32_MASKmskw_XMMu8_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSDS_XMMi32_MASKmskw_XMMu8_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSDS_XMMi32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPBUSDS_XMMi32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VPDPBUSDS_YMMi32_MASKmskw_YMMu8_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSDS_YMMi32_MASKmskw_YMMu8_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSDS_YMMi32_YMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPBUSDS_YMMi32_YMMu32_YMMu32,
    INTRINSIC_XED_IFORM_VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUSDS_ZMMi32_MASKmskw_ZMMu8_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPBUUD_XMMu32_XMM4u8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUD_XMMu32_XMM4u8_XMM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUD_YMMu32_YMM4u8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUD_YMMu32_YMM4u8_YMM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUDS_XMMu32_XMM4u8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUDS_XMMu32_XMM4u8_XMM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUDS_YMMu32_YMM4u8_MEM4u8,
    INTRINSIC_XED_IFORM_VPDPBUUDS_YMMu32_YMM4u8_YMM4u8,
    INTRINSIC_XED_IFORM_VPDPWSSD_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSD_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSD_XMMi32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPWSSD_XMMi32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VPDPWSSD_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSD_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSD_YMMi32_YMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPWSSD_YMMi32_YMMu32_YMMu32,
    INTRINSIC_XED_IFORM_VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSD_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSDS_XMMi32_MASKmskw_XMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSDS_XMMi32_MASKmskw_XMMi16_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSDS_XMMi32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPWSSDS_XMMi32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VPDPWSSDS_YMMi32_MASKmskw_YMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSDS_YMMi32_MASKmskw_YMMi16_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSDS_YMMi32_YMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VPDPWSSDS_YMMi32_YMMu32_YMMu32,
    INTRINSIC_XED_IFORM_VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSSDS_ZMMi32_MASKmskw_ZMMi16_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPDPWSUD_XMMi32_XMM2i16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUD_XMMi32_XMM2i16_XMM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUD_YMMi32_YMM2i16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUD_YMMi32_YMM2i16_YMM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUDS_XMMi32_XMM2i16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUDS_XMMi32_XMM2i16_XMM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUDS_YMMi32_YMM2i16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWSUDS_YMMi32_YMM2i16_YMM2u16,
    INTRINSIC_XED_IFORM_VPDPWUSD_XMMi32_XMM2u16_MEM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSD_XMMi32_XMM2u16_XMM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSD_YMMi32_YMM2u16_MEM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSD_YMMi32_YMM2u16_YMM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSDS_XMMi32_XMM2u16_MEM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSDS_XMMi32_XMM2u16_XMM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSDS_YMMi32_YMM2u16_MEM2i16,
    INTRINSIC_XED_IFORM_VPDPWUSDS_YMMi32_YMM2u16_YMM2i16,
    INTRINSIC_XED_IFORM_VPDPWUUD_XMMu32_XMM2u16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUD_XMMu32_XMM2u16_XMM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUD_YMMu32_YMM2u16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUD_YMMu32_YMM2u16_YMM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUDS_XMMu32_XMM2u16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUDS_XMMu32_XMM2u16_XMM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUDS_YMMu32_YMM2u16_MEM2u16,
    INTRINSIC_XED_IFORM_VPDPWUUDS_YMMu32_YMM2u16_YMM2u16,
    INTRINSIC_XED_IFORM_VPERM2F128_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERM2F128_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERM2I128_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERM2I128_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPERMD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPERMD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2B_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2B_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2B_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2B_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2B_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2B_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2D_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2D_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2D_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2D_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2D_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2D_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2Q_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2Q_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2Q_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2Q_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2Q_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2Q_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2W_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2W_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2W_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2W_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2W_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMI2W_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMIL2PD_XMMdq_XMMdq_MEMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PD_XMMdq_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PD_XMMdq_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PD_YMMqq_YMMqq_MEMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PD_YMMqq_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PD_YMMqq_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PS_XMMdq_XMMdq_MEMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PS_XMMdq_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PS_XMMdq_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PS_YMMqq_YMMqq_MEMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PS_YMMqq_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMIL2PS_YMMqq_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMf64_MASKmskw_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPERMILPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPERMILPD_ZMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_ZMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMf32_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMf32_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPERMILPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPERMILPS_ZMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_ZMMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMILPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_YMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_YMMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMPD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMPD_ZMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_ZMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPERMPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPERMPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMQ_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPERMQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2B_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2B_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2B_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2B_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2B_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2B_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2D_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2D_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2D_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2D_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2D_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2D_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2PS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2Q_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2Q_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2Q_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2Q_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2Q_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2Q_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2W_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2W_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2W_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2W_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2W_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMT2W_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPERMW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDB_XMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDB_XMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDB_YMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDB_YMMu8_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDB_ZMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDB_ZMMu8_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDD_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDD_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDD_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDD_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDD_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDD_ZMMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDQ_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDQ_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDQ_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDQ_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDQ_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDQ_ZMMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDW_XMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDW_XMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDW_YMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDW_YMMu16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDW_ZMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXPANDW_ZMMu16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRB_GPR32d_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRB_GPR32u8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRB_MEMb_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRB_MEMu8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRD_GPR32d_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRD_GPR32u32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRD_MEMd_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRD_MEMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRQ_GPR64q_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRQ_GPR64u64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRQ_MEMq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRQ_MEMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRW_GPR32d_XMMdq_IMMb_15,
    INTRINSIC_XED_IFORM_VPEXTRW_GPR32d_XMMdq_IMMb_C5,
    INTRINSIC_XED_IFORM_VPEXTRW_GPR32u16_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRW_MEMu16_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPEXTRW_MEMw_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPEXTRW_GPR32u16_XMMu16_IMM8_AVX512_C5,
    INTRINSIC_XED_IFORM_VPGATHERDD_XMMu32_MASKmskw_MEMu32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPGATHERDD_XMMu32_MEMd_XMMi32_VL128,
    INTRINSIC_XED_IFORM_VPGATHERDD_YMMu32_MASKmskw_MEMu32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPGATHERDD_YMMu32_MEMd_YMMi32_VL256,
    INTRINSIC_XED_IFORM_VPGATHERDD_ZMMu32_MASKmskw_MEMu32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPGATHERDQ_XMMu64_MASKmskw_MEMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPGATHERDQ_XMMu64_MEMq_XMMi64_VL128,
    INTRINSIC_XED_IFORM_VPGATHERDQ_YMMu64_MASKmskw_MEMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPGATHERDQ_YMMu64_MEMq_YMMi64_VL256,
    INTRINSIC_XED_IFORM_VPGATHERDQ_ZMMu64_MASKmskw_MEMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPGATHERQD_XMMu32_MASKmskw_MEMu32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPGATHERQD_XMMu32_MASKmskw_MEMu32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPGATHERQD_XMMu32_MEMd_XMMi32_VL128,
    INTRINSIC_XED_IFORM_VPGATHERQD_XMMu32_MEMd_XMMi32_VL256,
    INTRINSIC_XED_IFORM_VPGATHERQD_YMMu32_MASKmskw_MEMu32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPGATHERQQ_XMMu64_MASKmskw_MEMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPGATHERQQ_XMMu64_MEMq_XMMi64_VL128,
    INTRINSIC_XED_IFORM_VPGATHERQQ_YMMu64_MASKmskw_MEMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPGATHERQQ_YMMu64_MEMq_YMMi64_VL256,
    INTRINSIC_XED_IFORM_VPGATHERQQ_ZMMu64_MASKmskw_MEMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPHADDBD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDBD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDBQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDBQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDBW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPHADDD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPHADDDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDDQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPHADDSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPHADDUBD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDUBD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDUBQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDUBQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDUBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDUBW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDUDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDUDQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDUWD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDUWD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDUWQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDUWQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPHADDW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPHADDWD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDWD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHADDWQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHADDWQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHMINPOSUW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHMINPOSUW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHSUBBW_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHSUBBW_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHSUBD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHSUBD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHSUBD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPHSUBD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPHSUBDQ_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHSUBDQ_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHSUBSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHSUBSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHSUBSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPHSUBSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPHSUBW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHSUBW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPHSUBW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPHSUBW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPHSUBWD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPHSUBWD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPINSRB_XMMdq_XMMdq_GPR32d_IMMb,
    INTRINSIC_XED_IFORM_VPINSRB_XMMdq_XMMdq_MEMb_IMMb,
    INTRINSIC_XED_IFORM_VPINSRB_XMMu8_XMMu8_GPR32u8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRB_XMMu8_XMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRD_XMMdq_XMMdq_GPR32d_IMMb,
    INTRINSIC_XED_IFORM_VPINSRD_XMMdq_XMMdq_MEMd_IMMb,
    INTRINSIC_XED_IFORM_VPINSRD_XMMu32_XMMu32_GPR32u32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRD_XMMu32_XMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRQ_XMMdq_XMMdq_GPR64q_IMMb,
    INTRINSIC_XED_IFORM_VPINSRQ_XMMdq_XMMdq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_VPINSRQ_XMMu64_XMMu64_GPR64u64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRQ_XMMu64_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRW_XMMdq_XMMdq_GPR32d_IMMb,
    INTRINSIC_XED_IFORM_VPINSRW_XMMdq_XMMdq_MEMw_IMMb,
    INTRINSIC_XED_IFORM_VPINSRW_XMMu16_XMMu16_GPR32u16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPINSRW_XMMu16_XMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTD_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTD_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTD_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTD_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTD_ZMMu32_MASKmskw_MEMu32_AVX512CD,
    INTRINSIC_XED_IFORM_VPLZCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512CD,
    INTRINSIC_XED_IFORM_VPLZCNTQ_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTQ_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTQ_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTQ_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPLZCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512CD,
    INTRINSIC_XED_IFORM_VPLZCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512CD,
    INTRINSIC_XED_IFORM_VPMACSDD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSDD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSDQH_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSDQH_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSDQL_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSDQL_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSDD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSDD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSDQH_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSDQH_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSDQL_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSDQL_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSWD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSWD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSWW_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSSWW_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSWD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSWD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSWW_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMACSWW_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADCSSWD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADCSSWD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADCSWD_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADCSWD_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_XMMu64_XMMu64_MEMu64,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_XMMu64_XMMu64_XMMu64,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_YMMu64_YMMu64_MEMu64,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_YMMu64_YMMu64_YMMu64,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52HUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_XMMu64_XMMu64_MEMu64,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_XMMu64_XMMu64_XMMu64,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_YMMu64_YMMu64_MEMu64,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_YMMu64_YMMu64_YMMu64,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADD52LUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMADDUBSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMADDUBSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADDUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDUBSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMADDUBSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDWD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMADDWD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMADDWD_XMMi32_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDWD_XMMi32_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDWD_YMMi32_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDWD_YMMi32_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDWD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMADDWD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMADDWD_ZMMi32_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMADDWD_ZMMi32_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMASKMOVD_MEMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMASKMOVD_MEMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMASKMOVD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMASKMOVD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMASKMOVQ_MEMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMASKMOVQ_MEMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMASKMOVQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMASKMOVQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXSB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMAXSB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMAXSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXSB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMAXSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMAXSD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMAXSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXSD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMAXSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMAXSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMAXSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMAXSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMAXUB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMAXUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXUB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMAXUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMAXUD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMAXUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXUD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMAXUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMAXUW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMAXUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMAXUW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMAXUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMAXUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMINSB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMINSB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMINSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMINSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMINSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMINSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMINSB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMINSB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMINSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMINSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMINSD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMINSD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMINSD_XMMi32_MASKmskw_XMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMINSD_XMMi32_MASKmskw_XMMi32_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMINSD_YMMi32_MASKmskw_YMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMINSD_YMMi32_MASKmskw_YMMi32_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMINSD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMINSD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMINSD_ZMMi32_MASKmskw_ZMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMINSD_ZMMi32_MASKmskw_ZMMi32_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMINSQ_XMMi64_MASKmskw_XMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMINSQ_XMMi64_MASKmskw_XMMi64_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMINSQ_YMMi64_MASKmskw_YMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMINSQ_YMMi64_MASKmskw_YMMi64_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMINSQ_ZMMi64_MASKmskw_ZMMi64_MEMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMINSQ_ZMMi64_MASKmskw_ZMMi64_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMINSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMINSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMINSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMINSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMINSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMINSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMINSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMINSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMINSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMINSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMINUB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMINUB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMINUB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMINUB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMINUB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMINUB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMINUB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMINUB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMINUB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMINUB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMINUD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMINUD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMINUD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMINUD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMINUD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMINUD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMINUD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMINUD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMINUD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMINUD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMINUQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMINUQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMINUQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMINUQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMINUQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMINUQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMINUW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMINUW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMINUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMINUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMINUW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMINUW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMINUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMINUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMINUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMINUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVB2M_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVB2M_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVB2M_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVD2M_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVD2M_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVD2M_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDB_MEMu8_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDB_MEMu8_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDB_MEMu8_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDB_XMMu8_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDB_XMMu8_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDB_XMMu8_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDW_MEMu16_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDW_MEMu16_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDW_MEMu16_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDW_XMMu16_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDW_XMMu16_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVDW_YMMu16_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2B_XMMu8_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2B_YMMu8_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2B_ZMMu8_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2D_XMMu32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2D_YMMu32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2D_ZMMu32_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2Q_XMMu64_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2Q_YMMu64_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2Q_ZMMu64_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2W_XMMu16_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2W_YMMu16_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVM2W_ZMMu16_MASKmskw_AVX512,
    INTRINSIC_XED_IFORM_VPMOVMSKB_GPR32d_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVMSKB_GPR32d_YMMqq,
    INTRINSIC_XED_IFORM_VPMOVQ2M_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQ2M_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQ2M_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQB_MEMu8_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQB_MEMu8_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQB_MEMu8_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQB_XMMu8_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQB_XMMu8_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQB_XMMu8_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQD_MEMu32_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQD_MEMu32_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQD_MEMu32_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQD_XMMu32_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQD_XMMu32_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQD_YMMu32_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQW_MEMu16_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQW_MEMu16_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQW_MEMu16_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQW_XMMu16_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQW_XMMu16_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVQW_XMMu16_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDB_MEMi8_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDB_MEMi8_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDB_MEMi8_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDB_XMMi8_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDB_XMMi8_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDB_XMMi8_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDW_MEMi16_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDW_MEMi16_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDW_MEMi16_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDW_XMMi16_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDW_XMMi16_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSDW_YMMi16_MASKmskw_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQB_MEMi8_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQB_MEMi8_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQB_MEMi8_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQB_XMMi8_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQB_XMMi8_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQB_XMMi8_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQD_MEMi32_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQD_MEMi32_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQD_MEMi32_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQD_XMMi32_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQD_XMMi32_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQD_YMMi32_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQW_MEMi16_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQW_MEMi16_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQW_MEMi16_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQW_XMMi16_MASKmskw_XMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQW_XMMi16_MASKmskw_YMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSQW_XMMi16_MASKmskw_ZMMi64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSWB_MEMi8_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSWB_MEMi8_MASKmskw_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSWB_MEMi8_MASKmskw_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSWB_XMMi8_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSWB_XMMi8_MASKmskw_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSWB_YMMi8_MASKmskw_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VPMOVSXBD_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VPMOVSXBD_XMMi32_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBD_XMMi32_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBD_YMMi32_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBD_YMMi32_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBD_YMMqq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVSXBD_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVSXBD_ZMMi32_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBD_ZMMi32_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_XMMdq_MEMw,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_XMMdq_XMMw,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_XMMi64_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_XMMi64_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_YMMi64_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_YMMi64_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_YMMqq_MEMd,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_YMMqq_XMMd,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_ZMMi64_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBQ_ZMMi64_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBW_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVSXBW_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVSXBW_XMMi16_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBW_XMMi16_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBW_YMMi16_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBW_YMMi16_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBW_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPMOVSXBW_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVSXBW_ZMMi16_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXBW_ZMMi16_MASKmskw_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_XMMi64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_XMMi64_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_YMMi64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_YMMi64_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_ZMMi64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXDQ_ZMMi64_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVSXWD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVSXWD_XMMi32_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWD_XMMi32_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWD_YMMi32_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWD_YMMi32_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWD_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPMOVSXWD_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVSXWD_ZMMi32_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWD_ZMMi32_MASKmskw_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_XMMi64_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_XMMi64_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_YMMi64_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_YMMi64_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_YMMqq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_ZMMi64_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVSXWQ_ZMMi64_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDB_MEMu8_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDB_MEMu8_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDB_MEMu8_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDB_XMMu8_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDB_XMMu8_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDB_XMMu8_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDW_MEMu16_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDW_MEMu16_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDW_MEMu16_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDW_XMMu16_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDW_XMMu16_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSDW_YMMu16_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQB_MEMu8_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQB_MEMu8_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQB_MEMu8_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQB_XMMu8_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQB_XMMu8_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQB_XMMu8_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQD_MEMu32_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQD_MEMu32_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQD_MEMu32_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQD_XMMu32_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQD_XMMu32_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQD_YMMu32_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQW_MEMu16_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQW_MEMu16_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQW_MEMu16_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQW_XMMu16_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQW_XMMu16_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSQW_XMMu16_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSWB_MEMu8_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSWB_MEMu8_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSWB_MEMu8_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSWB_XMMu8_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSWB_XMMu8_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVUSWB_YMMu8_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVW2M_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVW2M_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVW2M_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVWB_MEMu8_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVWB_MEMu8_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVWB_MEMu8_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVWB_XMMu8_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVWB_XMMu8_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVWB_YMMu8_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBD_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VPMOVZXBD_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VPMOVZXBD_XMMi32_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBD_XMMi32_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBD_YMMi32_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBD_YMMi32_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBD_YMMqq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVZXBD_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVZXBD_ZMMi32_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBD_ZMMi32_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_XMMdq_MEMw,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_XMMdq_XMMw,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_XMMi64_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_XMMi64_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_YMMi64_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_YMMi64_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_YMMqq_MEMd,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_YMMqq_XMMd,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_ZMMi64_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBQ_ZMMi64_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBW_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVZXBW_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVZXBW_XMMi16_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBW_XMMi16_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBW_YMMi16_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBW_YMMi16_MASKmskw_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBW_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPMOVZXBW_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVZXBW_ZMMi16_MASKmskw_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXBW_ZMMi16_MASKmskw_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_XMMi64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_XMMi64_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_YMMi64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_YMMi64_MASKmskw_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_ZMMi64_MASKmskw_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXDQ_ZMMi64_MASKmskw_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVZXWD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVZXWD_XMMi32_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWD_XMMi32_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWD_YMMi32_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWD_YMMi32_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWD_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPMOVZXWD_YMMqq_XMMdq,
    INTRINSIC_XED_IFORM_VPMOVZXWD_ZMMi32_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWD_ZMMi32_MASKmskw_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_XMMi64_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_XMMi64_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_YMMi64_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_YMMi64_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_YMMqq_MEMq,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_ZMMi64_MASKmskw_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMOVZXWQ_ZMMi64_MASKmskw_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULDQ_XMMi64_MASKmskw_XMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMULDQ_XMMi64_MASKmskw_XMMi32_XMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMULDQ_YMMi64_MASKmskw_YMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMULDQ_YMMi64_MASKmskw_YMMi32_YMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMULDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULDQ_ZMMi64_MASKmskw_ZMMi32_MEMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMULDQ_ZMMi64_MASKmskw_ZMMi32_ZMMi32_AVX512,
    INTRINSIC_XED_IFORM_VPMULHRSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULHRSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULHRSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHRSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHRSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHRSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHRSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULHRSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHRSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHUW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULHUW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULHUW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHUW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHUW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULHUW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULHUW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHUW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHUW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHUW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULHW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULHW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULHW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULHW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULHW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULLD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULLD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULLD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULLD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULLD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULLD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULLQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULLQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULLW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULLW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULLW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULLW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULLW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULLW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPMULTISHIFTQB_XMMu8_MASKmskw_XMMu8_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULTISHIFTQB_XMMu8_MASKmskw_XMMu8_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULTISHIFTQB_YMMu8_MASKmskw_YMMu8_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULTISHIFTQB_YMMu8_MASKmskw_YMMu8_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULTISHIFTQB_ZMMu8_MASKmskw_ZMMu8_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULTISHIFTQB_ZMMu8_MASKmskw_ZMMu8_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPMULUDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPMULUDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPMULUDQ_XMMu64_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULUDQ_XMMu64_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULUDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPMULUDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPMULUDQ_YMMu64_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULUDQ_YMMu64_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPMULUDQ_ZMMu64_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTB_XMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTB_XMMu8_MASKmskw_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTB_YMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTB_YMMu8_MASKmskw_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTB_ZMMu8_MASKmskw_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTB_ZMMu8_MASKmskw_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTD_XMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTD_XMMu32_MASKmskw_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTD_YMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTD_YMMu32_MASKmskw_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTD_ZMMu32_MASKmskw_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTD_ZMMu32_MASKmskw_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTQ_XMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTQ_XMMu64_MASKmskw_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTQ_YMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTQ_YMMu64_MASKmskw_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTQ_ZMMu64_MASKmskw_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTQ_ZMMu64_MASKmskw_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTW_XMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTW_XMMu16_MASKmskw_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTW_YMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTW_YMMu16_MASKmskw_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTW_ZMMu16_MASKmskw_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPOPCNTW_ZMMu16_MASKmskw_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPOR_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPOR_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPOR_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPOR_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPPERM_XMMdq_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPPERM_XMMdq_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPPERM_XMMdq_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPROLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPROLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPROLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPROLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPROLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPROLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPROLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPROLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPROLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPROLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPROLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPROLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPRORD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPRORVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPRORVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPRORVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPRORVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPRORVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPRORVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPRORVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPRORVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPRORVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPRORVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPRORVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPRORVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPROTB_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTB_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTB_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPROTB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTD_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPROTD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTQ_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTQ_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTQ_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPROTQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTW_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPROTW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPROTW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPROTW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSADBW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSADBW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSADBW_XMMu16_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSADBW_XMMu16_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSADBW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSADBW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSADBW_YMMu16_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSADBW_YMMu16_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSADBW_ZMMu16_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSADBW_ZMMu16_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSCATTERDD_MEMu32_MASKmskw_XMMu32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPSCATTERDD_MEMu32_MASKmskw_YMMu32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPSCATTERDD_MEMu32_MASKmskw_ZMMu32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPSCATTERDQ_MEMu64_MASKmskw_XMMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPSCATTERDQ_MEMu64_MASKmskw_YMMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPSCATTERDQ_MEMu64_MASKmskw_ZMMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPSCATTERQD_MEMu32_MASKmskw_XMMu32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPSCATTERQD_MEMu32_MASKmskw_XMMu32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPSCATTERQD_MEMu32_MASKmskw_YMMu32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPSCATTERQQ_MEMu64_MASKmskw_XMMu64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VPSCATTERQQ_MEMu64_MASKmskw_YMMu64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VPSCATTERQQ_MEMu64_MASKmskw_ZMMu64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VPSHAB_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHAB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAD_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHAD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAQ_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHAQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAW_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHAW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHAW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLB_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHLB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLD_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHLD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLDD_XMMu32_MASKmskw_XMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDD_XMMu32_MASKmskw_XMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDD_YMMu32_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDD_YMMu32_MASKmskw_YMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDQ_XMMu64_MASKmskw_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDQ_XMMu64_MASKmskw_XMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDQ_YMMu64_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDQ_YMMu64_MASKmskw_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDW_XMMu16_MASKmskw_XMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDW_XMMu16_MASKmskw_XMMu16_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDW_YMMu16_MASKmskw_YMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDW_YMMu16_MASKmskw_YMMu16_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDW_ZMMu16_MASKmskw_ZMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLDW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHLQ_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHLQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLW_XMMdq_MEMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHLW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHLW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHRDD_XMMu32_MASKmskw_XMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDD_XMMu32_MASKmskw_XMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDD_YMMu32_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDD_YMMu32_MASKmskw_YMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDD_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDQ_XMMu64_MASKmskw_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDQ_XMMu64_MASKmskw_XMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDQ_YMMu64_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDQ_YMMu64_MASKmskw_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDW_XMMu16_MASKmskw_XMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDW_XMMu16_MASKmskw_XMMu16_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDW_YMMu16_MASKmskw_YMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDW_YMMu16_MASKmskw_YMMu16_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDW_ZMMu16_MASKmskw_ZMMu16_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHRDW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSHUFB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSHUFB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSHUFB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSHUFB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFBITQMB_MASKmskw_MASKmskw_XMMu64_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFBITQMB_MASKmskw_MASKmskw_XMMu64_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFBITQMB_MASKmskw_MASKmskw_YMMu64_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFBITQMB_MASKmskw_MASKmskw_YMMu64_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFBITQMB_MASKmskw_MASKmskw_ZMMu64_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFBITQMB_MASKmskw_MASKmskw_ZMMu64_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFD_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFHW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFHW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFHW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFHW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFHW_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFHW_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFHW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFHW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFHW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFHW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFLW_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFLW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFLW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFLW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFLW_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFLW_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSHUFLW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFLW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFLW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSHUFLW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSIGNB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSIGNB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSIGNB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSIGNB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSIGND_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSIGND_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSIGND_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSIGND_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSIGNW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSIGNW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSIGNW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSIGNW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSLLD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSLLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLD_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLD_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSLLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLDQ_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLDQ_XMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLDQ_XMMu8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLDQ_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLDQ_YMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLDQ_YMMu8_YMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLDQ_ZMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLDQ_ZMMu8_ZMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLVD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSLLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSLLVD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSLLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLVQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSLLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSLLVQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSLLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSLLW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSLLW_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSLLW_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSLLW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSLLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSRAD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRAD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRAD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSRAD_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRAD_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSRAD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRAVD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRAVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSRAVD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSRAVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSRAW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRAW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRAW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSRAW_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRAW_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSRAW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRAW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRLD_XMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_XMMu32_MASKmskw_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLD_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLD_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSRLD_YMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_YMMu32_MASKmskw_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_YMMu32_MASKmskw_YMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_ZMMu32_MASKmskw_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_ZMMu32_MASKmskw_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLD_ZMMu32_MASKmskw_ZMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLDQ_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLDQ_XMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLDQ_XMMu8_XMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLDQ_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLDQ_YMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLDQ_YMMu8_YMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLDQ_ZMMu8_MEMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLDQ_ZMMu8_ZMMu8_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMu64_MASKmskw_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMu64_MASKmskw_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_YMMu64_MASKmskw_YMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_ZMMu64_MASKmskw_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_ZMMu64_MASKmskw_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLQ_ZMMu64_MASKmskw_ZMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLVD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRLVD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSRLVD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSRLVD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLVQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRLVQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSRLVQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSRLVQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLVW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSRLW_XMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_XMMu16_MASKmskw_XMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VPSRLW_YMMqq_YMMqq_MEMdq,
    INTRINSIC_XED_IFORM_VPSRLW_YMMqq_YMMqq_XMMq,
    INTRINSIC_XED_IFORM_VPSRLW_YMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_YMMu16_MASKmskw_YMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_YMMu16_MASKmskw_YMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_ZMMu16_MASKmskw_MEMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_ZMMu16_MASKmskw_ZMMu16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSRLW_ZMMu16_MASKmskw_ZMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSUBD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSUBD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSUBD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSUBD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSUBD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPSUBQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSUBQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSUBQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSUBQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSUBQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSUBQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBSB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBSB_XMMi8_MASKmskw_XMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSB_XMMi8_MASKmskw_XMMi8_XMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSB_YMMi8_MASKmskw_YMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSB_YMMi8_MASKmskw_YMMi8_YMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBSB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBSB_ZMMi8_MASKmskw_ZMMi8_MEMi8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSB_ZMMi8_MASKmskw_ZMMi8_ZMMi8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBSW_XMMi16_MASKmskw_XMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSW_XMMi16_MASKmskw_XMMi16_XMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSW_YMMi16_MASKmskw_YMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSW_YMMi16_MASKmskw_YMMi16_YMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBSW_ZMMi16_MASKmskw_ZMMi16_MEMi16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBSW_ZMMi16_MASKmskw_ZMMi16_ZMMi16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSB_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBUSB_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBUSB_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSB_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSB_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBUSB_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBUSB_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSB_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSB_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBUSW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBUSW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBUSW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBUSW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBUSW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPSUBW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPSUBW_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBW_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPSUBW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPSUBW_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBW_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBW_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPSUBW_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGD_XMMu32_MASKmskw_XMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGD_XMMu32_MASKmskw_XMMu32_XMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGD_YMMu32_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGD_YMMu32_MASKmskw_YMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGD_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGQ_XMMu64_MASKmskw_XMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGQ_XMMu64_MASKmskw_XMMu64_XMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGQ_YMMu64_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGQ_YMMu64_MASKmskw_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTERNLOGQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VPTEST_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPTEST_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPTEST_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPTEST_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPTESTMB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTMW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMB_MASKmskw_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMB_MASKmskw_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMB_MASKmskw_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMB_MASKmskw_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMB_MASKmskw_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMB_MASKmskw_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMD_MASKmskw_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMD_MASKmskw_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMD_MASKmskw_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMD_MASKmskw_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMD_MASKmskw_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMD_MASKmskw_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMQ_MASKmskw_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMQ_MASKmskw_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMQ_MASKmskw_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMQ_MASKmskw_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMQ_MASKmskw_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMQ_MASKmskw_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMW_MASKmskw_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMW_MASKmskw_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMW_MASKmskw_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMW_MASKmskw_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMW_MASKmskw_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPTESTNMW_MASKmskw_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKHWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_XMMu8_MASKmskw_XMMu8_XMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_YMMu8_MASKmskw_YMMu8_YMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_MEMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLBW_ZMMu8_MASKmskw_ZMMu8_ZMMu8_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLDQ_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLQDQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_XMMu16_MASKmskw_XMMu16_XMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_YMMu16_MASKmskw_YMMu16_YMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_MEMu16_AVX512,
    INTRINSIC_XED_IFORM_VPUNPCKLWD_ZMMu16_MASKmskw_ZMMu16_ZMMu16_AVX512,
    INTRINSIC_XED_IFORM_VPXOR_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VPXOR_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VPXOR_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VPXOR_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VPXORD_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPXORD_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPXORD_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPXORD_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPXORD_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VPXORD_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VPXORQ_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPXORQ_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPXORQ_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPXORQ_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VPXORQ_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VPXORQ_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPD_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPS_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGEPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGESD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGESD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGESS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRANGESS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14PS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRCP14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRCP28PD_ZMMf64_MASKmskw_MEMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28PD_ZMMf64_MASKmskw_ZMMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28PS_ZMMf32_MASKmskw_MEMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28PS_ZMMf32_MASKmskw_ZMMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRCP28SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRCPPH_XMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPPH_XMMf16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPPH_YMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPPH_YMMf16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPPH_ZMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPPH_ZMMf16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VRCPPS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VRCPPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VRCPPS_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VRCPSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRCPSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VRCPSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VREDUCEPD_XMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPD_XMMf64_MASKmskw_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPD_YMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPD_YMMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPD_ZMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPD_ZMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPH_XMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPH_XMMf16_MASKmskw_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPH_YMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPH_YMMf16_MASKmskw_YMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPH_ZMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPH_ZMMf16_MASKmskw_ZMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPS_XMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPS_XMMf32_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPS_YMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPS_YMMf32_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPS_ZMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCEPS_ZMMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCESD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCESD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCESH_XMMf16_MASKmskw_XMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCESH_XMMf16_MASKmskw_XMMf16_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCESS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VREDUCESS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPD_XMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPD_XMMf64_MASKmskw_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPD_YMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPD_YMMf64_MASKmskw_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPD_ZMMf64_MASKmskw_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPD_ZMMf64_MASKmskw_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPH_XMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPH_XMMf16_MASKmskw_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPH_YMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPH_YMMf16_MASKmskw_YMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPH_ZMMf16_MASKmskw_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPH_ZMMf16_MASKmskw_ZMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPS_XMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPS_XMMf32_MASKmskw_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPS_YMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPS_YMMf32_MASKmskw_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPS_ZMMf32_MASKmskw_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALEPS_ZMMf32_MASKmskw_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALESD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALESD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALESH_XMMf16_MASKmskw_XMMf16_MEMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALESH_XMMf16_MASKmskw_XMMf16_XMMf16_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALESS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VRNDSCALESS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VROUNDPD_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPD_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPD_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPD_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPS_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPS_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPS_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDPS_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDSD_XMMdq_XMMdq_MEMq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDSD_XMMdq_XMMdq_XMMq_IMMb,
    INTRINSIC_XED_IFORM_VROUNDSS_XMMdq_XMMdq_MEMd_IMMb,
    INTRINSIC_XED_IFORM_VROUNDSS_XMMdq_XMMdq_XMMd_IMMb,
    INTRINSIC_XED_IFORM_VRSQRT14PD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14PS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT14SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VRSQRT28PD_ZMMf64_MASKmskw_MEMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28PD_ZMMf64_MASKmskw_ZMMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28PS_ZMMf32_MASKmskw_MEMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28PS_ZMMf32_MASKmskw_ZMMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28SD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28SD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28SS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRT28SS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512ER,
    INTRINSIC_XED_IFORM_VRSQRTPH_XMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTPH_XMMf16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTPH_YMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTPH_YMMf16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTPH_ZMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTPH_ZMMf16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VRSQRTPS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VRSQRTPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VRSQRTPS_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VRSQRTSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VRSQRTSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VRSQRTSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VSCALEFPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCALEFSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSCATTERDPD_MEMf64_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VSCATTERDPD_MEMf64_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VSCATTERDPD_MEMf64_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VSCATTERDPS_MEMf32_MASKmskw_XMMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VSCATTERDPS_MEMf32_MASKmskw_YMMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VSCATTERDPS_MEMf32_MASKmskw_ZMMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF0DPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF0DPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF0QPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF0QPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF1DPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF1DPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF1QPD_MEMf64_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERPF1QPS_MEMf32_MASKmskw_AVX512PF_VL512,
    INTRINSIC_XED_IFORM_VSCATTERQPD_MEMf64_MASKmskw_XMMf64_AVX512_VL128,
    INTRINSIC_XED_IFORM_VSCATTERQPD_MEMf64_MASKmskw_YMMf64_AVX512_VL256,
    INTRINSIC_XED_IFORM_VSCATTERQPD_MEMf64_MASKmskw_ZMMf64_AVX512_VL512,
    INTRINSIC_XED_IFORM_VSCATTERQPS_MEMf32_MASKmskw_XMMf32_AVX512_VL128,
    INTRINSIC_XED_IFORM_VSCATTERQPS_MEMf32_MASKmskw_XMMf32_AVX512_VL256,
    INTRINSIC_XED_IFORM_VSCATTERQPS_MEMf32_MASKmskw_YMMf32_AVX512_VL512,
    INTRINSIC_XED_IFORM_VSHA512MSG1_YMMu64_XMMu64,
    INTRINSIC_XED_IFORM_VSHA512MSG2_YMMu64_YMMu64,
    INTRINSIC_XED_IFORM_VSHA512RNDS2_YMMu64_YMMu64_XMMu64,
    INTRINSIC_XED_IFORM_VSHUFF32X4_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF32X4_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF32X4_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF32X4_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF64X2_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF64X2_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF64X2_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFF64X2_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI32X4_YMMu32_MASKmskw_YMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI32X4_YMMu32_MASKmskw_YMMu32_YMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI32X4_ZMMu32_MASKmskw_ZMMu32_MEMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI32X4_ZMMu32_MASKmskw_ZMMu32_ZMMu32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI64X2_YMMu64_MASKmskw_YMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI64X2_YMMu64_MASKmskw_YMMu64_YMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI64X2_ZMMu64_MASKmskw_ZMMu64_MEMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFI64X2_ZMMu64_MASKmskw_ZMMu64_ZMMu64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPD_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPD_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPD_XMMf64_MASKmskw_XMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPD_XMMf64_MASKmskw_XMMf64_XMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPD_YMMf64_MASKmskw_YMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPD_YMMf64_MASKmskw_YMMf64_YMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPD_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPD_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPS_XMMdq_XMMdq_MEMdq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPS_XMMdq_XMMdq_XMMdq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPS_XMMf32_MASKmskw_XMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPS_XMMf32_MASKmskw_XMMf32_XMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPS_YMMf32_MASKmskw_YMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPS_YMMf32_MASKmskw_YMMf32_YMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPS_YMMqq_YMMqq_MEMqq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPS_YMMqq_YMMqq_YMMqq_IMMb,
    INTRINSIC_XED_IFORM_VSHUFPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSHUFPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_IMM8_AVX512,
    INTRINSIC_XED_IFORM_VSM3MSG1_XMMu32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VSM3MSG1_XMMu32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VSM3MSG2_XMMu32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VSM3MSG2_XMMu32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VSM3RNDS2_XMMu32_XMMu32_MEMu32_IMM8,
    INTRINSIC_XED_IFORM_VSM3RNDS2_XMMu32_XMMu32_XMMu32_IMM8,
    INTRINSIC_XED_IFORM_VSM4KEY4_XMMu32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VSM4KEY4_XMMu32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VSM4KEY4_YMMu32_YMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VSM4KEY4_YMMu32_YMMu32_YMMu32,
    INTRINSIC_XED_IFORM_VSM4RNDS4_XMMu32_XMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VSM4RNDS4_XMMu32_XMMu32_XMMu32,
    INTRINSIC_XED_IFORM_VSM4RNDS4_YMMu32_YMMu32_MEMu32,
    INTRINSIC_XED_IFORM_VSM4RNDS4_YMMu32_YMMu32_YMMu32,
    INTRINSIC_XED_IFORM_VSQRTPD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VSQRTPD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VSQRTPD_XMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPD_XMMf64_MASKmskw_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPD_YMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPD_YMMf64_MASKmskw_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPD_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VSQRTPD_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VSQRTPD_ZMMf64_MASKmskw_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPD_ZMMf64_MASKmskw_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPH_XMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPH_XMMf16_MASKmskw_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPH_YMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPH_YMMf16_MASKmskw_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPH_ZMMf16_MASKmskw_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPH_ZMMf16_MASKmskw_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VSQRTPS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VSQRTPS_XMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPS_XMMf32_MASKmskw_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPS_YMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPS_YMMf32_MASKmskw_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VSQRTPS_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VSQRTPS_ZMMf32_MASKmskw_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTPS_ZMMf32_MASKmskw_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VSQRTSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VSQRTSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSQRTSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSQRTSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VSQRTSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VSQRTSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSQRTSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSTMXCSR_MEMd,
    INTRINSIC_XED_IFORM_VSUBPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VSUBPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VSUBPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VSUBPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VSUBPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBPH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBPH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBPH_YMMf16_MASKmskw_YMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBPH_YMMf16_MASKmskw_YMMf16_YMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBPH_ZMMf16_MASKmskw_ZMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBPH_ZMMf16_MASKmskw_ZMMf16_ZMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VSUBPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VSUBPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VSUBPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VSUBPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBSD_XMMdq_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VSUBSD_XMMdq_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VSUBSD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBSD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VSUBSH_XMMf16_MASKmskw_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBSH_XMMf16_MASKmskw_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VSUBSS_XMMdq_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VSUBSS_XMMdq_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VSUBSS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VSUBSS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VTESTPD_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VTESTPD_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VTESTPD_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VTESTPD_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VTESTPS_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VTESTPS_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VTESTPS_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VTESTPS_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VUCOMISD_XMMdq_MEMq,
    INTRINSIC_XED_IFORM_VUCOMISD_XMMdq_XMMq,
    INTRINSIC_XED_IFORM_VUCOMISD_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUCOMISD_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUCOMISH_XMMf16_MEMf16_AVX512,
    INTRINSIC_XED_IFORM_VUCOMISH_XMMf16_XMMf16_AVX512,
    INTRINSIC_XED_IFORM_VUCOMISS_XMMdq_MEMd,
    INTRINSIC_XED_IFORM_VUCOMISS_XMMdq_XMMd,
    INTRINSIC_XED_IFORM_VUCOMISS_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUCOMISS_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VUNPCKHPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VUNPCKHPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VUNPCKHPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VUNPCKHPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VUNPCKHPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VUNPCKHPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKHPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VUNPCKLPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VUNPCKLPD_XMMf64_MASKmskw_XMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPD_XMMf64_MASKmskw_XMMf64_XMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPD_YMMf64_MASKmskw_YMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPD_YMMf64_MASKmskw_YMMf64_YMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VUNPCKLPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_MEMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPD_ZMMf64_MASKmskw_ZMMf64_ZMMf64_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VUNPCKLPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VUNPCKLPS_XMMf32_MASKmskw_XMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPS_XMMf32_MASKmskw_XMMf32_XMMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPS_YMMf32_MASKmskw_YMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPS_YMMf32_MASKmskw_YMMf32_YMMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VUNPCKLPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_MEMf32_AVX512,
    INTRINSIC_XED_IFORM_VUNPCKLPS_ZMMf32_MASKmskw_ZMMf32_ZMMf32_AVX512,
    INTRINSIC_XED_IFORM_VXORPD_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VXORPD_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VXORPD_XMMu64_MASKmskw_XMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VXORPD_XMMu64_MASKmskw_XMMu64_XMMu64_AVX512,
    INTRINSIC_XED_IFORM_VXORPD_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VXORPD_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VXORPD_YMMu64_MASKmskw_YMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VXORPD_YMMu64_MASKmskw_YMMu64_YMMu64_AVX512,
    INTRINSIC_XED_IFORM_VXORPD_ZMMu64_MASKmskw_ZMMu64_MEMu64_AVX512,
    INTRINSIC_XED_IFORM_VXORPD_ZMMu64_MASKmskw_ZMMu64_ZMMu64_AVX512,
    INTRINSIC_XED_IFORM_VXORPS_XMMdq_XMMdq_MEMdq,
    INTRINSIC_XED_IFORM_VXORPS_XMMdq_XMMdq_XMMdq,
    INTRINSIC_XED_IFORM_VXORPS_XMMu32_MASKmskw_XMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VXORPS_XMMu32_MASKmskw_XMMu32_XMMu32_AVX512,
    INTRINSIC_XED_IFORM_VXORPS_YMMqq_YMMqq_MEMqq,
    INTRINSIC_XED_IFORM_VXORPS_YMMqq_YMMqq_YMMqq,
    INTRINSIC_XED_IFORM_VXORPS_YMMu32_MASKmskw_YMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VXORPS_YMMu32_MASKmskw_YMMu32_YMMu32_AVX512,
    INTRINSIC_XED_IFORM_VXORPS_ZMMu32_MASKmskw_ZMMu32_MEMu32_AVX512,
    INTRINSIC_XED_IFORM_VXORPS_ZMMu32_MASKmskw_ZMMu32_ZMMu32_AVX512,
    INTRINSIC_XED_IFORM_VZEROALL,
    INTRINSIC_XED_IFORM_VZEROUPPER,
    INTRINSIC_XED_IFORM_WBINVD,
    INTRINSIC_XED_IFORM_WBNOINVD,
    INTRINSIC_XED_IFORM_WRFSBASE_GPRy,
    INTRINSIC_XED_IFORM_WRGSBASE_GPRy,
    INTRINSIC_XED_IFORM_WRMSR,
    INTRINSIC_XED_IFORM_WRMSRLIST,
    INTRINSIC_XED_IFORM_WRMSRNS,
    INTRINSIC_XED_IFORM_WRPKRU,
    INTRINSIC_XED_IFORM_WRSSD_MEMu32_GPR32u32,
    INTRINSIC_XED_IFORM_WRSSD_MEMu32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_WRSSQ_MEMu64_GPR64u64,
    INTRINSIC_XED_IFORM_WRSSQ_MEMu64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_WRUSSD_MEMu32_GPR32u32,
    INTRINSIC_XED_IFORM_WRUSSD_MEMu32_GPR32u32_APX,
    INTRINSIC_XED_IFORM_WRUSSQ_MEMu64_GPR64u64,
    INTRINSIC_XED_IFORM_WRUSSQ_MEMu64_GPR64u64_APX,
    INTRINSIC_XED_IFORM_XABORT_IMMb,
    INTRINSIC_XED_IFORM_XADD_GPR8_GPR8,
    INTRINSIC_XED_IFORM_XADD_GPRv_GPRv,
    INTRINSIC_XED_IFORM_XADD_MEMb_GPR8,
    INTRINSIC_XED_IFORM_XADD_MEMv_GPRv,
    INTRINSIC_XED_IFORM_XADD_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_XADD_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_XBEGIN_RELBRz,
    INTRINSIC_XED_IFORM_XCHG_GPR8_GPR8,
    INTRINSIC_XED_IFORM_XCHG_GPRv_GPRv,
    INTRINSIC_XED_IFORM_XCHG_GPRv_OrAX,
    INTRINSIC_XED_IFORM_XCHG_MEMb_GPR8,
    INTRINSIC_XED_IFORM_XCHG_MEMv_GPRv,
    INTRINSIC_XED_IFORM_XEND,
    INTRINSIC_XED_IFORM_XGETBV,
    INTRINSIC_XED_IFORM_XLAT,
    INTRINSIC_XED_IFORM_XOR_AL_IMMb,
    INTRINSIC_XED_IFORM_XOR_GPR8_GPR8_30,
    INTRINSIC_XED_IFORM_XOR_GPR8_GPR8_32,
    INTRINSIC_XED_IFORM_XOR_GPR8_IMMb_80r6,
    INTRINSIC_XED_IFORM_XOR_GPR8_IMMb_82r6,
    INTRINSIC_XED_IFORM_XOR_GPR8_MEMb,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_GPR8i8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_MEMi8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_XOR_GPR8i8_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_31,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_33,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_GPRv_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_IMMb,
    INTRINSIC_XED_IFORM_XOR_GPRv_IMMz,
    INTRINSIC_XED_IFORM_XOR_GPRv_IMMz_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_MEMv,
    INTRINSIC_XED_IFORM_XOR_GPRv_MEMv_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_GPRv_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_XOR_MEMb_GPR8,
    INTRINSIC_XED_IFORM_XOR_MEMb_IMMb_80r6,
    INTRINSIC_XED_IFORM_XOR_MEMb_IMMb_82r6,
    INTRINSIC_XED_IFORM_XOR_MEMi8_GPR8i8_APX,
    INTRINSIC_XED_IFORM_XOR_MEMi8_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_MEMv_GPRv,
    INTRINSIC_XED_IFORM_XOR_MEMv_GPRv_APX,
    INTRINSIC_XED_IFORM_XOR_MEMv_IMM8_APX,
    INTRINSIC_XED_IFORM_XOR_MEMv_IMMb,
    INTRINSIC_XED_IFORM_XOR_MEMv_IMMz,
    INTRINSIC_XED_IFORM_XOR_MEMv_IMMz_APX,
    INTRINSIC_XED_IFORM_XOR_OrAX_IMMz,
    INTRINSIC_XED_IFORM_XORPD_XMMxuq_MEMxuq,
    INTRINSIC_XED_IFORM_XORPD_XMMxuq_XMMxuq,
    INTRINSIC_XED_IFORM_XORPS_XMMxud_MEMxud,
    INTRINSIC_XED_IFORM_XORPS_XMMxud_XMMxud,
    INTRINSIC_XED_IFORM_XOR_LOCK_MEMb_GPR8,
    INTRINSIC_XED_IFORM_XOR_LOCK_MEMb_IMMb_80r6,
    INTRINSIC_XED_IFORM_XOR_LOCK_MEMb_IMMb_82r6,
    INTRINSIC_XED_IFORM_XOR_LOCK_MEMv_GPRv,
    INTRINSIC_XED_IFORM_XOR_LOCK_MEMv_IMMb,
    INTRINSIC_XED_IFORM_XOR_LOCK_MEMv_IMMz,
    INTRINSIC_XED_IFORM_XRESLDTRK,
    INTRINSIC_XED_IFORM_XRSTOR_MEMmxsave,
    INTRINSIC_XED_IFORM_XRSTOR64_MEMmxsave,
    INTRINSIC_XED_IFORM_XRSTORS_MEMmxsave,
    INTRINSIC_XED_IFORM_XRSTORS64_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVE_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVE64_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVEC_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVEC64_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVEOPT_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVEOPT64_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVES_MEMmxsave,
    INTRINSIC_XED_IFORM_XSAVES64_MEMmxsave,
    INTRINSIC_XED_IFORM_XSETBV,
    INTRINSIC_XED_IFORM_XSTORE,
    INTRINSIC_XED_IFORM_XSUSLDTRK,
    INTRINSIC_XED_IFORM_XTEST,
    INTRINSIC_XED_IFORM_LAST,

    // Below are modified iforms
    // They are added to allow GetIntrinsicName() return an accurate intrinsic name.
    // The old intrinsics (e.g., INTRINSIC_XED_IFORM_TZCNT_GPRv_GPRv),
    // which are copied from xed include file is NOT deleted from above.
    // It is not used since no code will ever emit them.
    // Do NOT remove them since that will break the way we get intrinsic names.
    INTRINSIC_XED_IFORM_TZCNT_GPR64_GPRMEM64,
    INTRINSIC_XED_IFORM_TZCNT_GPR32_GPRMEM32,
    INTRINSIC_XED_IFORM_TZCNT_GPR16_GPRMEM16,

    INTRINSIC_XED_IFORM_LZCNT_GPR64_GPRMEM64,
    INTRINSIC_XED_IFORM_LZCNT_GPR32_GPRMEM32,
    INTRINSIC_XED_IFORM_LZCNT_GPR16_GPRMEM16,

    INTRINSIC_XED_IFORM_POPCNT_GPR64_GPRMEM64,
    INTRINSIC_XED_IFORM_POPCNT_GPR32_GPRMEM32,
    INTRINSIC_XED_IFORM_POPCNT_GPR16_GPRMEM16,

    INTRINSIC_LAST
};

bool GetLowLevelILForInstruction(BinaryNinja::Architecture* arch, const uint64_t addr, BinaryNinja::LowLevelILFunction& il, const xed_decoded_inst_t* const xedd);
