import fs from 'fs-extra';
import path from 'path';
import {
    assertHTML,
    assertRootChildren,
    createDataTransfer,
    createSnippet,
    ctrlOrCmd,
    dragMouse,
    enterUntilScrolled,
    expectUnchangedScrollPosition,
    focusEditor,
    getEditorStateJSON,
    html,
    initialize,
    insertCard,
    isMac,
    pasteHtml,
    pasteText
} from '../../utils/e2e';
import {expect, test} from '@playwright/test';
import {fileURLToPath} from 'url';
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

test.describe('Image card', async () => {
    let page;

    test.beforeAll(async ({browser}) => {
        page = await browser.newPage();
    });

    test.beforeEach(async () => {
        await initialize({page});
    });

    test.afterAll(async () => {
        await page.close();
    });

    test('can import serialized image card nodes', async function () {
        const contentParam = encodeURIComponent(JSON.stringify({
            root: {
                children: [{
                    type: 'image',
                    src: '/content/images/2022/11/koenig-lexical.jpg',
                    width: 3840,
                    height: 2160,
                    title: 'This is a title',
                    altText: 'This is some alt text',
                    caption: 'This is a <b>caption</b>',
                    cardWidth: 'wide'
                }],
                direction: null,
                format: '',
                indent: 0,
                type: 'root',
                version: 1
            }
        }));

        await initialize({page, uri: `/#/?content=${contentParam}`});

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false" data-kg-card-width="wide">
                <div data-kg-card-editing="false" data-kg-card-selected="false" data-kg-card="image">
                    <figure data-kg-card-width="wide">
                        <div><img alt="" src="/content/images/2022/11/koenig-lexical.jpg" /></div>
                        <figcaption>
                            <div data-kg-allow-clickthrough="true">
                                <div>
                                    <div data-kg="editor">
                                        <div
                                            contenteditable="true"
                                            role="textbox"
                                            spellcheck="true"
                                            data-lexical-editor="true"
                                        >
                                            <p dir="ltr">
                                                <span data-lexical-text="true">This is a</span>
                                                <strong data-lexical-text="true">caption</strong>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </figcaption>
                    </figure>
                </div>
            </div>
        `, {ignoreCardContents: false});
    });

    test('can upload image with `data:` url', async function () {
        await page.evaluate(() => {
            const serializedState = JSON.stringify({
                root: {
                    children: [{
                        type: 'image',
                        src: 'data:image/png;base64,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',
                        width: 248,
                        height: 248,
                        title: '',
                        altText: '',
                        caption: ''
                    }],
                    direction: null,
                    format: '',
                    indent: 0,
                    type: 'root',
                    version: 1
                }
            });
            const editor = window.lexicalEditor;
            const editorState = editor.parseEditorState(serializedState);
            editor.setEditorState(editorState);
        });

        await expect(await page.getByTestId('image-card-populated')).toBeVisible();
        await expect(page.locator('img')).toHaveAttribute('src', /blob:/);
    });

    test('renders image card node', async function () {
        await focusEditor(page);
        await page.keyboard.type('image! ');

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="false" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div data-testid="media-placeholder">
                            <div>
                                <button name="placeholder-button" type="button">
                                    <svg width="134" height="135" viewBox="0 0 134 135" xmlns="http://www.w3.org/2000/svg"></svg>
                                    <p>Click to select an image</p>
                                </button>
                            </div>
                        </div>
                        <form><input accept="image/*" hidden="" name="image-input" type="file" /></form>
                    </figure>
                </div>
            </div>
            <div contenteditable="false" data-lexical-cursor="true"></div>
        `);
    });

    test('can upload an image', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);

        await expect(await page.getByTestId('image-card-populated')).toBeVisible();
    });

    test('can get image dimensions from external URL', async function () {
        await page.route('https://example.com/large-image.png', route => route.fulfill({
            status: 200,
            contentType: 'image/png',
            body: fs.readFileSync(__dirname + '/../fixtures/large-image.png')
        }));
        await focusEditor(page);
        await page.keyboard.type('/image https://example.com/large-image.png');
        await page.keyboard.press('Enter');
        // Wait for card to be rendered before checking height and width
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();
        // Check that the image has the correct height and width populated
        await assertRootChildren(page, JSON.stringify([{
            type: 'image',
            version: 1,
            src: 'https://example.com/large-image.png',
            width: 248,
            height: 248,
            title: '',
            alt: '',
            caption: '',
            cardWidth: 'regular',
            href: ''
        },{
            children: [],
            direction: null,
            format: '',
            indent: 0,
            type: 'paragraph',
            version: 1
        }]));
    });

    test('can get image dimensions when rendering serialized node with missing data', async function () {
        await page.route('https://example.com/large-image.png', route => route.fulfill({
            status: 200,
            contentType: 'image/png',
            body: fs.readFileSync(__dirname + '/../fixtures/large-image.png')
        }));

        const contentParam = encodeURIComponent(JSON.stringify({
            root: {
                children: [{
                    type: 'image',
                    src: 'https://example.com/large-image.png',
                    title: 'This is a title',
                    alt: 'This is some alt text',
                    caption: 'This is a <b>caption</b>',
                    cardWidth: 'wide'
                }],
                direction: null,
                format: '',
                indent: 0,
                type: 'root',
                version: 1
            }
        }));

        await initialize({page, uri: `/#/?content=${contentParam}`});
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        const editorState = JSON.parse(await getEditorStateJSON(page));

        expect(editorState.root.children[0].type).toEqual('image');
        // missing width & height are extracted from the image
        expect(editorState.root.children[0].width, 'width').toEqual(248);
        expect(editorState.root.children[0].height, 'height').toEqual(248);
    });

    test('does not change existing image dimensions when rendering serialized node', async function () {
        await page.route('https://example.com/large-image.png', route => route.fulfill({
            status: 200,
            contentType: 'image/png',
            body: fs.readFileSync(__dirname + '/../fixtures/large-image.png')
        }));

        const contentParam = encodeURIComponent(JSON.stringify({
            root: {
                children: [{
                    type: 'image',
                    src: 'https://example.com/large-image.png',
                    title: 'This is a title',
                    alt: 'This is some alt text',
                    caption: 'This is a <b>caption</b>',
                    cardWidth: 'wide',
                    width: 1000,
                    height: 1000
                }],
                direction: null,
                format: '',
                indent: 0,
                type: 'root',
                version: 1
            }
        }));

        await initialize({page, uri: `/#/?content=${contentParam}`});
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        const editorState = JSON.parse(await getEditorStateJSON(page));

        expect(editorState.root.children[0].type).toEqual('image');
        // existing width & height are kept from the serialized state
        expect(editorState.root.children[0].width, 'width').toEqual(1000);
        expect(editorState.root.children[0].height, 'height').toEqual(1000);
    });

    test('can toggle to alt text', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);

        // placeholder is replaced with uploading image
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        // wait for upload to complete
        await expect(await page.getByTestId('progress-bar')).toBeHidden();

        await page.click('button[name="alt-toggle-button"]');

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div><img alt="" src="blob:..." /></div>
                        <figcaption>
                            <input placeholder="Type alt text for image (optional)" value=""/>
                            <button name="alt-toggle-button" type="button">Alt</button>
                        </figcaption>
                    </figure>
                    <div data-kg-card-toolbar="image"></div>
                </div>
            </div>
        `, {ignoreCardToolbarContents: true});
    });

    test('renders caption if present', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);

        // placeholder is replaced with uploading image
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        // wait for upload to complete
        await expect(await page.getByTestId('progress-bar')).toBeHidden();

        await page.click('[data-testid="image-caption-editor"]');
        await page.keyboard.type('This is a caption');
        await expect(await page.locator('text="This is a caption"')).toBeVisible();
    });

    // NOTE: still works, but it's a focus issue
    test.skip('can paste html to caption', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);

        await page.waitForSelector('[data-testid="image-caption-editor"]');
        await page.click('[data-testid="image-caption-editor"]');
        await pasteText(page, 'This is link <a href="https://ghost.org/changelog/markdown/">ghost.org/changelog/markdown/</a>', 'text/html');

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div><img alt="" src="blob:..." /></div>
                        <figcaption>
                            <div data-testid="image-caption-editor" data-kg-allow-clickthrough="true">
                                <div>
                                    <div data-kg="editor">
                                        <div contenteditable="true" role="textbox" spellcheck="true" data-lexical-editor="true" data-koenig-dnd-container="true">
                                            <p dir="ltr" data-koenig-dnd-droppable="true">
                                                <span data-lexical-text="true">This is link </span>
                                                <a href="https://ghost.org/changelog/markdown/" dir="ltr">
                                                    <span data-lexical-text="true">ghost.org/changelog/markdown/</span>
                                                </a>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <button name="alt-toggle-button" type="button">Alt</button>
                        </figcaption>
                    </figure>
                </div>
            </div>
        `, {ignoreCardToolbarContents: true});
    });

    test('renders image card toolbar', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"]')).not.toBeNull();
    });

    test('image card toolbar has Regular button', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"] button[aria-label="Regular"]')).not.toBeNull();
    });

    test('image card toolbar has Wide button', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"] button[aria-label="Wide"]')).not.toBeNull();
    });

    test('image card toolbar has Full button', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"] button[aria-label="Full"]')).not.toBeNull();
    });

    test('image card toolbar has Link button', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"] button[aria-label="Link"]')).not.toBeNull();
    });

    test('image card toolbar has Replace button', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"] button[aria-label="Replace"]')).not.toBeNull();
    });

    test('image card toolbar has Snippet button', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);
        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"] button[aria-label="Snippet"]')).not.toBeNull();
    });

    test('toolbar can toggle image sizes', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);

        // placeholder is replaced with uploading image
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        // wait for upload to complete
        await expect(await page.getByTestId('progress-bar')).toBeHidden();

        await page.click('[data-kg-card="image"]');

        expect(await page.locator('[data-kg-card-toolbar="image"]')).not.toBeNull();

        await page.click('[data-kg-card-toolbar="image"] button[aria-label="Wide width"]');
        expect (await page.locator('[data-kg-card-width="wide"]')).not.toBeNull();

        await page.click('[data-kg-card-toolbar="image"] button[aria-label="Full width"]');
        expect (await page.locator('[data-kg-card-width="full"]')).not.toBeNull();

        await page.click('[data-kg-card-toolbar="image"] button[aria-label="Regular width"]');
        expect (await page.locator('[data-kg-card-width="regular"]')).not.toBeNull();
    });

    test('toolbar does not disappear on click', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');

        await focusEditor(page);
        await page.keyboard.type('image! ');

        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            await page.click('button[name="placeholder-button"]')
        ]);
        await fileChooser.setFiles([filePath]);

        // placeholder is replaced with uploading image
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        // wait for upload to complete
        await expect(await page.getByTestId('progress-bar')).toBeHidden();

        await page.click('figure');

        await page.click('[data-kg-card-toolbar="image"] button[aria-label="Regular width"]');

        expect(await page.locator('[data-kg-card-toolbar="image"]')).not.toBeNull();
    });

    test('file input opens immediately when added via card menu', async function () {
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');
        const [fileChooser] = await Promise.all([
            page.waitForEvent('filechooser'),
            page.click('[data-kg-card-menu-item="Image"]')
        ]);

        expect(fileChooser).not.toBeNull();
    });

    test('can handle drag over & leave', async function () {
        await focusEditor(page);
        await page.keyboard.type('image! ');

        const imageCard = await page.locator('[data-kg-card="image"]');
        expect(imageCard).not.toBeNull();

        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');
        const dataTransfer = await createDataTransfer(page, [{filePath, fileName: 'large-image.png', fileType: 'image/png'}]);

        await page.locator('[data-kg-card="image"] [data-testid="media-placeholder"]').dispatchEvent('dragenter', {dataTransfer});

        expect(await page.locator('[data-kg-card-drag-text="true"]')).not.toBeNull();

        await page.locator('[data-kg-card="image"] [data-testid="media-placeholder"]').dispatchEvent('dragleave', {dataTransfer});

        await expect(await page.locator('[data-kg-card-drag-text="true"]')).toHaveCount(0);
    });

    test('can handle image drop on empty card', async function () {
        await focusEditor(page);
        await page.keyboard.type('image! ');

        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');
        const dataTransfer = await createDataTransfer(page, [{filePath, fileName: 'large-image.png', fileType: 'image/png'}]);

        await page.locator('[data-kg-card="image"] [data-testid="media-placeholder"]').dispatchEvent('dragenter', {dataTransfer});

        // Dragover text should be visible
        await expect(await page.locator('[data-kg-card-drag-text="true"]')).toBeVisible();

        await page.locator('[data-kg-card="image"] [data-testid="media-placeholder"]').dispatchEvent('drop', {dataTransfer});

        // placeholder is replaced with uploading image
        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        // wait for upload to complete
        await expect(await page.getByTestId('progress-bar')).toBeHidden();

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="false" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div><img alt="" src="blob:..." /></div>
                    </figure>
                </div>
            </div>
            <div contenteditable="false" data-lexical-cursor="true"></div>
        `);
    });

    test('replaces image when new image file dropped on populated card', async function () {
        await focusEditor(page);
        await insertImage(page);

        const originalSrc = await page.locator('[data-kg-card="image"] img').getAttribute('src');

        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image-1.png');
        const dataTransfer = await createDataTransfer(page, [{filePath, fileName: 'large-image-1.png', fileType: 'image/png'}]);

        await page.locator('[data-kg-card="image"] [data-testid="image-card-populated"]').dispatchEvent('dragenter', {dataTransfer});

        // Dragover text should be visible
        await expect(await page.locator('[data-kg-card="image"] [data-testid="drag-overlay"]')).toBeVisible();

        await page.locator('[data-kg-card="image"] [data-testid="image-card-populated"]').dispatchEvent('drop', {dataTransfer});

        // wait for upload to complete
        await expect(page.getByTestId('upload-progress')).toBeVisible();
        await expect(page.getByTestId('progress-bar')).toBeHidden();

        const newSrc = await page.locator('[data-kg-card="image"] img').getAttribute('src');

        expect(originalSrc).not.toEqual(newSrc);
    });

    test('adds extra paragraph when image is inserted at end of document', async function () {
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');

        await Promise.all([
            page.waitForEvent('filechooser'),
            page.click('[data-kg-card-menu-item="Image"]')
        ]);

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                </div>
            </div>
            <p><br /></p>
        `, {ignoreCardContents: true});
    });

    test('does not add extra paragraph when image is inserted mid-document', async function () {
        await focusEditor(page);
        await page.keyboard.press('Enter');
        await page.keyboard.type('Testing');
        await page.keyboard.press('ArrowUp');
        await page.click('[data-kg-plus-button]');

        await Promise.all([
            page.waitForEvent('filechooser'),
            page.click('[data-kg-card-menu-item="Image"]')
        ]);

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                </div>
            </div>
            <p dir="ltr"><span data-lexical-text="true">Testing</span></p>
        `, {ignoreCardContents: true});
    });

    test('can insert unsplash image', async () => {
        const testData = [
            {
                id: 'SgvrLyGKnHw',
                created_at: '2023-02-27T20:39:45Z',
                updated_at: '2023-03-01T06:08:01Z',
                promoted_at: '2023-03-01T06:08:01Z',
                width: 5504,
                height: 8256,
                color: '#8c8c8c',
                blur_hash: 'LHD]Vg4m%fIA_3D%%2MxIoWCs.s:',
                description: null,
                alt_description: 'a group of people walking down a street next to tall buildings',
                urls: {
                    raw: 'http://127.0.0.1:5173/Koenig-editor-1.png',
                    full: 'http://127.0.0.1:5173/Koenig-editor-1.png',
                    regular: 'http://127.0.0.1:5173/Koenig-editor-1.png',
                    small: 'http://127.0.0.1:5173/Koenig-editor-1.png',
                    thumb: 'http://127.0.0.1:5173/Koenig-editor-1.png',
                    small_s3: 'http://127.0.0.1:5173/Koenig-editor-1.png'
                },
                links: {
                    self: 'https://api.unsplash.com/photos/SgvrLyGKnHw',
                    html: 'https://unsplash.com/photos/SgvrLyGKnHw',
                    download: 'https://unsplash.com/photos/SgvrLyGKnHw/download?ixid=MnwxMTc3M3wwfDF8YWxsfDJ8fHx8fHwyfHwxNjc3NjUxMzk5',
                    download_location: 'https://api.unsplash.com/photos/SgvrLyGKnHw/download?ixid=MnwxMTc3M3wwfDF8YWxsfDJ8fHx8fHwyfHwxNjc3NjUxMzk5'
                },
                likes: 1,
                liked_by_user: false,
                current_user_collections: [],
                sponsorship: null,
                topic_submissions: {},
                user: {
                    id: '9_671Bq5l40',
                    updated_at: '2023-03-01T06:08:01Z',
                    username: 'jamillatrach',
                    name: 'Latrach Med Jamil',
                    first_name: 'Latrach',
                    last_name: 'Med Jamil',
                    twitter_username: null,
                    portfolio_url: null,
                    bio: 'Just trying to share what I have --\r\n\r\nInstagram.com/jamillatrach/',
                    location: 'Düsseldorf',
                    links: {
                        self: 'https://api.unsplash.com/users/jamillatrach',
                        html: 'https://unsplash.com/@jamillatrach',
                        photos: 'https://api.unsplash.com/users/jamillatrach/photos',
                        likes: 'https://api.unsplash.com/users/jamillatrach/likes',
                        portfolio: 'https://api.unsplash.com/users/jamillatrach/portfolio',
                        following: 'https://api.unsplash.com/users/jamillatrach/following',
                        followers: 'https://api.unsplash.com/users/jamillatrach/followers'
                    },
                    profile_image: {
                        small: 'https://images.unsplash.com/profile-fb-1570626489-2f1895a616ca.jpg?ixlib=rb-4.0.3\u0026crop=faces\u0026fit=crop\u0026w=32\u0026h=32',
                        medium: 'https://images.unsplash.com/profile-fb-1570626489-2f1895a616ca.jpg?ixlib=rb-4.0.3\u0026crop=faces\u0026fit=crop\u0026w=64\u0026h=64',
                        large: 'https://images.unsplash.com/profile-fb-1570626489-2f1895a616ca.jpg?ixlib=rb-4.0.3\u0026crop=faces\u0026fit=crop\u0026w=128\u0026h=128'
                    },
                    instagram_username: 'jamillatrach',
                    total_collections: 0,
                    total_likes: 4,
                    total_photos: 451,
                    accepted_tos: true,
                    for_hire: false,
                    social: {
                        instagram_username: 'jamillatrach',
                        portfolio_url: null,
                        twitter_username: null,
                        paypal_email: null
                    }
                }
            }
        ];
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');

        await page.click('button[data-kg-card-menu-item="Unsplash"]');

        // mock unsplash api
        await page.route('https://api.unsplash.com/photos?per_page=30', route => route.fulfill({
            status: 200,
            body: JSON.stringify(testData)
        }));
        await page.click('[data-kg-unsplash-insert-button]');
        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div>
                            <img
                                alt="a group of people walking down a street next to tall buildings"
                                src="http://127.0.0.1:5173/Koenig-editor-1.png" />
                        </div>
                        <figcaption>
                            <div data-testid="image-caption-editor" data-kg-allow-clickthrough="true">
                                <div>
                                    <div data-kg="editor">
                                        <div contenteditable="true" role="textbox" spellcheck="true" data-lexical-editor="true" data-koenig-dnd-container="true">
                                        <p dir="ltr">
                                        <span data-lexical-text="true">Photo by</span>
                                        <a href="https://unsplash.com/@jamillatrach" dir="ltr">
                                            <span data-lexical-text="true">Latrach Med Jamil</span>
                                        </a>
                                        <span data-lexical-text="true">/</span>
                                        <a
                                          href="https://unsplash.com/?utm_source=ghost&amp;utm_medium=referral&amp;utm_campaign=api-credit"
                                          dir="ltr">
                                          <span data-lexical-text="true">Unsplash</span>
                                        </a>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <button name="alt-toggle-button" type="button">Alt</button>
                        </figcaption>
                    </figure>
                    <div data-kg-card-toolbar="image"></div>
                </div>
            </div>
            <p><br /></p>
        `, {ignoreCardToolbarContents: true});
    });

    test('can insert tenor image', async () => {
        await mockTenorApi(page);
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');

        await page.click('button[data-kg-card-menu-item="GIF"]');

        // chose second gif from list
        await expect(await page.locator('[data-tenor-index="1"]')).toBeVisible();
        await page.click('[data-tenor-index="1"]');

        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div>
                            <img
                                alt=""
                                src="https://media.tenor.com/ocbMLlwniWQAAAAC/steve-harvey-oh.gif" />
                        </div>
                        <figcaption>
                            <div data-testid="image-caption-editor" data-kg-allow-clickthrough="true">
                                <div>
                                    <div data-kg="editor">
                                        <div contenteditable="true" role="textbox" spellcheck="true" data-lexical-editor="true" data-koenig-dnd-container="true">
                                            <p><br /></p>
                                        </div>
                                    </div>
                                    <div>Type caption for image (optional)</div>
                                </div>
                            </div>
                            <button name="alt-toggle-button" type="button">Alt</button>
                        </figcaption>
                    </figure>
                    <div data-kg-card-toolbar="image"></div>
                </div>
            </div>
            <p><br /></p>
        `, {ignoreCardToolbarContents: true});
    });

    test('can insert tenor image with key Tab', async () => {
        await mockTenorApi(page);
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');

        await page.click('button[data-kg-card-menu-item="GIF"]');

        // chose third gif from list
        await expect(await page.locator('[data-tenor-index="2"]')).toBeVisible();
        await page.keyboard.press('Tab');
        await page.keyboard.press('Tab');
        await page.keyboard.press('Tab');
        await page.keyboard.press('Enter');

        await expect(await page.getByTestId('image-card-populated')).toBeVisible();

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image">
                    <figure data-kg-card-width="regular">
                        <div>
                            <img
                                alt=""
                                src="https://media.tenor.com/Sm9aylrzSyMAAAAC/cats-animals.gif" />
                        </div>
                        <figcaption>
                            <div data-testid="image-caption-editor" data-kg-allow-clickthrough="true">
                                <div>
                                    <div data-kg="editor">
                                        <div contenteditable="true" role="textbox" spellcheck="true" data-lexical-editor="true" data-koenig-dnd-container="true">
                                            <p><br /></p>
                                        </div>
                                    </div>
                                    <div>Type caption for image (optional)</div>
                                </div>
                            </div>
                            <button name="alt-toggle-button" type="button">Alt</button>
                        </figcaption>
                    </figure>
                    <div data-kg-card-toolbar="image"></div>
                </div>
            </div>
            <p><br /></p>
        `, {ignoreCardToolbarContents: true});
    });

    test('can close tenor selector on Esc', async () => {
        await mockTenorApi(page);
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');

        await page.click('button[data-kg-card-menu-item="GIF"]');

        await expect(await page.getByTestId('tenor-selector')).toBeVisible();
        await page.keyboard.press('Escape');
        await expect(await page.getByTestId('tenor-selector')).toBeHidden();
    });

    test('can show tenor error', async () => {
        await mockTenorApi(page, {status: 400});
        await focusEditor(page);
        await page.click('[data-kg-plus-button]');

        await page.click('button[data-kg-card-menu-item="GIF"]');

        await expect(await page.getByTestId('tenor-selector-error')).toBeVisible();
    });

    test('can add snippet', async function () {
        // insert image
        await insertImage(page);

        // create snippet
        await page.keyboard.press('Escape');
        await createSnippet(page);

        // can insert card from snippet
        await page.keyboard.press('Enter');
        await page.keyboard.type('/snippet');
        await page.waitForSelector('[data-kg-cardmenu-selected="true"]');
        await page.keyboard.press('Enter');
        await expect(await page.locator('[data-kg-card="image"]')).toHaveCount(2);
    });

    test('can select caption text without scrolling', async function () {
        // Type in some text, so that we can scroll
        await focusEditor(page);
        await enterUntilScrolled(page);
        await insertImage(page);

        const paragraphCount = await page.locator('[data-kg="editor"] > div > p').count();

        await expectUnchangedScrollPosition(page, async () => {
            await page.keyboard.type('Captiontest--Captiontest');

            const captionEditor = page.locator('[data-testid="image-caption-editor"] [data-kg="editor"] p span');

            // Check contains text
            await expect(captionEditor).toHaveText('Captiontest--Captiontest');

            // Select the text
            // Get the bounding box of the span
            const box = await captionEditor.boundingBox();
            const y = box.y + box.height / 2;
            const startX = box.x + box.width / 2;
            const endX = box.x + box.width;

            await page.mouse.move(startX, y);
            await page.mouse.down();
            await page.mouse.move(endX, y);
            await page.mouse.up();

            await page.keyboard.type('world');

            // Check contains text
            await expect(captionEditor).toHaveText('Captiontest-world');

            // Press the enter key
            await page.keyboard.press('Enter');

            // Check if the image card is now deselected
            await expect(page.locator('[data-kg-card="image"]')).toHaveAttribute('data-kg-card-selected', 'false');

            // Check total paragraph count increased
            await expect(page.locator('[data-kg="editor"] > div > p')).toHaveCount(paragraphCount + 1);

            // Add some text
            await page.keyboard.type('last one');

            // Check contains text
            await expect(page.locator('[data-kg="editor"] > div > p:last-child').nth(1)).toHaveText('last one');
        });
    });

    test('can select caption text and make it italic', async function () {
        // Type in some text, so that we can scroll
        await focusEditor(page);
        await enterUntilScrolled(page);
        await insertImage(page);

        await expectUnchangedScrollPosition(page, async () => {
            await page.keyboard.type('Captiontest--Captiontest');

            const captionEditor = page.locator('[data-testid="image-caption-editor"] [data-kg="editor"] p span');

            // Check contains text
            await expect(captionEditor).toHaveText('Captiontest--Captiontest');

            // Select the left side of the text (deliberately a test in the other direction)
            // Get the bounding box of the span
            const box = await captionEditor.boundingBox();
            const y = box.y + box.height / 2;
            const startX = box.x + box.width / 2;
            const endX = box.x;

            await page.mouse.move(startX, y);
            await page.mouse.down();
            await page.mouse.move(endX, y);
            await page.mouse.up();

            // Click italic button
            await page.locator('[data-kg-toolbar-button="italic"]').click();

            // Check contains text
            await expect(captionEditor).toHaveText('-Captiontest');
            const italicSpan = page.locator('[data-testid="image-caption-editor"] [data-kg="editor"] p em').nth(0);
            await expect(italicSpan).toHaveText('Captiontest-');
        });
    });

    test('does not remove text when pressing ENTER in the middle of a caption', async function () {
        // Type in some text, so that we can scroll
        await focusEditor(page);
        await enterUntilScrolled(page);
        await insertImage(page);

        const paragraphCount = await page.locator('[data-kg="editor"] > div > p').count();

        await expectUnchangedScrollPosition(page, async () => {
            await page.keyboard.type('Captiontest--Captiontest');

            const captionEditor = page.locator('[data-testid="image-caption-editor"] [data-kg="editor"] p span');

            // Check contains text
            await expect(captionEditor).toHaveText('Captiontest--Captiontest');

            // Select the left side of the text (deliberately a test in the other direction)
            // Get the bounding box of the span
            const box = await captionEditor.boundingBox();
            const y = box.y + box.height / 2;
            const x = box.x + box.width / 2;

            await page.mouse.move(x, y);
            await page.mouse.down();
            await page.mouse.up();
            await page.keyboard.type('**'); // To test the cursor is at the middle, otherwise were not testing anything
            await expect(captionEditor).toHaveText('Captiontest-**-Captiontest');

            // Press the enter key
            await page.keyboard.press('Enter');
            await expect(captionEditor).toHaveText('Captiontest-**-Captiontest');

            // Check if the image card is now deselected
            await expect(page.locator('[data-kg-card="image"]')).toHaveAttribute('data-kg-card-selected', 'false');

            // Check total paragraph count increased
            await expect(page.locator('[data-kg="editor"] > div > p')).toHaveCount(paragraphCount + 1);

            // Add some text
            await page.keyboard.type('last one');

            // Check contains text
            await expect(page.locator('[data-kg="editor"] > div > p:last-child').nth(1)).toHaveText('last one');

            // Caption still ok?
            await expect(captionEditor).toHaveText('Captiontest-**-Captiontest');

            // Select the caption again
            // we need to do this again, because previously this caused a bug only if you selected the caption again
            await page.mouse.click(x, y);
            await page.keyboard.type('_'); // To test the cursor is at the middle, otherwise were not testing anything

            // Press the enter key
            await page.keyboard.press('Enter');
            await expect(captionEditor).toHaveText('Captiontest-*_*-Captiontest');
        });
    });

    test.describe('image should be a top-level element', () => {
        test('can insert image to nested list', async function () {
            await insertImage(page);
            const modifier = isMac ? 'Meta' : 'Control';
            await page.keyboard.press(`${modifier}+KeyC`);
            await page.keyboard.press('Enter');
            await page.keyboard.type('- First item');
            await page.keyboard.press('Enter');
            await page.keyboard.press('Tab');
            await page.keyboard.press(`${modifier}+KeyV`);

            // image should be top-level and shouldn't have ul as a parent
            await expect(page.locator('ul:has([data-kg-card="image"])')).toHaveCount(0);
        });

        test('can paste image with text from google doc', async function () {
            await focusEditor(page);
            await pasteHtml(
                page,
                '<meta charset=\'utf-8\'><meta charset="utf-8"><b style="font-weight:normal;" id="docs-internal-guid-338a96ba-7fff-6ecc-7aac-61f170b38b59"><p dir="ltr" style="line-height:1.38;margin-top:12pt;margin-bottom:12pt;"><span style="font-size:12pt;font-family:Arial;color:#000000;background-color:transparent;font-weight:400;font-style:normal;font-variant:normal;text-decoration:none;vertical-align:baseline;white-space:pre;white-space:pre-wrap;">Some text</span></p><br /><p dir="ltr" style="line-height:1.38;margin-top:12pt;margin-bottom:12pt;"><span style="font-size:12pt;font-family:Arial;color:#000000;background-color:transparent;font-weight:400;font-style:normal;font-variant:normal;text-decoration:none;vertical-align:baseline;white-space:pre;white-space:pre-wrap;"><span style="border:none;display:inline-block;overflow:hidden;width:248px;height:248px;"><img src="data:image/png;base64,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" width="248" height="248" style="margin-left:0px;margin-top:0px;" /></span></span></p><br /></b>',
                'text/html'
            );

            // image should be top-level and shouldn't have paragraph as a parent
            await expect(await page.getByTestId('image-card-populated')).toBeVisible();
            await expect(page.locator('p:has([data-kg-card="image"])')).toHaveCount(0);
        });
    });

    test.describe('caption', function () {
        test.beforeEach(async function () {
            const contentParam = encodeURIComponent(JSON.stringify({
                root: {
                    children: [{
                        type: 'image',
                        src: '/content/images/2022/11/koenig-lexical.jpg',
                        width: 3840,
                        height: 2160,
                        title: 'This is a title',
                        alt: 'This is some alt text',
                        caption: 'This is a <b>caption</b>',
                        cardWidth: 'wide'
                    }],
                    direction: null,
                    format: '',
                    indent: 0,
                    type: 'root',
                    version: 1
                }
            }));

            await initialize({page, uri: `/#/?content=${contentParam}`});
        });

        test('can delete node and undo without losing caption', async function () {
            await page.click('.koenig-lexical');
            await page.keyboard.press('ArrowUp');
            await page.waitForSelector('[data-kg-card="image"][data-kg-card-selected="true"]');
            await page.keyboard.press('Backspace');
            await expect(page.locator('[data-kg-card="image"]')).not.toBeVisible();
            await page.keyboard.press(`${ctrlOrCmd()}+z`);

            await expect(page.getByTestId('image-caption-editor')).toHaveText('This is a caption');
        });

        test('can toggle between alt and caption', async function () {
            await expect(page.getByTestId('image-caption-editor')).toHaveText('This is a caption');
            await page.click('[data-kg-card="image"]'); // alt toggle not shown until selected
            await page.click('[data-testid="alt-toggle-button"]');
            await expect(page.getByTestId('image-caption-editor')).toHaveValue('This is some alt text');
            await page.click('[data-testid="alt-toggle-button"]');
            await expect(page.getByTestId('image-caption-editor')).toHaveText('This is a caption');
        });
    });

    test.skip('can drag image card onto image card to create gallery', async function () {
        const filePath = path.relative(process.cwd(), __dirname + '/../fixtures/large-image.png');
        await focusEditor(page);

        const [fileChooser1] = await Promise.all([
            page.waitForEvent('filechooser'),
            await insertCard(page, {cardName: 'image', nth: 0})
        ]);
        await fileChooser1.setFiles([filePath]);

        await page.keyboard.press('Enter');

        const [fileChooser2] = await Promise.all([
            page.waitForEvent('filechooser'),
            await insertCard(page, {cardName: 'image', nth: 1})
        ]);
        await fileChooser2.setFiles([filePath]);

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="false" data-kg-card="image"></div>
            </div>
            <div data-lexical-decorator="true" contenteditable="false">
                <div data-kg-card-editing="false" data-kg-card-selected="true" data-kg-card="image"></div>
            </div>
            <p><br /></p>
        `, {ignoreCardContents: true});

        const imageCard1BBox = await page.locator('[data-kg-card="image"]').nth(0).boundingBox();
        const imageCard2BBox = await page.locator('[data-kg-card="image"]').nth(1).boundingBox();

        await dragMouse(page, imageCard2BBox, imageCard1BBox, 'middle', 'middle', true, 100, 100);

        await assertHTML(page, html`
            <div data-lexical-decorator="true" contenteditable="false" data-kg-card-width="wide">
                <div data-kg-card-editing="false" data-kg-card-selected="false" data-kg-card="gallery"></div>
            </div>
            <p><br /></p>
        `, {ignoreCardContents: true});
    });
});

async function insertImage(page, image = 'large-image.png') {
    const filePath = path.relative(process.cwd(), __dirname + `/../fixtures/${image}`);

    await focusEditor(page);
    await page.keyboard.type('image! ');

    const [fileChooser] = await Promise.all([
        page.waitForEvent('filechooser'),
        await page.click('button[name="placeholder-button"]')
    ]);
    await fileChooser.setFiles([filePath]);

    await expect(await page.getByTestId('image-card-populated')).toBeVisible();
}

function tenorTestData() {
    return (
        {
            locale: 'en',
            results: [
                {
                    id: '6897265628617702942',
                    title: '',
                    media_formats: {
                        tinygif: {
                            url: 'https://media.tenor.com/X7gCi8NE_h4AAAAM/cat-funny.gif',
                            duration: 0,
                            preview: '',
                            dims: [
                                220,
                                204
                            ],
                            size: 522164
                        },
                        gif: {
                            url: 'https://media.tenor.com/X7gCi8NE_h4AAAAC/cat-funny.gif',
                            duration: 0,
                            preview: '',
                            dims: [
                                498,
                                460
                            ],
                            size: 4870544
                        },
                        tinygifpreview: {
                            url: 'https://media.tenor.com/X7gCi8NE_h4AAAAF/cat-funny.png',
                            duration: 0,
                            preview: '',
                            dims: [
                                220,
                                204
                            ],
                            size: 21743
                        },
                        gifpreview: {
                            url: 'https://media.tenor.com/X7gCi8NE_h4AAAAe/cat-funny.png',
                            duration: 0,
                            preview: '',
                            dims: [
                                640,
                                592
                            ],
                            size: 141384
                        },
                        mp4: {
                            url: 'https://media.tenor.com/X7gCi8NE_h4AAAPo/cat-funny.mp4',
                            duration: 3.7,
                            preview: '',
                            dims: [
                                640,
                                592
                            ],
                            size: 754491
                        }
                    },
                    created: 1580334888.9161069,
                    content_description: 'Cat Funny GIF',
                    itemurl: 'https://tenor.com/view/cat-funny-fall-submit-play-gif-16179688',
                    url: 'https://tenor.com/bf3eS.gif',
                    tags: [
                        'cat',
                        'funny',
                        'fall',
                        'submit',
                        'play'
                    ],
                    flags: [],
                    hasaudio: false
                },
                {
                    id: '11657229184981764452',
                    title: '',
                    media_formats: {
                        tinygifpreview: {
                            url: 'https://media.tenor.com/ocbMLlwniWQAAAAF/steve-harvey-oh.png',
                            duration: 0,
                            preview: '',
                            dims: [
                                220,
                                124
                            ],
                            size: 11388
                        },
                        tinygif: {
                            url: 'https://media.tenor.com/ocbMLlwniWQAAAAM/steve-harvey-oh.gif',
                            duration: 0,
                            preview: '',
                            dims: [
                                220,
                                124
                            ],
                            size: 173121
                        },
                        gifpreview: {
                            url: 'https://media.tenor.com/ocbMLlwniWQAAAAe/steve-harvey-oh.png',
                            duration: 0,
                            preview: '',
                            dims: [
                                640,
                                360
                            ],
                            size: 65023
                        },
                        gif: {
                            url: 'https://media.tenor.com/ocbMLlwniWQAAAAC/steve-harvey-oh.gif',
                            duration: 0,
                            preview: '',
                            dims: [
                                498,
                                280
                            ],
                            size: 1669457
                        },
                        mp4: {
                            url: 'https://media.tenor.com/ocbMLlwniWQAAAPo/steve-harvey-oh.mp4',
                            duration: 2.4,
                            preview: '',
                            dims: [
                                640,
                                360
                            ],
                            size: 377541
                        }
                    },
                    created: 1600453059.6729331,
                    content_description: 'Steve Harvey Oh GIF',
                    itemurl: 'https://tenor.com/view/steve-harvey-oh-you-crazy-point-stop-gif-18502036',
                    url: 'https://tenor.com/bpNn6.gif',
                    tags: [
                        'Steve Harvey',
                        'oh',
                        'You Crazy',
                        'point',
                        'stop'
                    ],
                    flags: [],
                    hasaudio: false
                },
                {
                    id: '5363605506377337635',
                    title: '',
                    media_formats: {
                        gif: {
                            url: 'https://media.tenor.com/Sm9aylrzSyMAAAAC/cats-animals.gif',
                            duration: 0,
                            preview: '',
                            dims: [
                                498,
                                431
                            ],
                            size: 1574979
                        },
                        gifpreview: {
                            url: 'https://media.tenor.com/Sm9aylrzSyMAAAAe/cats-animals.png',
                            duration: 0,
                            preview: '',
                            dims: [
                                640,
                                554
                            ],
                            size: 153379
                        },
                        tinygifpreview: {
                            url: 'https://media.tenor.com/Sm9aylrzSyMAAAAF/cats-animals.png',
                            duration: 0,
                            preview: '',
                            dims: [
                                220,
                                190
                            ],
                            size: 25196
                        },
                        tinygif: {
                            url: 'https://media.tenor.com/Sm9aylrzSyMAAAAM/cats-animals.gif',
                            duration: 0,
                            preview: '',
                            dims: [
                                220,
                                190
                            ],
                            size: 236117
                        },
                        mp4: {
                            url: 'https://media.tenor.com/Sm9aylrzSyMAAAPo/cats-animals.mp4',
                            duration: 1.2,
                            preview: '',
                            dims: [
                                640,
                                554
                            ],
                            size: 265062
                        }
                    },
                    created: 1616817775.272332,
                    content_description: 'Cats Animals GIF',
                    itemurl: 'https://tenor.com/view/cats-animals-reaction-wow-surprised-gif-20914356',
                    url: 'https://tenor.com/bzUWu.gif',
                    tags: [
                        'cats',
                        'animals',
                        'reaction',
                        'wow',
                        'surprised'
                    ],
                    flags: [],
                    hasaudio: false
                }
            ]
        }
    );
}

const tenorUrl = 'https://tenor.googleapis.com/v2/featured?q=excited&media_filter=minimal&key=xxx&client_key=ghost-editor&contentfilter=off';
async function mockTenorApi(page, {status} = {status: 200}) {
    await page.route(tenorUrl, route => route.fulfill({
        status,
        body: JSON.stringify(tenorTestData())
    }));
}
