import { useCallback, useEffect } from "react";
import { Box, useTheme } from "@mui/material";
import { tokens } from "../../theme";
import { treeViewerSettings } from "../../data/settings";
import { insertDividerLines } from "./utils";
import ReactFlow, {
  // MiniMap,
  Controls,
  Background,
  useNodesState,
  useEdgesState,
  addEdge,
  MarkerType,
} from "reactflow";
import { useDrag } from "../DragProvider";

import "reactflow/dist/style.css";

import { DividerNode, PassiveNode, ActiveNode, SwitchNode } from "./NodeTypes";

const proOptions = { hideAttribution: true };

const typeToNodeType = {
  DIVIDER: "dividerNode",
  PASSIVE: "passiveNode",
  ACTIVE: "activeNode",
  SWITCH: "switchNode",
};
const nodeTypes = {
  dividerNode: DividerNode,
  passiveNode: PassiveNode,
  activeNode: ActiveNode,
  switchNode: SwitchNode,
};

const TreeObjectToFlowNode = (treeObject, assignedSkills, colors) => {
  var points = 0;
  if (treeObject.pre_filled === 1) {
    points = treeObject.max_points;
  } else {
    points = assignedSkills.hasOwnProperty(treeObject.order_id)
      ? assignedSkills[treeObject.order_id]
      : 0;
  }
  return {
    id: "n" + treeObject.order_id,
    type: typeToNodeType[treeObject.talent_type],
    position: {
      x: 0.5 * (treeObject.column + 1) * treeViewerSettings.gridSpace,
      y: 0.5 * (treeObject.row + 1) * treeViewerSettings.gridSpace,
    },
    data: {
      id: treeObject.order_id,
      size: treeViewerSettings.nodeSize,
      talentType: treeObject.talent_type,
      description: treeObject.description,
      descriptionSwitch: treeObject.description_switch,
      iconName: treeObject.icon_name,
      iconNameSwitch: treeObject.icon_name_switch,
      points: points,
      maxPoints: treeObject.max_points,
      name: treeObject.name,
      nameSwitch: treeObject.name_switch,
      nodeID: treeObject.node_id,
      preFilled: treeObject.pre_filled,
      requiredPoints: treeObject.required_points,
      row: treeObject.row,
      column: treeObject.column,
      childIDs: treeObject.child_ids,
      parentIDs: treeObject.parent_ids,
      isConnectable: false,
      // NodeTypes data
      borderColor:
        points === 0
          ? colors.treeColors.grey
          : points >= treeObject.max_points
          ? colors.treeColors.gold
          : colors.treeColors.green,
      pointsDisplayLow:
        points > treeObject.max_points ? treeObject.max_points : points,
      pointsDisplayHigh: treeObject.max_points,
      opacity: points === 0 ? 0.4 : 1.0,
    },
  };
};

const TreeObjectToFlowEdges = (
  treeObject,
  treeData,
  assignedSkills,
  colors
) => {
  var points = 0;
  if (treeObject.pre_filled === 1) {
    points = treeObject.max_points;
  } else {
    points = assignedSkills.hasOwnProperty(treeObject.order_id)
      ? assignedSkills[treeObject.order_id]
      : 0;
  }
  var edges = [];
  for (const childIndex of treeObject.child_ids) {
    if (!treeData.hasOwnProperty(childIndex)) {
      continue;
    }
    var arrowColor = colors.treeColors.greyDim;
    if (points >= treeObject.max_points) {
      if (assignedSkills.hasOwnProperty(childIndex)) {
        if (
          assignedSkills[childIndex] > 0 &&
          assignedSkills[childIndex] < treeData[childIndex].max_points
        ) {
          arrowColor = colors.treeColors.green;
        } else if (
          assignedSkills[childIndex] >= treeData[childIndex].max_points
        ) {
          arrowColor = colors.treeColors.gold;
        }
      } else if (treeData[childIndex].pre_filled === 1) {
        arrowColor = colors.treeColors.gold;
      }
    }
    edges.push({
      id: `e${treeObject.order_id}-${childIndex}`,
      source: `n${treeObject.order_id}`,
      target: `n${childIndex}`,
      type: "straight",
      markerEnd: {
        type: MarkerType.ArrowClosed,
        width: 10,
        height: 10,
        color: arrowColor,
      },
      style: {
        strokeWidth: 6,
        stroke: arrowColor,
      },
    });
  }
  return edges;
};

const BuildViewer = ({ treeData, assignedSkills, isClassTree }) => {
  const theme = useTheme();
  const colors = tokens(theme.palette.mode);
  const [nodes, setNodes, onNodesChange] = useNodesState([]);
  const [edges, setEdges, onEdgesChange] = useEdgesState([]);
  const { isDragging, isReadyToDrag, setIsDragging, setDragReady } = useDrag();

  useEffect(() => {
    setDragReady(false);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  useEffect(() => {
    if (!assignedSkills) return;
    var newNodes = [];
    var newEdges = [];
    for (const rawNode of Object.values(treeData)) {
      newNodes.push(
        TreeObjectToFlowNode(rawNode, assignedSkills[1 - isClassTree], colors)
      );
      newEdges.push(
        ...TreeObjectToFlowEdges(
          rawNode,
          treeData,
          assignedSkills[1 - isClassTree],
          colors
        )
      );
    }
    insertDividerLines(newNodes, newEdges, colors);
    setNodes(newNodes);
    setEdges(newEdges);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [treeData, assignedSkills]);

  const onConnect = useCallback(
    (params) => setEdges((eds) => addEdge(params, eds)),
    [setEdges]
  );

  const onMoveStart = () => {
    if (!isReadyToDrag) {
      setDragReady(true);
      return;
    }
    if (!isDragging) {
      setIsDragging(true);
    }
  };

  const onMoveEnd = () => {
    setIsDragging(false);
  };

  return (
    <Box sx={{ width: "100%", height: "100%" }} p="5px">
      <ReactFlow
        proOptions={proOptions}
        nodeTypes={nodeTypes}
        nodes={nodes}
        edges={edges}
        onNodesChange={onNodesChange}
        onEdgesChange={onEdgesChange}
        onConnect={onConnect}
        onMove={onMoveStart}
        onMoveEnd={onMoveEnd}
        snapGrid={[treeViewerSettings.gridSpace, treeViewerSettings.gridSpace]}
        snapToGrid
        fitView
        nodesDraggable={false}
        zoomOnScroll={false}
        panOnScroll={false}
        preventScrolling={false}
      >
        <Controls
          showInteractive={false}
          onZoomIn={() => {
            setTimeout(onMoveEnd, 500);
          }}
          onZoomOut={() => {
            setTimeout(onMoveEnd, 500);
          }}
          onFitView={() => {
            setTimeout(onMoveEnd, 500);
          }}
        />
        {/* <MiniMap /> */}
        <Background variant="dots" gap={24} size={1} />
      </ReactFlow>
    </Box>
  );
};

export default BuildViewer;
