/*
    WoW Talent Tree Manager is an application for creating/editing/sharing talent trees and setups.
    Copyright(C) 2022 Tobias Mielich

    This program is free software : you can redistribute it and /or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see < https://www.gnu.org/licenses/>.

    Contact via https://github.com/TobiasM95/WoW-Talent-Tree-Manager/discussions or BuffMePls#2973 on Discord
*/

#include "TalentTrees.h"
#include "TTMEnginePresets.h"

#include <regex>
#include <iostream>
#include <vector>
#include <algorithm>
#include <sstream> 
#include <fstream>
#include "Windows.h"
#include <chrono>
#include <thread>

namespace Engine {
    //Tree/talent helper functions

    /*
    DFS cyclicity check of a tree
    */
    bool checkIfTreeHasCycle(const TalentTree& tree) {
        TreeCycleCheckFormat tccf = createTreeCycleCheckFormat(tree);
        return checkCyclicity(tccf);
    }

    /*
    DFS cyclicity check of a tree if a talent would be inserted
    */
    bool checkIfTalentInsertsCycle(const TalentTree& tree, Talent_s talent) {
        TreeCycleCheckFormat tccf = createTreeCycleCheckFormat(tree, talent);
        return checkCyclicity(tccf);
    }

    /*
    DFS cyclicity check of a potential tree generated by a tree representation string
    */
    bool checkIfParseStringProducesCycle(std::string treeRep) {
        TreeCycleCheckFormat tccf = createTreeCycleCheckFormat(treeRep);
        return checkCyclicity(tccf);
    }

    /*
    Helper function to create a simple DFS cyclicity check data structure with a tree
    */
    TreeCycleCheckFormat createTreeCycleCheckFormat(const TalentTree& tree) {
        TreeCycleCheckFormat tccf;
        std::map<int, int> indexMap;
        int index = 0;
        for (auto& talent : tree.orderedTalents) {
            indexMap[talent.first] = index;
            index++;
        }
        for (auto& talent : tree.orderedTalents) {
            tccf.talents.push_back(0);
            std::vector<int> childVector;
            for (auto& child : talent.second->children) {
                childVector.push_back(indexMap[child->index]);
            }
            tccf.children.push_back(childVector);
        }
        return tccf;
    }

    /*
    Helper function to create a simple DFS cyclicity check data structure with a tree and a talent
    */
    TreeCycleCheckFormat createTreeCycleCheckFormat(const TalentTree& tree, Talent_s talent) {
        TreeCycleCheckFormat tccf;
        std::map<int, int> indexMap;
        int index = 0;
        for (auto& talent : tree.orderedTalents) {
            indexMap[talent.first] = index;
            index++;
        }
        for (auto& talent : tree.orderedTalents) {
            tccf.talents.push_back(0);
            std::vector<int> childVector;
            for (auto& child : talent.second->children) {
                childVector.push_back(indexMap[child->index]);
            }
            tccf.children.push_back(childVector);
        }
        tccf.talents.push_back(0);
        std::vector<int> childVector;
        for (auto& child : talent->children) {
            childVector.push_back(indexMap[child->index]);
        }
        tccf.children.push_back(childVector);
        for (auto& parent : talent->parents) {
            tccf.children[indexMap[parent->index]].push_back(static_cast<int>(tccf.talents.size() - 1));
        }
        return tccf;
    }

    TreeCycleCheckFormat createTreeCycleCheckFormat(std::string treeRep) {
        std::vector<std::string> treeComponents = splitString(treeRep, ";");
        TreeCycleCheckFormat tccf;
        std::map<int, int> indexMap;
        int numTalents = std::stoi(splitString(treeComponents[0], ":")[6]);
        for (int i = 1; i < numTalents + 1; i++) {
            if (treeComponents[i] == "")
                continue;
            int talentIndex = std::stoi(splitString(treeComponents[i], ":")[0]);
            indexMap[talentIndex] = i - 1;
        }
        for (int i = 1; i < numTalents + 1; i++) {
            if (treeComponents[i] == "")
                continue;
            tccf.talents.push_back(0);
            std::vector<int> childVector;
            std::vector<std::string> childIndexStrings = splitString(splitString(treeComponents[i], ":")[9], ",");
            for (auto& childIndex : childIndexStrings) {
                if (childIndex == "")
                    continue;
                childVector.push_back(indexMap[std::stoi(childIndex)]);
            }
            tccf.children.push_back(childVector);
        }
        return tccf;
    }

    /*
    Core DFS cyclicity check function, see https://en.wikipedia.org/wiki/Topological_sorting for pseudo code
    */
    bool checkCyclicity(TreeCycleCheckFormat tree) {
        bool hasCycle = false;
        bool done = false;
        while (!hasCycle) {
            done = true;
            for (int i = 0; i < tree.talents.size(); i++) {
                if (tree.talents[i] == 0) {
                    hasCycle = cycleCheckVisitTalent(i, tree.talents, tree.children);
                    done = false;
                    break;
                }
            }
            if (hasCycle || done) 
                break;
        }
        return hasCycle;
    }

    /*
    Auxilliary DFS cyclicity check function, see https://en.wikipedia.org/wiki/Topological_sorting for pseudo code
    */
    bool cycleCheckVisitTalent(int talentIndex, std::vector<int>& talents, vec2d<int> children) {
        bool hasCycle = false;
        if (talents[talentIndex] == 1)
            return true;
        if (talents[talentIndex] == 2)
            return false;
        talents[talentIndex] = 1;
        for (auto& child : children[talentIndex]) {
            hasCycle = cycleCheckVisitTalent(child, talents, children);
            if (hasCycle)
                return true;
        }

        talents[talentIndex] = 2;
        return false;
    }


    //TTMTODO: combine update node count with update ordered talent list
    void updateNodeCountAndMaxTalentPointsAndMaxID(TalentTree& tree) {
        std::unordered_map<int, int> nodeTalentPoints;
        std::unordered_map<int, std::unordered_set<int>> talentIndexInRow;
        int maxID = 0;
        int maxCol = 1;
        std::map<int, int> preFilledTalentCountMap;
        for (auto& root : tree.talentRoots) {
            countNodesRecursively(nodeTalentPoints, maxID, maxCol, preFilledTalentCountMap, talentIndexInRow, root);
        }
        tree.nodeCount = static_cast<int>(nodeTalentPoints.size());
        int maxTalentPoints = 0;
        for (auto& node : nodeTalentPoints) {
            maxTalentPoints += node.second;
        }
        tree.maxTalentPoints = maxTalentPoints;
        tree.maxID = maxID;
        tree.maxCol = maxCol;
        int preFilledTalentCount = 0;
        for (auto& tCount : preFilledTalentCountMap) {
            preFilledTalentCount += tCount.second;
        }
        tree.preFilledTalentPoints = preFilledTalentCount;
        tree.talentsPerRow.clear();
        for (auto& rowIndices : talentIndexInRow) {
            tree.talentsPerRow[rowIndices.first] = static_cast<int>(rowIndices.second.size());
        }

    }

    void countNodesRecursively(std::unordered_map<int, int>& nodeTalentPoints, int& maxID, int& maxCol, std::map<int, int>& preFilledTalentCountMap, std::unordered_map<int, std::unordered_set<int>>& talentsPerRow, Talent_s talent) {
        nodeTalentPoints[talent->index] = talent->maxPoints;
        if (talent->index >= maxID)
            maxID = talent->index + 1;

        if (talent->column + 1> maxCol)
            maxCol = talent->column + 1;

        talentsPerRow[talent->row].insert(talent->index);
        if (talent->preFilled) {
            preFilledTalentCountMap[talent->index] = talent->maxPoints;
        }
        /*if (talentsPerRow.count(talent->row)) {
        }
        else {
            talentsPerRow[talent->row] = std::unordered_set<int>();
        }*/

        for (auto& child : talent->children)
            countNodesRecursively(nodeTalentPoints, maxID, maxCol, preFilledTalentCountMap, talentsPerRow, child);
    }

    void updateOrderedTalentList(TalentTree& tree) {
        for (auto& root : tree.talentRoots) {
            orderTalentsRecursively(tree.orderedTalents, root);
        }
    }

    void updateRequirementSeparatorInfo(TalentTree& tree) {
        tree.requirementSeparatorInfo.clear();
        if (tree.orderedTalents.size() == 0) {
            return;
        }
        //stores the min and max height of each occurring point requirement (key of map) to calculate the separator positions next
        std::map<int, std::pair<int, int>> requirementRowMap;
        for (auto& talent : tree.orderedTalents) {
            int tRow = talent.second->row;
            int tReq = talent.second->pointsRequired;
            if (!requirementRowMap.count(tReq)) {
                requirementRowMap[tReq] = { tRow, tRow };
            }
            else {
                if (tRow < requirementRowMap[tReq].first) {
                    requirementRowMap[tReq].first = tRow;
                }
                if (tRow > requirementRowMap[tReq].second) {
                    requirementRowMap[tReq].second = tRow;
                }
            }
        }

        std::map<int, std::pair<int, int>>::iterator it1;
        std::map<int, std::pair<int, int>>::iterator it2;
        for (it1 = std::next(requirementRowMap.begin()); it1 != requirementRowMap.end();it1++) {
            bool valid = true;
            for (it2 = requirementRowMap.begin(); it2 != requirementRowMap.end(); it2++) {
                if (it2->first == it1->first) {
                    continue;
                }
                if (it2->first > it1->first) {
                    //regions with lower pts requirement should end before or at the same point as it2
                    if (it2->second.first < it1->second.first) {
                        valid = false;
                        break;
                    }
                }
                
                if (it2->first < it1->first) {
                    //regions with higher pts requirement should start after it2
                    if (it2->second.second >= it1->second.first) {
                        valid = false;
                        break;
                    }
                }
                
            }
            if (valid) {
                int closestRow = -INT_MAX;
                for (it2 = requirementRowMap.begin(); it2 != requirementRowMap.end(); it2++) {
                    //select the closest region that has a lower pts requirement
                    if (it2->first >= it1->first) {
                        continue;
                    }
                    if (it2->second.second > closestRow) {
                        closestRow = it2->second.second;
                    }
                }
                if (closestRow > -INT_MAX) {
                    tree.requirementSeparatorInfo.push_back({ it1->first, 0.5f * (it1->second.first + closestRow + 1) });
                }
            }
        }
    }

    void orderTalentsRecursively(std::map<int, Talent_s>& talentMap, Talent_s talent) {
        talentMap[talent->index] = talent;
        for (auto& child : talent->children) {
            orderTalentsRecursively(talentMap, child);
        }
    }

    void addChild(Talent_s parent, Talent_s child) {
        parent->children.push_back(child);
    }
    void addParent(Talent_s child, Talent_s parent) {
        child->parents.push_back(parent);
    }
    void pairTalents(Talent_s parent, Talent_s child) {
        parent->children.push_back(child);
        child->parents.push_back(parent);
    }

    /*
    Transforms a skilled talent tree into a string. That string does not contain the tree structure, just the selected talents.
    */
    std::string getTalentString(const TalentTree& tree) {
        std::unordered_map<std::string, int> treeRepresentation;
        for (auto& root : tree.talentRoots) {
            addTalentAndChildrenToMap(root, treeRepresentation);
        }
        return unorderedMapToString(treeRepresentation, true);
    }

    /*
    Prints some informations about a specific tree.
    */
    void printTree(const TalentTree& tree) {
        std::cout << tree.name << std::endl;
        std::cout << "Unspent talent points:\t" << tree.unspentTalentPoints << std::endl;
        std::cout << "Spent talent points:\t" << tree.spentTalentPoints << std::endl;
        std::cout << getTalentString(tree) << std::endl;
    }

    /*
    Helper function to replace all ":" with "__/__" to not disturb the parsing
    */
    inline std::string cleanString(std::string s) {
        s = std::regex_replace(s, std::regex(":"), "__cl__");
        s = std::regex_replace(s, std::regex("\n"), "__n__");
        s = std::regex_replace(s, std::regex(","), "__cm__");
        s = std::regex_replace(s, std::regex(";"), "__sc__");
        return s;
    }

    /*
    Helper function to replace all "__/__" with ":" to not disturb the parsing
    */
    inline std::string restoreString(std::string s) {
        s = std::regex_replace(s, std::regex("__cl__"), ":");
        s = std::regex_replace(s, std::regex("__n__"), "\n");
        s = std::regex_replace(s, std::regex("__cm__"), ",");
        s = std::regex_replace(s, std::regex("__sc__"), ";");
        return s;
    }

    /*
    Creates a full tree string representation that's compatible to import trees in parse tree
    */
    std::string createTreeStringRepresentation(TalentTree& tree) {
        std::stringstream treeRep;
        treeRep << Presets::TTM_VERSION << ":" << tree.presetName << ":" << static_cast<int>(tree.type) << ":" << tree.name << ":" << cleanString(tree.treeDescription) << ":" << cleanString(tree.loadoutDescription) << ":" << tree.orderedTalents.size() << ":" << tree.loadout.size() << ";";
        if (tree.presetName == "custom") {
            for (auto& talent : tree.orderedTalents) {
                treeRep << talent.first << ":" << cleanString(talent.second->name);
                if (talent.second->type == TalentType::SWITCH) {
                    treeRep << "," << cleanString(talent.second->nameSwitch);
                }
                treeRep << ":";
                for (int i = 0; i < talent.second->descriptions.size() - 1; i++) {
                    treeRep << cleanString(talent.second->descriptions[i]) << ",";
                }
                treeRep << cleanString(talent.second->descriptions[talent.second->descriptions.size() - 1]) << ":";
                treeRep << static_cast<int>(talent.second->type) << ":" << talent.second->row << ":" << talent.second->column << ":";
                treeRep << talent.second->maxPoints << ":" << talent.second->pointsRequired << ":" << static_cast<int>(talent.second->preFilled) << ":";
                if (talent.second->parents.size() > 0) {
                    for (int i = 0; i < talent.second->parents.size() - 1; i++) {
                        treeRep << talent.second->parents[i]->index << ",";
                    }
                    treeRep << talent.second->parents[talent.second->parents.size() - 1]->index;
                }
                treeRep << ":";
                if (talent.second->children.size() > 0) {
                    for (int i = 0; i < talent.second->children.size() - 1; i++) {
                        treeRep << talent.second->children[i]->index << ",";
                    }
                    treeRep << talent.second->children[talent.second->children.size() - 1]->index;
                }
                treeRep << ":" << cleanString(talent.second->iconName.first) << "," << cleanString(talent.second->iconName.second);
                treeRep << ";";
            }
        }
        for (auto& skillset : tree.loadout) {
            if (!validateSkillset(tree, skillset)) {
                continue;
            }
            treeRep << skillset->name << "," + std::to_string(skillset->levelCap) << "," + std::to_string(skillset->useLevelCap);
            for (auto& skillPoint : skillset->assignedSkillPoints) {
                treeRep << ":" << skillPoint.second;
            }
            treeRep << ";";
        }
        return treeRep.str();
    }

    /*
    Creates a human readable tree string that can be used to share the idea of a tree, not suitable for imports
    */
    std::string createReadableTreeString(TalentTree& tree) {
        std::stringstream treeStream;
        treeStream << tree.name << "\n";
        treeStream << "Created with TTM Version " << Presets::TTM_VERSION << "\n";
        treeStream << "Number of talents: " << tree.orderedTalents.size() << "\n";
        treeStream << "Description:\n" << tree.treeDescription << "\n";
        treeStream << "Loadout description:\n" << tree.loadoutDescription << "\n\n";

        treeStream << "Talents:" << "\n";
        for (auto& talent : tree.orderedTalents) {
            treeStream << "#" << talent.first << " " << talent.second->name;
            if (talent.second->type == TalentType::SWITCH) {
                treeStream << " / " << talent.second->nameSwitch;
            }
            treeStream << "\n";
            switch (talent.second->type) {
            case TalentType::ACTIVE: treeStream << "Talent type: active" << "\n"; break;
            case TalentType::PASSIVE: treeStream << "Talent type: passive" << "\n"; break;
            case TalentType::SWITCH: treeStream << "Talent type: choice" << "\n"; break;
            }
            treeStream << "Max points: " << talent.second->maxPoints << " - Points requirement: " << talent.second->pointsRequired << "\n";
            treeStream << "Descriptions:" << "\n";
            for (auto& desc : talent.second->descriptions) {
                treeStream << desc << "\n";
            }
            treeStream << "Parent talents: ";
            for (auto& parent : talent.second->parents) {
                treeStream << parent->index << ", ";
            }
            treeStream << "\n";
            treeStream << "Child talents: ";
            for (auto& child : talent.second->children) {
                treeStream << child->index << ", ";
            }
            treeStream << "\n\n";
        }

        return treeStream.str();
    }

    /*
    Helper function that adds a talent and its children recursively to a map (and adds talent switch information if existing).
    */
    void addTalentAndChildrenToMap(Talent_s talent, std::unordered_map<std::string, int>& treeRepresentation) {
        std::string talentIndex = std::to_string(talent->index);
        if (talent->type == TalentType::SWITCH) {
            talentIndex += std::to_string(talent->talentSwitch);
        }
        treeRepresentation[talentIndex] = talent->points;
        for (int i = 0; i < talent->children.size(); i++) {
            addTalentAndChildrenToMap(talent->children[i], treeRepresentation);
        }
    }

    /*
    Helper function that transforms a map that includes talents and their skill points to their respective string representation.
    */
    std::string unorderedMapToString(const std::unordered_map<std::string, int>& treeRepresentation, bool sortOutput) {
        std::vector<std::string> talentsAndPoints;
        talentsAndPoints.reserve(treeRepresentation.size());
        for (auto& talentRepresentation : treeRepresentation) {
            talentsAndPoints.push_back(talentRepresentation.first + std::to_string(talentRepresentation.second));
        }
        if (sortOutput) {
            std::sort(talentsAndPoints.begin(), talentsAndPoints.end());
        }
        std::stringstream treeString;
        for (auto& talentRepresentation : talentsAndPoints) {
            treeString << talentRepresentation << ";";
        }
        return treeString.str();
    }

    /*
    Helper function that creates a talent with the given index name and max points.
    */
    Talent_s createTalent(TalentTree& tree, std::string name, int maxPoints) {
        Talent_s talent = std::make_shared<Talent>();
        talent->index = tree.maxID;
        tree.nodeCount++;
        tree.maxTalentPoints += maxPoints;
        tree.maxID++;
        talent->maxPoints = maxPoints;
        talent->row = tree.maxID / 3;
        talent->column = tree.maxID % 3;
        tree.orderedTalents[talent->index] = talent;
        return talent;
    }

    TalentTree restorePreset(const TalentTree& tree, std::string treeRep) {
        if (!repairTreeStringFormat(treeRep)) {
            TalentTree emptyTree;
            emptyTree.name = "Load error";
            emptyTree.treeDescription = "Error when trying to load old version tree preset. It is recommended to update TTM and presets to the newest version.";
            emptyTree.loadoutDescription = "Error when trying to load old version tree preset. It is recommended to update TTM and presets to the newest version.";
            return emptyTree;
        }
        TalentTree presetTree = parseCustomTree(treeRep);
        presetTree.activeSkillsetIndex = tree.activeSkillsetIndex;
        presetTree.loadout.clear();
        for (auto& skillset : tree.loadout) {
            if (validateSkillset(presetTree, skillset)) {
                presetTree.loadout.push_back(skillset);
            }
        }
        if (presetTree.activeSkillsetIndex >= presetTree.loadout.size()) {
            presetTree.activeSkillsetIndex = static_cast<int>(presetTree.loadout.size() - 1);
        }
        presetTree.loadoutDescription = tree.loadoutDescription;
        return presetTree;
    }

    TalentTree loadTreePreset(std::string treeRep) {
        if (!repairTreeStringFormat(treeRep)) {
            TalentTree emptyTree;
            emptyTree.name = "Load error";
            emptyTree.treeDescription = "Error when trying to load old version tree preset. It is recommended to update TTM and presets to the newest version.";
            emptyTree.loadoutDescription = "Error when trying to load old version tree preset. It is recommended to update TTM and presets to the newest version.";
            return emptyTree;
        }
        return parseCustomTree(treeRep);
    }

    bool validateAndRepairTreeStringFormat(std::string treeRep) {
        if (!repairTreeStringFormat(treeRep)) {
            return false;
        }
        //first check tree meta info line
        std::vector<std::string> treeParts = splitString(treeRep, ";");
        std::vector<std::string> metaInfo = splitString(treeParts[0], ":");
        if (metaInfo.size() != 8) {
            return false;
        }
        if (metaInfo[0] != "custom") {
            try {
                Presets::LOAD_PRESETS()[metaInfo[0]];
            }
            catch (std::logic_error& e) {
                //TTMNOTE: do something with e?
                return false;
            }
        }
        if (metaInfo[2] != "0" && metaInfo[2] != "1") {
            return false;
        }
        if (metaInfo[3].find_first_not_of(
            "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 ():'-"
        ) != std::string::npos) {
            return false;
        }
        if (metaInfo[6].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        if (metaInfo[7].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        int numTalents = std::stoi(metaInfo[6]);
        int numSkillsets = std::stoi(metaInfo[7]);
        if (metaInfo[1] == "custom") {
            if (treeParts.size() != 1 + numTalents + numSkillsets && treeParts.size() != 2 + numTalents + numSkillsets) {
                return false;
            }
            //check all talents individually
            for (int i = 1; i < 1 + numTalents; i++) {
                if (treeParts[i] == "" && i != treeParts.size() - 1) {
                    return false;
                }
                if (!validateTalentStringFormat(treeParts[i])) {
                    return false;
                }
            }
        }
        else {
            if (treeParts.size() != 1 + numSkillsets && treeParts.size() != 2 + numSkillsets) {
                return false;
            }
        }
        //check all skillsets individually
        for (int i = 1 + numTalents; i < treeParts.size(); i++) {
            if (treeParts[i] == "" && i == treeParts.size() - 1) {
                break;
            }
            if (treeParts[i] == "" && i != treeParts.size() - 1) {
                return false;
            }
            if (!validateSkillsetStringFormat(numTalents, treeParts[i])) {
                return false;
            }
        }

        return true;
    }

    bool repairTreeStringFormat(std::string& treeRep) {
        std::vector<std::string> treeParts = splitString(treeRep, ";");
        std::vector<std::string> metaInfo = splitString(treeParts[0], ":");
        //This emulates a switch case with fallthrough with if statements and strings
        bool repairVersionStart = false;
        bool repairSuccess = false;
        if (repairVersionStart || metaInfo.size() != 8) {
            repairSuccess = repairToV120(treeRep);
            repairVersionStart = true;
            if (!repairSuccess) {
                return false;
            }
        }
        if (repairVersionStart || metaInfo[0] == "1.2.0") {
            repairSuccess = repairToV121(treeRep);
            repairVersionStart = true;
            if (!repairSuccess) {
                return false;
            }
        }
        /*
        In the future you can easily append:
        if (repairVersionStart || metaInfo[0] == "1.2.0" ) {
            repairSuccess = repairToV130(treeRep);
            repairVersionStart = true;
            if (!repairSuccess) {
                return false;
            }
        }
        */
        return true;
    }
    /*
    V. 1.2.1 didn't change any tree string formats. Just update the version number.
    */
    bool repairToV121(std::string& treeRep) {
        std::stringstream treeRepStream;
        std::stringstream treePartStream;
        std::vector<std::string> treeParts = splitString(treeRep, ";");
        //Expect meta info like: custom/preset : class/spec tree : tree name : tree desc : loadout desc : talent count : skillset count;
        std::vector<std::string> metaInfo = splitString(treeParts[0], ":");

        treePartStream << Presets::TTM_VERSION;
        for (int i = 1; i < metaInfo.size(); i++) {
            treePartStream << ":" << metaInfo[i];
        }
        treeParts[0] = treePartStream.str();
        treePartStream.str(std::string());
        treePartStream.clear();

        //put everything back into treeRep
        for (int i = 0; i < treeParts.size(); i++) {
            if (treeParts[i] != "") {
                treeRepStream << treeParts[i] << ";";
            }
        }

        treeRep = treeRepStream.str();
        return true;
    }

    /*
    V. 1.1.0 broke all tree strings that came before and gave no guarantee for them to work but guaranteed to make future versions not break v. 1.1.0 strings.
    Therefore, fix strings from version 1.1.0 to 1.2.0: include a new version string and include talent icon names.
    returns true if repair seemed to be successful, false if string definitely didn't match v. 1.1.0 format.
    */
    bool repairToV120(std::string& treeRep) {
        std::stringstream treeRepStream;
        std::stringstream treePartStream;
        std::vector<std::string> treeParts = splitString(treeRep, ";");
        //Expect meta info like: custom/preset : class/spec tree : tree name : tree desc : loadout desc : talent count : skillset count;
        std::vector<std::string> metaInfo = splitString(treeParts[0], ":");

        //Fix non-existing version number
        if (metaInfo.size() == 7) {
            treePartStream << Presets::TTM_VERSION;
            for (int i = 0; i < metaInfo.size(); i++) {
                treePartStream << ":" << metaInfo[i];
            }
        }
        else {
            return false;
        }
        treeParts[0] = treePartStream.str();
        treePartStream.str(std::string());
        treePartStream.clear();

        //fix non-exisiting talent icon names for non-preset trees
        if (metaInfo[0] == "custom") {
            for (int i = 1; i <= std::stoi(metaInfo[5]); i++) {
                //Expect talent parts like: id : name,switchName : desc,desc,.. : talent type : row : col : max points : point req : ?? : parents : children; 
                std::vector<std::string> talentParts = splitString(treeParts[i], ":");
                if (talentParts.size() == 11) {
                    for (int j = 0; j < talentParts.size(); j++) {
                        treePartStream << talentParts[j] << ":";
                    }
                    treePartStream << "default.png,default.png";
                    treeParts[i] = treePartStream.str();
                    treePartStream.str(std::string());
                    treePartStream.clear();
                }
                else {
                    return false;
                }
            }
        }


        //put everything back into treeRep
        for (int i = 0; i < treeParts.size(); i++) {
            if (treeParts[i] != "") {
                treeRepStream << treeParts[i] << ";";
            }
        }

        treeRep = treeRepStream.str();
        return true;
    }

    bool validateTalentStringFormat(std::string talentString) {
        std::vector<std::string> talentParts = splitString(talentString, ":");
        if (talentParts.size() != 12 && talentParts.size() != 13) {
            return false;
        }
        if (talentParts[0].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        if (talentParts[1].find_first_not_of(
            "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 _/()',-"
        ) != std::string::npos) {
            return false;
        }
        if (talentParts[3] != "0" && talentParts[3] != "1" && talentParts[3] != "2") {
            return false;
        }
        if (talentParts[4] == "" || talentParts[4].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        if (talentParts[5] == "" || talentParts[5].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        if (talentParts[6] == "" || talentParts[6].length() > 1 || talentParts[6].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        if (talentParts[7] == "" || talentParts[7].find_first_not_of("0123456789") != std::string::npos) {
            return false;
        }
        if (talentParts[8] != "0" && talentParts[8] != "1") {
            return false;
        }
        if (talentParts[9].find_first_not_of("0123456789,") != std::string::npos) {
            return false;
        }
        if (talentParts[10].find_first_not_of("0123456789,") != std::string::npos) {
            return false;
        }
        return true;
    }

        /*
    Parse a tree with a given format string. There are two variants, specified by the preset in the tree info string.
    Given a spec name only the preset and then a string of numbers with the spent talent points as well as a .X where x specifies the switch state is needed.
    Given "custom" as preset, the full tree details have to be given. This is how spec trees are built.
    The definition of a custom tree definition can be seen in parseCustomTree function.
    */
    TalentTree parseTree(std::string treeRep) {
        //TTMTODO: Wrap stuff in here in try except cause users are special
        std::vector<std::string> treeDefinitionParts = splitString(treeRep, ";");
        std::vector<std::string> treeInfoParts = splitString(treeDefinitionParts[0], ":");
        if (treeInfoParts[1] == "custom") {
            if (checkIfParseStringProducesCycle(treeRep))
                throw std::logic_error("Tree rep produces cyclic tree!");
            return parseCustomTree(treeRep);
        }
        else {
            return parseTreeFromPreset(treeRep, treeInfoParts[1]);
        }
    }

    /*
    Parse a import tree string that is based on a preset (same as regular tree representation but without talent information). 
    First load the preset then edit the meta info and clear the included skillsets, then insert saved (non preset) loadout
    */
    TalentTree parseTreeFromPreset(std::string treeRep, std::string presetName) {
        TalentTree tree = loadTreePreset(Presets::LOAD_PRESETS()[presetName]);
        std::vector<std::string> treeDefinitionParts = splitString(treeRep, ";");

        std::vector<std::string> treeInfoParts = splitString(treeDefinitionParts[0], ":");
        tree.presetName = treeInfoParts[1];
        tree.classID = Presets::CLASS_ID_FROM_PRESET_NAME(tree.presetName);
        tree.type = static_cast<TreeType>(std::stoi(treeInfoParts[2]));
        tree.name = restoreString(treeInfoParts[3]);
        tree.treeDescription = restoreString(treeInfoParts[4]);
        tree.loadoutDescription = restoreString(treeInfoParts[5]);
        int numTalents = std::stoi(treeInfoParts[6]);
        int numLoadouts = std::stoi(treeInfoParts[7]);
        tree.loadout.clear();

        for (int i = 1; i < numLoadouts + 1; i++) {
            if (treeDefinitionParts[i] == "")
                break;
            std::vector<std::string> skillsetParts = splitString(treeDefinitionParts[i], ":");
            std::shared_ptr<TalentSkillset> skillset = std::make_shared<TalentSkillset>();
            std::vector<std::string> skillsetMetadata = splitString(skillsetParts[0], ",");
            skillset->name = skillsetMetadata[0];
            if (skillsetMetadata.size() > 2) {
                skillset->levelCap = std::stoi(skillsetMetadata[1]);
                skillset->useLevelCap = static_cast<bool>(std::stoi(skillsetMetadata[2]));
            }

            if (skillsetParts.size() - 1 != numTalents)
                continue;

            int index = 1;
            std::map<int, Talent_s>::iterator it;
            for (it = tree.orderedTalents.begin(); it != tree.orderedTalents.end(); it++)
            {
                int points = std::stoi(skillsetParts[index]);
                skillset->assignedSkillPoints[it->first] = points;
                skillset->talentPointsSpent += points;
                it->second->points = points;
                index++;
            }

            if (validateSkillset(tree, skillset)) {
                tree.loadout.push_back(skillset);
            }
        }

        return tree;
    }

    TalentTree parseCustomTree(std::string treeRep) {
        std::vector<std::string> treeDefinitionParts = splitString(treeRep, ";");
        TalentVec roots;
        std::unordered_map<int, Talent_s> talentTree;
        TalentTree tree;

        std::vector<std::string> treeInfoParts = splitString(treeDefinitionParts[0], ":");
        tree.presetName = treeInfoParts[1];
        tree.classID = Presets::CLASS_ID_FROM_PRESET_NAME(tree.presetName);
        tree.type = static_cast<TreeType>(std::stoi(treeInfoParts[2]));
        tree.name = restoreString(treeInfoParts[3]);
        tree.treeDescription = restoreString(treeInfoParts[4]);
        tree.loadoutDescription = restoreString(treeInfoParts[5]);
        int numTalents = std::stoi(treeInfoParts[6]);
        int numLoadouts = std::stoi(treeInfoParts[7]);

        for (int i = 1; i < numTalents + 1; i++) {
            if (treeDefinitionParts[i] == "")
                break;
            std::vector<std::string> talentInfo = splitString(treeDefinitionParts[i], ":");
            Talent_s t;
            int talentIndex = std::stoi(talentInfo[0]);
            if (talentTree.count(talentIndex)) {
                t = talentTree[talentIndex];
            }
            else {
                t = std::make_shared<Talent>();
                t->index = talentIndex;
                talentTree[t->index] = t;
            }
            std::vector<std::string> names = splitString(talentInfo[1], ",");
            t->name = restoreString(names[0]);
            std::vector<std::string> tDescriptions;
            for (auto& desc : splitString(talentInfo[2], ",")) {
                tDescriptions.push_back(restoreString(desc));
            }
            t->descriptions = tDescriptions;
            t->type = static_cast<TalentType>(std::stoi(talentInfo[3]));
            if (t->type == TalentType::SWITCH) {
                if (names.size() <= 1) {
                    t->nameSwitch = "Undefined switch name";
                }
                else {
                    t->nameSwitch = restoreString(names[1]);
                }
                while (t->descriptions.size() < 2) {
                    t->descriptions.push_back("Undefined switch description");
                }
            }
            t->row = std::stoi(talentInfo[4]);
            t->row = t->row > tree.maxRowLimit ? tree.maxRowLimit : t->row;
            t->column = std::stoi(talentInfo[5]);
            t->column = t->column > tree.maxColumnLimit ? tree.maxColumnLimit : t->column;
            t->maxPoints = std::stoi(talentInfo[6]);
            while (t->descriptions.size() < t->maxPoints) {
                t->descriptions.push_back("Undefined rank description");
            }
            t->pointsRequired = std::stoi(talentInfo[7]);
            t->preFilled = static_cast<bool>(std::stoi(talentInfo[8]));
            for (auto& parent : splitString(talentInfo[9], ",")) {
                if (parent == "")
                    break;
                int parentIndex = std::stoi(parent);
                if (talentTree.count(parentIndex)) {
                    addParent(t, talentTree[parentIndex]);
                }
                else {
                    Talent_s parentTalent = std::make_shared<Talent>();
                    parentTalent->index = parentIndex;
                    talentTree[parentIndex] = parentTalent;
                    addParent(t, parentTalent);
                }
            }
            for (auto& child : splitString(talentInfo[10], ",")) {
                if (child == "")
                    break;
                int childIndex = std::stoi(child);
                if (talentTree.count(childIndex)) {
                    addChild(t, talentTree[childIndex]);
                }
                else {
                    Talent_s childTalent = std::make_shared<Talent>();
                    childTalent->index = childIndex;
                    talentTree[childIndex] = childTalent;
                    addChild(t, childTalent);
                }
            }
            std::vector<std::string> iconNames = splitString(talentInfo[11], ",");
            t->iconName.first = restoreString(iconNames[0]);
            if (iconNames.size() > 1) {
                t->iconName.second = restoreString(iconNames[1]);
            }
            if (talentInfo.size() > 12) {
                t->nodeID = std::stoi(talentInfo[12]);
            }
            if (t->preFilled && t->parents.size() > 0) {
                bool canBePreFilled = false;
                for (auto& parent : t->parents) {
                    if (parent->preFilled) {
                        canBePreFilled = true;
                    }
                }
                if (!canBePreFilled) {
                    t->preFilled = false;
                }
            }
            if (t->parents.size() == 0) {
                roots.push_back(t);
            }
        }
        
        tree.talentRoots = roots;

        updateNodeCountAndMaxTalentPointsAndMaxID(tree);
        updateOrderedTalentList(tree);
        updateRequirementSeparatorInfo(tree);

        for (int i = numTalents + 1; i < numTalents + numLoadouts + 1; i++) {
            if (treeDefinitionParts[i] == "")
                break;
            std::vector<std::string> skillsetParts = splitString(treeDefinitionParts[i], ":");
            std::shared_ptr<TalentSkillset> skillset = std::make_shared<TalentSkillset>();
            std::vector<std::string> skillsetMetadata = splitString(skillsetParts[0], ",");
            skillset->name = skillsetMetadata[0];
            if (skillsetMetadata.size() > 2) {
                skillset->levelCap = std::stoi(skillsetMetadata[1]);
                skillset->useLevelCap = static_cast<bool>(std::stoi(skillsetMetadata[2]));
            }

            if (skillsetParts.size() - 1 != numTalents)
                continue;

            for (int i = 1; i < skillsetParts.size(); i++) {
                int points = std::stoi(skillsetParts[i]);
                skillset->assignedSkillPoints[std::stoi(splitString(treeDefinitionParts[i], ":")[0])] = points;
                skillset->talentPointsSpent += points;
                Talent_s t = tree.orderedTalents[std::stoi(splitString(treeDefinitionParts[i], ":")[0])];
                if (t->type == TalentType::SWITCH) {
                    if (points > 0) {
                        t->points = 1;
                        t->talentSwitch = points > 1 ? 2 : 1;
                    }
                    else {
                        t->points = 0;
                    }
                }
                else {
                    t->points = points;
                }
            }

            if (validateSkillset(tree, skillset)) {
                tree.loadout.push_back(skillset);
            }
        }

        return tree;
    }

    bool validateLoadout(TalentTree& tree, bool addNote) {
        bool changed = false;
        std::vector<std::shared_ptr<TalentSkillset>>::iterator it = tree.loadout.begin();

        while (it != tree.loadout.end()) {

            if (!validateSkillset(tree, *it)) {

                it = tree.loadout.erase(it);
                changed = true;
            }
            else ++it;
        }

        if (addNote && changed) {
            if (tree.loadoutDescription.substr(0, 9) != "TTM Note:") {
                tree.loadoutDescription = (
                    "TTM Note:\nDue to tree changes loadout has been revalidated and one or more skillset was invalidated!\n\n"
                    + tree.loadoutDescription
                    );
            }
        }
        return changed;
    }

    bool validateSkillset(TalentTree& tree, std::shared_ptr<TalentSkillset> skillset) {
        if (skillset->assignedSkillPoints.size() != tree.orderedTalents.size()) {
            return false;
        }
        for (auto& indexPointsPair : skillset->assignedSkillPoints) {
            //check if point is in talent with index that is not present in tree
            if (!tree.orderedTalents.count(indexPointsPair.first)) {
                return false;
            }
            if (indexPointsPair.second == 0) {
                //check if talent is pre filled but has 0 in skillset
                if (tree.orderedTalents[indexPointsPair.first]->preFilled) {
                    return false;
                }
                continue;
            }
            //first, check if talent has more or fewer points than allowed
            if (indexPointsPair.second < 0) {
                return false;
            }
            if ((tree.orderedTalents[indexPointsPair.first]->type == TalentType::SWITCH && indexPointsPair.second > 2)
                || (tree.orderedTalents[indexPointsPair.first]->type != TalentType::SWITCH 
                    && indexPointsPair.second > tree.orderedTalents[indexPointsPair.first]->maxPoints)) {
                return false;
            }
            //check if talent is pre filled but is not fully skilled
            if (tree.orderedTalents[indexPointsPair.first]->preFilled && indexPointsPair.second != tree.orderedTalents[indexPointsPair.first]->maxPoints) {
                return false;
            }
            //check if at least 1 parent is fully skilled
            //TTMTODO: this should be unnecessary since the same check is performed in points requirement check
            if (tree.orderedTalents[indexPointsPair.first]->parents.size() > 0) {
                bool parentFilled = false;
                for (auto& parent : tree.orderedTalents[indexPointsPair.first]->parents) {
                    if (parent->type == TalentType::SWITCH && skillset->assignedSkillPoints[parent->index] > 0) {
                        parentFilled = true;
                        break;
                    }
                    else if (parent->type != TalentType::SWITCH && skillset->assignedSkillPoints[parent->index] >= parent->maxPoints) {
                        parentFilled = true;
                        break;
                    }
                }
                if (!parentFilled) {
                    return false;
                }
            }
        }
        //check if points requirement is met by virtually assigning each point
        std::map<int, int> tempASP = skillset->assignedSkillPoints;
        int pointsSpent = 0;
        while (true) {
            int talentsStart = static_cast<int>(tempASP.size());
            if (talentsStart == 0) {
                break;
            }
            std::vector<int> talentsSelected;
            for (auto& indexPointsPair : tempASP) {
                if (indexPointsPair.second == 0) {
                    talentsSelected.push_back(indexPointsPair.first);
                    continue;
                }
                //check if point requirement is met
                if (pointsSpent < tree.orderedTalents[indexPointsPair.first]->pointsRequired) {
                    continue;
                }
                //check if parent is fully skilled if it exists
                if (tree.orderedTalents[indexPointsPair.first]->parents.size() > 0) {
                    bool parentFilled = false;
                    for (auto& parent : tree.orderedTalents[indexPointsPair.first]->parents) {
                        if (parent->type == TalentType::SWITCH && skillset->assignedSkillPoints[parent->index] > 0) {
                            parentFilled = true;
                            break;
                        }
                        else if (parent->type != TalentType::SWITCH && skillset->assignedSkillPoints[parent->index] >= parent->maxPoints) {
                            parentFilled = true;
                            break;
                        }
                    }
                    if (!parentFilled) {
                        continue;
                    }
                }
                //assign point
                pointsSpent += indexPointsPair.second;
                talentsSelected.push_back(indexPointsPair.first);
            }
            for (auto& index : talentsSelected) {
                tempASP.erase(index);
            }
            if (talentsStart == tempASP.size() && talentsStart > 0) {
                return false;
            }
            talentsSelected.clear();
        }

        //TTMTODO: Not sure why this exists, maybe failsafe? probably best to investigate and delete
        pointsSpent = 0;
        for (auto& pointSpentPair : skillset->assignedSkillPoints) {
            if (tree.orderedTalents[pointSpentPair.first]->type == TalentType::SWITCH) {
                if (pointSpentPair.second > 0) {
                    pointsSpent += 1;
                }
            }
            else {
                pointsSpent += pointSpentPair.second;
            }
        }
        skillset->talentPointsSpent = pointsSpent;

        return true;
    }

    /*
    Helper function that checks if a skillset string is in the correct format
    */
    bool validateSkillsetStringFormat(TalentTree& tree, std::string skillsetString) {
        return validateSkillsetStringFormat(tree.orderedTalents.size(), skillsetString);
    }

    /*
    Helper function that checks if a skillset string is in the correct format
    */
    bool validateSkillsetStringFormat(size_t numTalents, std::string skillsetString) {
        if (skillsetString.find(";") != std::string::npos) {
            return false;
        }
        std::vector<std::string> skillsetParts = splitString(skillsetString, ":");
        if (skillsetParts.size() - 1 != numTalents) {
            return false;
        }
        std::string skillsetName = skillsetParts[0];
        if (skillsetName.find_first_not_of(
            "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 :()'-"
        ) != std::string::npos) {
            //accept either no "," or exactly 2 "," to define levelCap and useLevelCap for backwards compatibility
            if (splitString(skillsetName, ",").size() != 3) {
                return false;
            }
        }
        for (int i = 1; i < skillsetParts.size(); i++) {
            if ((skillsetParts[i] == "" && i < skillsetParts.size() - 1)
                || skillsetParts[i].find_first_not_of("0123456789") != std::string::npos) {
                return false;
            }
        }
        return true;
    }

    /*
    Helper function that splits a string given the delimiter, if string does not contain delimiter then whole string is returned.
    */
    std::vector<std::string> splitString(std::string s, std::string delimiter) {
        std::vector<std::string> stringSplit;

        size_t pos = 0;
        std::string token;
        while ((pos = s.find(delimiter)) != std::string::npos) {
            token = s.substr(0, pos);
            stringSplit.push_back(token);
            s.erase(0, pos + delimiter.length());
        }
        stringSplit.push_back(s);

        return stringSplit;
    }

    /*
    Helper function to extract the original talent name of an expanded talent.
    */
    std::string extractOrigTalentName(std::string name) {
        std::string origName = "";
        std::vector<std::string> parts = splitString(name, "_");
        if (parts.size() == 1) {
            throw std::logic_error("There is no talent expansion character in talent name!");
        }
        for (int i = 0; i < parts.size() - 1; i++) {
            origName += parts[i];
            if (i < parts.size() - 2) {
                origName += "_";
            }
        }
        return origName;
    }


    /*
    Visualizes a given tree with graphviz. Needs to be installed and the paths have to exist. Generally not safe to use without careful skimming through it.
    */
    void visualizeTree(const TalentTree& tree, std::string suffix) {
        std::cout << "Visualize tree " << tree.name << " " << suffix << std::endl;
        std::string directory = "C:\\Users\\Tobi\\Documents\\Programming\\CodeSnippets\\WowTalentTrees\\TreesInputsOutputs";

        std::stringstream output;
        output << "strict digraph " << tree.name << " {\n";
        output << "label=\"" + tree.name + "\"\n";
        output << "fontname=\"Arial\"\n";
        //define each node individually
        output << "{\n";
        output << "node [fontname=\"Arial\" width=1 height=1 fixedsize=shape style=filled]\n";
        output << visualizeTalentInformation(tree);
        output << "}\n";
        //define node connections
        for (auto& root : tree.talentRoots) {
            std::stringstream connections;
            visualizeTalentConnections(root, connections);
            output << connections.str();
        }
        output << "}";

        //output txt file in graphviz format
        std::ofstream f;
        f.open(directory + "\\tree_" + tree.name + suffix + ".txt");
        f << output.str();
        f.close();

        std::string command = "\"\"C:\\Program Files\\Graphviz\\bin\\dot.exe\" -Tpng \"" + directory + "\\tree_" + tree.name + suffix + ".txt\" -o \"" + directory + "\\tree_" + tree.name + suffix + ".png\"\"";
        system(command.c_str());
    }

    /*
    Helper function that gathers all individual graphviz talent visualization strings in graphviz language into one string.
    */
    std::string visualizeTalentInformation(TalentTree tree) {
        std::unordered_map<int, std::string> talentInfos;

        for (auto& root : tree.talentRoots) {
            getTalentInfos(root, talentInfos);
        }

        std::stringstream talentInfosStream;
        for (auto& talent : talentInfos) {
            talentInfosStream << talent.first << " " << talent.second << std::endl;
        }

        return talentInfosStream.str();
    }

    /*
    Helper function that recursively gets the specific graphviz visualization string for a talent and its children.
    */
    void getTalentInfos(Talent_s talent, std::unordered_map<int, std::string>& talentInfos) {
        talentInfos[talent->index] = "[label=\"" + std::to_string(talent->index) + " " + std::to_string(talent->points) + "/" + std::to_string(talent->maxPoints) + getSwitchLabel(talent) + "\" fillcolor=" + getFillColor(talent) + " shape=" + getShape(talent->type) + "]";
        for (auto& child : talent->children) {
            getTalentInfos(child, talentInfos);
        }
    }

    /*
    Helper function that returns the appropriate fill color for different types of talents and points allocations.
    */
    std::string getFillColor(Talent_s talent) {
        if (talent->points == 0) {
            return "white";
        }
        else if (talent->type != TalentType::SWITCH) {
            if (talent->points == talent->maxPoints) {
                return "darkgoldenrod2";
            }
            else {
                return "chartreuse3";
            }
        }
        else if (talent->talentSwitch == 0)
            return "aquamarine3";
        else if (talent->talentSwitch == 1)
            return "coral";
        else
            return "fuchsia";

    }

    /*
    Helper function that defines the shape of each talent type.
    */
    std::string getShape(TalentType type) {
        switch (type) {
        case TalentType::ACTIVE: return "square";
        case TalentType::PASSIVE: return "circle";
        case TalentType::SWITCH: return "octagon";
        default: throw std::logic_error("Wrong talent type (illegal cast to TalentType enum?)!");
        }
    }

    /*
    Helper function that displays the switch state of a given talent.
    */
    std::string getSwitchLabel(Talent_s talent) {
        if (talent->type == TalentType::SWITCH)
            return "";
        if (talent->talentSwitch == 0)
            return "\nleft";
        else if (talent->talentSwitch == 1)
            return "\nright";
        else
            return "";
    }

    /*
    Helper function that recursively creates the graphviz visualization string of all talent connections. (We only need connections from parents to children.
    Also, graph is defined as strict digraph so we don't need to take care of duplicate arrows.
    */
    void visualizeTalentConnections(Talent_s root, std::stringstream& connections) {
        if (root->children.size() == 0)
            return;
        connections << root->index << " -> {";
        for (auto& child : root->children) {
            connections << child->index << " ";
        }
        connections << "}\n";
        for (auto& child : root->children) {
            visualizeTalentConnections(child, connections);
        }
    }

    /*
    Automatically decides the row,col position of tree nodes and reindexes the tree
    */
    void autoPositionTreeNodes(TalentTree& tree) {
        //TTMTODO: Completely rework this system or maybe don't include it at all
        //find depth for each talent
        //depth is the longest path from a root talent that reaches the given talent
        std::unordered_map<Talent_s, int> maxDepthMap;
        for (auto& root : tree.talentRoots) {
            findDepthRecursively(1, root, maxDepthMap);
        }

        //create a map that holds all talent per depth value
        std::map<int, TalentVec> talentDepths;
        for (auto& talentDepthPair : maxDepthMap) {
            talentDepths[talentDepthPair.second].push_back(talentDepthPair.first);
        }

        //from now on treat depth not as connection length from root but as a row index
        //move talents with high point requirements further down in rows
        bool ordered = false;
        while (!ordered) {
            ordered = true;
            for (auto& depthTalentsPair : talentDepths) {
                //talents at the bottom can't move further down
                if (!talentDepths.count(depthTalentsPair.first + 1))
                    break;
                for (auto& talent : depthTalentsPair.second) {
                    for (auto& lowerTalent : talentDepths[depthTalentsPair.first + 1]) {
                        if (talent->pointsRequired > lowerTalent->pointsRequired) {
                            ordered = false;
                            break;
                        }
                    }
                    if (!ordered) {
                        //move talent one row further down
                        depthTalentsPair.second.erase(std::remove(depthTalentsPair.second.begin(), depthTalentsPair.second.end(), talent), depthTalentsPair.second.end());
                        talentDepths[depthTalentsPair.first + 1].push_back(talent);
                        break;
                    }
                }
                if (!ordered)
                    break;
            }
        }

        //recursively position row nodes
        bool isPositioned = autoPositionRowNodes(1, talentDepths, tree.maxRowLimit, tree.maxColumnLimit);

        return;
    }

    bool autoPositionRowNodes(int row, std::map<int, TalentVec>& talentDepths, int maxRow, int maxColumn) {
        vec2d<int> positions = createPositionIndices(row, talentDepths);
        bool isPositioned = false;
        for (auto& positionVec : positions) {
            do
            {
                for (int i = 0; i < positionVec.size(); i++) {
                    talentDepths[row][i]->row = row > maxRow ? maxRow : row;
                    talentDepths[row][i]->column = positionVec[i] > maxColumn ? maxColumn : positionVec[i];
                }
                bool isCrossing = checkForCrossing(row, talentDepths);
                if (isCrossing) {
                    continue;
                }
                if (talentDepths.count(row + 1)) {
                    isPositioned = autoPositionRowNodes(row + 1, talentDepths, maxRow, maxColumn);
                }
                else {
                    isPositioned = !isCrossing;
                }
                if (isPositioned) {
                    break;
                }
            } while (std::next_permutation(positionVec.begin(), positionVec.end()));
            if (isPositioned) {
                break;
            }
        }
        return isPositioned;
    }

    vec2d<int> createPositionIndices(int row, std::map<int, TalentVec>& talentDepths) {
        vec2d<int> positions;
        std::vector<int> positionVec;
        expandPosition(0, 1, talentDepths[row].size(), 3, positionVec, positions);

        return positions;
    }

    void expandPosition(int currentIndex, int currentPos, size_t indexSize, size_t additionalPos, std::vector<int> positionVec, vec2d<int>& positions) {
        for (int i = currentPos; i < additionalPos + currentIndex + 2; i++) {
            positionVec.push_back(i);
            if (positionVec.size() == indexSize) {
                positions.push_back(positionVec);
                positionVec.pop_back();
                continue;
            }
            expandPosition(currentIndex + 1, i + 1, indexSize, additionalPos, positionVec, positions);
            positionVec.pop_back();
        }
    }

    bool checkForCrossing(int row, std::map<int, TalentVec>& talentDepths) {
        vec2d<int> edges;
        appendEdges(edges, row, talentDepths);
        bool crossing = false;
        for (int i = 0; i < static_cast<int>(edges.size()) - 1; i++) {
            for (int j = i + 1; j < edges.size(); j++) {
                crossing = intersects(edges[i], edges[j]);
                if (crossing) {
                    return true;
                }
            }
        }
        return crossing;
    }

    void appendEdges(vec2d<int>& edges, int row, std::map<int, TalentVec>& talentDepths) {
        if (!talentDepths.count(row)) {
            return;
        }
        for (auto& talent : talentDepths[row]) {
            for (auto& parent : talent->parents) {
                edges.push_back(std::vector<int>{talent->column, talent->row, parent->column, parent->row});
            }
        }
        appendEdges(edges, row - 1, talentDepths);
    }

    /*
     returns true if the line from(a, b)->(c, d) intersects with(p, q)->(r, s)
    */
    bool intersects(std::vector<int> edge1, std::vector<int> edge2) {
        int a = edge1[0];
        int b = edge1[1];
        int c = edge1[2];
        int d = edge1[3];
        int p = edge2[0];
        int q = edge2[1];
        int r = edge2[2];
        int s = edge2[3];
        if (a == 0 && b == 8 && c == 0 && d == 5)
            int i = 0;
        int det;
        float gamma, lambda, alpha, beta;
        det = (c - a) * (s - q) - (r - p) * (d - b);
        if (det == 0) {
            //line segments parallel
            if (r - p != 0) {
                gamma = (a - p) * 1.0f / (r - p);
                alpha = (p - a) * 1.0f / (c - a);
            }
            else {
                gamma = (b - q) * 1.0f / (s - q);
                alpha = (q - b) * 1.0f / (d - b);
            }
            if (r - p != 0) {
                lambda = (c - p) * 1.0f / (r - p);
                beta = (r - a) * 1.0f / (c - a);
            }
            else {
                lambda = (d - q) * 1.0f / (s - q);
                beta = (s - b) * 1.0f / (d - b);
            }
            return (((std::abs(p + (r - p) * gamma - a) < 0.001f && std::abs(q + (s - q) * gamma - b) < 0.001f)
                 || (std::abs(p + (r - p) * lambda - c) < 0.001f && std::abs(q + (s - q) * lambda - d) < 0.001f))
                && ((0.001f < gamma && gamma < .999f) || (0.001f < lambda && lambda < .999f)
                    || (0.001f < alpha && alpha < .999f) || (0.001f < beta && beta < .999f)));
        }
        else {
            lambda = ((s - q) * (r - a) + (p - r) * (s - b)) * 1.0f / det;
            gamma = ((b - d) * (r - a) + (c - a) * (s - b)) * 1.0f / det;
            return (0 < lambda && lambda < 1) && (0 < gamma && gamma < 1);
        }
    };

    void findDepthRecursively(int depth, Talent_s talent, std::unordered_map<Talent_s, int>& maxDepthMap) {
        if(depth > maxDepthMap[talent])
            maxDepthMap[talent] = depth;
        for (auto& child : talent->children) {
            findDepthRecursively(depth + 1, child, maxDepthMap);
        }
    }


    /*
    Changes the indices of a tree to go from top left to bottom right.
    */
    void reindexTree(TalentTree& tree) {
        //create sortable talent vector
        TalentVec talents;
        for (std::map<int, Talent_s>::iterator it = tree.orderedTalents.begin(); it != tree.orderedTalents.end(); ++it) {
            talents.push_back(it->second);
        }
        //sort talent vector by row and column
        std::sort(talents.begin(), talents.end(),
            [](const Talent_s& a, const Talent_s& b) -> bool
            {
                if (a->row != b->row) {
                    return a->row < b->row;
                }
                return a->column < b->column;
            });
        //create map with old->new indices
        std::map<int, int> indexMap;
        for (int i = 0; i < talents.size(); i++) {
            indexMap[talents[i]->index] = i;
        }

        //reindex talents (only index has to change)
        for (auto& talent : talents) {
            talent->index = indexMap[talent->index];
        }
        //recreate tree.orderedTalents
        std::map<int, Talent_s> newOrderedTalents;
        for (auto& talent : talents) {
            newOrderedTalents[talent->index] = talent;
        }
        tree.orderedTalents = newOrderedTalents;
        //clear loadout
        tree.loadout.clear();
    }

    /*
    Auto sets the point requirements for every talent according to Blizzards example trees.
    */
    void autoPointRequirements(TalentTree& tree) {
        std::vector<int> uniqueRows;
        for (auto& talent : tree.orderedTalents) {
            if (std::find(uniqueRows.begin(), uniqueRows.end(), talent.second->row) == uniqueRows.end()) {
                uniqueRows.push_back(talent.second->row);
            }
        }
        std::sort(uniqueRows.begin(), uniqueRows.end());
        for (auto& talent : tree.orderedTalents) {
            if (talent.second->row > uniqueRows[6]) {
                talent.second->pointsRequired = 20;
                continue;
            }
            if (talent.second->row > uniqueRows[3]) {
                talent.second->pointsRequired = 8;
                continue;
            }
            talent.second->pointsRequired = 0;
        }
    }

    /*
    Auto shifts the tree such that min col and min row are 1
    */
    void autoShiftTreeToCorner(TalentTree& tree) {
        int minCol = 1337;
        int minRow = 1337;
        for (auto& talent : tree.orderedTalents) {
            if (talent.second->row < minRow) {
                minRow = talent.second->row;
            }
            if (talent.second->column < minCol) {
                minCol = talent.second->column;
            }
        }
        tree.maxCol = 0;
        for (auto& talent : tree.orderedTalents) {
            talent.second->row -= minRow - 1;
            talent.second->column -= minCol - 1;
            if (talent.second->column > tree.maxCol) {
                tree.maxCol = talent.second->column;
            }
        }
        tree.maxCol++;
    }

    /*
    Tries to automatically insert icon names based on talent names
    */
    void autoInsertIconNames(std::vector<std::string> iconNames, TalentTree& tree) {
        //we do not try to wrangle any unicode business, simple transformation functions, if that fails then no icon change
        std::vector<std::string> formattedIconNames;
        for (auto& name : iconNames) {
            std::string formattedIconName;
            formattedIconName.reserve(name.length());
            for (auto& ch : name) {
                if (ch == '.') {
                    break;
                }
                if (ch >= 'A' && ch <= 'Z') {
                    formattedIconName += static_cast<char>(ch + 32);
                }
                if (ch >= 'a' && ch <= 'z') {
                    formattedIconName += ch;
                }
            }
            formattedIconNames.push_back(formattedIconName);
        }
        for (auto& talent : tree.orderedTalents) {
            std::string talentName(talent.second->name);
            std::string formattedTalentName;
            formattedTalentName.reserve(talentName.size());
            for (auto& ch : talentName) {
                if (ch >= 'A' && ch <= 'Z') {
                    formattedTalentName += static_cast<char>(ch + 32);
                }
                if (ch >= 'a' && ch <= 'z') {
                    formattedTalentName += ch;
                }
            }

            for (int i = 0; i < formattedIconNames.size(); i++) {
                //much faster but worse in some edge cases
                /*size_t maxLength = 15;
                maxLength = formattedIconNames[i].length() < maxLength ? formattedIconNames[i].length() : maxLength;
                maxLength = formattedTalentName.length() < maxLength ? formattedTalentName.length() : maxLength;
                if (formattedIconNames[i].substr(0, maxLength) == formattedTalentName.substr(0, maxLength)) {
                    talent.second->iconName.first = iconNames[i];
                }*/
                std::vector<double> similarityRanking = getSimilarityRanking(formattedTalentName, formattedIconNames);
                int bestMatch = 0;
                double bestRanking = -1;
                for (i = 0; i < similarityRanking.size(); i++) {
                    if (similarityRanking[i] > bestRanking) {
                        bestMatch = i;
                        bestRanking = similarityRanking[i];
                    }
                }
                talent.second->iconName.first = iconNames[bestMatch];
            }


            if (talent.second->type == TalentType::SWITCH) {
                std::string switchTalentName(talent.second->nameSwitch);
                formattedTalentName = "";
                formattedTalentName.reserve(switchTalentName.size());
                for (auto& ch : switchTalentName) {
                    if (ch >= 'A' && ch <= 'Z') {
                        formattedTalentName += static_cast<char>(ch + 32);
                    }
                    if (ch >= 'a' && ch <= 'z') {
                        formattedTalentName += ch;
                    }
                }

                for (int i = 0; i < formattedIconNames.size(); i++) {
                    //much faster but worse in some edge cases
                    /*size_t maxLength = 15;
                    maxLength = formattedIconNames[i].length() < maxLength ? formattedIconNames[i].length() : maxLength;
                    maxLength = formattedTalentName.length() < maxLength ? formattedTalentName.length() : maxLength;
                    if (formattedIconNames[i].substr(0, maxLength) == formattedTalentName.substr(0, maxLength)) {
                        talent.second->iconName.second = iconNames[i];
                    }*/
                    std::vector<double> similarityRanking = getSimilarityRanking(formattedTalentName, formattedIconNames);
                    int bestMatch = 0;
                    double bestRanking = -1;
                    for (i = 0; i < similarityRanking.size(); i++) {
                        if (similarityRanking[i] > bestRanking) {
                            bestMatch = i;
                            bestRanking = similarityRanking[i];
                        }
                    }
                    talent.second->iconName.second = iconNames[bestMatch];
                }
            }
        }
    }

    /*
    Transforms tree with "complex" talents (that can hold mutliple skill points) to "simple" tree with only talents that can hold a single talent point.
    In the process, all pre filled talents get deleted as they are irrelevant for loadout solving.
    */
    void expandTreeTalents(TalentTree& tree) {
        //We use a very clever trick to easily incorporate pre filled talents in the tree solve process.
        //Pre filled talents are fully filled and enable every child to be picked next. Therefore every child acts as a root node.
        //We simply have to cut all parent connections (and therefore remove from parent's children) and add to root list to transform talent to root node.
        //Then solving works the same as before.
        int rootIndex = 0;
        while(rootIndex < tree.talentRoots.size()) {
            Talent_s& root = tree.talentRoots[rootIndex];
            if (root->preFilled) {
                int talentIndex = root->index;
                //check children for new root talents &
                //delete from children's parents lists
                TalentVec origChildren = root->children;
                for (auto& child : origChildren) {
                    for (auto& childParent : child->parents) {
                        childParent->children.erase(
                            std::remove(
                                childParent->children.begin(),
                                childParent->children.end(),
                                child),
                            childParent->children.end());
                    }
                    child->parents.clear();
                    tree.talentRoots.push_back(child);
                }
                //delete from tree.orderedTalents
                tree.orderedTalents.erase(talentIndex);
                rootIndex++;
                continue;
            }
            expandTalentAndAdvance(tree, root, tree.maxTalentPoints);
            rootIndex++;
        }
        tree.talentRoots.clear();
        for (auto& indexTalentPair : tree.orderedTalents) {
            if (indexTalentPair.second->parents.size() == 0) {
                tree.talentRoots.push_back(indexTalentPair.second);
            }
        }
        updateNodeCountAndMaxTalentPointsAndMaxID(tree);
        updateOrderedTalentList(tree);
        updateRequirementSeparatorInfo(tree);
    }

    /*
    Creates all the necessary single point talents to replace a multi point talent and inserts them with correct parents/children
    */
    void expandTalentAndAdvance(TalentTree& tree, Talent_s talent, int maxTalentPoints) {
        if (talent->preFilled) {
            //TTMNOTE: This should not be possible, only root nodes or nodes with pre filled parent (which are transformed to roots) can be pre filled
            /*
            int talentIndex = talent->index;
            //check children for new root talents &
            //delete from children's parents lists
            for (auto& child : talent->children) {
                child->parents.erase(
                    std::remove(
                        child->parents.begin(),
                        child->parents.end(),
                        talent),
                    child->parents.end());
            }
            //delete from parents' children lists
            for (auto& parent : talent->parents) {
                parent->children.erase(
                    std::remove(
                        parent->children.begin(),
                        parent->children.end(),
                        talent),
                    parent->children.end());
            }
            //delete from tree.orderedTalents
            tree.orderedTalents.erase(talentIndex);
            for (auto& child : talent->children) {
                expandTalentAndAdvance(tree, child, maxTalentPoints);
            }
            talent->children.clear();
            talent->parents.clear();
            */
            bool hasPreFilledParent = false;
            for (auto& parent : talent->parents) {
                if (parent->preFilled) {
                    hasPreFilledParent = true;
                }
            }
            if (hasPreFilledParent) {
                return;
            }
            else {
                throw std::logic_error("Non-root nodes cannot be pre filled!");
            }
        }
        if (talent->maxPoints > 1) {
            TalentVec talentParts;
            TalentVec originalChildren;
            for (auto& child : talent->children) {
                originalChildren.push_back(child);
            }
            talent->children.clear();
            talentParts.push_back(talent);
            for (int i = 0; i < talent->maxPoints - 1; i++) {
                Talent_s t = std::make_shared<Talent>();
                //TTMNOTE: if this changes, also change TreeSolver.cpp->filterSolvedSkillsets and TreeSolver.cpp->skillsetIndexToSkillset
                t->index = (talent->index + 1) * maxTalentPoints + i;
                t->expansionIndex = i + 1;
                t->isExpanded = true;
                t->name = talent->name + "_" + std::to_string(i + 1);
                t->type = talent->type;
                t->row = talent->row;
                t->column = talent->column;
                t->points = 0;
                t->maxPoints = 1;
                t->pointsRequired = talent->pointsRequired;
                t->talentSwitch = talent->talentSwitch;
                t->parents.push_back(talentParts[i]);
                talentParts.push_back(t);
                talentParts[i]->children.push_back(t);
            }
            talent->isExpanded = true;
            for (auto& child : originalChildren) {
                talentParts[talent->maxPoints - 1]->children = originalChildren;
            }
            for (auto& child : originalChildren) {
                TalentVec::iterator i = std::find(child->parents.begin(), child->parents.end(), talent);
                if (i != child->parents.end()) {
                    (*i) = talentParts[talent->maxPoints - 1];
                }
                else {
                    //If this happens then n has m as child but m does not have n as parent! Bug!
                    throw std::logic_error("child has missing parent");
                }
            }
            talent->maxPoints = 1;
            for (auto& child : originalChildren) {
                expandTalentAndAdvance(tree, child, maxTalentPoints);
            }
        }
        else {
            for (auto& child : talent->children) {
                expandTalentAndAdvance(tree, child, maxTalentPoints);
            }
        }
    }

    /*
    Transforms tree with "simple" talents (that can only hold one skill point) to "complex" tree with multi-skill-point talents
    */
    void contractTreeTalents(TalentTree& tree) {
        for (auto& root : tree.talentRoots) {
            contractTalentAndAdvance(root);
        }
        updateNodeCountAndMaxTalentPointsAndMaxID(tree);
        updateOrderedTalentList(tree);
        updateRequirementSeparatorInfo(tree);
    }

    /*
    Creates all the necessary multi point talents to replace a single point talent and inserts them with correct parents/children
    */
    void contractTalentAndAdvance(Talent_s& talent) {
        //std::vector<std::string> splitIndex = splitString(talent->index, "_");
        if (talent->isExpanded) {
            //std::vector<std::string> splitName = splitString(talent->name, "_");
            //talent has to be contracted
            //expanded Talents have 1 child at most and talent chain is at least 2 talents long
            //std::string baseIndex = splitIndex[0];
            TalentVec talentParts;
            Talent_s currTalent = talent;
            talentParts.push_back(talent);
            Talent_s childTalent = talent->children[0];
            int minIndex = talent->index;
            while (extractOrigTalentName(childTalent->name) == talent->name) {
                minIndex = minIndex > childTalent->index ? childTalent->index : minIndex;
                talentParts.push_back(childTalent);
                currTalent = childTalent;
                if (currTalent->children.size() == 0)
                    break;
                childTalent = currTalent->children[0];
            }
            Talent_s t = std::make_shared<Talent>();
            t->index = minIndex;
            t->name = talent->name;
            t->type = talent->type;
            t->points = 0;
            for (auto& talent : talentParts) {
                t->points += talent->points;
            }
            t->maxPoints = static_cast<int>(talentParts.size());
            t->talentSwitch = talent->talentSwitch;
            t->parents = talentParts[0]->parents;
            t->children = talentParts[talentParts.size() - 1]->children;

            //replace talent pointer
            talent = t;
            //iterate through children
            for (auto& child : talent->children) {
                contractTalentAndAdvance(child);
            }
        }
        else {
            for (auto& child : talent->children) {
                contractTalentAndAdvance(child);
            }
        }
    }

    void clearTree(TalentTree& tree) {
        for (auto& talent : tree.orderedTalents) {
            talent.second->points = 0;
            talent.second->talentSwitch = 0;
        }
    }

    void createSkillset(TalentTree& tree) {
        std::shared_ptr<TalentSkillset> skillset = std::make_shared<TalentSkillset>();
        skillset->name = "New skillset";
        for (auto& talent : tree.orderedTalents) {
            skillset->assignedSkillPoints[talent.first] = 0;
        }
        tree.loadout.push_back(skillset);
    }

    void copySkillset(TalentTree& tree, std::shared_ptr<TalentSkillset> skillset) {
        std::shared_ptr<TalentSkillset> skillsetCopy = std::make_shared<TalentSkillset>();
        skillsetCopy->name = skillset->name;
        skillsetCopy->talentPointsSpent = skillset->talentPointsSpent;
        for (auto& indexPointsPair : skillset->assignedSkillPoints) {
            skillsetCopy->assignedSkillPoints[indexPointsPair.first] = indexPointsPair.second;
        }
        tree.loadout.push_back(skillsetCopy);
    }

    void activateSkillset(TalentTree& tree, int index) {
        if (index < 0 || index >= tree.loadout.size()) {
            throw std::logic_error("Skillset index is -1 or larger than loadout size!");
        }
        tree.activeSkillsetIndex = index;
        tree.loadout[index]->talentPointsSpent = 0;
        for (auto& indexPointsPair : tree.loadout[index]->assignedSkillPoints) {
            if (indexPointsPair.second < 0 
                || (tree.orderedTalents[indexPointsPair.first]->type != TalentType::SWITCH &&
                    indexPointsPair.second > tree.orderedTalents[indexPointsPair.first]->maxPoints)
                || (tree.orderedTalents[indexPointsPair.first]->type == TalentType::SWITCH &&
                    indexPointsPair.second > 2)) {
                throw std::logic_error("Skillset allocates <0 points or more than max points!");
            }
            if (tree.orderedTalents[indexPointsPair.first]->type != TalentType::SWITCH) {
                tree.orderedTalents[indexPointsPair.first]->points = indexPointsPair.second;
                tree.loadout[index]->talentPointsSpent += indexPointsPair.second;
            }
            else {
                if (indexPointsPair.second > 0) {
                    tree.orderedTalents[indexPointsPair.first]->points = 1;
                    tree.orderedTalents[indexPointsPair.first]->talentSwitch = indexPointsPair.second;
                    tree.loadout[index]->talentPointsSpent += 1;
                }
                else {
                    tree.orderedTalents[indexPointsPair.first]->points = 0;
                    //If we don't reset this, talentSwitches can linger around which might be better UX
                    //tree.orderedTalents[indexPointsPair.first]->talentSwitch = 0;
                }
            }
        }
        if (tree.loadout[index]->talentPointsSpent > tree.loadout[index]->levelCap) {
            tree.loadout[index]->useLevelCap = false;
        }
    }

    void applyPreselectedTalentsToSkillset(TalentTree& tree, std::shared_ptr<TalentSkillset> skillset) {
        for (auto& indexTalentPair : tree.orderedTalents) {
            if (indexTalentPair.second->preFilled) {
                int additionalTalentPoints = indexTalentPair.second->maxPoints - skillset->assignedSkillPoints[indexTalentPair.first];
                skillset->assignedSkillPoints[indexTalentPair.first] += additionalTalentPoints;
                skillset->talentPointsSpent += additionalTalentPoints;
            }
        }
    }

    std::pair<int, int> importSkillsets(TalentTree& tree, std::string importString) {
        std::pair<int, int> importedSkillsets = { 0,0 };
        std::vector<std::string> skillsetsString = splitString(importString, ";");
        for (int i = 0; i < skillsetsString.size(); i++) {
            if (skillsetsString[i] == "") {
                break;
            }
            if (!validateSkillsetStringFormat(tree, skillsetsString[i])) {
                importedSkillsets.second += 1;
                continue;
            }
            std::vector<std::string> skillsetParts = splitString(skillsetsString[i], ":");
            std::shared_ptr<TalentSkillset> skillset = std::make_shared<TalentSkillset>();
            std::vector<std::string> skillsetMetadata = splitString(skillsetParts[0], ",");
            skillset->name = skillsetMetadata[0];
            if (skillsetMetadata.size() > 2) {
                skillset->levelCap = std::stoi(skillsetMetadata[1]);
                skillset->useLevelCap = static_cast<bool>(std::stoi(skillsetMetadata[2]));
            }

            if (skillsetParts.size() - 1 != tree.orderedTalents.size()) {
                importedSkillsets.second += 1;
            }

            int index = 1;
            std::map<int, Talent_s>::iterator it;
            for (it = tree.orderedTalents.begin(); it != tree.orderedTalents.end(); it++)
            {
                int points = std::stoi(skillsetParts[index]);
                skillset->assignedSkillPoints[it->first] = points;
                skillset->talentPointsSpent += points;
                index++;
            }

            if (validateSkillset(tree, skillset)) {
                tree.loadout.push_back(skillset);
                importedSkillsets.first += 1;
            }
            else {
                importedSkillsets.second += 1;
            }
        }
        tree.activeSkillsetIndex = static_cast<int>(tree.loadout.size() - 1);
        if (tree.activeSkillsetIndex >= 0) {
            activateSkillset(tree, tree.activeSkillsetIndex);
        }
        return importedSkillsets;
    }

    std::string createSkillsetStringRepresentation(std::shared_ptr<TalentSkillset> skillset) {
        std::string rep = skillset->name + "," + std::to_string(skillset->levelCap) + "," + std::to_string(skillset->useLevelCap);
        for (auto& indexPointsPair : skillset->assignedSkillPoints) {
            rep += ":" + std::to_string(indexPointsPair.second);
        }
        rep += ";";
        return rep;
    }

    std::string createSkillsetSimcStringRepresentation(std::shared_ptr<TalentSkillset> skillset, const TalentTree& tree) {
        std::string rep = tree.type == TreeType::CLASS ? "class_talents=" : "spec_talents=";
        for (auto& indexPointsPair : skillset->assignedSkillPoints) {
            if (indexPointsPair.second == 0) {
                continue;
            }
            if (tree.orderedTalents.at(indexPointsPair.first)->type != TalentType::SWITCH) {
                rep += simcTokenizeName(tree.orderedTalents.at(indexPointsPair.first)->name) + ":" + std::to_string(indexPointsPair.second) + "/";
            }
            else {
                if (indexPointsPair.second == 1) {
                    rep += simcTokenizeName(tree.orderedTalents.at(indexPointsPair.first)->name) + ":1/";
                }
                else {
                    rep += simcTokenizeName(tree.orderedTalents.at(indexPointsPair.first)->nameSwitch) + ":1/";
                }
            }
        }
        return rep.substr(0, rep.size() - 1);
    }

    std::string createActiveSkillsetStringRepresentation(TalentTree& tree) {
        if (tree.loadout.size() <= tree.activeSkillsetIndex || !validateSkillset(tree, tree.loadout[tree.activeSkillsetIndex])) {
            return "Invalid skillset!";
        }
        return createSkillsetStringRepresentation(tree.loadout[tree.activeSkillsetIndex]);
    }

    std::string createAllSkillsetsStringRepresentation(TalentTree& tree) {
        std::string rep;
        for (auto& skillset : tree.loadout) {
            if (!validateSkillset(tree, skillset)) {
                return "At least skillset " + skillset->name + " is invalid!";
            }
            rep += createSkillsetStringRepresentation(skillset);
        }
        return rep;
    }

    std::string createSingleTalentsSimcString(TalentTree& tree) {
        std::string rep;
        for (auto& talent : tree.orderedTalents) {
            if (talent.second->type == TalentType::SWITCH) {
                std::string tokenName = simcTokenizeName(talent.second->name);
                rep += "profileset.\"" + tokenName + " - 1" + "\"+=\"" + (tree.type == TreeType::CLASS ? "class_talents=" : "spec_talents=") + tokenName + ":1\"\n";
                tokenName = simcTokenizeName(talent.second->nameSwitch);
                rep += "profileset.\"" + tokenName + " - 1" + "\"+=\"" + (tree.type == TreeType::CLASS ? "class_talents=" : "spec_talents=") + tokenName + ":1\"\n";
            }
            else {
                for (int i = 1; i <= talent.second->maxPoints; i++) {
                    std::string tokenName = simcTokenizeName(talent.second->name);
                    rep += "profileset.\"" + tokenName + " - " + std::to_string(i) + "\"+=\"" + (tree.type == TreeType::CLASS ? "class_talents=" : "spec_talents=") + tokenName + ":" + std::to_string(i) + "\"\n";
                }
            }
        }
        return rep;
    }

    std::string createSingleTalentComparisonSimcString(TalentTree& tree) {
        std::string rep;
        TalentSkillset skillset = *tree.loadout[tree.activeSkillsetIndex];
        for (const auto& talent : tree.orderedTalents) {
            int currentPoints = skillset.assignedSkillPoints[talent.first];
            int maxPoints = talent.second->type == TalentType::SWITCH ? 2 : talent.second->maxPoints;
            for (int points = 0; points <= maxPoints; points++) {
                if (points == currentPoints) {
                    continue;
                }
                skillset.assignedSkillPoints[talent.first] = points;
                std::string talentName;
                if (talent.second->type == TalentType::SWITCH) {
                    if (points == 2) {
                        talentName = talent.second->nameSwitch;
                    }
                    else {
                        talentName = talent.second->name;
                    }
                }
                else {
                    talentName = talent.second->name;
                }
                rep += 
                    "profileset.\"" 
                    + talent.second->name + " " + std::to_string(currentPoints) + " to " + std::to_string(points)
                    + "\"+=\"" + createSkillsetSimcStringRepresentation(std::make_shared<TalentSkillset>(skillset), tree) + "\"\n";
            }
            skillset.assignedSkillPoints[talent.first] = currentPoints;
        }

        return rep;
    }

    std::string createActiveSkillsetSimcStringRepresentation(TalentTree& tree, bool createProfileset) {
        if (tree.loadout.size() <= tree.activeSkillsetIndex || !validateSkillset(tree, tree.loadout[tree.activeSkillsetIndex])) {
            return "Invalid skillset!";
        }
        if (createProfileset) {
            return "profileset.\"" + tree.loadout[tree.activeSkillsetIndex]->name + "\"+=\"" + createSkillsetSimcStringRepresentation(tree.loadout[tree.activeSkillsetIndex], tree) + "\"";
        }
        else {
            return createSkillsetSimcStringRepresentation(tree.loadout[tree.activeSkillsetIndex], tree);
        }
    }

    std::string createAllSkillsetsSimcStringRepresentation(TalentTree& tree) {
        return createAllSkillsetsSimcStringRepresentation(tree, tree.loadout);
    }

    std::string createAllSkillsetsSimcStringRepresentation(TalentTree& tree, std::vector<std::shared_ptr<TalentSkillset>> loadout) {
        std::string rep;
        for (auto& skillset : loadout) {
            if (!validateSkillset(tree, skillset)) {
                return "At least skillset " + skillset->name + " is invalid!";
            }
            rep += "profileset.\"" + skillset->name + "\"+=\"" + createSkillsetSimcStringRepresentation(skillset, tree) + "\"\n";
        }
        return rep;
    }

    // part of the blizz hash import/export methodology taken from 
    // https://github.com/simulationcraft/simc/blob/7cfe69501aff528096516246ad40471ae1b468ed/engine/player/player.cpp
    namespace
    {
        const std::string base64_char = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
        // hardcoded values from Interface/AddOns/Blizzard_ClassTalentUI/Blizzard_ClassTalentImportExport.lua
        constexpr unsigned LOADOUT_SERIALIZATION_VERSION = 1;
        constexpr size_t version_bits = 8;    // serialization version
        constexpr size_t spec_bits = 16;   // specialization id
        constexpr size_t tree_bits = 128;  // C_Traits.GetTreeHash(), optionally can be 0-filled
        constexpr size_t rank_bits = 6;    // ranks purchased if node is partially filled
        constexpr size_t choice_bits = 2;    // choice index, 0-based
        // hardcoded value from Interface/SharedXML/ExportUtil.lua
        constexpr size_t byte_size = 6;
    }

    void exportBlizzardHash(
        const TalentTree& tree, 
        const TalentTree* complementaryTree, 
        const std::shared_ptr<TalentSkillset> complementarySkillset, 
        std::string& hash_string) {
        if (tree.presetName == "custom") {
            hash_string = "Custom trees cannot be exported to ingame import strings!";
            return;
        }

        // prepare node order
        std::vector<std::string> rawNodeIDOrder = splitString(Presets::LOAD_RAW_NODE_ID_ORDER(tree.presetName), ":");
        int classID = std::stoi(rawNodeIDOrder[1]);
        int specID = std::stoi(rawNodeIDOrder[2]);
        std::vector<int> order;
        for (auto& nodeIDStr : splitString(rawNodeIDOrder[3], ",")) {
            order.push_back(std::stoi(nodeIDStr));
        }

        // prepare class and spec skillsets and trees
        const TalentTree * classTree, * specTree;
        std::shared_ptr<TalentSkillset> classSkillset, specSkillset;
        if (tree.type == TreeType::CLASS) {
            classTree = &tree;
            classSkillset = tree.loadout[tree.activeSkillsetIndex];

            specTree = complementaryTree;
            specSkillset = complementarySkillset;
        }
        else {
            specTree = &tree;
            specSkillset = tree.loadout[tree.activeSkillsetIndex];

            classTree = complementaryTree;
            classSkillset = complementarySkillset;
        }

        // prepare map that holds talent and assigned points with node ids as key
        std::map<int, std::pair<std::shared_ptr<Talent>, unsigned int>> nodeIDTalentMap;
        if (classTree) {
            for (auto& indexTalentPair : classTree->orderedTalents) {
                nodeIDTalentMap[indexTalentPair.second->nodeID] = std::pair<std::shared_ptr<Talent>, int>{
                    indexTalentPair.second, static_cast<unsigned int>(classSkillset->assignedSkillPoints[indexTalentPair.first])
                };
            }
        }
        if (specTree) {
            for (auto& indexTalentPair : specTree->orderedTalents) {
                nodeIDTalentMap[indexTalentPair.second->nodeID] = std::pair<std::shared_ptr<Talent>, int>{
                    indexTalentPair.second, static_cast<unsigned int>(specSkillset->assignedSkillPoints[indexTalentPair.first])
                };
            }
        }

        // start creating the export string
        std::string export_str;

        size_t head = 0;
        size_t byte = 0;
        auto put_bit = [&export_str, &head, &byte](size_t bits, unsigned value) {
            for (size_t i = 0; i < bits; i++)
            {
                size_t bit = head % byte_size;
                head++;
                byte += (value >> i & 0b1) << bit;
                if (bit == byte_size - 1)
                {
                    export_str += base64_char[byte];
                    byte = 0;
                }
            }
        };

        put_bit(version_bits, LOADOUT_SERIALIZATION_VERSION);
        put_bit(spec_bits, static_cast<unsigned>(specID));
        put_bit(tree_bits, 0);  // 0-filled to bypass validation, as GetTreeHash() is unavailable externally

        for (int& nodeID : order) {
            if (!nodeIDTalentMap.count(nodeID)) {
                put_bit(1, 0);
                continue;
            }
            const unsigned int& rank = nodeIDTalentMap[nodeID].second;
            const unsigned int& maxRank = static_cast<unsigned int>(nodeIDTalentMap[nodeID].first->maxPoints);
            const bool isSwitch = nodeIDTalentMap[nodeID].first->type == TalentType::SWITCH;
            if (rank == 0)  // is node selected?
            {
                put_bit(1, 0);
                continue;
            }
            else
            {
                put_bit(1, 1);
            }

            if (rank >= maxRank)  // is node partially ranked?
            {
                put_bit(1, 0);
            }
            else
            {
                put_bit(1, 1);
                put_bit(rank_bits, rank);
            }

            if (!isSwitch)
            {
                put_bit(1, 0);
            }
            else
            {
                put_bit(1, 1);
                put_bit(choice_bits, rank - 1);
            }
        }

        if (head % byte_size)
            export_str += base64_char[byte];

        hash_string = export_str;
    }

    size_t get_bit(std::string& hash_string, size_t& head, size_t& byte, const size_t& bits) {
        size_t val = 0;
        for (size_t i = 0; i < bits; i++)
        {
            size_t bit = head % byte_size;
            head++;
            val += (byte >> bit & 0b1) << i;
            if (bit == byte_size - 1)
            {
                byte = base64_char.find(hash_string[head / byte_size]);
            }
        }
        return val;
    }

    std::pair<std::string, std::string> getClassSpecPresetsFromBlizzHash(std::map<std::string, std::string> presets, std::string& hash_string) {
        if (hash_string.find_first_not_of(base64_char) != std::string::npos)
        {
            return {"", ""};
        }

        if (version_bits + spec_bits + tree_bits > hash_string.size() * byte_size)
        {
            return {"", ""};
        }

        size_t head = 0;
        size_t byte = base64_char.find(hash_string[0]);

        size_t version_id = get_bit(hash_string, head, byte, version_bits);
        size_t spec_id = get_bit(hash_string, head, byte, spec_bits);

        return Presets::CLASS_SPEC_PRESETS_FROM_BLIZZ_SPEC_ID(presets, spec_id);
    }

    bool verifyTreeIDWithBlizzHash(const TalentTree& tree, std::string hash_string) {
        if (hash_string.find_first_not_of(base64_char) != std::string::npos)
        {
            return false;
        }

        if (version_bits + spec_bits + tree_bits > hash_string.size() * byte_size)
        {
            return false;
        }

        size_t head = 0;
        size_t byte = base64_char.find(hash_string[0]);

        size_t version_id = get_bit(hash_string, head, byte, version_bits);
        size_t spec_id = get_bit(hash_string, head, byte, spec_bits);

        std::string presetName = Presets::PRESET_NAME_FROM_BLIZZ_SPEC_ID(spec_id);

        if (tree.type == TreeType::CLASS) {
            std::vector<std::string> presetNameParts = splitString(tree.presetName, "_class");
            return presetNameParts[0] + presetNameParts[1] == presetName;
        }
        else {
            return tree.presetName == presetName;
        }
    }

    bool importBlizzardHash(
        TalentTree& tree,
        TalentTree* complementaryTree,
        std::string& hash_string,
        bool extractComplementarySkillset
    ) {
        extractComplementarySkillset = extractComplementarySkillset && complementaryTree != nullptr;
        // shouldn't be necessary but doesn't hurt either way
        std::string paddedHash = hash_string + "AAAAAAAAAAAA";

        if (paddedHash.find_first_not_of(base64_char) != std::string::npos)
        {
            return false;
        }

        if (version_bits + spec_bits + tree_bits > paddedHash.size() * byte_size)
        {
            return false;
        }

        size_t head = 0;
        size_t byte = base64_char.find(paddedHash[0]);

        auto version_id = get_bit(paddedHash, head, byte, version_bits);
        auto spec_id = get_bit(paddedHash, head, byte, spec_bits);

        if (version_id != LOADOUT_SERIALIZATION_VERSION)
        {
            return false;
        }

        // complete overkill, rework this at some point
        std::vector<std::string> rawNodeIDOrder = splitString(Presets::LOAD_RAW_NODE_ID_ORDER(tree.presetName), ":");
        int classID = std::stoi(rawNodeIDOrder[1]);
        int specID = std::stoi(rawNodeIDOrder[2]);
        std::vector<int> order;
        for (auto& nodeIDStr : splitString(rawNodeIDOrder[3], ",")) {
            order.push_back(std::stoi(nodeIDStr));
        }

        if (spec_id != specID)
        {
            return false;
        }

        // As per Interface/AddOns/Blizzard_ClassTalentUI/Blizzard_ClassTalentImportExport.lua: treeHash is a 128bit hash,
        // passed as an array of 16, 8-bit values. For SimC purposes we can ignore it, as invalid/outdated strings can error
        // in later checks
        get_bit(paddedHash, head, byte, tree_bits);

        std::shared_ptr<TalentSkillset> classSkillset = std::make_shared<TalentSkillset>();
        classSkillset->name = "Ingame imported skillset";
        std::shared_ptr<TalentSkillset> specSkillset = std::make_shared<TalentSkillset>();
        specSkillset->name = "Ingame imported skillset";

        TalentTree* classTree, * specTree;
        if (tree.type == TreeType::CLASS) {
            classTree = &tree;
            specTree = complementaryTree;
        }
        else {
            specTree = &tree;
            classTree = complementaryTree;
        }

        // prepare map that holds talent with node ids as key
        std::map<int, std::pair<std::shared_ptr<TalentSkillset>, std::shared_ptr<Talent>>> nodeIDTalentMap;
        if (classTree) {
            for (auto& indexTalentPair : classTree->orderedTalents) {
                nodeIDTalentMap[indexTalentPair.second->nodeID] = std::pair<std::shared_ptr<TalentSkillset>, std::shared_ptr<Talent>>(classSkillset, indexTalentPair.second);
                classSkillset->assignedSkillPoints[indexTalentPair.first] = 0;
            }
        }
        if (specTree) {
            for (auto& indexTalentPair : specTree->orderedTalents) {
                nodeIDTalentMap[indexTalentPair.second->nodeID] = std::pair<std::shared_ptr<TalentSkillset>, std::shared_ptr<Talent>>(specSkillset, indexTalentPair.second);
                specSkillset->assignedSkillPoints[indexTalentPair.first] = 0;
            }
        }


        for (int& nodeID : order) {
            if (get_bit(paddedHash, head, byte, 1)) { // selected
                size_t rank = nodeIDTalentMap.count(nodeID) ? nodeIDTalentMap[nodeID].second->maxPoints : 0;
                if (get_bit(paddedHash, head, byte, 1))  // partially ranked normal trait
                {
                    rank = get_bit(paddedHash, head, byte, rank_bits);
                }

                if (get_bit(paddedHash, head, byte, 1))  // choice trait
                {
                    size_t choice = get_bit(paddedHash, head, byte, choice_bits);

                    rank += choice;
                }

                if (nodeIDTalentMap.count(nodeID)) {
                    nodeIDTalentMap[nodeID].first->assignedSkillPoints[nodeIDTalentMap[nodeID].second->index] = static_cast<int>(rank);
                }
            }
        }

        if (classTree) {
            for (auto& indexPointsPair : classSkillset->assignedSkillPoints) {
                if (classTree->orderedTalents.at(indexPointsPair.first)->type == TalentType::SWITCH) {
                    classSkillset->talentPointsSpent += indexPointsPair.second > 0 ? 1 : 0;
                }
                else {
                    classSkillset->talentPointsSpent += indexPointsPair.second;
                }
            }
        }
        if (specTree) {
            for (auto& indexPointsPair : specSkillset->assignedSkillPoints) {
                if (specTree->orderedTalents.at(indexPointsPair.first)->type == TalentType::SWITCH) {
                    specSkillset->talentPointsSpent += indexPointsPair.second > 0 ? 1 : 0;
                }
                else {
                    specSkillset->talentPointsSpent += indexPointsPair.second;
                }
            }
        }

        if (tree.type == TreeType::CLASS) {
            if (validateSkillset(tree, classSkillset)) {
                tree.loadout.push_back(classSkillset);
                tree.activeSkillsetIndex = static_cast<int>(tree.loadout.size() - 1);
                activateSkillset(tree, tree.activeSkillsetIndex);
            }
            if (extractComplementarySkillset) {
                if (validateSkillset(*complementaryTree, specSkillset)) {
                    complementaryTree->loadout.push_back(specSkillset);
                    complementaryTree->activeSkillsetIndex = static_cast<int>(specTree->loadout.size() - 1);
                    activateSkillset(*complementaryTree, specTree->activeSkillsetIndex);
                }
            }
        }
        else {
            if (validateSkillset(tree, specSkillset)) {
                tree.loadout.push_back(specSkillset);
                tree.activeSkillsetIndex = static_cast<int>(tree.loadout.size() - 1);
                activateSkillset(tree, tree.activeSkillsetIndex);
            }
            if (extractComplementarySkillset) {
                if (validateSkillset(*complementaryTree, classSkillset)) {
                    complementaryTree->loadout.push_back(classSkillset);
                    complementaryTree->activeSkillsetIndex = static_cast<int>(complementaryTree->loadout.size() - 1);
                    activateSkillset(*complementaryTree, complementaryTree->activeSkillsetIndex);
                }
            }
        }

        return true;
    }

    int getLevelRequirement(const TalentSkillset& sk, const TalentTree& tree, int offset) {
        return getLevelRequirement(sk.talentPointsSpent, tree, offset);
    }

    int getLevelRequirement(const int& pointsSpent, const TalentTree& tree, int offset) {
        int minLevel = 0;
        if (pointsSpent > tree.preFilledTalentPoints) {
            minLevel += 10;
            if (tree.type == TreeType::CLASS) {
                minLevel += (pointsSpent + offset - tree.preFilledTalentPoints) * 2 - 2;
            }
            else {
                minLevel += (pointsSpent + offset - tree.preFilledTalentPoints) * 2 - 1;
            }
        }
        return minLevel;
    }

    void ImportSimData(std::vector<std::string>& simOutputNames, vec2d<std::string>& simOutputs, TalentTree& tree) {
        if (simOutputNames.size() != simOutputs.size()) {
            return;
        }
        for (size_t i = 0; i < simOutputNames.size(); i++) {
            SimResult res = ImportSimResult(simOutputNames[i], simOutputs[i], tree);
            tree.simAnalysisRawResults.push_back(res);
        }
    }

    SimResult ImportSimResult(std::string& simOutputName, std::vector<std::string>& simOutput, TalentTree& tree) {
        SimResult res;
        res.name = simOutputName;
        std::vector<std::pair<std::string, float>> extractedSimRuns;

        for(size_t i = 0; i < simOutput.size(); i++)
        {
            std::string& line = simOutput[i];
            //grab main run
            if (line.substr(0, 7) == "Player:") {
                std::string trimLine{ line.substr(8) };
                int counter = 4;
                for (int i = static_cast<int>(trimLine.length() - 1); i >= 0; i--) {
                    const char& ch = trimLine[i];
                    if (ch == ' ') {
                        counter--;
                    }
                    if (counter == 0) {
                        counter = i;
                        break;
                    }
                }
                std::string name = trimLine.substr(0, counter);
                if (i >= simOutput.size() - 1) {
                    break;
                }
                line = simOutput[++i];
                size_t start = line.find("DPS=") + 4;
                size_t end = line.find("DPS-Error") - 1;
                float dps = static_cast<float>(std::stod(line.substr(start, end - start + 1)));
                extractedSimRuns.push_back(std::pair{ name, dps });
            }
            //grab all profilesets
            else if (line.substr(0, 39) == "Profilesets (median Damage per Second):") {
                if (i >= simOutput.size() - 1) {
                    break;
                }
                line = simOutput[++i];
                while (line.substr(0, 21) != "Baseline Performance:" && line != "") {
                    std::vector<std::string> content = splitString(line, ":");
                    float dps = static_cast<float>(std::stod(content[0]));
                    std::string name = trim(content[1]);
                    extractedSimRuns.push_back(std::pair{ name, dps });
                    if (i >= simOutput.size() - 1) {
                        break;
                    }
                    line = simOutput[++i];
                }
            }
        }

        //fetch skillsets and put them in simresult
        for (auto& run : extractedSimRuns) {
            for (auto& skillset : tree.loadout) {
                if (skillset->name == run.first) {
                    res.skillsets.push_back(*skillset);
                    res.dps.push_back(run.second);
                }
            }
        }
        return res;
    }

    std::string trim(const std::string& str)
    {
        size_t first = str.find_first_not_of(' ');
        if (std::string::npos == first)
        {
            return str;
        }
        size_t last = str.find_last_not_of(' ');
        return str.substr(first, (last - first + 1));
    }

    bool checkTalentValidity(const TalentTree& tree) {
        int maxPointsRequirement = 0;
        //first check if at least one parent is filled
        for (auto& talent : tree.orderedTalents) {
            maxPointsRequirement = talent.second->pointsRequired > maxPointsRequirement ? talent.second->pointsRequired : maxPointsRequirement;
            if (talent.second->points == 0) {
                continue;
            }
            bool isParentFilled = talent.second->parents.size() == 0;
            for (auto& parent : talent.second->parents) {
                if (parent->points == parent->maxPoints) {
                    isParentFilled = true;
                    break;
                }
            }
            if (!isParentFilled) {
                return false;
            }
        }

        //second check if all talents fulfill their points requirement
        std::vector<TalentVec> reqSortedTalents;
        reqSortedTalents.resize(maxPointsRequirement + 1, TalentVec());
        for (auto& talent : tree.orderedTalents) {
            reqSortedTalents[talent.second->pointsRequired].push_back(talent.second);
        }
        for (int i = 1; i < reqSortedTalents.size(); i++) {
            int talentPointsSpentUntil = 0;
            for (int k = 0; k < i; k++) {
                for (int l = 0; l < reqSortedTalents[k].size(); l++) {
                    talentPointsSpentUntil += reqSortedTalents[k][l]->points;
                }
            }
            talentPointsSpentUntil -= tree.preFilledTalentPoints;
            for (int j = 0; j < reqSortedTalents[i].size(); j++) {
                if (reqSortedTalents[i][j]->points > 0 && talentPointsSpentUntil < i) {
                    return false;
                }
            }
        }

        return true;
    }

    std::vector<double> getSimilarityRanking(std::string formattedTalentName, std::vector<std::string> formattedIconNames) {
        std::vector<double> ranking;
        for (auto& iconName : formattedIconNames) {
            std::vector<std::string> talentNamePairs = createWordLetterPairs(formattedTalentName);
            std::vector<std::string> iconNamePairs = createWordLetterPairs(iconName);

            size_t pairIntersection = 0;
            size_t pairUnion = talentNamePairs.size()  + iconNamePairs.size();

            for (int i = 0; i < talentNamePairs.size(); i++)
            {
                for (int j = 0; j < iconNamePairs.size(); j++)
                {
                    if (talentNamePairs[i] == iconNamePairs[j])
                    {
                        pairIntersection++;
                        iconNamePairs.erase(iconNamePairs.begin() + j);

                        break;
                    }
                }
            }

            ranking.push_back((2.0 * pairIntersection) / pairUnion);
        }
        return ranking;
    }

    std::vector<std::string> createWordLetterPairs(std::string name) {
        std::vector<std::string> pairs;
        for (int i = 0; i < name.length() - 1; i++) {
            pairs.push_back(name.substr(i, 2));
        }
        return pairs;
    }

    /*
    Helper function that transforms a string to a version with only lowercase letters stripping everything else
    */
    std::string simplifyString(const std::string& s) {
        std::string ret = "";
        ret.reserve(s.length());
        for (auto& ch : s) {
            if (ch >= 'A' && ch <= 'Z') {
                ret += static_cast<char>(ch + 32);
            }
            if (ch >= 'a' && ch <= 'z') {
                ret += ch;
            }
        }
        return ret;
    }

    void filterTalentSearch(const std::string& search, TalentVec& filteredTalents, const TalentTree& tree) {
        //TTMNOTE: Apparently we don't need to do any fancy search register caching and can just brute force everything all the time
        std::string formattedSearch = simplifyString(search);

        for (auto& talent : tree.orderedTalents) {
            std::string formattedTalentName = simplifyString(talent.second->name + talent.second->nameSwitch);
            std::string concatTalentDescriptions = "";
            int maxP = talent.second->type == TalentType::SWITCH ? 2 : talent.second->maxPoints;
            for (int i = 0; i < maxP; i++) {
                concatTalentDescriptions += talent.second->descriptions[i];
            }
            std::string formattedTalentDescription = simplifyString(concatTalentDescriptions);
            bool typeMatched = false;
            switch (talent.second->type) {
            case TalentType::ACTIVE: {
                typeMatched = std::string("active").find(formattedSearch) != std::string::npos;
            }break;
            case TalentType::PASSIVE: {
                typeMatched = std::string("passive").find(formattedSearch) != std::string::npos;
            }break;
            case TalentType::SWITCH: {
                typeMatched = std::string("switch").find(formattedSearch) != std::string::npos;
            }break;
            }
            if (formattedTalentName.find(formattedSearch) != std::string::npos 
                || formattedTalentDescription.find(formattedSearch) != std::string::npos
                || typeMatched) {
                filteredTalents.push_back(talent.second);
            }
        }
    }

    std::string simcTokenizeName(const std::string& s) {
        std::string name {s};
        if (name.empty()) return "";

        // remove leading '_' or '+'
        std::string::size_type n = name.find_first_not_of("_+");
        std::string::iterator it;
        if (n != std::string::npos)
        {
            it = name.erase(name.begin(), name.begin() + n);
        }
        else
        {
            it = name.begin();
        }

        for (; it != name.end(); )
        {
            unsigned char c = *it;

            if (c >= 0x80)
            {
                it = name.erase(it);
            }
            else if (std::isalpha(c))
            {
                *it++ = std::tolower(c);
            }
            else if (c == ' ')
            {
                *it++ = '_';
            }
            else if (c != '_' &&
                c != '+' &&
                c != '.' &&
                c != '%' &&
                !std::isdigit(c))
            {
                it = name.erase(it);
            }
            else
            {
                ++it; // Just skip it
            }
        }
        return name;
    }
}
