package net.twisterrob.gradle.checkstyle.test

/**
 * Test resources for Checkstyle static analyser.
 *
 * Usage:
 * ```
 * private lateinit var gradle: GradleRunnerRule
 * private val checkstyle = CheckstyleTestResources()
 *
 * @Test fun test() {
 *     gradle.file(checkstyle.….config, "config", "checkstyle", "checkstyle.xml")
 * }
 * ```
 */
@Suppress("detekt.UseDataClass") // https://github.com/detekt/detekt/issues/5339
class CheckstyleTestResources {

	val empty: EmptyConfiguration = object : EmptyConfiguration {}
	val simple: SimpleFailure = object : SimpleFailure {}
	val multi: MultiFailure = object : MultiFailure {}

	interface EmptyConfiguration {
		val config: String
			get() = read("empty/empty-checkstyle.xml")
	}

	interface SimpleFailure {
		val config: String
			get() = read("simple_failure/simple-checkstyle.xml")

		val content: String
			get() = read("simple_failure/empty.java")

		val message: Regex
			get() = Regex(""".*src.main.java.Checkstyle\.java:1: .*? \[Header]""")
	}

	interface MultiFailure {
		val config: String
			get() = read("multiple_violations/multi-checkstyle.xml")

		val contents: Map<String, String>
			get() =
				listOf(
					"EmptyBlock_3.java",
					"MemberName_2.java",
					"UnusedImports_4.java"
				).associateWith { read("multiple_violations/$it") }
	}

	companion object {
		private fun read(path: String): String =
			fileFromResources(CheckstyleTestResources::class.java, path)
	}
}

private fun fileFromResources(loader: Class<*>, path: String): String {
	val container = "/${loader.`package`.name.replace(".", "/")}"
	val fullPath = "${container}/${path}"
	val resource = loader.getResource(fullPath)
		?: throw IllegalArgumentException("Cannot find ${fullPath}, trying to load ${path} near ${loader}.")
	return resource.readText()
}
