<?php

/*
 * This file is part of the Sylius package.
 *
 * (c) Paweł Jędrzejewski
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

declare(strict_types=1);

namespace spec\Sylius\Component\Core\Promotion\Checker\Rule;

use PhpSpec\ObjectBehavior;
use Sylius\Component\Core\Model\CustomerInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Sylius\Component\Core\Repository\OrderRepositoryInterface;
use Sylius\Component\Promotion\Checker\Rule\RuleCheckerInterface;
use Sylius\Component\Promotion\Model\PromotionSubjectInterface;
use Sylius\Component\Resource\Exception\UnexpectedTypeException;

final class NthOrderRuleCheckerSpec extends ObjectBehavior
{
    function let(OrderRepositoryInterface $ordersRepository): void
    {
        $this->beConstructedWith($ordersRepository);
    }

    function it_implements_a_rule_checker_interface(): void
    {
        $this->shouldImplement(RuleCheckerInterface::class);
    }

    function it_recognizes_a_subject_without_customer_as_not_eligible(OrderInterface $subject): void
    {
        $subject->getCustomer()->willReturn(null);

        $this->isEligible($subject, ['nth' => 10])->shouldReturn(false);
    }

    function it_recognizes_a_subject_as_not_eligible_if_nth_order_is_zero(
        CustomerInterface $customer,
        OrderInterface $subject,
        OrderRepositoryInterface $ordersRepository
    ): void {
        $subject->getCustomer()->willReturn($customer);
        $customer->getId()->willReturn(1);

        $ordersRepository->countByCustomer($customer)->willReturn(0);

        $this->isEligible($subject, ['nth' => 10])->shouldReturn(false);
    }

    function it_recognizes_a_subject_as_not_eligible_if_nth_order_is_less_then_configured(
        CustomerInterface $customer,
        OrderInterface $subject,
        OrderRepositoryInterface $ordersRepository
    ): void {
        $subject->getCustomer()->willReturn($customer);
        $customer->getId()->willReturn(1);

        $ordersRepository->countByCustomer($customer)->willReturn(5);

        $this->isEligible($subject, ['nth' => 10])->shouldReturn(false);
    }

    function it_recognizes_a_subject_as_not_eligible_if_nth_order_is_greater_than_configured(
        CustomerInterface $customer,
        OrderInterface $subject,
        OrderRepositoryInterface $ordersRepository
    ): void {
        $subject->getCustomer()->willReturn($customer);
        $customer->getId()->willReturn(1);

        $ordersRepository->countByCustomer($customer)->willReturn(12);

        $this->isEligible($subject, ['nth' => 10])->shouldReturn(false);
    }

    function it_recognizes_a_subject_as_eligible_if_nth_order_is_equal_with_configured(
        CustomerInterface $customer,
        OrderInterface $subject,
        OrderRepositoryInterface $ordersRepository
    ): void {
        $subject->getCustomer()->willReturn($customer);
        $customer->getId()->willReturn(1);

        $ordersRepository->countByCustomer($customer)->willReturn(9);

        $this->isEligible($subject, ['nth' => 10])->shouldReturn(true);
    }

    function it_recognizes_a_subject_as_eligible_if_nth_order_is_one_and_customer_is_not_in_database(
        CustomerInterface $customer,
        OrderInterface $subject
    ): void {
        $subject->getCustomer()->willReturn($customer);
        $customer->getId()->willReturn(null);

        $this->isEligible($subject, ['nth' => 1])->shouldReturn(true);
    }

    function it_recognizes_a_subject_as_not_eligible_if_it_is_first_order_of_new_customer_and_promotion_is_for_more_than_one_order(
        CustomerInterface $customer,
        OrderInterface $subject
    ): void {
        $subject->getCustomer()->willReturn($customer);
        $customer->getId()->willReturn(null);

        $this->isEligible($subject, ['nth' => 10])->shouldReturn(false);
    }

    function it_recognizes_a_subject_as_not_eligible_if_configuration_is_invalid(OrderInterface $subject): void
    {
        $this->isEligible($subject, [])->shouldReturn(false);
        $this->isEligible($subject, ['nth' => 'string'])->shouldReturn(false);
    }

    function it_throws_an_exception_if_subject_is_not_order(PromotionSubjectInterface $subject): void
    {
        $this
            ->shouldThrow(new UnexpectedTypeException($subject->getWrappedObject(), OrderInterface::class))
            ->during('isEligible', [$subject, ['nth' => 10]])
        ;
    }
}
