﻿#nullable disable
using System.ComponentModel;
using System.Linq;
using System.Threading.Tasks;
using Dapper;
using Microsoft.Data.Sqlite;
using Microsoft.EntityFrameworkCore;
using NewtonsoftJsonSerializer = Newtonsoft.Json.JsonConvert;
using SystemTextJsonSerializer = System.Text.Json.JsonSerializer;
using Vogen.IntegrationTests.TestTypes.StructVos;
using LinqToDB;
using LinqToDB.Data;
using LinqToDB.DataProvider.SQLite;
using LinqToDB.Mapping;

namespace Vogen.IntegrationTests.SerializationAndConversionTests.StructVos;

[ValueObject(underlyingType: typeof(long))]
public partial struct AnotherLongVo { }

public class LongVoTests
{
    [Fact]
    public void equality_between_same_value_objects()
    {
        LongVo.From(18).Equals(LongVo.From(18)).Should().BeTrue();
        (LongVo.From(18) == LongVo.From(18)).Should().BeTrue();

        (LongVo.From(18) != LongVo.From(19)).Should().BeTrue();
        (LongVo.From(18) == LongVo.From(19)).Should().BeFalse();

        LongVo.From(18).Equals(LongVo.From(18)).Should().BeTrue();
        (LongVo.From(18) == LongVo.From(18)).Should().BeTrue();

        var original = LongVo.From(18);
        var other = LongVo.From(18);

        ((original as IEquatable<LongVo>).Equals(other)).Should().BeTrue();
        ((other as IEquatable<LongVo>).Equals(original)).Should().BeTrue();
    }

    [Fact]
    public void equality_between_different_value_objects()
    {
        LongVo.From(18).Equals(AnotherLongVo.From(18)).Should().BeFalse();
    }

    [Fact]
    public void CanSerializeToLong_WithNewtonsoftJsonProvider()
    {
        var vo = NewtonsoftJsonLongVo.From(123L);

        string serializedVo = NewtonsoftJsonSerializer.SerializeObject(vo);
        string serializedLong = NewtonsoftJsonSerializer.SerializeObject(vo.Value);

        Assert.Equal(serializedVo, serializedLong);
    }

    [Fact]
    public void CanSerializeToLong_WithSystemTextJsonProvider()
    {
        var vo = SystemTextJsonLongVo.From(123L);

        string serializedVo = SystemTextJsonSerializer.Serialize(vo);
        string serializedLong = SystemTextJsonSerializer.Serialize(vo.Value);

        serializedVo.Equals(serializedLong).Should().BeTrue();
    }

    [Fact]
    public void CanDeserializeFromLong_WithNewtonsoftJsonProvider()
    {
        var value = 123L;
        var vo = NewtonsoftJsonLongVo.From(value);
        var serializedLong = NewtonsoftJsonSerializer.SerializeObject(value);

        var deserializedVo = NewtonsoftJsonSerializer.DeserializeObject<NewtonsoftJsonLongVo>(serializedLong);

        Assert.Equal(vo, deserializedVo);
    }

    [Fact]
    public void CanDeserializeFromLong_WithSystemTextJsonProvider()
    {
        var value = 123L;
        var vo = SystemTextJsonLongVo.From(value);
        var serializedLong = SystemTextJsonSerializer.Serialize(value);

        var deserializedVo = SystemTextJsonSerializer.Deserialize<SystemTextJsonLongVo>(serializedLong);

        Assert.Equal(vo, deserializedVo);
    }

    [Fact]
    public void CanSerializeToLong_WithBothJsonConverters()
    {
        var vo = BothJsonLongVo.From(123L);

        var serializedVo1 = NewtonsoftJsonSerializer.SerializeObject(vo);
        var serializedLong1 = NewtonsoftJsonSerializer.SerializeObject(vo.Value);

        var serializedVo2 = SystemTextJsonSerializer.Serialize(vo);
        var serializedLong2 = SystemTextJsonSerializer.Serialize(vo.Value);

        Assert.Equal(serializedVo1, serializedLong1);
        Assert.Equal(serializedVo2, serializedLong2);
    }

    [Fact]
    public void WhenNoJsonConverter_SystemTextJsonSerializesWithValueProperty()
    {
        var vo = NoJsonLongVo.From(123L);

        var serialized = SystemTextJsonSerializer.Serialize(vo);

        var expected = "{\"Value\":" + vo.Value + "}";

        Assert.Equal(expected, serialized);
    }

    [Fact]
    public void WhenNoJsonConverter_NewtonsoftSerializesWithoutValueProperty()
    {
        var vo = NoJsonLongVo.From(123L);

        var serialized = NewtonsoftJsonSerializer.SerializeObject(vo);

        var expected = $"\"{vo.Value}\"";

        Assert.Equal(expected, serialized);
    }

    [Fact]
    public void WhenNoTypeConverter_SerializesWithValueProperty()
    {
        var vo = NoConverterLongVo.From(123L);

        var newtonsoft = SystemTextJsonSerializer.Serialize(vo);
        var systemText = SystemTextJsonSerializer.Serialize(vo);

        var expected = "{\"Value\":" + vo.Value + "}";

        Assert.Equal(expected, newtonsoft);
        Assert.Equal(expected, systemText);
    }

    [Fact]
    public void WhenEfCoreValueConverterUsesValueConverter()
    {
        var connection = new SqliteConnection("DataSource=:memory:");
        connection.Open();

        var options = new DbContextOptionsBuilder<TestDbContext>()
            .UseSqlite(connection)
            .Options;

        var original = new EfCoreTestEntity { Id = EfCoreLongVo.From(123L) };
        using (var context = new TestDbContext(options))
        {
            context.Database.EnsureCreated();
            context.Entities.Add(original);
            context.SaveChanges();
        }
        using (var context = new TestDbContext(options))
        {
            var all = context.Entities.ToList();
            var retrieved = Assert.Single(all);
            Assert.Equal(original.Id, retrieved.Id);
        }
    }

    [Fact]
    public async Task WhenDapperValueConverterUsesValueConverter()
    {
        using var connection = new SqliteConnection("DataSource=:memory:");
        await connection.OpenAsync();

        IEnumerable<DapperLongVo> results = await connection.QueryAsync<DapperLongVo>("SELECT 123");

        var value = Assert.Single(results);
        Assert.Equal(DapperLongVo.From(123L), value);
    }

    [Fact]
    public void WhenLinqToDbValueConverterUsesValueConverter()
    {
        var connection = new SqliteConnection("DataSource=:memory:");
        connection.Open();

        var original = new LinqToDbTestEntity { Id = LinqToDbLongVo.From(123) };
        using (var context = new DataConnection(
                   SQLiteTools.GetDataProvider("SQLite.MS"),
                   connection,
                   disposeConnection: false))
        {
            context.CreateTable<LinqToDbTestEntity>();
            context.Insert(original);
        }
        using (var context = new DataConnection(
                   SQLiteTools.GetDataProvider("SQLite.MS"),
                   connection,
                   disposeConnection: false))
        {
            var all = context.GetTable<LinqToDbTestEntity>().ToList();
            var retrieved = Assert.Single(all);
            Assert.Equal(original.Id, retrieved.Id);
        }
    }

    [Theory]
    [InlineData(123L)]
    [InlineData("123")]
    public void TypeConverter_CanConvertToAndFrom(object value)
    {
        var converter = TypeDescriptor.GetConverter(typeof(NoJsonLongVo));
        var id = converter.ConvertFrom(value);
        Assert.IsType<NoJsonLongVo>(id);
        Assert.Equal(NoJsonLongVo.From(123L), id);

        var reconverted = converter.ConvertTo(id, value.GetType());
        Assert.Equal(value, reconverted);
    }

    public class TestDbContext : DbContext
    {
        public DbSet<EfCoreTestEntity> Entities { get; set; }

        public TestDbContext(DbContextOptions options) : base(options)
        {
        }

        protected override void OnModelCreating(ModelBuilder modelBuilder)
        {
            modelBuilder
                .Entity<EfCoreTestEntity>(builder =>
                {
                    builder
                        .Property(x => x.Id)
                        .HasConversion(new EfCoreLongVo.EfCoreValueConverter())
                        .ValueGeneratedNever();
                });
        }
    }

    public class EfCoreTestEntity
    {
        public EfCoreLongVo Id { get; set; }
    }

    public class LinqToDbTestEntity
    {
        [Column(DataType = DataType.Int64)]
        [ValueConverter(ConverterType = typeof(LinqToDbLongVo.LinqToDbValueConverter))]
        public LinqToDbLongVo Id { get; set; }
    }
}