﻿#nullable disable
using System.ComponentModel;
using System.Linq;
using System.Threading.Tasks;
using Dapper;
using Microsoft.Data.Sqlite;
using Microsoft.EntityFrameworkCore;
using NewtonsoftJsonSerializer = Newtonsoft.Json.JsonConvert;
using SystemTextJsonSerializer = System.Text.Json.JsonSerializer;
using Vogen.IntegrationTests.TestTypes.ClassVos;
using LinqToDB;
using LinqToDB.Data;
using LinqToDB.DataProvider.SQLite;
using LinqToDB.Mapping;

// ReSharper disable RedundantOverflowCheckingContext
// ReSharper disable ConvertToLocalFunction

namespace Vogen.IntegrationTests.SerializationAndConversionTests.ClassVos;

[ValueObject(underlyingType: typeof(char))]
public partial struct AnotherCharVo { }

public class CharVoTests
{
    [Fact]
    public void equality_between_same_value_objects()
    {
        CharVo.From('a').Equals(CharVo.From('a')).Should().BeTrue();
        (CharVo.From('a') == CharVo.From('a')).Should().BeTrue();

        (CharVo.From('a') != CharVo.From('b')).Should().BeTrue();
        (CharVo.From('a') == CharVo.From('b')).Should().BeFalse();

        CharVo.From('a').Equals(CharVo.From('a')).Should().BeTrue();
        (CharVo.From('a') == CharVo.From('a')).Should().BeTrue();

        var original = CharVo.From('a');
        var other = CharVo.From('a');

        ((original as IEquatable<CharVo>).Equals(other)).Should().BeTrue();
        ((other as IEquatable<CharVo>).Equals(original)).Should().BeTrue();
    }


    [Fact]
    public void equality_between_different_value_objects()
    {
        CharVo.From('a').Equals(AnotherCharVo.From('a')).Should().BeFalse();
    }

    [Fact]
    public void CanSerializeToShort_WithNewtonsoftJsonProvider()
    {
        var vo = NewtonsoftJsonCharVo.From('a');

        string serializedVo = NewtonsoftJsonSerializer.SerializeObject(vo);
        string serializedShort = NewtonsoftJsonSerializer.SerializeObject(vo.Value);

        Assert.Equal(serializedVo, serializedShort);
    }

    
    [Fact]
    public void CanDeserializeFromShort_WithNewtonsoftJsonProvider()
    {
        char value = 'a';
        var vo = NewtonsoftJsonCharVo.From(value);
        var serializedShort = NewtonsoftJsonSerializer.SerializeObject(value);

        var deserializedVo = NewtonsoftJsonSerializer.DeserializeObject<NewtonsoftJsonCharVo>(serializedShort);

        Assert.Equal(vo, deserializedVo);
    }

    [Fact]
    public void CanDeserializeFromShort_WithSystemTextJsonProvider()
    {
        char value = 'a';
        var vo = SystemTextJsonCharVo.From(value);
        var serializedShort = SystemTextJsonSerializer.Serialize(value);

        var deserializedVo = SystemTextJsonSerializer.Deserialize<SystemTextJsonCharVo>(serializedShort);

        Assert.Equal(vo, deserializedVo);
    }

    [Fact]
    public void CanSerializeToShort_WithBothJsonConverters()
    {
        var vo = BothJsonCharVo.From('a');

        var serializedVo1 = NewtonsoftJsonSerializer.SerializeObject(vo);
        var serializedShort1 = NewtonsoftJsonSerializer.SerializeObject(vo.Value);

        var serializedVo2 = SystemTextJsonSerializer.Serialize(vo);
        var serializedShort2 = SystemTextJsonSerializer.Serialize(vo.Value);

        Assert.Equal(serializedVo1, serializedShort1);
        Assert.Equal(serializedVo2, serializedShort2);
    }

    [Fact]
    public void WhenNoJsonConverter_SystemTextJsonSerializesWithValueProperty()
    {
        var vo = NoJsonCharVo.From('a');

        var serialized = SystemTextJsonSerializer.Serialize(vo);

        var expected = "{\"Value\":\"a\"}";

        Assert.Equal(expected, serialized);
    }

    [Fact]
    public void WhenNoJsonConverter_NewtonsoftSerializesWithoutValueProperty()
    {
        var vo = NoJsonCharVo.From('a');

        var serialized = NewtonsoftJsonSerializer.SerializeObject(vo);

        var expected = $"\"{vo.Value}\"";

        Assert.Equal(expected, serialized);
    }

    [Fact]
    public void WhenNoTypeConverter_SerializesWithValueProperty()
    {
        var vo = NoConverterCharVo.From('a');

        var newtonsoft = SystemTextJsonSerializer.Serialize(vo);
        var systemText = SystemTextJsonSerializer.Serialize(vo);

        var expected = "{\"Value\":\"a\"}";

        Assert.Equal(expected, newtonsoft);
        Assert.Equal(expected, systemText);
    }

    [Fact]
    public void WhenEfCoreValueConverterUsesValueConverter()
    {
        var connection = new SqliteConnection("DataSource=:memory:");
        connection.Open();

        var options = new DbContextOptionsBuilder<TestDbContext>()
            .UseSqlite(connection)
            .Options;

        var original = new EfCoreTestEntity { Id = EfCoreCharVo.From('a') };
        using (var context = new TestDbContext(options))
        {
            context.Database.EnsureCreated();
            context.Entities.Add(original);
            context.SaveChanges();
        }
        using (var context = new TestDbContext(options))
        {
            var all = context.Entities.ToList();
            var retrieved = Assert.Single(all);
            Assert.Equal(original.Id, retrieved.Id);
        }
    }

    [Fact]
    public async Task WhenDapperValueConverterUsesValueConverter()
    {
        using var connection = new SqliteConnection("DataSource=:memory:");
        await connection.OpenAsync();

        IEnumerable<DapperCharVo> results = await connection.QueryAsync<DapperCharVo>("SELECT 'a'");

        var value = Assert.Single(results);
        Assert.Equal(DapperCharVo.From('a'), value);
    }

    [Fact]
    public void WhenLinqToDbValueConverterUsesValueConverter()
    {
        var connection = new SqliteConnection("DataSource=:memory:");
        connection.Open();

        var original = new LinqToDbTestEntity { Id = LinqToDbCharVo.From('a') };
        using (var context = new DataConnection(
                   SQLiteTools.GetDataProvider("SQLite.MS"),
                   connection,
                   disposeConnection: false))
        {
            context.CreateTable<LinqToDbTestEntity>();
            context.Insert(original);
        }
        using (var context = new DataConnection(
                   SQLiteTools.GetDataProvider("SQLite.MS"),
                   connection,
                   disposeConnection: false))
        {
            var all = context.GetTable<LinqToDbTestEntity>().ToList();
            var retrieved = Assert.Single(all);
            Assert.Equal(original.Id, retrieved.Id);
        }
    }

    [Theory]
    [InlineData((char) 97)]
    public void TypeConverter_CanConvertToAndFrom(object value)
    {
        var converter = TypeDescriptor.GetConverter(typeof(NoJsonCharVo));
        var id = converter.ConvertFrom(value);
        Assert.IsType<NoJsonCharVo>(id);
        Assert.Equal(NoJsonCharVo.From('a'), id);

        var reconverted = converter.ConvertTo(id, value.GetType());
        Assert.Equal(value, reconverted);
    }

    public class TestDbContext : DbContext
    {
        public DbSet<EfCoreTestEntity> Entities { get; set; }

        public TestDbContext(DbContextOptions options) : base(options)
        {
        }

        protected override void OnModelCreating(ModelBuilder modelBuilder)
        {
            modelBuilder
                .Entity<EfCoreTestEntity>(builder =>
                {
                    builder
                        .Property(x => x.Id)
                        .HasConversion(new EfCoreCharVo.EfCoreValueConverter())
                        .ValueGeneratedNever();
                });
        }
    }

    public class EfCoreTestEntity
    {
        public EfCoreCharVo Id { get; set; }
    }

    public class LinqToDbTestEntity
    {
        [Column(DataType = DataType.Char)]
        [ValueConverter(ConverterType = typeof(LinqToDbCharVo.LinqToDbValueConverter))]
        public LinqToDbCharVo Id { get; set; }
    }
}