/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
//  TEST    Foundation::Containers::SortedAssociation
//      \note Code-Status:  <a href="Code-Status.md#Beta">Beta</a>
#include "Stroika/Foundation/StroikaPreComp.h"

#include <iostream>

#include "Stroika/Foundation/Containers/Concrete/SortedAssociation_SkipList.h"
#include "Stroika/Foundation/Containers/Concrete/SortedAssociation_stdmultimap.h"
#include "Stroika/Foundation/Containers/SortedAssociation.h"
#include "Stroika/Foundation/Debug/Assertions.h"
#include "Stroika/Foundation/Debug/Trace.h"
#include "Stroika/Foundation/Debug/Visualizations.h"
#include "Stroika/Foundation/Memory/Optional.h"

#include "Stroika/Frameworks/Test/ArchtypeClasses.h"
#include "Stroika/Frameworks/Test/TestHarness.h"

#include "../TestCommon/CommonTests_Association.h"

using namespace Stroika::Foundation;
using namespace Stroika::Foundation::Containers;

using namespace Stroika::Frameworks;

using Test::ArchtypeClasses::AsIntsEqualsComparer;
using Test::ArchtypeClasses::AsIntsLessComparer;
using Test::ArchtypeClasses::AsIntsThreeWayComparer;
using Test::ArchtypeClasses::OnlyCopyableMoveable;
using Test::ArchtypeClasses::OnlyCopyableMoveableAndTotallyOrdered;

using Concrete::SortedAssociation_SkipList;
using Concrete::SortedAssociation_stdmultimap;

#if qStroika_HasComponent_googletest
namespace {
    template <typename CONCRETE_CONTAINER>
    void DoTestForConcreteContainer_ ()
    {
        using namespace CommonTests::AssociationTests;
        auto testSchema                      = DEFAULT_TESTING_SCHEMA<CONCRETE_CONTAINER>{};
        testSchema.ApplyToContainerExtraTest = [] (const typename CONCRETE_CONTAINER::ArchetypeContainerType& m) {
            // verify in sorted order
            EXPECT_TRUE (m.IsOrderedBy ([comparer = m.GetKeyInOrderComparer ()] (auto l, auto r) { return comparer (l.fKey, r.fKey); }));
        };
        SimpleAssociationTest_All_ (testSchema);
        SimpleAssociationTest_WithDefaultEqComparer_ (testSchema);
    }
    template <typename CONCRETE_CONTAINER, typename FACTORY, typename VALUE_EQUALS_COMPARER_TYPE>
    void DoTestForConcreteContainer_ (FACTORY factory, VALUE_EQUALS_COMPARER_TYPE valueEqualsComparer)
    {
        using namespace CommonTests::AssociationTests;
        auto testSchema = DEFAULT_TESTING_SCHEMA<CONCRETE_CONTAINER, FACTORY, VALUE_EQUALS_COMPARER_TYPE>{factory, valueEqualsComparer};
        testSchema.ApplyToContainerExtraTest = [] (const typename CONCRETE_CONTAINER::ArchetypeContainerType& m) {
            // verify in sorted order
            using value_type       = typename CONCRETE_CONTAINER::value_type;
            using key_type         = typename CONCRETE_CONTAINER::key_type;
            using INORDER_COMPARER = decltype (m.GetKeyInOrderComparer ());
            optional<value_type> last;
            for (value_type i : m) {
                if (last.has_value ()) {
                    EXPECT_TRUE ((Common::ThreeWayComparerAdapter<key_type, INORDER_COMPARER>{m.GetKeyInOrderComparer ()}(last->fKey, i.fKey) <= 0));
                }
                last = i;
            }
        };
        SimpleAssociationTest_All_ (testSchema);
    }
}

namespace {
    GTEST_TEST (Foundation_Containers_SortedAssociation, FACTORY_DEFAULT)
    {
        DoTestForConcreteContainer_<SortedAssociation<size_t, size_t>> ();
        DoTestForConcreteContainer_<SortedAssociation<OnlyCopyableMoveableAndTotallyOrdered, OnlyCopyableMoveableAndTotallyOrdered>> ();
        DoTestForConcreteContainer_<SortedAssociation<OnlyCopyableMoveable, OnlyCopyableMoveable>> (
            [] () { return SortedAssociation<OnlyCopyableMoveable, OnlyCopyableMoveable> (AsIntsLessComparer<OnlyCopyableMoveable>{}); },
            AsIntsEqualsComparer<OnlyCopyableMoveable>{});
    }
}

namespace {
    GTEST_TEST (Foundation_Containers_SortedAssociation, SortedAssociation_stdmultimap)
    {
        DoTestForConcreteContainer_<SortedAssociation_stdmultimap<size_t, size_t>> ();
        DoTestForConcreteContainer_<SortedAssociation_stdmultimap<OnlyCopyableMoveableAndTotallyOrdered, OnlyCopyableMoveableAndTotallyOrdered>> ();
        DoTestForConcreteContainer_<SortedAssociation_stdmultimap<OnlyCopyableMoveable, OnlyCopyableMoveable>> (
            [] () {
                return SortedAssociation_stdmultimap<OnlyCopyableMoveable, OnlyCopyableMoveable> (AsIntsLessComparer<OnlyCopyableMoveable>{});
            },
            AsIntsEqualsComparer<OnlyCopyableMoveable>{});
    }
}

namespace {
    GTEST_TEST (Foundation_Containers_SortedAssociation, SortedAssociation_SkipList)
    {
        DoTestForConcreteContainer_<SortedAssociation_SkipList<size_t, size_t>> ();
        DoTestForConcreteContainer_<SortedAssociation_SkipList<OnlyCopyableMoveableAndTotallyOrdered, OnlyCopyableMoveableAndTotallyOrdered>> ();
        DoTestForConcreteContainer_<SortedAssociation_SkipList<OnlyCopyableMoveable, OnlyCopyableMoveable>> (
            [] () {
                return SortedAssociation_SkipList<OnlyCopyableMoveable, OnlyCopyableMoveable> (AsIntsThreeWayComparer<OnlyCopyableMoveable>{});
            },
            AsIntsEqualsComparer<OnlyCopyableMoveable>{});
    }
}

namespace {
    GTEST_TEST (Foundation_Containers_SortedAssociation, Test3_ConvertAssociation2SortedAssociation)
    {
        Association<int, int>       m{pair<int, int>{1, 2}, pair<int, int>{2, 4}};
        SortedAssociation<int, int> ms{m};
        EXPECT_EQ (ms.size (), 2u);
        EXPECT_TRUE ((*ms.begin () == pair<int, int>{1, 2}));
    }
}

namespace {
    GTEST_TEST (Foundation_Containers_SortedAssociation, Cleanup)
    {
        EXPECT_TRUE (OnlyCopyableMoveableAndTotallyOrdered::GetTotalLiveCount () == 0 and OnlyCopyableMoveable::GetTotalLiveCount () == 0); // simple portable leak check
    }
}
#endif

int main (int argc, const char* argv[])
{
    Test::Setup (argc, argv);
#if qStroika_HasComponent_googletest
    return RUN_ALL_TESTS ();
#else
    cerr << "Stroika regression tests require building with google test feature [  PASSED  ]" << endl;
#endif
}
