/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#include "Stroika/Frameworks/StroikaPreComp.h"

#include <iostream>

#include "Stroika/Foundation/Characters/FloatConversion.h"
#include "Stroika/Foundation/Characters/String2Int.h"
#include "Stroika/Foundation/Characters/ToString.h"
#include "Stroika/Foundation/Common/Property.h"
#include "Stroika/Foundation/DataExchange/InternetMediaTypeRegistry.h"
#include "Stroika/Foundation/IO/Network/HTTP/Exception.h"
#include "Stroika/Foundation/IO/Network/HTTP/Headers.h"
#include "Stroika/Foundation/IO/Network/HTTP/Methods.h"
#include "Stroika/Foundation/Streams/TextReader.h"

#include "Stroika/Frameworks/WebServer/ConnectionManager.h"
#include "Stroika/Frameworks/WebServer/Router.h"
#include "Stroika/Frameworks/WebService/Server/Basic.h"
#include "Stroika/Frameworks/WebService/Server/ObjectRequestHandler.h"
#include "Stroika/Frameworks/WebService/Server/VariantValue.h"

#include "WebServer.h"

#include "AppVersion.h"

using namespace std;

using namespace Stroika::Foundation;
using namespace Stroika::Foundation::Characters;
using namespace Stroika::Foundation::DataExchange;
using namespace Stroika::Foundation::IO::Network;
using namespace Stroika::Foundation::Traversal;

using namespace Stroika::Frameworks::WebServer;
using namespace Stroika::Frameworks::WebService;
using namespace Stroika::Frameworks::WebService::Server;
using namespace Stroika::Frameworks::WebService::Server::VariantValue;

using Memory::BLOB;
using Stroika::Frameworks::WebServer::Request;
using Stroika::Frameworks::WebServer::Response;
using Stroika::Frameworks::WebService::Server::VariantValue::ExtractArgumentsAsVariantValue;

using namespace StroikaSample::WebServices;

namespace {
    const Common::ConstantProperty<Headers> kDefaultResponseHeaders_{[] () {
        Headers h;
        h.server = "Stroika-Sample-WebServices/"_k + AppVersion::kVersion.AsMajorMinorString ();
        return h;
    }};
}

namespace {
    const ObjectRequestHandler::Options kBinaryOpObjRequestOptions_ = {.fObjectMapper = Model::kMapper,
                                                                       .fExtractVariantValueFromRequest = ExtractArgumentsAsVariantValue::FromRequest,
                                                                       .fTreatBodyAsListOfArguments = Iterable<String>{"arg1"sv, "arg2"sv}};
}

/*
 *  It's often helpful to structure together, routes, special interceptors, with your connection manager, to package up
 *  all the logic /options for HTTP interface.
 *
 *  This particular organization also makes it easy to save instance variables with the webserver (like a pointer to a handler)
 *  and access them from the Route handler functions.
 */
class WebServer::Rep_ {
public:
    const Sequence<Route> kRoutes_;        // rules saying how to map urls to code
    shared_ptr<IWSAPI>    fWSImpl_;        // application logic actually handling webservices
    ConnectionManager     fConnectionMgr_; // manage http connection objects, thread pool, etc

    static const WebServiceMethodDescription kVariables_;

    static const WebServiceMethodDescription kPlus_;
    static const WebServiceMethodDescription kMinus;
    static const WebServiceMethodDescription kTimes;
    static const WebServiceMethodDescription kDivide;

    Rep_ (uint16_t portNumber, const shared_ptr<IWSAPI>& wsImpl)
        : kRoutes_{Route{""_RegEx, DefaultPage_},

                   Route{HTTP::MethodsRegEx::kPost, "SetAppState"_RegEx, SetAppState_},

                   Route{"FRED"_RegEx,
                         [] (Request&, Response& response) {
                             response.write ("FRED");
                             response.contentType = InternetMediaTypes::kText_PLAIN;
                         }},

                   /*
                    * the 'variable' API demonstrates a typical REST style CRUD usage - where the 'arguments' mainly come from 
                    * the URL itself.
                    */
                   Route{"variables(/?)"_RegEx,
                         [this] (Message& m) {
                             WriteResponse (m.rwResponse (), kVariables_, kMapper.FromObject (fWSImpl_->Variables_GET ()));
                         }},
                   Route{"variables/(.+)"_RegEx,
                         [this] (Message& m, const String& varName) {
                             WriteResponse (m.rwResponse (), kVariables_, kMapper.FromObject (fWSImpl_->Variables_GET (varName)));
                         }},
                   Route{HTTP::MethodsRegEx::kPostOrPut, "variables/(.+)"_RegEx,
                         [this] (Message& m, const String& varName) {
                             optional<Number> number;
                             // demo getting argument from the body
                             if (not number) {
                                 // read if content-type is text (not json)
                                 if (m.request ().contentType () and
                                     InternetMediaTypeRegistry::sThe->IsA (InternetMediaTypes::kText_PLAIN, *m.request ().contentType ())) {
                                     String argsAsString = Streams::TextReader::New (m.rwRequest ().GetBody ()).ReadAll ();
                                     number              = kMapper.ToObject<Number> (DataExchange::VariantValue{argsAsString});
                                 }
                             }
                             // demo getting argument from the query argument
                             if (not number) {
                                 static const String                         kValueParamName_ = "value"sv;
                                 Mapping<String, DataExchange::VariantValue> args             = PickoutParamValuesFromURL (m.request ());
                                 number = Model::kMapper.ToObject<Number> (args.LookupValue (kValueParamName_));
                             }
                             // demo getting either query arg, or url encoded arg
                             if (not number) {
                                 static const String kValueParamName_ = "value"sv;
                                 // NOTE - PickoutParamValues combines PickoutParamValuesFromURL, and PickoutParamValuesFromBody. You can use
                                 // Either one of those instead. PickoutParamValuesFromURL assumes you know the name of the parameter, and its
                                 // encoded in the query string. PickoutParamValuesFromBody assumes you have something equivalent you can parse ouf
                                 // of the body, either json encoded or form-encoded (as of 2.1d23, only json encoded supported)
                                 Mapping<String, DataExchange::VariantValue> args = PickoutParamValues (m.rwRequest ());
                                 number = Model::kMapper.ToObject<Number> (args.LookupValue (kValueParamName_));
                             }
                             if (not number) {
                                 Execution::Throw (HTTP::ClientErrorException{"Expected argument to PUT/POST variable"sv});
                             }
                             fWSImpl_->Variables_SET (varName, *number);
                             WriteResponse (m.rwResponse (), kVariables_);
                         }},
                   Route{HTTP::MethodsRegEx::kDelete, "variables/(.+)"_RegEx,
                         /*
                          * varName - the first string argument to the request handler - comes from the first (and only) capture in the regexp.
                          */
                         [this] (Message& m, const String& varName) {
                             fWSImpl_->Variables_DELETE (varName);
                             WriteResponse (m.rwResponse (), kVariables_);
                         }},

                   /*
                    *    plus, minus, times, and divide, test-void-return all all demonstrate passing in variables through either the POST body, or query-arguments.
                    */
                   Route{HTTP::MethodsRegEx::kPost, "plus"_RegEx,
                         ObjectRequestHandler::Factory{kBinaryOpObjRequestOptions_,
                                                       [this] (Number arg1, Number arg2) { return fWSImpl_->plus (arg1, arg2); }}},
                   Route{HTTP::MethodsRegEx::kPost, "minus"_RegEx,
                         ObjectRequestHandler::Factory{kBinaryOpObjRequestOptions_,
                                                       [this] (Number arg1, Number arg2) { return fWSImpl_->minus (arg1, arg2); }}},
                   Route{HTTP::MethodsRegEx::kPost, "times"_RegEx,
                         ObjectRequestHandler::Factory{kBinaryOpObjRequestOptions_,
                                                       [this] (Number arg1, Number arg2) { return fWSImpl_->times (arg1, arg2); }}},
                   Route{HTTP::MethodsRegEx::kPost, "divide"_RegEx,
                         ObjectRequestHandler::Factory{kBinaryOpObjRequestOptions_,
                                                       [this] (Number arg1, Number arg2) { return fWSImpl_->divide (arg1, arg2); }}},
                   Route{HTTP::MethodsRegEx::kPost, "test-void-return"_RegEx,
                         ObjectRequestHandler::Factory{
                             ObjectRequestHandler::Options{.fExtractVariantValueFromRequest = ExtractArgumentsAsVariantValue::FromRequest,
                                                           .fTreatBodyAsListOfArguments     = Iterable<String>{"err-if-more-than-10"sv}},
                             [] (double check) {
                                 if (check > 10) {
                                     Execution::Throw (Execution::Exception{"more than 10"sv});
                                 }
                             }}}}
        , fWSImpl_{wsImpl}
        , fConnectionMgr_{SocketAddresses (InternetAddresses_Any (), portNumber), kRoutes_,
                          ConnectionManager::Options{.fDefaultResponseHeaders = kDefaultResponseHeaders_}}
    {
    }
    // Can declare arguments as Request&,Response&
    static void DefaultPage_ (Request&, Response& response)
    {
        WriteDocsPage (response,
                       Sequence<WebServiceMethodDescription>{
                           kVariables_,
                           kPlus_,
                           kMinus,
                           kTimes,
                           kDivide,
                       },
                       DocsOptions{"Stroika Sample WebService - Web Methods"_k, "Note - curl lines all in bash quoting syntax"_k});
    }
    static void SetAppState_ (Message& message)
    {
        String argsAsString = Streams::TextReader::New (message.rwRequest ().GetBody ()).ReadAll ();
        message.rwResponse ().writeln ("<html><body><p>Hi SetAppState ("sv + argsAsString + ")</p></body></html>"sv);
        message.rwResponse ().contentType = InternetMediaTypes::kHTML;
    }
};

/*
 *  Documentation on WSAPIs
 */
const WebServiceMethodDescription WebServer::Rep_::kVariables_{
    "variables"_k,
    Set<String>{HTTP::Methods::kGet, HTTP::Methods::kPost, HTTP::Methods::kDelete},
    InternetMediaTypes::kJSON,
    {},
    Sequence<String>{"curl http://localhost:8080/variables -v --output -", "curl http://localhost:8080/variables/x -v --output -",
                     "curl  -X POST http://localhost:8080/variables/x -v --output -",
                     "curl -H \"Content-Type: application/json\" -X POST -d '{\"value\": 3}' http://localhost:8080/variables/x --output -",
                     "curl -H \"Content-Type: text/plain\" -X POST -d 3 http://localhost:8080/variables/x --output -"},
    Sequence<String>{"@todo - this is a rough draft (but functional). It could use alot of cleanup and review to see WHICH way I recommend "
                     "using, and just provide the recommended ways in samples"},
};

const WebServiceMethodDescription WebServer::Rep_::kPlus_{
    "plus"_k,
    Set<String>{HTTP::Methods::kPost},
    InternetMediaTypes::kJSON,
    {},
    Sequence<String>{
        "curl -H \"Content-Type: application/json\" -X POST -d '{\"arg1\": 3, \"arg2\": 5 }' http://localhost:8080/plus --output -",
        "curl -X POST 'http://localhost:8080/plus?arg1=3&arg2=5' --output -",
    },
    Sequence<String>{"add the two argument numbers"},
};
const WebServiceMethodDescription WebServer::Rep_::kMinus{
    "minus"_k,
    Set<String>{HTTP::Methods::kPost},
    InternetMediaTypes::kJSON,
    {},
    Sequence<String>{
        "curl -H \"Content-Type: application/json\" -X POST -d '{\"arg1\": 4.5, \"arg2\": -3.23 }' http://localhost:8080/minus --output -",
    },
    Sequence<String>{"subtract the two argument numbers"},
};
const WebServiceMethodDescription WebServer::Rep_::kTimes{
    "times"_k,
    Set<String>{HTTP::Methods::kPost},
    InternetMediaTypes::kJSON,
    {},
    Sequence<String>{
        "curl -H \"Content-Type: application/json\" -X POST -d '{\"arg1\":\"2 + 4i\", \"arg2\": 3.2 }' http://localhost:8080/times "
        "--output -",
        "curl -H \"Content-Type: application/json\" -X POST -d '{\"arg1\":\"2 + i\", \"arg2\": \"2 - i\" }' http://localhost:8080/times "
        "--output -",
    },
    Sequence<String>{"multiply the two argument numbers"},
};
const WebServiceMethodDescription WebServer::Rep_::kDivide{
    "divide"_k,
    Set<String>{HTTP::Methods::kPost},
    InternetMediaTypes::kJSON,
    {},
    Sequence<String>{
        "curl -H \"Content-Type: application/json\" -X POST -d '{\"arg1\":\"2 + i\", \"arg2\": 0 }' http://localhost:8080/divide --output "
        "-",
    },
    Sequence<String>{"divide the two argument numbers"},
};

WebServer::WebServer (uint16_t portNumber, const shared_ptr<IWSAPI>& wsImpl)
    : fRep_{make_shared<Rep_> (portNumber, wsImpl)}
{
}