/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#include "Stroika/Frameworks/StroikaPreComp.h"

#include "Stroika/Foundation/Characters/FloatConversion.h"
#include "Stroika/Foundation/Characters/String2Int.h"
#include "Stroika/Foundation/Common/SystemConfiguration.h"
#include "Stroika/Foundation/Containers/Mapping.h"
#include "Stroika/Foundation/Containers/Sequence.h"
#include "Stroika/Foundation/Containers/Set.h"
#include "Stroika/Foundation/DataExchange/Variant/CharacterDelimitedLines/Reader.h"
#include "Stroika/Foundation/DataExchange/Variant/JSON/Writer.h"
#include "Stroika/Foundation/Debug/AssertExternallySynchronizedMutex.h"
#include "Stroika/Foundation/Debug/Assertions.h"
#include "Stroika/Foundation/Debug/Trace.h"
#include "Stroika/Foundation/Execution/Exceptions.h"
#include "Stroika/Foundation/Execution/ProcessRunner.h"
#include "Stroika/Foundation/Execution/Synchronized.h"
#include "Stroika/Foundation/IO/FileSystem/FileInputStream.h"
#include "Stroika/Foundation/Streams/InputStream.h"
#include "Stroika/Foundation/Streams/MemoryStream.h"
#include "Stroika/Foundation/Streams/TextReader.h"

#include "Stroika/Frameworks/SystemPerformance/Support/InstrumentHelpers.h"

#include "Memory.h"

using namespace Stroika::Foundation;
using namespace Stroika::Foundation::Characters::Literals;
using namespace Stroika::Foundation::Containers;
using namespace Stroika::Foundation::DataExchange;
using namespace Stroika::Foundation::Execution;
using namespace Stroika::Foundation::Memory;
using namespace Stroika::Foundation::IO::FileSystem;

using namespace Stroika::Frameworks;
using namespace Stroika::Frameworks::SystemPerformance;

using Characters::Character;
using Containers::Mapping;
using Containers::Sequence;
using Containers::Set;

using Instruments::Memory::Info;
using Instruments::Memory::Options;

// Comment this in to turn on aggressive noisy DbgTrace in this module
//#define   USE_NOISY_TRACE_IN_THIS_MODULE_       1

#ifndef qUseWMICollectionSupport_
#define qUseWMICollectionSupport_ qStroika_Foundation_Common_Platform_Windows
#endif

#if qUseWMICollectionSupport_
#include "Stroika/Frameworks/SystemPerformance/Support/WMICollector.h"

using SystemPerformance::Support::WMICollector;
#endif

#if qUseWMICollectionSupport_
namespace {
    const String kInstanceName_{"_Total"sv};

    const String kCommittedBytes_{"Committed Bytes"sv};
    const String kCommitLimit_{"Commit Limit"sv};
    const String kHardPageFaultsPerSec_{"Pages/sec"sv};
    const String kPagesOutPerSec_{"Pages Output/sec"sv};

    const String kFreeMem_{"Free & Zero Page List Bytes"sv};

    // Something of an empirical WAG (kHardwareReserved*) but not super important to get right -- LGP 2015-09-24
    const String kHardwareReserved1_{"System Driver Resident Bytes"sv};
    const String kHardwareReserved2_{"System Driver Total Bytes"sv};
}
#endif

/*
 ********************************************************************************
 ************ Instruments::Memory::Info::PhysicalRAMDetailsType *****************
 ********************************************************************************
 */
String Instruments::Memory::Info::PhysicalRAMDetailsType::ToString () const
{
    return DataExchange::Variant::JSON::Writer{}.WriteAsString (Instrument::kObjectVariantMapper.FromObject (*this));
}

/*
 ********************************************************************************
 ************ Instruments::Memory::Info::VirtualMemoryDetailsType ***************
 ********************************************************************************
 */
String Instruments::Memory::Info::VirtualMemoryDetailsType::ToString () const
{
    return DataExchange::Variant::JSON::Writer{}.WriteAsString (Instrument::kObjectVariantMapper.FromObject (*this));
}

/*
 ********************************************************************************
 **************** Instruments::Memory::Info::PagingDetailsType ******************
 ********************************************************************************
 */
String Instruments::Memory::Info::PagingDetailsType::ToString () const
{
    return DataExchange::Variant::JSON::Writer{}.WriteAsString (Instrument::kObjectVariantMapper.FromObject (*this));
}

/*
 ********************************************************************************
 ********************** Instruments::Memory::Info *******************************
 ********************************************************************************
 */
String Instruments::Memory::Info::ToString () const
{
    return DataExchange::Variant::JSON::Writer{}.WriteAsString (Instrument::kObjectVariantMapper.FromObject (*this));
}

namespace {
    template <typename CONTEXT>
    using InstrumentRepBase_ = SystemPerformance::Support::InstrumentRep_COMMON<Options, CONTEXT>;
}

#if qStroika_Foundation_Common_Platform_Linux
namespace {
    struct _Context : SystemPerformance::Support::Context {
        uint64_t               fSaved_MajorPageFaultsSinceBoot{};
        uint64_t               fSaved_MinorPageFaultsSinceBoot{};
        uint64_t               fSaved_PageOutsSinceBoot{};
        Time::TimePointSeconds fSaved_VMPageStats_At{};
    };

    struct InstrumentRep_Linux_ : InstrumentRepBase_<_Context> {

        using InstrumentRepBase_<_Context>::InstrumentRepBase_;

        Instruments::Memory::Info _InternalCapture ()
        {
            Instruments::Memory::Info result;
            Read_ProcMemInfo_ (&result);
            try {
                Read_ProcVMStat_ (&result);
            }
            catch (...) {
                DbgTrace ("Ignoring error in Read_ProcVMStat_: {}"_f, current_exception ());
            }
            _NoteCompletedCapture ();
            return result;
        }

    private:
        static void Read_ProcMemInfo_ (Instruments::Memory::Info* updateResult)
        {
#if USE_NOISY_TRACE_IN_THIS_MODULE_
            Debug::TraceContextBumper ctx{"Read_ProcMemInfo_"};
#endif
            auto readMemInfoLine = [] (optional<uint64_t>* result, const String& n, const Sequence<String>& line) {
                if (line.size () >= 3 and line[0] == n) {
                    String unit   = line[2];
                    double factor = (unit == L"kB") ? 1024 : 1;
                    *result       = Math::Round<uint64_t> (Characters::FloatConversion::ToFloat<double> (line[1]) * factor);
#if USE_NOISY_TRACE_IN_THIS_MODULE_
                    DbgTrace (L"Set %s = %ld", n.c_str (), static_cast<long> (**result));
#endif
                }
            };
            /*
             *  @todo   minor performance note: we current do about 10 (tha many strings * 45 (about that many lines in file) compares.
             *          We couuld read data and form a map so lookups faster. Or at least keep a list of items alreayd found and not
             *          look for them more, and stop when none left to look for (wont work if some like sreclaimable not found).
             */
            static const filesystem::path                          kProcMemInfoFileName_{"/proc/meminfo"sv};
            DataExchange::Variant::CharacterDelimitedLines::Reader reader{{':', ' ', '\t'}};
            // Note - /procfs files always unseekable
            optional<uint64_t> memTotal;
            optional<uint64_t> slabReclaimable;
            optional<uint64_t> slab; // older kernels don't have slabReclaimable
            for (const Sequence<String>& line :
                 reader.ReadMatrix (IO::FileSystem::FileInputStream::New (kProcMemInfoFileName_, IO::FileSystem::FileInputStream::eNotSeekable))) {
#if USE_NOISY_TRACE_IN_THIS_MODULE_
                DbgTrace (L"***in Instruments::Memory::Info capture_ linesize=%d, line[0]=%s", line.size (), line.empty () ? L"" : line[0].c_str ());
#endif
                static const String kMemTotalLabel_{"MemTotal"sv};
                static const String kMemFreelLabel_{"MemFree"sv};
                static const String kMemAvailableLabel_{"MemAvailable"sv};
                static const String kActiveLabel_{"Active"sv};
                static const String kInactiveLabel_{"Inactive"sv};
                static const String kCommitLimitLabel_{"CommitLimit"sv};
                static const String kCommitted_ASLabel_{"Committed_AS"sv};
                static const String kSwapTotalLabel_{"SwapTotal"sv};
                static const String kSReclaimableLabel_{"SReclaimable"sv};
                static const String kSlabLabel_{"Slab"sv};
                readMemInfoLine (&memTotal, kMemTotalLabel_, line);
                readMemInfoLine (&updateResult->fPhysicalMemory.fFree, kMemFreelLabel_, line);
                readMemInfoLine (&updateResult->fPhysicalMemory.fAvailable, kMemAvailableLabel_, line);
                readMemInfoLine (&updateResult->fPhysicalMemory.fActive, kActiveLabel_, line);
                readMemInfoLine (&updateResult->fPhysicalMemory.fInactive, kInactiveLabel_, line);
                readMemInfoLine (&updateResult->fVirtualMemory.fCommitLimit, kCommitLimitLabel_, line);
                /*
                 *  From docs on https://github.com/torvalds/linux/blob/master/Documentation/filesystems/proc.txt about
                 *  Commited_AS - its unclear if this is the best measure of commited bytes.
                 */
                readMemInfoLine (&updateResult->fVirtualMemory.fCommittedBytes, kCommitted_ASLabel_, line);
                readMemInfoLine (&updateResult->fVirtualMemory.fPagefileTotalSize, kSwapTotalLabel_, line);
                readMemInfoLine (&slabReclaimable, kSReclaimableLabel_, line);
                readMemInfoLine (&slab, kSlabLabel_, line);
            }
            if (memTotal and updateResult->fPhysicalMemory.fFree and updateResult->fPhysicalMemory.fInactive and
                updateResult->fPhysicalMemory.fActive) {
                updateResult->fPhysicalMemory.fOSReserved = *memTotal - *updateResult->fPhysicalMemory.fFree -
                                                            *updateResult->fPhysicalMemory.fInactive - *updateResult->fPhysicalMemory.fActive;
            }
            if (not updateResult->fPhysicalMemory.fAvailable.has_value () and updateResult->fPhysicalMemory.fFree and
                updateResult->fPhysicalMemory.fInactive) {
                if (not slabReclaimable.has_value ()) {
                    // wag
                    slabReclaimable = NullCoalesce (slab) / 2;
                }
                updateResult->fPhysicalMemory.fAvailable =
                    *updateResult->fPhysicalMemory.fFree + *updateResult->fPhysicalMemory.fInactive + NullCoalesce (slabReclaimable);
            }
        }
        nonvirtual void Read_ProcVMStat_ (Instruments::Memory::Info* updateResult)
        {
#if USE_NOISY_TRACE_IN_THIS_MODULE_
            Debug::TraceContextBumper ctx{"Read_ProcVMStat_"};
#endif
            auto readVMStatLine = [] (optional<uint64_t>* result, const String& n, const Sequence<String>& line) -> unsigned int {
                if (line.size () >= 2 and line[0] == n) {
                    *result = Characters::String2Int<uint64_t> (line[1]);
#if USE_NOISY_TRACE_IN_THIS_MODULE_
                    DbgTrace (L"Set %s = %ld", n.c_str (), static_cast<long> (**result));
#endif
                    return 1;
                }
                return 0;
            };
            {
                static const filesystem::path kProcVMStatFileName_{"/proc/vmstat"sv};
                optional<uint64_t>            pgfault;
                optional<uint64_t>            pgpgout;
                {
                    unsigned int nFound{};
                    // Note - /procfs files always unseekable
                    DataExchange::Variant::CharacterDelimitedLines::Reader reader{{' ', '\t'}};
                    for (const Sequence<String>& line : reader.ReadMatrix (
                             IO::FileSystem::FileInputStream::New (kProcVMStatFileName_, IO::FileSystem::FileInputStream::eNotSeekable))) {
#if USE_NOISY_TRACE_IN_THIS_MODULE_
                        DbgTrace ("in Instruments::Memory::Info capture_ linesize={}, line[0]={}", line.size (), line.empty () ? ""_k : line[0]);
#endif
                        static const String kpgfaultLabel_{"pgfault"sv};
                        static const String kpgpgoutLabel_{"pgpgout"sv};
                        static const String kpgmajfaultLabel_{"pgmajfault"sv};
                        nFound += readVMStatLine (&pgfault, kpgfaultLabel_, line);
                        // Unsure if this should be pgpgout or pgpgout, or none of the above. On a system with no swap, I seem to get both happening,
                        // which makes no sense
                        nFound += readVMStatLine (&pgpgout, kpgpgoutLabel_, line); // tried pgpgout but I don't know what it is but doesn't appear to be pages out - noneof this well documented
                        nFound += readVMStatLine (&updateResult->fPaging.fMajorPageFaultsSinceBoot, kpgmajfaultLabel_, line);
                        if (nFound >= 3) {
                            break; // avoid reading rest if all found
                        }
                    }
                }
                Time::TimePointSeconds now               = Time::GetTickCount ();
                updateResult->fPaging.fPageOutsSinceBoot = pgpgout;
                if (pgfault and updateResult->fPaging.fMajorPageFaultsSinceBoot) {
                    updateResult->fPaging.fMinorPageFaultsSinceBoot = *pgfault - *updateResult->fPaging.fMajorPageFaultsSinceBoot;
                }
                auto doAve_ = [this] (Time::TimePointSeconds savedVMPageStatsAt, Time::TimePointSeconds now, uint64_t* savedBaseline,
                                      optional<uint64_t> faultsSinceBoot, optional<double>* faultsPerSecond) {
                    if (faultsSinceBoot) {
                        if (now - savedVMPageStatsAt >= _fOptions.fMinimumAveragingInterval) {
                            *faultsPerSecond = (*faultsSinceBoot - *savedBaseline) / (now - savedVMPageStatsAt).count ();
                        }
                        *savedBaseline = *faultsSinceBoot;
                    }
                };
                auto ctxLock = scoped_lock{_fContext};
                auto ctx     = _fContext.rwget ().rwref ();
                doAve_ (ctx->fSaved_VMPageStats_At, now, &ctx->fSaved_MinorPageFaultsSinceBoot,
                        updateResult->fPaging.fMinorPageFaultsSinceBoot, &updateResult->fPaging.fMinorPageFaultsPerSecond);
                doAve_ (ctx->fSaved_VMPageStats_At, now, &ctx->fSaved_MajorPageFaultsSinceBoot,
                        updateResult->fPaging.fMajorPageFaultsSinceBoot, &updateResult->fPaging.fMajorPageFaultsPerSecond);
                doAve_ (ctx->fSaved_VMPageStats_At, now, &ctx->fSaved_PageOutsSinceBoot, updateResult->fPaging.fPageOutsSinceBoot,
                        &updateResult->fPaging.fPageOutsPerSecond);
                ctx->fSaved_VMPageStats_At = now;
            }
        }
    };
}
#endif

#if qStroika_Foundation_Common_Platform_Windows
namespace {
    struct _Context : SystemPerformance::Support::Context {
#if qUseWMICollectionSupport_
        WMICollector fMemoryWMICollector_{"Memory"sv,
                                          {kInstanceName_},
                                          {kCommittedBytes_, kCommitLimit_, kHardPageFaultsPerSec_, kPagesOutPerSec_, kFreeMem_,
                                           kHardwareReserved1_, kHardwareReserved2_}};
#endif
    };
    struct InstrumentRep_Windows_ : InstrumentRepBase_<_Context> {

        InstrumentRep_Windows_ (const Options& options, const shared_ptr<_Context>& context)
            : InstrumentRepBase_<_Context>{options, context}
        {
#if USE_NOISY_TRACE_IN_THIS_MODULE_
            for (const String& i : _fContext.cget ().cref ()->fMemoryWMICollector_.GetAvailableCounters ()) {
                DbgTrace (L"Memory:Countername: %s", i.c_str ());
            }
#endif
        }

        nonvirtual Instruments::Memory::Info _InternalCapture ()
        {
            Instruments::Memory::Info result;
            uint64_t                  totalRAM{};
            Read_GlobalMemoryStatusEx_ (&result, &totalRAM);
#if qUseWMICollectionSupport_
            Read_WMI_ (&result, totalRAM);
#endif
            // I've found no docs to clearly state one way or another, but empirically from looking at the graph in
            // Resource Monitor, the amount reported as 'hardware' - which I'm thinking is roughly 'osreserved' is
            // subtracted from 'standby'.
            if (result.fPhysicalMemory.fOSReserved) {
                Memory::AccumulateIf (&result.fPhysicalMemory.fInactive, result.fPhysicalMemory.fOSReserved, std::minus{});
            }
            _NoteCompletedCapture ();
            return result;
        }

    private:
        static void Read_GlobalMemoryStatusEx_ (Instruments::Memory::Info* updateResult, uint64_t* totalRAM)
        {
            RequireNotNull (totalRAM);
            MEMORYSTATUSEX statex{};
            statex.dwLength = sizeof (statex);
            Verify (::GlobalMemoryStatusEx (&statex) != 0);
            updateResult->fPhysicalMemory.fFree = statex.ullAvailPhys; // overridden later, but a good first estimate if we don't use WMI
            *totalRAM                           = statex.ullTotalPhys;
            updateResult->fVirtualMemory.fPagefileTotalSize = statex.ullTotalPageFile;

            /*
             *  dwMemoryLoad
             *  A number between 0 and 100 that specifies the approximate percentage of physical
             *  memory that is in use (0 indicates no memory use and 100 indicates full memory use)
             */
            updateResult->fPhysicalMemory.fActive = statex.ullTotalPhys * statex.dwMemoryLoad / 100;
        }
#if qUseWMICollectionSupport_
        nonvirtual void Read_WMI_ (Instruments::Memory::Info* updateResult, uint64_t totalRAM)
        {
            auto                 lock      = scoped_lock{_fContext};
            shared_ptr<_Context> rwContext = _fContext.rwget ().rwref ();
            rwContext->fMemoryWMICollector_.Collect ();
            Memory::CopyToIf (&updateResult->fVirtualMemory.fCommittedBytes,
                              rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kCommittedBytes_));
            Memory::CopyToIf (&updateResult->fVirtualMemory.fCommitLimit,
                              rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kCommitLimit_));
            Memory::CopyToIf (&updateResult->fPaging.fMajorPageFaultsPerSecond,
                              rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kHardPageFaultsPerSec_));
            Memory::CopyToIf (&updateResult->fPaging.fPageOutsPerSecond,
                              rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kPagesOutPerSec_));
            Memory::CopyToIf (&updateResult->fPhysicalMemory.fFree, rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kFreeMem_));
            if (optional<double> freeMem = rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kFreeMem_)) {
                if (updateResult->fPhysicalMemory.fActive) {
                    // Active + Inactive + Free == TotalRAM
                    updateResult->fPhysicalMemory.fInactive = totalRAM - *updateResult->fPhysicalMemory.fActive - static_cast<uint64_t> (*freeMem);
                }
            }
            updateResult->fPhysicalMemory.fOSReserved = nullopt;
            Memory::AccumulateIf (&updateResult->fPhysicalMemory.fOSReserved,
                                  rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kHardwareReserved1_));
            Memory::AccumulateIf (&updateResult->fPhysicalMemory.fOSReserved,
                                  rwContext->fMemoryWMICollector_.PeekCurrentValue (kInstanceName_, kHardwareReserved2_));
            // fPhysicalMemory.fAvailable WAG TMPHACK - probably should add "hardware in use" memory + private WS of each process + shared memory "WS" - but not easy to compute...
            updateResult->fPhysicalMemory.fAvailable = updateResult->fPhysicalMemory.fFree + updateResult->fPhysicalMemory.fInactive;
        }
#endif
    };
}
#endif

namespace {
    const MeasurementType kMemoryUsageMeasurement_ = MeasurementType{"Memory-Usage"sv};
}

namespace {
    struct MemoryInstrumentRep_
#if qStroika_Foundation_Common_Platform_Linux
        : InstrumentRep_Linux_
#elif qStroika_Foundation_Common_Platform_Windows
        : InstrumentRep_Windows_
#else
        : InstrumentRepBase_<SystemPerformance::Support::Context>
#endif
    {
#if qStroika_Foundation_Common_Platform_Linux
        using inherited = InstrumentRep_Linux_;
#elif qStroika_Foundation_Common_Platform_Windows
        using inherited = InstrumentRep_Windows_;
#else
        using inherited = InstrumentRepBase_<SystemPerformance::Support::Context>;
#endif
        MemoryInstrumentRep_ (const Options& options, const shared_ptr<_Context>& context = make_shared<_Context> ())
            : inherited{options, context}
        {
            Require (_fOptions.fMinimumAveragingInterval > 0s);
        }
        virtual MeasurementSet Capture () override
        {
            Debug::TraceContextBumper ctx{"SystemPerformance::Instrument...Memory...MemoryInstrumentRep_::Capture ()"};
            MeasurementSet            results;
            results.fMeasurements.Add (Measurement{
                kMemoryUsageMeasurement_, Instruments::Memory::Instrument::kObjectVariantMapper.FromObject (Capture_Raw (&results.fMeasuredAt))});
            return results;
        }
        nonvirtual Info Capture_Raw (Range<TimePointSeconds>* outMeasuredAt)
        {
            return Do_Capture_Raw<Info> ([this] () { return _InternalCapture (); }, outMeasuredAt);
        }
        virtual unique_ptr<IRep> Clone () const override
        {
            return make_unique<MemoryInstrumentRep_> (_fOptions, _fContext.load ());
        }
        nonvirtual Info _InternalCapture ()
        {
            AssertExternallySynchronizedMutex::WriteContext declareContext{*this};
#if USE_NOISY_TRACE_IN_THIS_MODULE_
            Debug::TraceContextBumper ctx{"Instruments::Memory::Info _InternalCapture"};
#endif
#if qStroika_Foundation_Common_Platform_Linux or qStroika_Foundation_Common_Platform_Windows
            Info result = inherited::_InternalCapture ();
#else
            Info result;
#endif
            Ensure (NullCoalesce (result.fPhysicalMemory.fActive) + NullCoalesce (result.fPhysicalMemory.fInactive) +
                        NullCoalesce (result.fPhysicalMemory.fFree) + NullCoalesce (result.fPhysicalMemory.fOSReserved) <=
                    Common::GetSystemConfiguration_Memory ().fTotalPhysicalRAM);
            if (result.fPhysicalMemory.fActive and result.fPhysicalMemory.fInactive and result.fPhysicalMemory.fFree and
                result.fPhysicalMemory.fOSReserved) {
                Ensure (NullCoalesce (result.fPhysicalMemory.fActive) + NullCoalesce (result.fPhysicalMemory.fInactive) +
                            NullCoalesce (result.fPhysicalMemory.fFree) + NullCoalesce (result.fPhysicalMemory.fOSReserved) ==
                        Common::GetSystemConfiguration_Memory ().fTotalPhysicalRAM);
            }
            return result;
        }
    };
}

/*
 ********************************************************************************
 ********************** Instruments::Memory::Instrument *************************
 ********************************************************************************
 */
const ObjectVariantMapper Instruments::Memory::Instrument::kObjectVariantMapper = [] () -> ObjectVariantMapper {
    ObjectVariantMapper mapper;
    mapper.AddClass<Info::PhysicalRAMDetailsType> ({
        {"Available"sv, &Info::PhysicalRAMDetailsType::fAvailable},
        {"Active"sv, &Info::PhysicalRAMDetailsType::fActive},
        {"Inactive"sv, &Info::PhysicalRAMDetailsType::fInactive},
        {"Free"sv, &Info::PhysicalRAMDetailsType::fFree},
        {"OS-Reserved"sv, &Info::PhysicalRAMDetailsType::fOSReserved},
    });
    mapper.AddClass<Info::VirtualMemoryDetailsType> ({
        {"Commit-Limit"sv, &Info::VirtualMemoryDetailsType::fCommitLimit},
        {"Committed-Bytes"sv, &Info::VirtualMemoryDetailsType::fCommittedBytes},
        {"Pagefile-Total-Size"sv, &Info::VirtualMemoryDetailsType::fPagefileTotalSize},
    });
    mapper.AddClass<Info::PagingDetailsType> ({
        {"Major-Faults-Since-Boot"sv, &Info::PagingDetailsType::fMajorPageFaultsSinceBoot},
        {"Minor-Faults-Since-Boot"sv, &Info::PagingDetailsType::fMinorPageFaultsSinceBoot},
        {"Page-Outs-Since-Boot"sv, &Info::PagingDetailsType::fPageOutsSinceBoot},
        {"Major-Faults-Per-Second"sv, &Info::PagingDetailsType::fMajorPageFaultsPerSecond},
        {"Minor-Faults-Per-Second"sv, &Info::PagingDetailsType::fMinorPageFaultsPerSecond},
        {"Page-Outs-Per-Second"sv, &Info::PagingDetailsType::fPageOutsPerSecond},
    });
    mapper.AddClass<Info> ({
        {"Physical-Memory"sv, &Info::fPhysicalMemory},
        {"Virtual-Memory"sv, &Info::fVirtualMemory},
        {"Paging"sv, &Info::fPaging},
    });
    return mapper;
}();

Instruments::Memory::Instrument::Instrument (const Options& options)
    : SystemPerformance::Instrument{InstrumentNameType{"Memory"sv},
                                    make_unique<MemoryInstrumentRep_> (options),
                                    {kMemoryUsageMeasurement_},
                                    {KeyValuePair<type_index, MeasurementType>{typeid (Info), kMemoryUsageMeasurement_}},
                                    kObjectVariantMapper}
{
}

/*
 ********************************************************************************
 ********* SystemPerformance::Instrument::CaptureOneMeasurement *****************
 ********************************************************************************
 */
template <>
Instruments::Memory::Info SystemPerformance::Instrument::CaptureOneMeasurement (Range<TimePointSeconds>* measurementTimeOut)
{
    Debug::TraceContextBumper ctx{"SystemPerformance::Instrument::CaptureOneMeasurement<Memory::Info>"};
    MemoryInstrumentRep_*     myCap = dynamic_cast<MemoryInstrumentRep_*> (fCaptureRep_.get ());
    AssertNotNull (myCap);
    return myCap->Capture_Raw (measurementTimeOut);
}
