/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */

namespace Stroika::Foundation::Memory {

    /*
     ********************************************************************************
     ************************* BlockAllocationUseHelper<T> **************************
     ********************************************************************************
     */
    template <typename T>
    inline void* BlockAllocationUseHelper<T>::operator new ([[maybe_unused]] size_t n)
    {
        Require (n == sizeof (T));
        return BlockAllocator<T>{}.allocate (1);
    }
    template <typename T>
    inline void* BlockAllocationUseHelper<T>::operator new ([[maybe_unused]] size_t n, int, const char*, int)
    {
        Require (n == sizeof (T));
        return BlockAllocator<T>{}.allocate (1);
    }
    template <typename T>
    inline void BlockAllocationUseHelper<T>::operator delete (void* p)
    {
        BlockAllocator<T>{}.deallocate (reinterpret_cast<T*> (p), 1);
    }
    template <typename T>
    inline void BlockAllocationUseHelper<T>::operator delete (void* p, int, const char*, int)
    {
        BlockAllocator<T>{}.deallocate (reinterpret_cast<T*> (p), 1);
    }

    /*
     ********************************************************************************
     *************************** UsesBlockAllocation<T> *****************************
     ********************************************************************************
     */
    template <typename T>
    constexpr bool UsesBlockAllocation ()
    {
        return derived_from<T, BlockAllocationUseHelper<T>>;
    }

    /*
     ********************************************************************************
     ********************************* MakeSharedPtr<T> *****************************
     ********************************************************************************
     */
    template <typename T, typename... ARGS_TYPE>
    inline auto MakeSharedPtr (ARGS_TYPE&&... args) -> shared_ptr<T>
    {
        if constexpr (UsesBlockAllocation<T> ()) {
            return allocate_shared<T> (BlockAllocator<T>{}, forward<ARGS_TYPE> (args)...);
        }
        else {
            return make_shared<T> (forward<ARGS_TYPE> (args)...);
        }
    }

    /*
     ********************************************************************************
     ******************* BlockAllocationUseGlobalAllocatorHelper<T> *****************
     ********************************************************************************
     */
    template <typename T>
    inline void* BlockAllocationUseGlobalAllocatorHelper<T>::operator new (size_t n)
    {
        return ::operator new (n);
    }
    template <typename T>
    inline void* BlockAllocationUseGlobalAllocatorHelper<T>::operator new (size_t n, int, const char*, int)
    {
        return ::operator new (n);
    }
    template <typename T>
    inline void BlockAllocationUseGlobalAllocatorHelper<T>::operator delete (void* p)
    {
        ::operator delete (p);
    }
    template <typename T>
    inline void BlockAllocationUseGlobalAllocatorHelper<T>::operator delete (void* p, int, const char*, int)
    {
        ::operator delete (p);
    }

    /*
     ********************************************************************************
     *************************** ManuallyBlockAllocated<T> **************************
     ********************************************************************************
     */
    template <typename T>
    template <typename... ARGS>
    inline T* ManuallyBlockAllocated<T>::New (ARGS&&... args)
    {
#if qStroika_Foundation_Memory_PreferBlockAllocation
        return new (BlockAllocator<T>{}.allocate (1)) T{forward<ARGS> (args)...};
#else
        return new T{forward<ARGS> (args)...};
#endif
    }
    template <typename T>
    inline void ManuallyBlockAllocated<T>::Delete (T* p) noexcept
    {
#if qStroika_Foundation_Memory_PreferBlockAllocation
        if (p != nullptr) {
            destroy_at (p);
            BlockAllocator<T>{}.deallocate (p, 1);
        }
#else
        delete p;
#endif
    }

}
