/*
* Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
*/
#ifndef _Stroika_Foundation_Math_LinearAlgebra_Vector_h_
#define _Stroika_Foundation_Math_LinearAlgebra_Vector_h_ 1

#include "Stroika/Foundation/StroikaPreComp.h"

#include "Stroika/Foundation/Characters/String.h"
#include "Stroika/Foundation/Containers/Sequence.h"
#include "Stroika/Foundation/Memory/SharedByValue.h"
#include "Stroika/Foundation/Traversal/Iterable.h"

/**
 *  \file
 *
 *  \note Code-Status:  <a href="Code-Status.md#Alpha">Alpha</a>
 */

#define Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION 1
#ifndef Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION
#define Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION 0
#endif

namespace Stroika::Foundation::Math::LinearAlgebra {

    /**
     */
    template <typename T>
    class Vector {
    public:
        /**
         */
        Vector (size_t dimension);
        Vector (size_t dimension, Common::ArgByValueType<T> fillValue);
        Vector (size_t dimension, const function<T ()>& filler);
        template <Traversal::IIterableOfTo<T> CONTAINER_OF_T>
        Vector (const CONTAINER_OF_T& c);

#if Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION
    public:
        /**
         *  \note - Armadillo calls the Fill overload with a function argument 'imbue'
         */
        nonvirtual void Fill (T value);
        nonvirtual void Fill (function<T ()> filler);
#endif

    public:
        /**
         */
        nonvirtual size_t GetDimension () const;

    public:
        /**
         * construct a new vector by applying the argument function to each element and collecting the results.
         */
        nonvirtual Vector<T> Transform (const function<T (T)>& f) const;

    public:
        /**
         *  Euclidian norm = sqrt (sum (xi^2))
         */
        nonvirtual T Norm () const;

    public:
        /**
         */
        nonvirtual Containers::Sequence<T> GetItems () const;

#if Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION
    public:
        /**
         */
        template <typename CONTAINER>
        nonvirtual void SetItems (const CONTAINER& s);
#endif

    public:
        /**
         */
        nonvirtual T GetAt (size_t i) const;

#if Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION
    public:
        /**
         */
        nonvirtual void SetAt (size_t i, Common::ArgByValueType<T> v);
#endif

    public:
        /**
         */
        nonvirtual T operator[] (size_t i) const;

#if Stroika_Foundation_Math_LinearAlgebra_Vector_ALLOW_MUTATION
    private:
        struct TMP_ {
            Vector<T>& fV;
            size_t     fIndex;
            T          fValue;
            ~TMP_ ()
            {
                fV.SetAt (fIndex, fValue);
            }
            operator T& ()
            {
                return fValue;
            }
        };

    public:
        /**
         */
        nonvirtual TMP_ operator[] (size_t i)
        {
            return TMP_{*this, i, GetAt (i)};
        }
#endif

    public:
        nonvirtual Characters::String ToString () const;

    private:
        class IRep_;

    private:
        Memory::SharedByValue<IRep_> fRep_;
    };

    template <typename T>
    Vector<T> operator* (T lhs, const Vector<T>& rhs)
    {
        vector<T> tmp;
        for (const auto& i : rhs.GetItems ()) {
            tmp.push_back (lhs * i);
        }
        return tmp;
    }
    template <typename T>
    Vector<T> operator* (const Vector<T>& lhs, T rhs)
    {
        vector<T> tmp;
        for (const auto& i : lhs.GetItems ()) {
            tmp.push_back (i * rhs);
        }
        return tmp;
    }

    template <typename T>
    Vector<T> operator+ (const Vector<T>& lhs, const Vector<T>& rhs)
    {
        Require (lhs.GetDimension () == rhs.GetDimension ());
        vector<T> tmp;
        for (size_t i = 0; i < lhs.GetDimension (); ++i) {
            tmp.push_back (lhs[i] + rhs[i]);
        }
        return tmp;
    }
    template <typename T>
    Vector<T> operator+ (T lhs, const Vector<T>& rhs)
    {
        vector<T> tmp;
        for (const auto& i : rhs.GetItems ()) {
            tmp.push_back (lhs + i);
        }
        return tmp;
    }
    template <typename T>
    Vector<T> operator+ (const Vector<T>& lhs, T rhs)
    {
        vector<T> tmp;
        for (const auto& i : lhs.GetItems ()) {
            tmp.push_back (i + rhs);
        }
        return tmp;
    }

    template <typename T>
    Vector<T> operator- (const Vector<T>& lhs, const Vector<T>& rhs)
    {
        Require (lhs.GetDimension () == rhs.GetDimension ());
        vector<T> tmp;
        for (size_t i = 0; i < lhs.GetDimension (); ++i) {
            tmp.push_back (lhs[i] - rhs[i]);
        }
        return tmp;
    }
    template <typename T>
    Vector<T> operator- (T lhs, const Vector<T>& rhs)
    {
        vector<T> tmp;
        for (const T& i : rhs.GetItems ()) {
            tmp.push_back (lhs - i);
        }
        return tmp;
    }
    template <typename T>
    Vector<T> operator- (const Vector<T>& lhs, T rhs)
    {
        vector<T> tmp;
        for (const auto& i : lhs.GetItems ()) {
            tmp.push_back (i - rhs);
        }
        return tmp;
    }

}

/*
 ********************************************************************************
 ***************************** Implementation Details ***************************
 ********************************************************************************
 */
#include "Vector.inl"

#endif /*_Stroika_Foundation_Math_LinearAlgebra_Vector_h_*/
