/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#include "Stroika/Foundation/StroikaPreComp.h"

#include "Stroika/Foundation/Characters/String.h"
#include "Stroika/Foundation/Characters/StringBuilder.h"
#include "Stroika/Foundation/Memory/StackBuffer.h"

#include "PathName.h"

using namespace Stroika::Foundation;
using namespace Stroika::Foundation::Characters;
using namespace Stroika::Foundation::IO;
using namespace Stroika::Foundation::IO::FileSystem;

using Memory::StackBuffer;

/*
 ********************************************************************************
 ************** FileSystem::AssureDirectoryPathSlashTerminated ******************
 ********************************************************************************
 */
String FileSystem::AssureDirectoryPathSlashTerminated (const String& dirPath)
{
    if (dirPath.empty ()) {
        AssertNotReached (); // not sure if this is an error or not. Not sure how code used.
        // put assert in there to find out... Probably should THROW!
        //      -- LGP 2009-05-12
        return String{filesystem::path::preferred_separator};
    }
    else {
        Character lastChar = dirPath[dirPath.size () - 1];
        if (lastChar == filesystem::path::preferred_separator) {
            return dirPath;
        }
        StringBuilder result = dirPath;
        result += filesystem::path::preferred_separator;
        return result;
    }
}

/*
 ********************************************************************************
 *********************** FileSystem::SafeFilenameChars **************************
 ********************************************************************************
 */
String FileSystem::SafeFilenameChars (const String& s)
{
    // analyze as wide-char string so we don't mis-identify characters (by looking at lead bytes etc)
    wstring tmp = s.As<wstring> ();
Again:
    for (auto i = tmp.begin (); i != tmp.end (); ++i) {
        switch (*i) {
            case '\\':
                *i = '_';
                break;
            case '/':
                *i = '_';
                break;
            case ':':
                *i = ' ';
                break;
            case '.':
                tmp.erase (i);
                goto Again;
        }
    }
    return tmp;
}

/*
 ********************************************************************************
 ********************* FileSystem::AssureLongFileName ***************************
 ********************************************************************************
 */
String FileSystem::AssureLongFileName (const String& fileName)
{
#if qStroika_Foundation_Common_Platform_Windows
    DWORD r = ::GetLongPathNameW (fileName.As<wstring> ().c_str (), nullptr, 0);
    if (r != 0) {
        StackBuffer<wchar_t> buf{Memory::eUninitialized, r};
        r = ::GetLongPathNameW (fileName.As<wstring> ().c_str (), buf.data (), r);
        if (r != 0) {
            return static_cast<const wchar_t*> (buf);
        }
    }
#endif
    return fileName;
}
