/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#ifndef _Stroika_Foundation_IO_FileSystem_Disk_h_
#define _Stroika_Foundation_IO_FileSystem_Disk_h_ 1

#include "Stroika/Foundation/StroikaPreComp.h"

#include <filesystem>
#include <optional>

#include "Stroika/Foundation/Characters/String.h"
#include "Stroika/Foundation/Common/Common.h"
#include "Stroika/Foundation/Containers/KeyedCollection.h"

/**
 *  \file
 *
 *  \note Code-Status:  <a href="Code-Status.md#Alpha">Alpha</a>
 *
 * TODO:
 *      @todo   Underneath DiskInfoType - include partitions (which I think maybe like volumnes for windows - maybe not)
 *
 *      @todo   KeyedCollection<DiskInfoType> FileSystem::GetAvailableDisks ()
 *              for linux
 *              https://github.com/karelzak/util-linux/blob/master/misc-utils/lsblk.c
 *              iterate_block_devices
 */

namespace Stroika::Foundation::IO::FileSystem {

    using Characters::String;

    /**
     *  \note   Common::DefaultNames<> supported
     *  \note   These print names are mostly for display and debugging purposes, and they are not guaranteed to be safe for
     *          persistence (so be sure to version).
     */
    enum class BlockDeviceKind {
        /**
         *  On Windoze, corresponds to https://msdn.microsoft.com/en-us/library/aa394173%28v=vs.85%29.aspx?f=255&MSPPError=-2147217396 "Removable Disk" or
         *  https://msdn.microsoft.com/en-us/library/windows/desktop/aa364939%28v=vs.85%29.aspx DRIVE_REMOVABLE
         */
        eRemovableDisk,

        /**
         *  On Windoze, corresponds to https://msdn.microsoft.com/en-us/library/aa394173%28v=vs.85%29.aspx?f=255&MSPPError=-2147217396 "Local Disk" or
         *  https://msdn.microsoft.com/en-us/library/windows/desktop/aa364939%28v=vs.85%29.aspx DRIVE_FIXED
         */
        eLocalDisk,

        /**
         *  On Windoze, corresponds to https://msdn.microsoft.com/en-us/library/aa394173%28v=vs.85%29.aspx?f=255&MSPPError=-2147217396 "Network Drive" or
         *  https://msdn.microsoft.com/en-us/library/windows/desktop/aa364939%28v=vs.85%29.aspx DRIVE_REMOTE
         */
        eNetworkDrive,

        /**
         *  On Windoze, corresponds to https://msdn.microsoft.com/en-us/library/aa394173%28v=vs.85%29.aspx?f=255&MSPPError=-2147217396 "RAM Disk" or
         *  https://msdn.microsoft.com/en-us/library/windows/desktop/aa364939%28v=vs.85%29.aspx DRIVE_RAMDISK
         *  On Linux, this is tmpfs
         */
        eTemporaryFiles,

        /**
         *  On Windoze, corresponds to https://msdn.microsoft.com/en-us/library/aa394173%28v=vs.85%29.aspx?f=255&MSPPError=-2147217396 "Compact Disc" or
         *  https://msdn.microsoft.com/en-us/library/windows/desktop/aa364939%28v=vs.85%29.aspx DRIVE_CDROM
         */
        eReadOnlyEjectable,

        /**
         *  e.g. Linux procfs
         */
        eSystemInformation,

        Stroika_Define_Enum_Bounds (eRemovableDisk, eSystemInformation)
    };

    /**
     *  Information for a physical disk (not for a partition).
     */
    struct DiskInfoType {
        // DeviceName was string until Stroika v2.1b2, but then switched to path (hope thats right)??? --LGP 2020-06/30
        filesystem::path fDeviceName;

        /*
         *  Is the 'disk' a 'remote' device (network),  CD-ROM, direct-attached hard disk (e.g. internal) or removable drive,
         */
        optional<BlockDeviceKind> fDeviceKind;

        /*
         *  This is the size of the physical block device. All the filesystems must fit in it.
         */
        optional<uint64_t> fSizeInBytes;

        /**
         *  @see Characters::ToString ();
         */
        nonvirtual String ToString () const;
    };

    /**
     *  Fetch all the available disks (DiskInfoType) installed on the system, keyed by fDeviceName
     */
    Containers::KeyedCollection<DiskInfoType, filesystem::path> GetAvailableDisks ();

}

/*
 ********************************************************************************
 ***************************** Implementation Details ***************************
 ********************************************************************************
 */
#include "Disk.inl"

#endif /*_Stroika_Foundation_IO_FileSystem_Disk_h_*/
