/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#ifndef _Stroika_Foundation_Debug_Sanitizer_h_
#define _Stroika_Foundation_Debug_Sanitizer_h_ 1

#include "Stroika/Foundation/StroikaPreComp.h"

#if __has_include(<sanitizer/asan_interface.h>)
#include <sanitizer/asan_interface.h>
#endif
#if __has_include(<sanitizer/lsan_interface.h>)
#include <sanitizer/lsan_interface.h>
#endif
#if __has_include(<sanitizer/tsan_interface.h>)
#include <sanitizer/tsan_interface.h>
#endif

#if qCompilerAndStdLib_undefined_behavior_macro_Buggy
extern "C" void __attribute__ ((weak)) __ubsan_handle_builtin_unreachable ();
#endif

/**
 *  \file
 *
 *  \note Code-Status:  <a href="Code-Status.md#Beta">Beta</a>
 *
 */

#if defined(__cplusplus)
namespace Stroika::Foundation::Debug {

    /**
     *  Detect if Address Sanitizer is enabled in this compilation.
     *
     *      Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer
     *
     *  \note this is defined as a macro, since sometimes its very hard to use constexpr to disable bunches of code
     *        BUT - use Debug::kBuiltWithAddressSanitizer in preference to this where you can.
     */
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer)
#if defined(__SANITIZE_ADDRESS__)
#define Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer 1
#endif
#endif
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer)
#if defined(__has_feature)
// @ see https://clang.llvm.org/docs/AddressSanitizer.html#conditional-compilation-with-has-feature-address-sanitizer
#define Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer __has_feature (address_sanitizer)
#endif
#endif
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer)
#define Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer 0
#endif

    /**
     *  \brief kBuiltWithAddressSanitizer can be checked in compiled code to see if the address sanitizer
     *         support is compiled into this executable
     */
    constexpr bool kBuiltWithAddressSanitizer = Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer;

    /*
     *  Macro: Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS
     *
     *  \par Example Usage
     *      \code
     *          #if qCompilerAndStdLib_arm_asan_FaultStackUseAfterScope_Buggy
     *              Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS
     *          #endif
     *          void Debug::Private_::Emitter::DoEmit_ (const wchar_t* p, const wchar_t* e) noexcept
     *      \endcode
     * 
     *  \par Example Usage
     *      \code
     *          // OLD EXAMPLE - NOT SURE THIS SYNTAX WORKS - MAYBE HAS TO GO BEFORE [] not after function()??
     *          ToObjectMapperType<CLASS> toObjectMapper = [fields, preflightBeforeToObject] (const ObjectVariantMapper& mapper, const VariantValue& d, CLASS* intoObjOfTypeT) 
     *              Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS 
     *              -> void {...}
     *      \endcode
     */
#if Stroika_Foundation_Debug_Sanitizer_HAS_AddressSanitizer
#if defined(__clang__)
    // using [[gnu::no_sanitize_undefined]] syntax on clang++-10 on lambdas produces warning of no_sanitize_address undefined but this seems to work?
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS __attribute__ ((no_sanitize_address))
#elif defined(__GNUC__)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS [[gnu::no_sanitize_address]]
#elif defined(_MSC_VER)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS __declspec (no_sanitize_address)
#endif
#endif
#if !defined(Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_ADDRESS
#endif

    /**
     *  Detect if Thread Sanitizer is enabled in this compilation.
     *
     *      Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer
     *
     *  \note this is defined as a macro, since sometimes its very hard to use constexpr to disable bunchs of code
     *        BUT - use Debug::kBuiltWithThreadSanitizer in preference to this where you can.
     */
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer)
#if defined(__SANITIZE_THREAD__)
#define Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer 1
#endif
#endif
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer)
// see https://clang.llvm.org/docs/ThreadSanitizer.html#has-feature-thread-sanitizer
#if defined(__has_feature)
#define Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer __has_feature (thread_sanitizer)
#endif
#endif
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer)
#define Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer 0
#endif

    /**
     *  \brief kBuiltWithThreadSanitizer can be checked in compiled code to see if the thread sanitizer
     *         support is compiled into this executable
     */
    constexpr bool kBuiltWithThreadSanitizer = Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer;

    /**
     *  Macro: Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_THREAD
     *
     *  \par Example Usage
     *      \code
     *          #if qCompiler_ThreadSantizer_SPR_717_Buggy
     *              Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_THREAD
     *          #endif
     *          static void DoIt (void* ignored) {...}
     *      \endcode
     */
#if Stroika_Foundation_Debug_Sanitizer_HAS_ThreadSanitizer
#if defined(__clang__) || defined(__GNUC__)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_THREAD [[gnu::no_sanitize_thread]]
#endif
#endif
#if !defined(Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_THREAD)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_THREAD
#endif

    /**
     *  Detect if Undefined Behavior Sanitizer is enabled in this compilation.
     *
     *      Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer
     *
     *  \note this is defined as a macro, since sometimes its very hard to use constexpr to disable bunchs of code
     *        BUT - use Debug::kBuiltWithUndefinedBehaviorSanitizer in preference to this where you can.
     * 
     *  \note sure this works with GCC - https://github.com/google/sanitizers/issues/765
     */
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer)
#if defined(__SANITIZE_UNDEFINED__)
#define Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer 1
#endif
#endif
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer)
#if defined(__has_feature)
    // not documented - https://clang.llvm.org/docs/UndefinedBehaviorSanitizer.html - not sure if ever works
    // appears to work on apple XCode 11 clang, at least
#define Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer __has_feature (undefined_behavior_sanitizer)
#endif
#endif
#if !defined(Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer)
#define Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer 0
#endif

/**
     *  \brief kBuiltWithUndefinedBehaviorSanitizer can be checked in compiled code to see if the undfined behavior sanitizer
     *         support is compiled into this executable
     * 
     *  \note WARNING: This incorrectly reports false on GCC builds, at least up to gcc 12, it appears.
     */
#if qCompilerAndStdLib_undefined_behavior_macro_Buggy
    namespace Private_ {
        inline bool isUndefinedBehavorSanitizerRunning_ ()
        {
            // trick from https://stackoverflow.com/questions/39371798/how-can-i-determine-if-ubsan-has-been-compiled-in-using-clang-or-gcc
            return &__ubsan_handle_builtin_unreachable;
        }
    }
    inline const bool kBuiltWithUndefinedBehaviorSanitizer = Private_::isUndefinedBehavorSanitizerRunning_ ();
#else
    constexpr bool kBuiltWithUndefinedBehaviorSanitizer = Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer;
#endif

    /**
     *  Macro: Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_UNDEFINED
     *
     *  \par Example Usage
     *      \code
     *          #if qSomeBugFlag
     *              Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_UNDEFINED
     *          #endif
     *              Interface GetInterfaces_POSIX_mkInterface_ (int sd, const ifreq* i) {...}
     *      \endcode
     */
#if Stroika_Foundation_Debug_Sanitizer_HAS_UndefinedBehaviorSanitizer
#if defined(__clang__)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_UNDEFINED __attribute__ ((no_sanitize ("undefined")))
#elif defined(__GNUC__)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_UNDEFINED [[gnu::no_sanitize_undefined]]
// NO UBSAN YET ON VS
#endif
#endif
#if !defined(Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_UNDEFINED)
#define Stroika_Foundation_Debug_ATTRIBUTE_NO_SANITIZE_UNDEFINED
#endif

}
#endif

/*
 ********************************************************************************
 ***************************** Implementation Details ***************************
 ********************************************************************************
 */
#include "Sanitizer.inl"

#endif /*_Stroika_Foundation_Debug_Sanitizer_h_*/
