/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#include "Stroika/Foundation/StroikaPreComp.h"

#include "Stroika/Foundation/Database/SQL/Statement.h"

#include "Versioning.h"

using namespace Stroika::Foundation;
using namespace Stroika::Foundation::Characters;
using namespace Stroika::Foundation::Database;
using namespace Stroika::Foundation::Database::SQL;
using namespace Stroika::Foundation::Debug;

// Comment this in to turn on aggressive noisy DbgTrace in this module
// #define USE_NOISY_TRACE_IN_THIS_MODULE_ 1

/*
 ********************************************************************************
 *************************** ORM::ProvisionForVersion ***************************
 ********************************************************************************
 */
void ORM::ProvisionForVersion (SQL::Connection::Ptr conn, Common::Version targetDBVersion, const Traversal::Iterable<Schema::Table>& tables)
{
    Collection<TableProvisioner> provisioners;
    for (const auto& ti : tables) {
        provisioners += TableProvisioner{
            ti.fName, [=] (SQL::Connection::Ptr conn, optional<Common::Version> existingVersion, [[maybe_unused]] Common::Version targetDBVersion) -> void {
                // properly upgrade - for now just create if doesn't exist
                if (!existingVersion) {
                    conn.Exec (Schema::StandardSQLStatements{ti}.CreateTable ());
                }
            }};
    }
    ProvisionForVersion (conn, targetDBVersion, provisioners);
}
void ORM::ProvisionForVersion (SQL::Connection::Ptr conn, Common::Version targetDBVersion, const Traversal::Iterable<TableProvisioner>& tables)
{
    TraceContextBumper ctx{"ORM::ProvisionForVersion", "conn={}"_f, conn};
    SQL::Statement doesTableExist = conn.mkStatement (conn.GetEngineProperties ()->GetSQL (SQL::EngineProperties::NonStandardSQL::eDoesTableExist));
    for (const auto& ti : tables) {
        doesTableExist.Reset ();
        doesTableExist.Bind (SQL::EngineProperties::kDoesTableExistParameterName, ti.fTableName);
        if (not doesTableExist.GetNextRow ()) {
            ti.fProvisioner (conn, nullopt, targetDBVersion);
        }
        else {
            // @todo store / pass along version
            ti.fProvisioner (conn, Common::Version{1, 0, Common::VersionStage::Alpha, 0}, targetDBVersion);
        }
    }
}