/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#include "Stroika/Foundation/StroikaPreComp.h"

#include "Stroika/Foundation/Cryptography/Encoding/OpenSSLCryptoStream.h"

#include "AES.h"

using std::byte;

using namespace Stroika::Foundation;
using namespace Stroika::Foundation::Containers;
using namespace Stroika::Foundation::Cryptography;
using namespace Stroika::Foundation::Cryptography::Encoding;
using namespace Stroika::Foundation::Cryptography::Encoding::Algorithm;
using namespace Stroika::Foundation::Cryptography::OpenSSL;

using Memory::BLOB;

#if qStroika_HasComponent_OpenSSL
namespace {
    OpenSSLCryptoParams cvt_ (const OpenSSL::DerivedKey& key, AESOptions options)
    {
        switch (options) {
            case AESOptions::e128_CBC:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_128_CBC, key};
            case AESOptions::e128_ECB:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_128_ECB, key};
            case AESOptions::e128_OFB:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_128_OFB, key};
            case AESOptions::e128_CFB1:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_128_CFB1, key};
            case AESOptions::e128_CFB8:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_128_CFB8, key};
            case AESOptions::e128_CFB128:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_128_CFB128, key};
            case AESOptions::e192_CBC:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_192_CBC, key};
            case AESOptions::e192_ECB:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_192_ECB, key};
            case AESOptions::e192_OFB:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_192_OFB, key};
            case AESOptions::e192_CFB1:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_192_CFB1, key};
            case AESOptions::e192_CFB8:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_192_CFB8, key};
            case AESOptions::e192_CFB128:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_192_CFB128, key};
            case AESOptions::e256_CBC:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_CBC, key};
            case AESOptions::e256_ECB:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_ECB, key};
            case AESOptions::e256_OFB:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_OFB, key};
            case AESOptions::e256_CFB1:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_CFB1, key};
            case AESOptions::e256_CFB8:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_CFB8, key};
            case AESOptions::e256_CFB128:
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_CFB128, key};
            default:
                RequireNotReached ();
                return OpenSSLCryptoParams{CipherAlgorithms::kAES_256_CFB128, key};
        }
    }
}
#endif

#if qStroika_HasComponent_OpenSSL
/*
 ********************************************************************************
 ***************************** Algorithm::DecodeAES *****************************
 ********************************************************************************
 */
Streams::InputStream::Ptr<byte> Algorithm::DecodeAES (const OpenSSL::DerivedKey& key, const Streams::InputStream::Ptr<byte>& in, AESOptions options)
{
    return OpenSSLInputStream::New (cvt_ (key, options), Direction::eDecrypt, in);
}
Memory::BLOB Algorithm::DecodeAES (const OpenSSL::DerivedKey& key, const Memory::BLOB& in, AESOptions options)
{
    return DecodeAES (key, in.As<Streams::InputStream::Ptr<byte>> (), options).ReadAll ();
}
#endif

#if qStroika_HasComponent_OpenSSL
/*
 ********************************************************************************
 ****************************** Algorithm::EncodeAES ****************************
 ********************************************************************************
 */
Streams::InputStream::Ptr<byte> Algorithm::EncodeAES (const OpenSSL::DerivedKey& key, const Streams::InputStream::Ptr<byte>& in, AESOptions options)
{
    return OpenSSLInputStream::New (cvt_ (key, options), Direction::eEncrypt, in);
}
Memory::BLOB Algorithm::EncodeAES (const OpenSSL::DerivedKey& key, const Memory::BLOB& in, AESOptions options)
{
    return EncodeAES (key, in.As<Streams::InputStream::Ptr<byte>> (), options).ReadAll ();
}
#endif

#if qStroika_HasComponent_OpenSSL
/*
 ********************************************************************************
 **************************** Algorithm::AESEncoder *****************************
 ********************************************************************************
 */
Streams::OutputStream::Ptr<byte> Algorithm::AESDecoder (const OpenSSL::DerivedKey& key, const Streams::OutputStream::Ptr<byte>& out, AESOptions options)
{
    return OpenSSLOutputStream::New (cvt_ (key, options), Direction::eDecrypt, out);
}
#endif

#if qStroika_HasComponent_OpenSSL
/*
 ********************************************************************************
 ****************************** Algorithm::AESEncoder ***************************
 ********************************************************************************
 */
Streams::OutputStream::Ptr<byte> Algorithm::AESEncoder (const OpenSSL::DerivedKey& key, const Streams::OutputStream::Ptr<byte>& out, AESOptions options)
{
    return OpenSSLOutputStream::New (cvt_ (key, options), Direction::eEncrypt, out);
}
#endif
