/*
 * Copyright(c) Sophist Solutions, Inc. 1990-2024.  All rights reserved
 */
#ifndef _Stroika_Foundation_Common_Platform_Windows_Registry_h_
#define _Stroika_Foundation_Common_Platform_Windows_Registry_h_ 1

#include "Stroika/Foundation/StroikaPreComp.h"

#if qStroika_Foundation_Common_Platform_Windows
#include <Windows.h>
#else
#error "WINDOWS REQUIRED FOR THIS MODULE"
#endif

#include "Stroika/Foundation/Characters/String.h"
#include "Stroika/Foundation/Common/Common.h"
#include "Stroika/Foundation/DataExchange/VariantValue.h"

namespace Stroika::Foundation::Common::Platform::Windows {

    /**
     *  Simple utility class to allow (read-only) access to the windows registry.
     *
     *  I considered making this support modification, but as JSON configuration files seem a better path than using
     *  the registry, and this is mainly for backward compatibility with older systems, I chose to keep this read-only.
     *
     *  \par Example Usage
     *      \code
     *          if (auto tmp = RegistryKey{HKEY_CLASSES_ROOT}.Lookup (Format (L"MIME\\Database\\Content Type\\%s\\Extension", ct.c_str ()))) {
     *              return tmp.As<String> ();
     *          }
     *      \endcode
     *
     *  \note It would be convenient to be able to retrieve the parent key and name for this given HKEY,
     *        but this doesn't appear well supported (but possible):
     *          https://stackoverflow.com/questions/937044/determine-path-to-registry-key-from-hkey-handle-in-c
     */
    class RegistryKey {
    public:
        /**
         *  Require HKEY argument (parent or actual) to be != null, and != INVALID_HANDLE_VALUE
         *
         *  If the KEY/String overload is used, and the key cannot be opened, and exception is raised: you can
         *  never get an invalid RegistryKey object.
         *
         *  The overload taking 'owned' argument determines if the key is destroyed at the end of ownership.
         *
         *  'samDesired' must be KEY_READ or some other permission that doesn't cause creation.
         */
        constexpr RegistryKey (HKEY hkey, bool owned = false);
        RegistryKey (HKEY parentKey, const Characters::String& path, REGSAM samDesired = KEY_READ);
        RegistryKey ()                   = delete;
        RegistryKey (const RegistryKey&) = delete;

    public:
        /**
         */
        const RegistryKey& operator= (const RegistryKey&) const = delete;

    public:
        /**
         */
        ~RegistryKey ();

    public:
        /**
         */
        nonvirtual operator HKEY () const;

    public:
        /**
         *  Returns \-delimited string, for example:
         *      L"\\REGISTRY\\USER\\S-1-5-21-1083356233-3619400068-4035759904-1001_Classes\\*"
         */
        nonvirtual Characters::String GetFullPathOfKey () const;

    public:
        /**
         *  Returns empty object (null) if missing. Can also return String type, or numeric for DWORDs.
         *
         *  \note @todo - should return empty VariantValue if detectably missing, but for permissions issues, raise exception
         *
         *  \note Unlike the underlying Windows registry APIs, you can pass in a prefName with backslashes, and it will be
         *        interpreted as lookup relative to 'this' registry key
         */
        nonvirtual DataExchange::VariantValue Lookup (const Characters::String& valuePath) const;

    public:
        /**
         */
        nonvirtual Traversal::Iterable<shared_ptr<RegistryKey>> EnumerateSubKeys () const;

    public:
        /**
         */
        nonvirtual Containers::Mapping<Characters::String, DataExchange::VariantValue> EnumerateValues () const;

    private:
        static HKEY OpenPath_ (HKEY parentKey, const Characters::String& path, REGSAM samDesired);

    private:
        bool fOwned_;
        HKEY fKey_;
    };

}

/*
 ********************************************************************************
 ***************************** Implementation Details ***************************
 ********************************************************************************
 */
#include "Registry.inl"

#endif /*_Stroika_Foundation_Common_Platform_Windows_Registry_h_*/
