/*
 * SonarQube Java
 * Copyright (C) 2012-2024 SonarSource SA
 * mailto:info AT sonarsource DOT com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the Sonar Source-Available License Version 1, as published by SonarSource SA.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Sonar Source-Available License for more details.
 *
 * You should have received a copy of the Sonar Source-Available License
 * along with this program; if not, see https://sonarsource.com/license/ssal/
 */
package org.sonar.java.model;

import java.io.File;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Nullable;
import org.junit.jupiter.api.Test;
import org.sonar.plugins.java.api.semantic.Symbol;
import org.sonar.plugins.java.api.tree.AssignmentExpressionTree;
import org.sonar.plugins.java.api.tree.BaseTreeVisitor;
import org.sonar.plugins.java.api.tree.BinaryExpressionTree;
import org.sonar.plugins.java.api.tree.ClassTree;
import org.sonar.plugins.java.api.tree.CompilationUnitTree;
import org.sonar.plugins.java.api.tree.ConditionalExpressionTree;
import org.sonar.plugins.java.api.tree.ExpressionStatementTree;
import org.sonar.plugins.java.api.tree.ExpressionTree;
import org.sonar.plugins.java.api.tree.MethodInvocationTree;
import org.sonar.plugins.java.api.tree.MethodTree;
import org.sonar.plugins.java.api.tree.ReturnStatementTree;
import org.sonar.plugins.java.api.tree.StaticInitializerTree;
import org.sonar.plugins.java.api.tree.Tree;
import org.sonar.plugins.java.api.tree.VariableTree;

import static java.lang.reflect.Modifier.isFinal;
import static java.lang.reflect.Modifier.isPrivate;
import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.sonar.java.model.ExpressionUtils.isInvocationOnVariable;
import static org.sonar.java.model.assertions.TreeAssert.assertThat;

class ExpressionUtilsTest {

  @Test
  void test_skip_parenthesis() {
    File file = new File("src/test/files/model/ExpressionUtilsTestSample.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    MethodTree methodTree = (MethodTree) ((ClassTree) tree.types().get(0)).members().get(0);
    ExpressionTree parenthesis = ((ReturnStatementTree) methodTree.block().body().get(0)).expression();

    assertThat(parenthesis).is(Tree.Kind.PARENTHESIZED_EXPRESSION);
    ExpressionTree skipped = ExpressionUtils.skipParentheses(parenthesis);
    assertThat(skipped).is(Tree.Kind.CONDITIONAL_AND);
    assertThat(ExpressionUtils.skipParentheses(((BinaryExpressionTree) skipped).leftOperand())).is(Tree.Kind.IDENTIFIER);
  }

  @Test
  void test_simple_assignments() {
    File file = new File("src/test/files/model/ExpressionUtilsTestSample.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    MethodTree methodTree = (MethodTree) ((ClassTree) tree.types().get(0)).members().get(1);
    List<AssignmentExpressionTree> assignments = findAssignmentExpressionTrees(methodTree);

    assertThat(assignments).hasSize(4);
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(0))).isTrue();
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(1))).isTrue();
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(2))).isFalse();
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(3))).isFalse();
  }

  @Test
  void method_name() {
    File file = new File("src/test/files/model/ExpressionUtilsMethodNameTest.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    MethodTree methodTree = (MethodTree) ((ClassTree) tree.types().get(0)).members().get(0);

    MethodInvocationTree firstMIT = (MethodInvocationTree) ((ExpressionStatementTree) methodTree.block().body().get(0)).expression();
    MethodInvocationTree secondMIT = (MethodInvocationTree) ((ExpressionStatementTree) methodTree.block().body().get(1)).expression();

    assertThat(ExpressionUtils.methodName(firstMIT).name()).isEqualTo("foo");
    assertThat(ExpressionUtils.methodName(secondMIT).name()).isEqualTo("foo");
  }

  @Test
  void private_constructor() throws Exception {
    assertThat(isFinal(ExpressionUtils.class.getModifiers())).isTrue();
    Constructor<ExpressionUtils> constructor = ExpressionUtils.class.getDeclaredConstructor();
    assertThat(isPrivate(constructor.getModifiers())).isTrue();
    assertThat(constructor.isAccessible()).isFalse();
    constructor.setAccessible(true);
    constructor.newInstance();
  }

  @Test
  void test_extract_identifier_mixed_access() {
    File file = new File("src/test/files/model/ExpressionUtilsTest.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    MethodTree methodTree = (MethodTree) ((ClassTree) tree.types().get(0)).members().get(1);
    List<AssignmentExpressionTree> assignments = findAssignmentExpressionTrees(methodTree);

    // This should reflect method 'mixedReference'.
    assertThat(assignments).hasSize(5);
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(0))).isTrue();
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(1))).isTrue();
    // Contains method invocation.
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(2))).isFalse();
    // Compound assignment
    assertThat(ExpressionUtils.isSimpleAssignment(assignments.get(2))).isFalse();

    // The returned identifier should have the same symbol regardless of the explicit usage of this.
    assertThat(ExpressionUtils.extractIdentifier(assignments.get(0)).symbol())
      .isEqualTo(ExpressionUtils.extractIdentifier(assignments.get(1)).symbol());

  }

  @Test
  void test_cannot_extract_identifier() {
    File file = new File("src/test/files/model/ExpressionUtilsTest.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    MethodTree methodTree = (MethodTree) ((ClassTree) tree.types().get(0)).members().get(1);
    List<AssignmentExpressionTree> assignments = findAssignmentExpressionTrees(methodTree);
    AssignmentExpressionTree assignment = assignments.get(4);
    assertThrows(IllegalArgumentException.class, () -> ExpressionUtils.extractIdentifier(assignment));
  }

  @Test
  void test_get_assigned_symbol() {
    File file = new File("src/test/files/model/ExpressionUtilsTest.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    MethodTree methodTree = (MethodTree) ((ClassTree) tree.types().get(0)).members().get(1);
    List<AssignmentExpressionTree> assignments = findAssignmentExpressionTrees(methodTree);

    // This should reflect method 'mixedReference'.
    assertThat(assignments).hasSize(5);

    assertThat(ExpressionUtils.getAssignedSymbol(assignments.get(0).expression())).isPresent();
    assertThat(ExpressionUtils.getAssignedSymbol(assignments.get(1).expression())).isPresent();

    assertThat(ExpressionUtils.getAssignedSymbol(assignments.get(0).expression())).
      contains(ExpressionUtils.getAssignedSymbol(assignments.get(1).expression()).get());

    assertThat(ExpressionUtils.getAssignedSymbol(assignments.get(2).expression())).isNotPresent();
    assertThat(ExpressionUtils.getAssignedSymbol(assignments.get(3).expression())).isNotPresent();
    assertThat(ExpressionUtils.getAssignedSymbol(assignments.get(4).expression())).isNotPresent();

    List<VariableTree> variables = findVariableTrees(methodTree);
    assertThat(variables).hasSize(2);
    assertThat(ExpressionUtils.getAssignedSymbol(variables.get(1).initializer())).isPresent();

  }

  @Test
  void test_invocation_on_same_variable() {
    CompilationUnitTree tree = JParserTestUtils.parse("""
        class A {
          static {
            String s1 = "a";
            String s2 = "b";
            s1.toString();
            s2.toString();
            toString();
            Optional.of(s1).get().toString();
            Optional.of(s2).get().toString();
          }
        }
        """);

    StaticInitializerTree staticInitializer = (StaticInitializerTree) ((ClassTree) tree.types().get(0)).members().get(0);
    List<Symbol> variablesSymbols = staticInitializer.body().stream()
      .filter(t -> t instanceof VariableTree)
      .map(VariableTree.class::cast)
      .map(VariableTree::symbol)
      .toList();

    List<MethodInvocationTree> invocations = staticInitializer.body().stream()
      .filter(t -> t instanceof ExpressionStatementTree)
      .map(ExpressionStatementTree.class::cast)
      .map(ExpressionStatementTree::expression)
      .map(MethodInvocationTree.class::cast)
      .toList();

    assertThat(isInvocationOnVariable(invocations.get(0), variablesSymbols.get(0), true)).isTrue();
    assertThat(isInvocationOnVariable(invocations.get(0), variablesSymbols.get(1), true)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(1), variablesSymbols.get(0), true)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(1), variablesSymbols.get(1), true)).isTrue();

    // We report the default value if we can not compare two symbol
    assertThat(isInvocationOnVariable(invocations.get(2), variablesSymbols.get(0), true)).isTrue();
    assertThat(isInvocationOnVariable(invocations.get(2), variablesSymbols.get(1), true)).isTrue();
    assertThat(isInvocationOnVariable(invocations.get(3), variablesSymbols.get(0), true)).isTrue();
    assertThat(isInvocationOnVariable(invocations.get(3), variablesSymbols.get(1), true)).isTrue();
    assertThat(isInvocationOnVariable(invocations.get(4), variablesSymbols.get(0), true)).isTrue();
    assertThat(isInvocationOnVariable(invocations.get(4), variablesSymbols.get(1), true)).isTrue();

    assertThat(isInvocationOnVariable(invocations.get(2), variablesSymbols.get(0), false)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(2), variablesSymbols.get(1), false)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(3), variablesSymbols.get(0), false)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(3), variablesSymbols.get(1), false)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(4), variablesSymbols.get(0), false)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(4), variablesSymbols.get(1), false)).isFalse();

    assertThat(isInvocationOnVariable(invocations.get(4), null, false)).isFalse();
    assertThat(isInvocationOnVariable(invocations.get(4), null, true)).isTrue();
  }


  @Test
  void securing_byte() {
    CompilationUnitTree tree = JParserTestUtils.parse("""
        class A {
          static {
            int i1 = 12;
            int i2 = 12 & 0xFF;
            int i3 = 0xff & 12;
            int i4 = 12 & 12;
          }
        }
        """);

    StaticInitializerTree staticInitializer = (StaticInitializerTree) ((ClassTree) tree.types().get(0)).members().get(0);
    List<ExpressionTree> expressions = staticInitializer.body().stream()
      .map(VariableTree.class::cast)
      .map(VariableTree::initializer)
      .toList();

    assertThat(ExpressionUtils.isSecuringByte(expressions.get(0))).isFalse();
    assertThat(ExpressionUtils.isSecuringByte(expressions.get(1))).isTrue();
    assertThat(ExpressionUtils.isSecuringByte(expressions.get(2))).isTrue();
    assertThat(ExpressionUtils.isSecuringByte(expressions.get(3))).isFalse();
  }

  private List<AssignmentExpressionTree> findAssignmentExpressionTrees(MethodTree methodTree) {
    return methodTree.block().body().stream()
      .filter(s -> s.is(Tree.Kind.EXPRESSION_STATEMENT))
      .map(ExpressionStatementTree.class::cast)
      .map(ExpressionStatementTree::expression)
      .filter(e -> e instanceof AssignmentExpressionTree)
      .map(AssignmentExpressionTree.class::cast)
      .toList();
  }

  private List<VariableTree> findVariableTrees(MethodTree methodTree) {
    return methodTree.block().body().stream()
      .filter(s -> s.is(Tree.Kind.VARIABLE))
      .map(VariableTree.class::cast)
      .toList();
  }

  @Test
  void enclosing_method_test() {
    File file = new File("src/test/files/model/ExpressionEnclosingMethodTest.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    FindAssignment findAssignment = new FindAssignment();
    tree.accept(findAssignment);
    findAssignment.assignments.forEach(a -> {
      String expectedName = a.firstToken().trivias().get(0).comment().substring(3);
      MethodTree enclosingMethod = ExpressionUtils.getEnclosingMethod(a);
      if ("null".equals(expectedName)) {
        assertThat(enclosingMethod).isNull();
      } else {
        assertThat(enclosingMethod.simpleName().name()).isEqualTo(expectedName);
      }
    });
  }

  @Test
  void parent_of_type_method_test() {
    File file = new File("src/test/files/model/ExpressionEnclosingMethodTest.java");
    CompilationUnitTree tree = JParserTestUtils.parse(file);
    FindAssignment findAssignment = new FindAssignment();
    tree.accept(findAssignment);
    findAssignment.assignments.forEach(a -> {
      ClassTree parent = (ClassTree) ExpressionUtils.getParentOfType(a, Tree.Kind.CLASS, Tree.Kind.INTERFACE);
      if (parent.is(Tree.Kind.CLASS)) {
        assertThat(parent.simpleName().name()).isEqualTo("Clazz");
      } else if (parent.is(Tree.Kind.INTERFACE)) {
        assertThat(parent.simpleName().name()).isEqualTo("I1");
      }
      assertThat(ExpressionUtils.getParentOfType(tree, Tree.Kind.COMPILATION_UNIT)).isNull();
    });
  }

  private static class FindAssignment extends BaseTreeVisitor {
    private List<AssignmentExpressionTree> assignments = new ArrayList<>();

    @Override
    public void visitAssignmentExpression(AssignmentExpressionTree tree) {
      assignments.add(tree);
      super.visitAssignmentExpression(tree);
    }
  }

  @Test
  void resolve_as_int_constant() {
    assertResolveAsConstant("0", 0);
    assertResolveAsConstant("1", 1);
    assertResolveAsConstant("+1", +1);
    assertResolveAsConstant("0x01 | 0xF0", 0x01 | 0xF0);
    assertResolveAsConstant("-1", -1);
    assertResolveAsConstant("(1)", (1));
    assertResolveAsConstant("~42", ~42);
  }

  @Test
  void resolve_as_long_constant() {
    assertResolveAsConstant("-(0x01 + 2L)", -(0x01 + 2L));
    assertResolveAsConstant("0L", 0L);
    assertResolveAsConstant("1L", 1L);
    assertResolveAsConstant("-1L", -1L);
    assertResolveAsConstant("-(1L)", -(1L));
    assertResolveAsConstant("-(-1L)", -(-1L));
    assertResolveAsConstant("-(-(1L))", -(-(1L)));
    assertResolveAsConstant("-0x25L", -0x25L);
    assertResolveAsConstant("~42L", ~42L);
  }

  @Test
  void resolve_as_boolean_constant() {
    assertResolveAsConstant("true", true);
    assertResolveAsConstant("!true", !true);
    assertResolveAsConstant("false", false);
    assertResolveAsConstant("!false", !false);
    assertResolveAsConstant("Boolean.TRUE", true);
    assertResolveAsConstant("Boolean.FALSE", false);
  }

  @Test
  void resolve_as_string_constant() {
    assertResolveAsConstant("\"abc\"", "abc");
    assertResolveAsConstant("(\"abc\")", ("abc"));
  }

  @Test
  void resolve_as_constant_not_yet_supported() {
    assertResolveAsConstant("true || true", null);
  }

  @Test
  void resolve_as_constant_arithmetic_operations() {
    assertResolveAsConstant("1 + 1 - 1", 1);
    assertResolveAsConstant("8 - 3 + 2 * 2", 9);
    assertResolveAsConstant("8 - (3 + 2) * 2", -2);
    assertResolveAsConstant("8 - (3 + 2) / 5 * 2", 6);
    assertResolveAsConstant("8 - (3 + 2) % 5 * 2", 8);
    assertResolveAsConstant("8 - (x + 2) % 5 * 2", null);
    assertResolveAsConstant("8 - (3 + x) % 5 * 2", null);
    assertResolveAsConstant("8 - (x + x) % 5 * 2", null);
  }

  @Test
  void resolve_as_constant_division_by_zero() {
    assertResolveAsConstant("5 / 0", null);
    assertResolveAsConstant("5L / 0", null);
    assertResolveAsConstant("5D / 0", null);

    assertResolveAsConstant("5 / 0L", null);
    assertResolveAsConstant("5L / 0L", null);
    assertResolveAsConstant("5D / 0L", null);

    assertResolveAsConstant("5 / 0D", null);
    assertResolveAsConstant("5L / 0D", null);
    assertResolveAsConstant("5D / 0D", null);
  }

  @Test
  void resolve_as_constant_unknown_symbol() {
    assertResolveAsConstant("x", null);
    assertResolveAsConstant("-x", null);
    assertResolveAsConstant("~x", null);
    assertResolveAsConstant("!x", null);
    assertResolveAsConstant("++x", null);
    assertResolveAsConstant("x.y", null);
  }

  private void assertResolveAsConstant(String code, @Nullable Object expected) {
    CompilationUnitTree unit = JParserTestUtils.parse("class A { Object f = " + code + "; }");
    ExpressionTree expression = ((VariableTree) ((ClassTree) unit.types().get(0)).members().get(0)).initializer();
    Object actual = ExpressionUtils.resolveAsConstant(expression);
    if (expected == null) {
      assertThat(actual).isNull();
    } else {
      assertThat(actual)
        .hasSameClassAs(expected)
        .isEqualTo(expected);
    }
  }

  @Test
  void areVariablesSame_identifier_assert_true() {
    var unit = JParserTestUtils.parse("class A { void m(int min, int max, int value) { int conditionalValue = value == max ? max : value; }}");
    var classTree = (ClassTree) unit.types().get(0);
    var methodTree = (MethodTree) classTree.members().get(0);
    var variableTree = (VariableTree) methodTree.block().body().get(0);
    var initializer = (ConditionalExpressionTree) variableTree.initializer();
    assertThat(ExpressionUtils.areVariablesSame(((BinaryExpressionTree) initializer.condition()).leftOperand(), initializer.falseExpression(), false)).isTrue();
  }

  @Test
  void areVariablesSame_identifier_assert_false() {
    var unit = JParserTestUtils.parse("class A { void m(int min, int max, int value) { int conditionalValue = value == max ? max : value; }}");
    var classTree = (ClassTree) unit.types().get(0);
    var methodTree = (MethodTree) classTree.members().get(0);
    var variableTree = (VariableTree) methodTree.block().body().get(0);
    var initializer = (ConditionalExpressionTree) variableTree.initializer();
    assertThat(ExpressionUtils.areVariablesSame(((BinaryExpressionTree) initializer.condition()).leftOperand(), initializer.trueExpression(), false)).isFalse();
  }

  @Test
  void areVariablesSame_member_select_assert_true() {
    var unit = JParserTestUtils.parse("class A { int min; void m(int min, int max, int value) { int conditionalValue = value == this.min ? min : value; }}");
    var classTree = (ClassTree) unit.types().get(0);
    var methodTree = (MethodTree) classTree.members().get(1);
    var variableTree = (VariableTree) methodTree.block().body().get(0);
    var initializer = (ConditionalExpressionTree) variableTree.initializer();
    assertThat(ExpressionUtils.areVariablesSame(((BinaryExpressionTree) initializer.condition()).rightOperand(), initializer.trueExpression(), false)).isTrue();
  }

  @Test
  void areVariablesSame_member_select_assert_false() {
    var unit = JParserTestUtils.parse("class A { int min; void m(int min, int max, int value) { int conditionalValue = value == this.min ? max : value; }}");
    var classTree = (ClassTree) unit.types().get(0);
    var methodTree = (MethodTree) classTree.members().get(1);
    var variableTree = (VariableTree) methodTree.block().body().get(0);
    var initializer = (ConditionalExpressionTree) variableTree.initializer();
    assertThat(ExpressionUtils.areVariablesSame(((BinaryExpressionTree) initializer.condition()).rightOperand(), initializer.trueExpression(), false)).isFalse();
  }

  @Test
  void areVariablesSame_unknown_symbol() {
    var unit = JParserTestUtils.parse("class A { void m(int min, int max, int value) { int conditionalValue = getValue() == Math.MIN ? max : value; }}");
    var classTree = (ClassTree) unit.types().get(0);
    var methodTree = (MethodTree) classTree.members().get(0);
    var variableTree = (VariableTree) methodTree.block().body().get(0);
    var initializer = (ConditionalExpressionTree) variableTree.initializer();
    var condition = (BinaryExpressionTree) initializer.condition();
    assertThat(ExpressionUtils.areVariablesSame(condition.leftOperand(), initializer.trueExpression(), false)).isFalse();
    assertThat(ExpressionUtils.areVariablesSame(condition.rightOperand(), initializer.trueExpression(), false)).isFalse();
    assertThat(ExpressionUtils.areVariablesSame(initializer.trueExpression(), condition.leftOperand(), false)).isFalse();
    assertThat(ExpressionUtils.areVariablesSame(initializer.falseExpression(), condition.rightOperand(), false)).isFalse();
  }

}
