﻿// Copyright (c) Tunnel Vision Laboratories, LLC. All Rights Reserved.
// Licensed under the MIT License. See LICENSE in the project root for license information.

namespace StyleCop.Analyzers.CodeGeneration
{
    using System;
    using System.Collections.Generic;
    using System.Collections.Immutable;
    using System.Collections.ObjectModel;
    using System.Globalization;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Xml.Linq;
    using System.Xml.XPath;
    using Microsoft.CodeAnalysis;
    using Microsoft.CodeAnalysis.CSharp;
    using Microsoft.CodeAnalysis.CSharp.Syntax;

    [Generator]
    internal sealed class OperationLightupGenerator : IIncrementalGenerator
    {
        public void Initialize(IncrementalGeneratorInitializationContext context)
        {
            var operationInterfacesFiles = context.AdditionalTextsProvider.Where(static x => Path.GetFileName(x.Path) == "OperationInterfaces.xml");
            context.RegisterSourceOutput(operationInterfacesFiles, this.Execute);
        }

        private void Execute(SourceProductionContext context, AdditionalText operationInterfacesFile)
        {
            var operationInterfacesText = operationInterfacesFile.GetText(context.CancellationToken);
            if (operationInterfacesText is null)
            {
                throw new InvalidOperationException("Failed to read OperationInterfaces.xml");
            }

            var operationInterfaces = XDocument.Parse(operationInterfacesText.ToString());
            this.GenerateOperationInterfaces(in context, operationInterfaces);
        }

        private void GenerateOperationInterfaces(in SourceProductionContext context, XDocument operationInterfaces)
        {
            var tree = operationInterfaces.XPathSelectElement("/Tree");
            if (tree is null)
            {
                throw new InvalidOperationException("Failed to find the IOperation root.");
            }

            var documentData = new DocumentData(operationInterfaces);
            foreach (var pair in documentData.Interfaces)
            {
                this.GenerateOperationInterface(in context, pair.Value);
            }

            this.GenerateOperationWrapperHelper(in context, documentData.Interfaces.Values.ToImmutableArray());
            this.GenerateOperationKindEx(in context, documentData.Interfaces.Values.ToImmutableArray());
        }

        private void GenerateOperationInterface(in SourceProductionContext context, InterfaceData node)
        {
            var members = SyntaxFactory.List<MemberDeclarationSyntax>();

            // internal const string WrappedTypeName = "Microsoft.CodeAnalysis.Operations.IArgumentOperation";
            members = members.Add(SyntaxFactory.FieldDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.ConstKeyword)), // Sonar
                declaration: SyntaxFactory.VariableDeclaration(
                    type: SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.StringKeyword)),
                    variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator(
                        identifier: SyntaxFactory.Identifier("WrappedTypeName"),
                        argumentList: null,
                        initializer: SyntaxFactory.EqualsValueClause(SyntaxFactory.LiteralExpression(SyntaxKind.StringLiteralExpression, SyntaxFactory.Literal($"{node.Namespace}.{node.InterfaceName}"))))))));

            if (node.InterfaceName != "IOperation")
            {
                // private static readonly Type WrappedType;
                members = members.Add(SyntaxFactory.FieldDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PrivateKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword), SyntaxFactory.Token(SyntaxKind.ReadOnlyKeyword)),
                    declaration: SyntaxFactory.VariableDeclaration(
                        type: SyntaxFactory.IdentifierName("Type"),
                        variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator("WrappedType")))));
            }

            foreach (var property in node.Properties)
            {
                if (property.IsSkipped || !property.NeedsAccessor)
                {
                    continue;
                }

                // private static readonly Func<IOperation, IMethodSymbol> ConstructorAccessor;
                members = members.Add(SyntaxFactory.FieldDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PrivateKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword), SyntaxFactory.Token(SyntaxKind.ReadOnlyKeyword)),
                    declaration: SyntaxFactory.VariableDeclaration(
                        type: SyntaxFactory.GenericName(
                            identifier: SyntaxFactory.Identifier("Func"),
                            typeArgumentList: SyntaxFactory.TypeArgumentList(SyntaxFactory.SeparatedList(
                                new[]
                                {
                                    SyntaxFactory.IdentifierName("IOperation"),
                                    property.AccessorResultType,
                                }))),
                        variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator(property.AccessorName)))));
            }

            // private readonly IOperation operation;
            members = members.Add(SyntaxFactory.FieldDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PrivateKeyword), SyntaxFactory.Token(SyntaxKind.ReadOnlyKeyword)),
                declaration: SyntaxFactory.VariableDeclaration(
                    type: SyntaxFactory.IdentifierName("IOperation"),
                    variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator("operation")))));

            var staticCtorStatements = SyntaxFactory.List<StatementSyntax>();

            if (node.InterfaceName != "IOperation")
            {
                staticCtorStatements = staticCtorStatements.Add(
                    SyntaxFactory.ExpressionStatement(SyntaxFactory.AssignmentExpression(
                        SyntaxKind.SimpleAssignmentExpression,
                        left: SyntaxFactory.IdentifierName("WrappedType"),
                        right: SyntaxFactory.InvocationExpression(
                            expression: SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.IdentifierName("OperationWrapperHelper"),
                                name: SyntaxFactory.IdentifierName("GetWrappedType")),
                            argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                                SyntaxFactory.TypeOfExpression(SyntaxFactory.IdentifierName(node.WrapperName)))))))));
            }

            foreach (var property in node.Properties)
            {
                if (property.IsSkipped || !property.NeedsAccessor)
                {
                    continue;
                }

                SimpleNameSyntax helperName;
                if (property.IsDerivedOperationArray)
                {
                    // CreateOperationListPropertyAccessor<IOperation>
                    helperName = SyntaxFactory.GenericName(
                        identifier: SyntaxFactory.Identifier("CreateOperationListPropertyAccessor"),
                        typeArgumentList: SyntaxFactory.TypeArgumentList(SyntaxFactory.SingletonSeparatedList<TypeSyntax>(SyntaxFactory.IdentifierName("IOperation"))));
                }
                else
                {
                    // CreateOperationPropertyAccessor<IOperation, IMethodSymbol>
                    helperName = SyntaxFactory.GenericName(
                        identifier: SyntaxFactory.Identifier("CreateOperationPropertyAccessor"),
                        typeArgumentList: SyntaxFactory.TypeArgumentList(SyntaxFactory.SeparatedList(
                            new[]
                            {
                                SyntaxFactory.IdentifierName("IOperation"),
                                property.AccessorResultType,
                            })));
                }

                // ConstructorAccessor = LightupHelpers.CreateOperationPropertyAccessor<IOperation, IMethodSymbol>(WrappedType, nameof(Constructor));
                ExpressionSyntax wrappedType = node.InterfaceName == "IOperation"
                    ? SyntaxFactory.TypeOfExpression(SyntaxFactory.IdentifierName("IOperation"))
                    : SyntaxFactory.IdentifierName("WrappedType");
                staticCtorStatements = staticCtorStatements.Add(SyntaxFactory.ExpressionStatement(SyntaxFactory.AssignmentExpression(
                    SyntaxKind.SimpleAssignmentExpression,
                    left: SyntaxFactory.IdentifierName(property.AccessorName),
                    right: SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.IdentifierName("LightupHelpers"),
                            name: helperName),
                        argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SeparatedList(
                            new[]
                            {
                                SyntaxFactory.Argument(wrappedType),
                                SyntaxFactory.Argument(SyntaxFactory.InvocationExpression(
                                    expression: SyntaxFactory.IdentifierName("nameof"),
                                    argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(SyntaxFactory.IdentifierName(property.Name)))))),
                            }))))));
            }

            // static IArgumentOperationWrapper()
            // {
            //     WrappedType = OperationWrapperHelper.GetWrappedType(typeof(IObjectCreationOperationWrapper));
            // }
            members = members.Add(SyntaxFactory.ConstructorDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.StaticKeyword)),
                identifier: SyntaxFactory.Identifier(node.WrapperName),
                parameterList: SyntaxFactory.ParameterList(),
                initializer: null,
                body: SyntaxFactory.Block(staticCtorStatements),
                expressionBody: null));

            // private IArgumentOperationWrapper(IOperation operation)
            // {
            //     this.operation = operation;
            // }
            members = members.Add(SyntaxFactory.ConstructorDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PrivateKeyword)),
                identifier: SyntaxFactory.Identifier(node.WrapperName),
                parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                    attributeLists: default,
                    modifiers: default,
                    type: SyntaxFactory.IdentifierName("IOperation"),
                    identifier: SyntaxFactory.Identifier("operation"),
                    @default: null))),
                initializer: null,
                body: SyntaxFactory.Block(
                    SyntaxFactory.ExpressionStatement(SyntaxFactory.AssignmentExpression(
                        SyntaxKind.SimpleAssignmentExpression,
                        left: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.ThisExpression(),
                            name: SyntaxFactory.IdentifierName("operation")),
                        right: SyntaxFactory.IdentifierName("operation")))),
                expressionBody: null));

            // public IOperation WrappedOperation => this.operation;
            members = members.Add(SyntaxFactory.PropertyDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword)),
                type: SyntaxFactory.IdentifierName("IOperation"),
                explicitInterfaceSpecifier: null,
                identifier: SyntaxFactory.Identifier("WrappedOperation"),
                accessorList: null,
                expressionBody: SyntaxFactory.ArrowExpressionClause(SyntaxFactory.MemberAccessExpression(
                    SyntaxKind.SimpleMemberAccessExpression,
                    expression: SyntaxFactory.ThisExpression(),
                    name: SyntaxFactory.IdentifierName("operation"))),
                initializer: null,
                semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));

            if (node.InterfaceName != "IOperation")
            {
                // public ITypeSymbol Type => this.WrappedOperation.Type;
                members = members.Add(SyntaxFactory.PropertyDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword)),
                    type: SyntaxFactory.IdentifierName("ITypeSymbol"),
                    explicitInterfaceSpecifier: null,
                    identifier: SyntaxFactory.Identifier("Type"),
                    accessorList: null,
                    expressionBody: SyntaxFactory.ArrowExpressionClause(SyntaxFactory.MemberAccessExpression(
                        SyntaxKind.SimpleMemberAccessExpression,
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.ThisExpression(),
                            name: SyntaxFactory.IdentifierName("WrappedOperation")),
                        name: SyntaxFactory.IdentifierName("Type"))),
                    initializer: null,
                    semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));
            }

            foreach (var property in node.Properties)
            {
                if (property.IsSkipped)
                {
                    // Generate a NotImplementedException for public properties that do not have a supported type
                    if (property.IsPublicProperty && !property.IsOverride) // Sonar
                    {
                        // public object Constructor => throw new NotImplementedException("Property 'Type.Property' has unsupported type 'Type'");
                        members = members.Add(SyntaxFactory.PropertyDeclaration(
                            attributeLists: default,
                            modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword)),
                            type: SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.ObjectKeyword)),
                            explicitInterfaceSpecifier: null,
                            identifier: SyntaxFactory.Identifier(property.Name),
                            accessorList: null,
                            expressionBody: SyntaxFactory.ArrowExpressionClause(SyntaxFactory.ThrowExpression(SyntaxFactory.ObjectCreationExpression(
                                type: SyntaxFactory.IdentifierName(nameof(NotImplementedException)),
                                argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                                    SyntaxFactory.LiteralExpression(
                                        SyntaxKind.StringLiteralExpression,
                                        SyntaxFactory.Literal($"Property '{node.InterfaceName}.{property.Name}' has unsupported type '{property.Type}'"))))),
                                initializer: null))),
                            initializer: null,
                            semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));
                    }

                    continue;
                }

                var propertyType = property.NeedsWrapper ? SyntaxFactory.IdentifierName(property.WrappedType) : property.AccessorResultType; // Sonar

                // The value is accessed in one of the following ways:
                //   ConstructorAccessor(this.WrappedOperation)
                //   this.WrappedOperation.Constructor
                ExpressionSyntax evaluatedAccessor;
                if (property.NeedsAccessor)
                {
                    evaluatedAccessor = SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.IdentifierName(property.AccessorName),
                        argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                            expression: SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.ThisExpression(),
                                name: SyntaxFactory.IdentifierName("WrappedOperation"))))));
                }
                else
                {
                    evaluatedAccessor = SyntaxFactory.MemberAccessExpression(
                        SyntaxKind.SimpleMemberAccessExpression,
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.ThisExpression(),
                            name: SyntaxFactory.IdentifierName("WrappedOperation")),
                        name: SyntaxFactory.IdentifierName(property.Name));
                }

                ExpressionSyntax convertedResult;
                if (property.NeedsWrapper)
                {
                    // IObjectOrCollectionInitializerOperationWrapper.FromOperation(...)
                    convertedResult = SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: propertyType,
                            name: SyntaxFactory.IdentifierName("FromOperation")),
                        argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(evaluatedAccessor))));
                }
                else
                {
                    convertedResult = evaluatedAccessor;
                }

                // public IMethodSymbol Constructor => ConstructorAccessor(this.WrappedOperation);
                members = members.Add(SyntaxFactory.PropertyDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword)),
                    type: propertyType,
                    explicitInterfaceSpecifier: null,
                    identifier: SyntaxFactory.Identifier(property.Name),
                    accessorList: null,
                    expressionBody: SyntaxFactory.ArrowExpressionClause(convertedResult),
                    initializer: null,
                    semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));
            }

            foreach (var baseDefinition in node.InheritedInterfaces)
            {
                // For now, don't inherit properties from IOperationWrapper
                var inheritedProperties = baseDefinition.InterfaceName != "IOperation" ? baseDefinition.Properties : ImmutableArray<PropertyData>.Empty;
                foreach (var property in inheritedProperties)
                {
                    if (node.Properties.Any(derivedProperty => derivedProperty.Name == property.Name && derivedProperty.IsNew))
                    {
                        continue;
                    }

                    if (!property.IsPublicProperty)
                    {
                        continue;
                    }

                    var propertyType = property.IsSkipped
                        ? SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.ObjectKeyword))
                        : property.NeedsWrapper ? SyntaxFactory.IdentifierName(property.WrappedType) : property.AccessorResultType; // Sonar

                    // public IOperation Instance => ((IMemberReferenceOperationWrapper)this).Instance;
                    members = members.Add(SyntaxFactory.PropertyDeclaration(
                        attributeLists: default,
                        modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword)),
                        type: propertyType,
                        explicitInterfaceSpecifier: null,
                        identifier: SyntaxFactory.Identifier(property.Name),
                        accessorList: null,
                        expressionBody: SyntaxFactory.ArrowExpressionClause(SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.ParenthesizedExpression(SyntaxFactory.CastExpression(
                                type: SyntaxFactory.IdentifierName(baseDefinition.WrapperName),
                                expression: SyntaxFactory.ThisExpression())),
                            name: SyntaxFactory.IdentifierName(property.Name))),
                        initializer: null,
                        semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));
                }

                // public static explicit operator IFieldReferenceOperationWrapper(IMemberReferenceOperationWrapper wrapper)
                //     => FromOperation(wrapper.WrappedOperation);
                members = members.Add(SyntaxFactory.ConversionOperatorDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword)),
                    implicitOrExplicitKeyword: SyntaxFactory.Token(SyntaxKind.ExplicitKeyword),
                    operatorKeyword: SyntaxFactory.Token(SyntaxKind.OperatorKeyword),
                    type: SyntaxFactory.IdentifierName(node.WrapperName),
                    parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                        attributeLists: default,
                        modifiers: default,
                        type: SyntaxFactory.IdentifierName(baseDefinition.WrapperName),
                        identifier: SyntaxFactory.Identifier("wrapper"),
                        @default: null))),
                    body: null,
                    expressionBody: SyntaxFactory.ArrowExpressionClause(SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.IdentifierName("FromOperation"),
                        argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                            SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.IdentifierName("wrapper"),
                                name: SyntaxFactory.IdentifierName("WrappedOperation"))))))),
                    semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));

                // public static implicit operator IMemberReferenceOperationWrapper(IFieldReferenceOperationWrapper wrapper)
                //     => IMemberReferenceOperationWrapper.FromUpcast(wrapper.WrappedOperation);
                members = members.Add(SyntaxFactory.ConversionOperatorDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword)),
                    implicitOrExplicitKeyword: SyntaxFactory.Token(SyntaxKind.ImplicitKeyword),
                    operatorKeyword: SyntaxFactory.Token(SyntaxKind.OperatorKeyword),
                    type: SyntaxFactory.IdentifierName(baseDefinition.WrapperName),
                    parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                        attributeLists: default,
                        modifiers: default,
                        type: SyntaxFactory.IdentifierName(node.WrapperName),
                        identifier: SyntaxFactory.Identifier("wrapper"),
                        @default: null))),
                    body: null,
                    expressionBody: SyntaxFactory.ArrowExpressionClause(SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.IdentifierName(baseDefinition.WrapperName),
                            name: SyntaxFactory.IdentifierName("FromUpcast")),
                        argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                            SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.IdentifierName("wrapper"),
                                name: SyntaxFactory.IdentifierName("WrappedOperation"))))))),
                    semicolonToken: SyntaxFactory.Token(SyntaxKind.SemicolonToken)));
            }

            // public static IArgumentOperationWrapper FromOperation(IOperation operation)
            // {
            //     if (operation == null)
            //     {
            //         return default;
            //     }
            //
            //     if (!IsInstance(operation))
            //     {
            //         throw new InvalidCastException($"Cannot cast '{operation.GetType().FullName}' to '{WrappedTypeName}'");
            //     }
            //
            //     return new IArgumentOperationWrapper(operation);
            // }
            var fromOperationStatements = new List<StatementSyntax>();
            fromOperationStatements.Add(SyntaxFactory.IfStatement(
                condition: SyntaxFactory.BinaryExpression(
                    SyntaxKind.EqualsExpression,
                    left: SyntaxFactory.IdentifierName("operation"),
                    right: SyntaxFactory.LiteralExpression(SyntaxKind.NullLiteralExpression)),
                statement: SyntaxFactory.Block(
                    SyntaxFactory.ReturnStatement(SyntaxFactory.LiteralExpression(SyntaxKind.DefaultLiteralExpression)))));

            if (node.InterfaceName != "IOperation")
            {
                fromOperationStatements.Add(SyntaxFactory.IfStatement(
                    condition: SyntaxFactory.PrefixUnaryExpression(
                        SyntaxKind.LogicalNotExpression,
                        operand: SyntaxFactory.InvocationExpression(
                            expression: SyntaxFactory.IdentifierName("IsInstance"),
                            argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(SyntaxFactory.IdentifierName("operation")))))),
                    statement: SyntaxFactory.Block(
                        SyntaxFactory.ThrowStatement(SyntaxFactory.ObjectCreationExpression(
                            type: SyntaxFactory.IdentifierName("InvalidCastException"),
                            argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                                SyntaxFactory.InterpolatedStringExpression(
                                    SyntaxFactory.Token(SyntaxKind.InterpolatedStringStartToken),
                                    SyntaxFactory.List(new InterpolatedStringContentSyntax[]
                                    {
                                        SyntaxFactory.InterpolatedStringText(SyntaxFactory.Token(
                                            leading: default,
                                            SyntaxKind.InterpolatedStringTextToken,
                                            "Cannot cast '",
                                            "Cannot cast '",
                                            trailing: default)),
                                        SyntaxFactory.Interpolation(SyntaxFactory.MemberAccessExpression(
                                            SyntaxKind.SimpleMemberAccessExpression,
                                            expression: SyntaxFactory.InvocationExpression(
                                                expression: SyntaxFactory.MemberAccessExpression(
                                                    SyntaxKind.SimpleMemberAccessExpression,
                                                    expression: SyntaxFactory.IdentifierName("operation"),
                                                    name: SyntaxFactory.IdentifierName("GetType")),
                                                argumentList: SyntaxFactory.ArgumentList()),
                                            name: SyntaxFactory.IdentifierName("FullName"))),
                                        SyntaxFactory.InterpolatedStringText(SyntaxFactory.Token(
                                            leading: default,
                                            SyntaxKind.InterpolatedStringTextToken,
                                            "' to '",
                                            "' to '",
                                            trailing: default)),
                                        SyntaxFactory.Interpolation(SyntaxFactory.IdentifierName("WrappedTypeName")),
                                        SyntaxFactory.InterpolatedStringText(SyntaxFactory.Token(
                                            leading: default,
                                            SyntaxKind.InterpolatedStringTextToken,
                                            "'",
                                            "'",
                                            trailing: default)),
                                    }))))),
                            initializer: null)))));
            }

            fromOperationStatements.Add(SyntaxFactory.ReturnStatement(SyntaxFactory.ObjectCreationExpression(
                type: SyntaxFactory.IdentifierName(node.WrapperName),
                argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(SyntaxFactory.IdentifierName("operation")))),
                initializer: null)));

            members = members.Add(SyntaxFactory.MethodDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword)),
                returnType: SyntaxFactory.IdentifierName(node.WrapperName),
                explicitInterfaceSpecifier: null,
                identifier: SyntaxFactory.Identifier("FromOperation"),
                typeParameterList: null,
                parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                    attributeLists: default,
                    modifiers: default,
                    type: SyntaxFactory.IdentifierName("IOperation"),
                    identifier: SyntaxFactory.Identifier("operation"),
                    @default: null))),
                constraintClauses: default,
                body: SyntaxFactory.Block(fromOperationStatements),
                expressionBody: null));

            // public static bool IsInstance(IOperation operation)
            // {
            //     return operation != null && LightupHelpers.CanWrapOperation(operation, WrappedType);
            // }
            ExpressionSyntax isInstanceExpression = SyntaxFactory.BinaryExpression(
                SyntaxKind.NotEqualsExpression,
                left: SyntaxFactory.IdentifierName("operation"),
                right: SyntaxFactory.LiteralExpression(SyntaxKind.NullLiteralExpression));
            if (node.InterfaceName != "IOperation")
            {
                isInstanceExpression = SyntaxFactory.BinaryExpression(
                    SyntaxKind.LogicalAndExpression,
                    left: isInstanceExpression,
                    right: SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.IdentifierName("LightupHelpers"),
                            name: SyntaxFactory.IdentifierName("CanWrapOperation")),
                        argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SeparatedList(
                            new[]
                            {
                                SyntaxFactory.Argument(SyntaxFactory.IdentifierName("operation")),
                                SyntaxFactory.Argument(SyntaxFactory.IdentifierName("WrappedType")),
                            }))));
            }

            members = members.Add(SyntaxFactory.MethodDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword)),
                returnType: SyntaxFactory.PredefinedType(SyntaxFactory.Token(SyntaxKind.BoolKeyword)),
                explicitInterfaceSpecifier: null,
                identifier: SyntaxFactory.Identifier("IsInstance"),
                typeParameterList: null,
                parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                    attributeLists: default,
                    modifiers: default,
                    type: SyntaxFactory.IdentifierName("IOperation"),
                    identifier: SyntaxFactory.Identifier("operation"),
                    @default: null))),
                constraintClauses: default,
                body: SyntaxFactory.Block(SyntaxFactory.ReturnStatement(isInstanceExpression)),
                expressionBody: null));

            if (node.IsAbstract)
            {
                // internal static IMemberReferenceOperationWrapper FromUpcast(IOperation operation)
                // {
                //     return new IMemberReferenceOperationWrapper(operation);
                // }
                members = members.Add(SyntaxFactory.MethodDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword)), // Sonar
                    returnType: SyntaxFactory.IdentifierName(node.WrapperName),
                    explicitInterfaceSpecifier: null,
                    identifier: SyntaxFactory.Identifier("FromUpcast"),
                    typeParameterList: null,
                    parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                        attributeLists: default,
                        modifiers: default,
                        type: SyntaxFactory.IdentifierName("IOperation"),
                        identifier: SyntaxFactory.Identifier("operation"),
                        @default: null))),
                    constraintClauses: default,
                    body: SyntaxFactory.Block(
                        SyntaxFactory.ReturnStatement(SyntaxFactory.ObjectCreationExpression(
                            type: SyntaxFactory.IdentifierName(node.WrapperName),
                            argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(SyntaxFactory.IdentifierName("operation")))),
                            initializer: null))),
                    expressionBody: null));
            }

            var wrapperStruct = SyntaxFactory.StructDeclaration(
                attributeLists: default,
                modifiers: SyntaxTokenList.Create(SyntaxFactory.Token(SyntaxKind.PublicKeyword)).Add(SyntaxFactory.Token(SyntaxKind.ReadOnlyKeyword)), // Sonar
                identifier: SyntaxFactory.Identifier(node.WrapperName),
                typeParameterList: null,
                baseList: SyntaxFactory.BaseList(SyntaxFactory.SingletonSeparatedList<BaseTypeSyntax>(SyntaxFactory.SimpleBaseType(SyntaxFactory.IdentifierName("IOperationWrapper")))), // Sonar
                constraintClauses: default,
                members: members);

            var usingDirectives = new List<UsingDirectiveSyntax>();
            usingDirectives.Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System")));
            if (node.InterfaceName == "IOperation")
            {
                usingDirectives.Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Collections.Generic")));
            }

            usingDirectives.Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Collections.Immutable")));
            usingDirectives.Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Microsoft.CodeAnalysis")));

            var wrapperNamespace = SyntaxFactory.NamespaceDeclaration(
                name: SyntaxFactory.ParseName("StyleCop.Analyzers.Lightup"),
                externs: default,
                usings: SyntaxFactory.List(usingDirectives),
                members: SyntaxFactory.SingletonList<MemberDeclarationSyntax>(wrapperStruct));

            wrapperNamespace = wrapperNamespace
                .NormalizeWhitespace()
                .WithLeadingTrivia(
                    SyntaxFactory.Comment("// Copyright (c) Tunnel Vision Laboratories, LLC. All Rights Reserved."),
                    SyntaxFactory.CarriageReturnLineFeed,
                    SyntaxFactory.Comment("// Licensed under the MIT License. See LICENSE in the project root for license information."),
                    SyntaxFactory.CarriageReturnLineFeed,
                    SyntaxFactory.CarriageReturnLineFeed)
                .WithTrailingTrivia(
                    SyntaxFactory.CarriageReturnLineFeed);

            context.AddSource(node.WrapperName + ".g.cs", wrapperNamespace.GetText(Encoding.UTF8));
        }

        private void GenerateOperationWrapperHelper(in SourceProductionContext context, ImmutableArray<InterfaceData> wrapperTypes)
        {
            // private static readonly ImmutableDictionary<Type, Type> WrappedTypes;
            var wrappedTypes = SyntaxFactory.FieldDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PrivateKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword), SyntaxFactory.Token(SyntaxKind.ReadOnlyKeyword)),
                declaration: SyntaxFactory.VariableDeclaration(
                    type: SyntaxFactory.GenericName(
                        identifier: SyntaxFactory.Identifier("ImmutableDictionary"),
                        typeArgumentList: SyntaxFactory.TypeArgumentList(
                            SyntaxFactory.SeparatedList<TypeSyntax>(
                                new[]
                                {
                                    SyntaxFactory.IdentifierName("Type"),
                                    SyntaxFactory.IdentifierName("Type"),
                                }))),
                    variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator(SyntaxFactory.Identifier("WrappedTypes")))));

            // var codeAnalysisAssembly = typeof(SyntaxNode).GetTypeInfo().Assembly;
            // var builder = ImmutableDictionary.CreateBuilder<Type, Type>();
            var staticCtorStatements = SyntaxFactory.List<StatementSyntax>()
                .Add(SyntaxFactory.LocalDeclarationStatement(SyntaxFactory.VariableDeclaration(
                    type: SyntaxFactory.IdentifierName("var"),
                    variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator(
                        identifier: SyntaxFactory.Identifier("codeAnalysisAssembly"),
                        argumentList: null,
                        initializer: SyntaxFactory.EqualsValueClause(
                            SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.InvocationExpression(
                                    SyntaxFactory.MemberAccessExpression(
                                        SyntaxKind.SimpleMemberAccessExpression,
                                        expression: SyntaxFactory.TypeOfExpression(SyntaxFactory.IdentifierName("SyntaxNode")),
                                        name: SyntaxFactory.IdentifierName("GetTypeInfo"))),
                                name: SyntaxFactory.IdentifierName("Assembly"))))))))
                .Add(SyntaxFactory.LocalDeclarationStatement(SyntaxFactory.VariableDeclaration(
                    type: SyntaxFactory.IdentifierName("var"),
                    variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator(
                        identifier: SyntaxFactory.Identifier("builder"),
                        argumentList: null,
                        initializer: SyntaxFactory.EqualsValueClause(
                            SyntaxFactory.InvocationExpression(
                                SyntaxFactory.MemberAccessExpression(
                                    SyntaxKind.SimpleMemberAccessExpression,
                                    expression: SyntaxFactory.IdentifierName("ImmutableDictionary"),
                                    name: SyntaxFactory.GenericName(
                                        identifier: SyntaxFactory.Identifier("CreateBuilder"),
                                        typeArgumentList: SyntaxFactory.TypeArgumentList(
                                            SyntaxFactory.SeparatedList<TypeSyntax>(
                                                new[]
                                                {
                                                    SyntaxFactory.IdentifierName("Type"),
                                                    SyntaxFactory.IdentifierName("Type"),
                                                })))))))))));

            foreach (var node in wrapperTypes)
            {
                // For the base IOperation node:
                //   builder.Add(typeof(IArgumentOperationWrapper), typeof(IOperation));
                //
                // For all other nodes:
                //   builder.Add(typeof(IArgumentOperationWrapper), codeAnalysisAssembly.GetType(IArgumentOperationWrapper.WrappedTypeName));
                ArgumentSyntax typeArgument;
                if (node.InterfaceName == "IOperation")
                {
                    typeArgument = SyntaxFactory.Argument(SyntaxFactory.TypeOfExpression(SyntaxFactory.IdentifierName("IOperation")));
                }
                else
                {
                    typeArgument = SyntaxFactory.Argument(
                        SyntaxFactory.InvocationExpression(
                            expression: SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.IdentifierName("codeAnalysisAssembly"),
                                name: SyntaxFactory.IdentifierName("GetType")),
                            argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Argument(
                                SyntaxFactory.MemberAccessExpression(
                                    SyntaxKind.SimpleMemberAccessExpression,
                                    expression: SyntaxFactory.IdentifierName(node.WrapperName),
                                    name: SyntaxFactory.IdentifierName("WrappedTypeName")))))));
                }

                staticCtorStatements = staticCtorStatements.Add(SyntaxFactory.ExpressionStatement(
                    SyntaxFactory.InvocationExpression(
                        expression: SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.IdentifierName("builder"),
                            name: SyntaxFactory.IdentifierName("Add")),
                        argumentList: SyntaxFactory.ArgumentList(
                            SyntaxFactory.SeparatedList(
                                new[]
                                {
                                    SyntaxFactory.Argument(SyntaxFactory.TypeOfExpression(SyntaxFactory.IdentifierName(node.WrapperName))),
                                    typeArgument,
                                })))));
            }

            // WrappedTypes = builder.ToImmutable();
            staticCtorStatements = staticCtorStatements.Add(SyntaxFactory.ExpressionStatement(
                SyntaxFactory.AssignmentExpression(
                    SyntaxKind.SimpleAssignmentExpression,
                    left: SyntaxFactory.IdentifierName("WrappedTypes"),
                    right: SyntaxFactory.InvocationExpression(
                        SyntaxFactory.MemberAccessExpression(
                            SyntaxKind.SimpleMemberAccessExpression,
                            expression: SyntaxFactory.IdentifierName("builder"),
                            name: SyntaxFactory.IdentifierName("ToImmutable"))))));

            var staticCtor = SyntaxFactory.ConstructorDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.StaticKeyword)),
                identifier: SyntaxFactory.Identifier("OperationWrapperHelper"),
                parameterList: SyntaxFactory.ParameterList(),
                initializer: null,
                body: SyntaxFactory.Block(staticCtorStatements),
                expressionBody: null);

            // /// <summary>
            // /// Gets the type that is wrapped by the given wrapper.
            // /// </summary>
            // /// <param name="wrapperType">Type of the wrapper for which the wrapped type should be retrieved.</param>
            // /// <returns>The wrapped type, or null if there is no info.</returns>
            // internal static Type GetWrappedType(Type wrapperType)
            // {
            //     if (WrappedTypes.TryGetValue(wrapperType, out Type wrappedType))
            //     {
            //         return wrappedType;
            //     }
            //
            //     return null;
            // }
            var getWrappedType = SyntaxFactory.MethodDeclaration(
                attributeLists: default,
                modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.StaticKeyword)), // Sonar
                returnType: SyntaxFactory.IdentifierName("Type"),
                explicitInterfaceSpecifier: null,
                identifier: SyntaxFactory.Identifier("GetWrappedType"),
                typeParameterList: null,
                parameterList: SyntaxFactory.ParameterList(SyntaxFactory.SingletonSeparatedList(SyntaxFactory.Parameter(
                    attributeLists: default,
                    modifiers: default,
                    type: SyntaxFactory.IdentifierName("Type"),
                    identifier: SyntaxFactory.Identifier("wrapperType"),
                    @default: null))),
                constraintClauses: default,
                body: SyntaxFactory.Block(
                    SyntaxFactory.IfStatement(
                        condition: SyntaxFactory.InvocationExpression(
                            expression: SyntaxFactory.MemberAccessExpression(
                                SyntaxKind.SimpleMemberAccessExpression,
                                expression: SyntaxFactory.IdentifierName("WrappedTypes"),
                                name: SyntaxFactory.IdentifierName("TryGetValue")),
                            argumentList: SyntaxFactory.ArgumentList(SyntaxFactory.SeparatedList(
                                new[]
                                {
                                    SyntaxFactory.Argument(SyntaxFactory.IdentifierName("wrapperType")),
                                    SyntaxFactory.Argument(
                                        nameColon: null,
                                        refKindKeyword: SyntaxFactory.Token(SyntaxKind.OutKeyword),
                                        expression: SyntaxFactory.DeclarationExpression(
                                            type: SyntaxFactory.IdentifierName("Type"),
                                            designation: SyntaxFactory.SingleVariableDesignation(SyntaxFactory.Identifier("wrappedType")))),
                                }))),
                        statement: SyntaxFactory.Block(
                            SyntaxFactory.ReturnStatement(SyntaxFactory.IdentifierName("wrappedType")))),
                    SyntaxFactory.ReturnStatement(SyntaxFactory.LiteralExpression(SyntaxKind.NullLiteralExpression))),
                expressionBody: null);

            getWrappedType = getWrappedType.WithLeadingTrivia(SyntaxFactory.TriviaList(
                SyntaxFactory.Trivia(SyntaxFactory.DocumentationComment(
                    SyntaxFactory.XmlText(" "),
                    SyntaxFactory.XmlSummaryElement(
                        SyntaxFactory.XmlText(XmlSyntaxFactory.XmlCarriageReturnLineFeedWithContinuation),
                        SyntaxFactory.XmlText(" Gets the type that is wrapped by the given wrapper."),
                        SyntaxFactory.XmlText(XmlSyntaxFactory.XmlCarriageReturnLineFeedWithContinuation),
                        SyntaxFactory.XmlText(" ")),
                    SyntaxFactory.XmlText(XmlSyntaxFactory.XmlCarriageReturnLineFeedWithContinuation),
                    SyntaxFactory.XmlText(" "),
                    SyntaxFactory.XmlParamElement(
                        "wrapperType",
                        SyntaxFactory.XmlText("Type of the wrapper for which the wrapped type should be retrieved.")),
                    SyntaxFactory.XmlText(XmlSyntaxFactory.XmlCarriageReturnLineFeedWithContinuation),
                    SyntaxFactory.XmlText(" "),
                    SyntaxFactory.XmlReturnsElement(
                        SyntaxFactory.XmlText("The wrapped type, or null if there is no info.")),
                    SyntaxFactory.XmlText(XmlSyntaxFactory.XmlCarriageReturnLineFeedWithContinuation).WithoutTrailingTrivia()))));

            var wrapperHelperClass = SyntaxFactory.ClassDeclaration(
                attributeLists: default,
                modifiers: SyntaxTokenList.Create(SyntaxFactory.Token(SyntaxKind.PublicKeyword)).Add(SyntaxFactory.Token(SyntaxKind.StaticKeyword)), // Sonar
                identifier: SyntaxFactory.Identifier("OperationWrapperHelper"),
                typeParameterList: null,
                baseList: null,
                constraintClauses: default,
                members: SyntaxFactory.List<MemberDeclarationSyntax>()
                    .Add(wrappedTypes)
                    .Add(staticCtor)
                    .Add(getWrappedType));
            var wrapperNamespace = SyntaxFactory.NamespaceDeclaration(
                name: SyntaxFactory.ParseName("StyleCop.Analyzers.Lightup"),
                externs: default,
                usings: SyntaxFactory.List<UsingDirectiveSyntax>()
                    .Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System")))
                    .Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Collections.Immutable")))
                    .Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("System.Reflection")))
                    .Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Microsoft.CodeAnalysis"))),
                members: SyntaxFactory.SingletonList<MemberDeclarationSyntax>(wrapperHelperClass));

            wrapperNamespace = wrapperNamespace
                .NormalizeWhitespace()
                .WithLeadingTrivia(
                    SyntaxFactory.Comment("// Copyright (c) Tunnel Vision Laboratories, LLC. All Rights Reserved."),
                    SyntaxFactory.CarriageReturnLineFeed,
                    SyntaxFactory.Comment("// Licensed under the MIT License. See LICENSE in the project root for license information."),
                    SyntaxFactory.CarriageReturnLineFeed,
                    SyntaxFactory.CarriageReturnLineFeed)
                .WithTrailingTrivia(
                    SyntaxFactory.CarriageReturnLineFeed);

            context.AddSource("OperationWrapperHelper.g.cs", wrapperNamespace.GetText(Encoding.UTF8));
        }

        private void GenerateOperationKindEx(in SourceProductionContext context, ImmutableArray<InterfaceData> wrapperTypes)
        {
            var operationKinds = wrapperTypes
                .SelectMany(type => type.OperationKinds)
                .OrderBy(kind => kind.value)
                .ToImmutableArray();

            var members = SyntaxFactory.List<MemberDeclarationSyntax>();
            foreach (var operationKind in operationKinds)
            {
                // public const OperationKind FieldReference = (OperationKind)26;
                members = members.Add(SyntaxFactory.FieldDeclaration(
                    attributeLists: default,
                    modifiers: SyntaxFactory.TokenList(SyntaxFactory.Token(SyntaxKind.PublicKeyword), SyntaxFactory.Token(SyntaxKind.ConstKeyword)),
                    declaration: SyntaxFactory.VariableDeclaration(
                        type: SyntaxFactory.IdentifierName("OperationKind"),
                        variables: SyntaxFactory.SingletonSeparatedList(SyntaxFactory.VariableDeclarator(
                            identifier: SyntaxFactory.Identifier(operationKind.name),
                            argumentList: null,
                            initializer: SyntaxFactory.EqualsValueClause(SyntaxFactory.CastExpression(
                                type: SyntaxFactory.IdentifierName("OperationKind"),
                                expression: SyntaxFactory.LiteralExpression(SyntaxKind.NumericLiteralExpression, SyntaxFactory.Literal($"0x{operationKind.value:x}", operationKind.value)))))))));
            }

            var operationKindExClass = SyntaxFactory.ClassDeclaration(
                attributeLists: default,
                modifiers: SyntaxTokenList.Create(SyntaxFactory.Token(SyntaxKind.PublicKeyword)).Add(SyntaxFactory.Token(SyntaxKind.StaticKeyword)), // Sonar
                identifier: SyntaxFactory.Identifier("OperationKindEx"),
                typeParameterList: null,
                baseList: null,
                constraintClauses: default,
                members: members);
            var wrapperNamespace = SyntaxFactory.NamespaceDeclaration(
                name: SyntaxFactory.ParseName("StyleCop.Analyzers.Lightup"),
                externs: default,
                usings: SyntaxFactory.List<UsingDirectiveSyntax>()
                    .Add(SyntaxFactory.UsingDirective(SyntaxFactory.ParseName("Microsoft.CodeAnalysis"))),
                members: SyntaxFactory.SingletonList<MemberDeclarationSyntax>(operationKindExClass));

            wrapperNamespace = wrapperNamespace
                .NormalizeWhitespace()
                .WithLeadingTrivia(
                    SyntaxFactory.Comment("// Copyright (c) Tunnel Vision Laboratories, LLC. All Rights Reserved."),
                    SyntaxFactory.CarriageReturnLineFeed,
                    SyntaxFactory.Comment("// Licensed under the MIT License. See LICENSE in the project root for license information."),
                    SyntaxFactory.CarriageReturnLineFeed,
                    SyntaxFactory.CarriageReturnLineFeed)
                .WithTrailingTrivia(
                    SyntaxFactory.CarriageReturnLineFeed);

            context.AddSource("OperationKindEx.g.cs", wrapperNamespace.GetText(Encoding.UTF8));
        }

        private sealed class DocumentData
        {
            public DocumentData(XDocument document)
            {
                var operationKinds = GetOperationKinds(document);

                var interfaces = new Dictionary<string, InterfaceData>();
                foreach (var node in document.XPathSelectElements("/Tree/AbstractNode"))
                {
                    if (node.Attribute("Internal")?.Value == "true")
                    {
                        continue;
                    }

                    if (!operationKinds.TryGetValue(node.RequiredAttribute("Name").Value, out var kinds))
                    {
                        kinds = ImmutableArray<(string name, int value, string? extraDescription)>.Empty;
                    }

                    var interfaceData = new InterfaceData(this, node, kinds);
                    interfaces.Add(interfaceData.InterfaceName, interfaceData);
                }

                foreach (var node in document.XPathSelectElements("/Tree/Node"))
                {
                    if (node.Attribute("Internal")?.Value == "true")
                    {
                        continue;
                    }

                    if (!operationKinds.TryGetValue(node.RequiredAttribute("Name").Value, out var kinds))
                    {
                        kinds = ImmutableArray<(string name, int value, string? extraDescription)>.Empty;
                    }

                    var interfaceData = new InterfaceData(this, node, kinds);
                    interfaces.Add(interfaceData.InterfaceName, interfaceData);
                }

                this.Interfaces = new ReadOnlyDictionary<string, InterfaceData>(interfaces);
            }

            public ReadOnlyDictionary<string, InterfaceData> Interfaces { get; }

            private static ImmutableDictionary<string, ImmutableArray<(string name, int value, string? extraDescription)>> GetOperationKinds(XDocument document)
            {
                var skippedOperationKinds = GetSkippedOperationKinds(document);

                var builder = ImmutableDictionary.CreateBuilder<string, ImmutableArray<(string name, int value, string? extraDescription)>>();

                int operationKind = 0;
                foreach (var node in document.XPathSelectElements("/Tree/AbstractNode|/Tree/Node"))
                {
                    if (node.Attribute("Internal")?.Value == "true")
                    {
                        continue;
                    }

                    if (node.XPathSelectElement("OperationKind") is { } explicitKind)
                    {
                        if (node.Name == "AbstractNode" && explicitKind.Attribute("Include")?.Value != "true")
                        {
                            continue;
                        }
                        else if (explicitKind.Attribute("Include")?.Value == "false")
                        {
                            // The node is explicitly excluded
                            continue;
                        }
                        else if (explicitKind.XPathSelectElements("Entry").Any())
                        {
                            var nodeBuilder = ImmutableArray.CreateBuilder<(string name, int value, string? extraDescription)>();
                            foreach (var entry in explicitKind.XPathSelectElements("Entry"))
                            {
                                if (entry.Attribute("EditorBrowsable")?.Value == "false")
                                {
                                    // Skip code generation for this operation kind
                                    continue;
                                }

                                int parsedValue = ParsePrefixHexValue(entry.RequiredAttribute("Value").Value);
                                nodeBuilder.Add((entry.RequiredAttribute("Name").Value, parsedValue, entry.Attribute("ExtraDescription")?.Value));
                            }

                            builder.Add(node.RequiredAttribute("Name").Value, nodeBuilder.ToImmutable());
                            continue;
                        }
                    }
                    else if (node.Name == "AbstractNode")
                    {
                        // Abstract nodes without explicit Include="true" are skipped
                        continue;
                    }

                    // Implicit operation kind
                    operationKind++;
                    while (skippedOperationKinds.Contains(operationKind))
                    {
                        operationKind++;
                    }

                    var nodeName = node.RequiredAttribute("Name").Value;
                    var kindName = nodeName.Substring("I".Length, nodeName.Length - "I".Length - "Operation".Length);
                    builder.Add(nodeName, ImmutableArray.Create((kindName, operationKind, (string?)null)));
                }

                return builder.ToImmutable();
            }

            private static ImmutableHashSet<int> GetSkippedOperationKinds(XDocument document)
            {
                var builder = ImmutableHashSet.CreateBuilder<int>();
                foreach (var skippedKind in document.XPathSelectElements("/Tree/UnusedOperationKinds/Entry"))
                {
                    builder.Add(ParsePrefixHexValue(skippedKind.RequiredAttribute("Value").Value));
                }

                foreach (var explicitKind in document.XPathSelectElements("/Tree/*/OperationKind/Entry"))
                {
                    builder.Add(ParsePrefixHexValue(explicitKind.RequiredAttribute("Value").Value));
                }

                return builder.ToImmutable();
            }

            private static int ParsePrefixHexValue(string value)
            {
                if (!value.StartsWith("0x"))
                {
                    throw new InvalidOperationException($"Unexpected number format: '{value}'");
                }

                return int.Parse(value.Substring("0x".Length), NumberStyles.AllowHexSpecifier);
            }
        }

        private sealed class InterfaceData
        {
            private readonly DocumentData documentData;

            public InterfaceData(DocumentData documentData, XElement node, ImmutableArray<(string name, int value, string? extraDescription)> operationKinds)
            {
                this.documentData = documentData;

                this.OperationKinds = operationKinds;
                this.InterfaceName = node.RequiredAttribute("Name").Value;

                if (node.Attribute("Namespace") is { } namespaceNode)
                {
                    if (namespaceNode.Value == string.Empty)
                    {
                        this.Namespace = "Microsoft.CodeAnalysis";
                    }
                    else
                    {
                        this.Namespace = $"Microsoft.CodeAnalysis.{namespaceNode.Value}";
                    }
                }
                else
                {
                    this.Namespace = "Microsoft.CodeAnalysis.Operations";
                }

                this.Name = this.InterfaceName.Substring("I".Length, this.InterfaceName.Length - "I".Length - "Operation".Length);
                this.WrapperName = this.InterfaceName + "Wrapper";
                this.BaseInterfaceName = node.Attribute("Base")?.Value;
                this.IsAbstract = node.Name == "AbstractNode";
                this.Properties = node.XPathSelectElements("Property").Select(property => new PropertyData(property)).ToImmutableArray();
            }

            public ImmutableArray<(string name, int value, string? extraDescription)> OperationKinds { get; }

            public string InterfaceName { get; }

            public string Namespace { get; }

            public string Name { get; }

            public string WrapperName { get; }

            public string? BaseInterfaceName { get; }

            public bool IsAbstract { get; }

            public ImmutableArray<PropertyData> Properties { get; }

            public InterfaceData? BaseInterface
            {
                get
                {
                    if (this.BaseInterfaceName is not null
                        && this.documentData.Interfaces.TryGetValue(this.BaseInterfaceName, out var baseInterface))
                    {
                        return baseInterface;
                    }

                    return null;
                }
            }

            public IEnumerable<InterfaceData> InheritedInterfaces
            {
                get
                {
                    var inheritedInterfaces = new List<InterfaceData>();
                    for (var baseDefinition = this.BaseInterface; baseDefinition is not null; baseDefinition = baseDefinition.BaseInterface)
                    {
                        inheritedInterfaces.Add(baseDefinition);
                    }

                    inheritedInterfaces.Reverse();
                    return inheritedInterfaces;
                }
            }
        }

        private sealed class PropertyData
        {
            public PropertyData(XElement node)
            {
                this.Name = node.RequiredAttribute("Name").Value;
                this.AccessorName = this.Name + "Accessor";
                this.Type = node.RequiredAttribute("Type").Value;
                this.TypeNonNullable = Type.TrimEnd('?'); // Sonar - When comparing types as strings, the nullable suffix should be ignored.
                this.WrappedType = TypeNonNullable + "Wrapper"; // Sonar

                this.IsNew = node.Attribute("New")?.Value == "true";
                this.IsPublicProperty = node.Attribute("Internal")?.Value != "true";
                this.IsOverride = node.Attribute("Override")?.Value == "true"; // Sonar

                this.IsSkipped = this.TypeNonNullable switch // Sonar
                {
                    "ArgumentKind" => true,
                    "BranchKind" => true,
                    "CaseKind" => true,
                    "CommonConversion" => true,
                    "ForEachLoopOperationInfo" => true,
                    "IDiscardSymbol" => true,
                    "InstanceReferenceKind" => true,
                    "InterpolatedStringArgumentPlaceholderKind" => true,    // Sonar: Skipped because it's not available
                    "LoopKind" => true,
                    "PlaceholderKind" => true,
                    _ => !this.IsPublicProperty || this.IsOverride, // Sonar
                };

                this.NeedsAccessor = this.Name switch
                {
                    nameof(IOperation.Kind) => false,
                    nameof(IOperation.Syntax) => false,
                    nameof(IOperation.Type) => false,
                    nameof(IOperation.ConstantValue) => false,
                    _ => true,
                };
                this.NeedsWrapper = IsAnyOperation(TypeNonNullable) && TypeNonNullable != "IOperation"; // Sonar
                this.IsDerivedOperationArray = IsAnyOperationArray(TypeNonNullable) && TypeNonNullable != "ImmutableArray<IOperation>"; // Sonar

                if (this.IsDerivedOperationArray)
                {
                    this.AccessorResultType = SyntaxFactory.GenericName(
                        identifier: SyntaxFactory.Identifier("ImmutableArray"),
                        typeArgumentList: SyntaxFactory.TypeArgumentList(SyntaxFactory.SingletonSeparatedList<TypeSyntax>(SyntaxFactory.IdentifierName("IOperation"))));
                }
                else if (IsAnyOperation(TypeNonNullable)) // Sonar
                {
                    this.AccessorResultType = SyntaxFactory.IdentifierName("IOperation");
                }
                else
                {
                    this.AccessorResultType = SyntaxFactory.ParseTypeName(this.Type);
                }
            }

            public bool IsNew { get; }

            public bool IsPublicProperty { get; }

            public bool IsOverride { get; } // Added by Sonar. Usages are also by Sonar.

            public bool IsSkipped { get; }

            public string Name { get; }

            public string AccessorName { get; }

            public string Type { get; }

            public bool NeedsAccessor { get; }

            public string TypeNonNullable { get; } // Sonar

            public string WrappedType { get; } // Sonar

            public bool NeedsWrapper { get; }

            public bool IsDerivedOperationArray { get; }

            public TypeSyntax AccessorResultType { get; }

            private static bool IsAnyOperation(string type)
            {
                return type.StartsWith("I") && type.EndsWith("Operation");
            }

            private static bool IsAnyOperationArray(string type)
            {
                return type.StartsWith("ImmutableArray<I") && type.EndsWith("Operation>");
            }
        }
    }
}
