﻿/*
 * SonarAnalyzer for .NET
 * Copyright (C) 2014-2024 SonarSource SA
 * mailto:info AT sonarsource DOT com
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the Sonar Source-Available License Version 1, as published by SonarSource SA.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Sonar Source-Available License for more details.
 *
 * You should have received a copy of the Sonar Source-Available License
 * along with this program; if not, see https://sonarsource.com/license/ssal/
 */

using Microsoft.CodeAnalysis.Text;

namespace SonarAnalyzer.Rules
{
    public abstract class SingleStatementPerLineBase<TSyntaxKind, TStatementSyntax> : SonarDiagnosticAnalyzer<TSyntaxKind>
        where TSyntaxKind : struct
        where TStatementSyntax : SyntaxNode
    {
        protected const string DiagnosticId = "S122";
        protected override string MessageFormat => "Reformat the code to have only one statement per line.";

        protected abstract GeneratedCodeRecognizer GeneratedCodeRecognizer { get; }
        protected abstract bool StatementShouldBeExcluded(TStatementSyntax statement);

        protected SingleStatementPerLineBase() : base(DiagnosticId) { }

        protected sealed override void Initialize(SonarAnalysisContext context) =>
            context.RegisterTreeAction(
                GeneratedCodeRecognizer,
                c =>
                {
                    var statements = c.Tree.GetRoot()
                        .DescendantNodesAndSelf()
                        .OfType<TStatementSyntax>()
                        .Where(st => !StatementShouldBeExcluded(st));

                    var statementsByLines = MultiValueDictionary<int, TStatementSyntax>.Create<HashSet<TStatementSyntax>>();
                    foreach (var statement in statements)
                    {
                        AddStatementToLineCache(statement, statementsByLines);
                    }

                    var lines = c.Tree.GetText().Lines;
                    foreach (var statementsByLine in statementsByLines.Where(pair => pair.Value.Count > 1))
                    {
                        var location = CalculateLocationForLine(lines[statementsByLine.Key], c.Tree, statementsByLine.Value);
                        c.ReportIssue(SupportedDiagnostics[0], location);
                    }
                });

        private static Location CalculateLocationForLine(TextLine line, SyntaxTree tree, ICollection<TStatementSyntax> statements)
        {
            var lineSpan = line.Span;

            var min = statements.Min(st => lineSpan.Intersection(st.Span).Value.Start);
            var max = statements.Max(st => lineSpan.Intersection(st.Span).Value.End);

            return Location.Create(tree, TextSpan.FromBounds(min, max));
        }

        private void AddStatementToLineCache(TStatementSyntax statement, MultiValueDictionary<int, TStatementSyntax> statementsByLines)
        {
            var startLine = statement.GetLocation().StartLine();
            statementsByLines.AddWithKey(startLine, statement);

            var lastToken = statement.GetLastToken();
            var tokenBelonsTo = GetContainingStatement(lastToken);
            if (tokenBelonsTo == statement)
            {
                var endLine = statement.GetLocation().EndLine();
                statementsByLines.AddWithKey(endLine, statement);
            }
        }

        private TStatementSyntax GetContainingStatement(SyntaxToken token)
        {
            var node = token.Parent;
            var statement = node as TStatementSyntax;
            while (node != null
                && (statement == null || !StatementShouldBeExcluded(statement)))
            {
                node = node.Parent;
                statement = node as TStatementSyntax;
            }
            return statement;
        }
    }
}
