﻿/*
 * SonarAnalyzer for .NET
 * Copyright (C) 2014-2024 SonarSource SA
 * mailto:info AT sonarsource DOT com
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the Sonar Source-Available License Version 1, as published by SonarSource SA.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Sonar Source-Available License for more details.
 *
 * You should have received a copy of the Sonar Source-Available License
 * along with this program; if not, see https://sonarsource.com/license/ssal/
 */

using Microsoft.CodeAnalysis.Text;
using SonarAnalyzer.CFG.Sonar;
using SonarAnalyzer.CSharp.Core.LiveVariableAnalysis;
using SonarAnalyzer.CSharp.Walkers;

namespace SonarAnalyzer.Rules.CSharp
{
    public partial class DeadStores : SonarDiagnosticAnalyzer
    {
        private class SonarChecker : CheckerBase<IControlFlowGraph, Block>
        {
            private readonly SyntaxNode node;

            public SonarChecker(SonarSyntaxNodeReportingContext context, SonarCSharpLiveVariableAnalysis lva, SyntaxNode node) : base(context, lva) =>
                this.node = node;

            protected override State CreateState(Block block) =>
                new SonarState(this, block, node);

            private class SonarState : State
            {
                private readonly ISet<SyntaxNode> assignmentLhs = new HashSet<SyntaxNode>();
                private readonly SyntaxNode node;

                public SonarState(SonarChecker owner, Block block, SyntaxNode node) : base(owner, block) =>
                    this.node = node;

                public override void AnalyzeBlock()
                {
                    foreach (var instruction in block.Instructions.Reverse())
                    {
                        switch (instruction.Kind())
                        {
                            case SyntaxKind.IdentifierName:
                                ProcessIdentifier(instruction);
                                break;

                            case SyntaxKind.AddAssignmentExpression:
                            case SyntaxKind.SubtractAssignmentExpression:
                            case SyntaxKind.MultiplyAssignmentExpression:
                            case SyntaxKind.DivideAssignmentExpression:
                            case SyntaxKind.ModuloAssignmentExpression:
                            case SyntaxKind.AndAssignmentExpression:
                            case SyntaxKind.ExclusiveOrAssignmentExpression:
                            case SyntaxKind.OrAssignmentExpression:
                            case SyntaxKind.LeftShiftAssignmentExpression:
                            case SyntaxKind.RightShiftAssignmentExpression:
                            case SyntaxKindEx.CoalesceAssignmentExpression:
                                ProcessOpAssignment(instruction);
                                break;

                            case SyntaxKind.SimpleAssignmentExpression:
                                ProcessSimpleAssignment(instruction);
                                break;

                            case SyntaxKind.VariableDeclarator:
                                ProcessVariableDeclarator(instruction);
                                break;

                            case SyntaxKind.PreIncrementExpression:
                            case SyntaxKind.PreDecrementExpression:
                                ProcessPrefixExpression(instruction);
                                break;

                            case SyntaxKind.PostIncrementExpression:
                            case SyntaxKind.PostDecrementExpression:
                                ProcessPostfixExpression(instruction);
                                break;
                        }
                    }
                }

                private void ProcessIdentifier(SyntaxNode instruction)
                {
                    var identifier = (IdentifierNameSyntax)instruction;
                    var symbol = SemanticModel.GetSymbolInfo(identifier).Symbol;
                    if (IsSymbolRelevant(symbol)
                        && !identifier.GetSelfOrTopParenthesizedExpression().IsInNameOfArgument(SemanticModel)
                        && IsLocal(symbol))
                    {
                        if (SonarCSharpLiveVariableAnalysis.IsOutArgument(identifier))
                        {
                            liveOut.Remove(symbol);
                        }
                        else if (!assignmentLhs.Contains(identifier))
                        {
                            liveOut.Add(symbol);
                        }
                    }
                }

                private void ProcessOpAssignment(SyntaxNode instruction)
                {
                    var assignment = (AssignmentExpressionSyntax)instruction;
                    var left = assignment.Left.RemoveParentheses();
                    if (IdentifierRelevantSymbol(left) is { } symbol)
                    {
                        ReportOnAssignment(assignment, left, symbol);
                    }
                }

                private void ProcessSimpleAssignment(SyntaxNode instruction)
                {
                    var assignment = (AssignmentExpressionSyntax)instruction;
                    var left = assignment.Left.RemoveParentheses();
                    if (IdentifierRelevantSymbol(left) is { } symbol)
                    {
                        ReportOnAssignment(assignment, left, symbol);
                        liveOut.Remove(symbol);
                    }
                }

                private void ProcessVariableDeclarator(SyntaxNode instruction)
                {
                    var declarator = (VariableDeclaratorSyntax)instruction;
                    if (SemanticModel.GetDeclaredSymbol(declarator) is ILocalSymbol symbol && IsSymbolRelevant(symbol))
                    {
                        if (declarator.Initializer != null
                            && !IsAllowedInitializationValue(declarator.Initializer.Value)
                            && !symbol.IsConst
                            && symbol.RefKind() == RefKind.None
                            && !liveOut.Contains(symbol)
                            && !IsUnusedLocal(symbol)
                            && !IsMuted(declarator, symbol))
                        {
                            var location = GetFirstLineLocationFromToken(declarator.Initializer.EqualsToken, declarator.Initializer);
                            ReportIssue(location, symbol);
                        }
                        liveOut.Remove(symbol);
                    }
                }

                private bool IsUnusedLocal(ISymbol declaredSymbol) =>
                    node.DescendantNodes()
                        .OfType<IdentifierNameSyntax>()
                        .SelectMany(x => VariableUnusedBase.GetUsedSymbols(x, SemanticModel))
                        .All(x => !x.Equals(declaredSymbol));

                private void ProcessPrefixExpression(SyntaxNode instruction)
                {
                    var prefixExpression = (PrefixUnaryExpressionSyntax)instruction;
                    var parent = prefixExpression.GetSelfOrTopParenthesizedExpression();
                    var operand = prefixExpression.Operand.RemoveParentheses();
                    if (parent.Parent is ExpressionStatementSyntax
                        && IdentifierRelevantSymbol(operand) is { } symbol
                        && IsLocal(symbol)
                        && !liveOut.Contains(symbol)
                        && !IsMuted(operand))
                    {
                        ReportIssue(prefixExpression.GetLocation(), symbol);
                    }
                }

                private void ProcessPostfixExpression(SyntaxNode instruction)
                {
                    var postfixExpression = (PostfixUnaryExpressionSyntax)instruction;
                    var operand = postfixExpression.Operand.RemoveParentheses();
                    if (IdentifierRelevantSymbol(operand) is { } symbol
                        && IsLocal(symbol)
                        && !liveOut.Contains(symbol)
                        && !IsMuted(operand))
                    {
                        ReportIssue(postfixExpression.GetLocation(), symbol);
                    }
                }

                private void ReportOnAssignment(AssignmentExpressionSyntax assignment, ExpressionSyntax left, ISymbol symbol)
                {
                    if (IsLocal(symbol)
                        && !liveOut.Contains(symbol)
                        && !IsMuted(left))
                    {
                        var location = GetFirstLineLocationFromToken(assignment.OperatorToken, assignment.Right);
                        ReportIssue(location, symbol);
                    }

                    assignmentLhs.Add(left);
                }

                private bool IsMuted(SyntaxNode node) =>
                    new MutedSyntaxWalker(SemanticModel, node).IsMuted();

                private static Location GetFirstLineLocationFromToken(SyntaxToken issueStartToken, SyntaxNode wholeIssue)
                {
                    var line = wholeIssue.SyntaxTree.GetText().Lines[issueStartToken.GetLocation().StartLine()];
                    var rightSingleLine = line.Span.Intersection(TextSpan.FromBounds(issueStartToken.SpanStart, wholeIssue.Span.End));
                    return Location.Create(wholeIssue.SyntaxTree, TextSpan.FromBounds(issueStartToken.SpanStart, rightSingleLine.HasValue ? rightSingleLine.Value.End : issueStartToken.Span.End));
                }

                private ISymbol IdentifierRelevantSymbol(SyntaxNode node) =>
                    node.IsKind(SyntaxKind.IdentifierName)
                    && SemanticModel.GetSymbolInfo(node).Symbol is { } symbol
                    && IsSymbolRelevant(symbol)
                    ? symbol
                    : null;
            }
        }
    }
}
