window.addEventListener("load", function() {
  // global variables
  // They must be reset to their initial values when back-forward cache kicks in.
  var FORM_SUBMITTED;

  function initializeGlobals() {
    FORM_SUBMITTED = false;
  }
  initializeGlobals();

  window.addEventListener("pageshow", function(e) {
    if (e.persisted) {
      initializeGlobals();
    }
  });

  function togglePasswordVisibility() {
    var pwd = document.querySelector("#password");
    if (pwd == null) {
      return;
    }
    if (pwd.type === "password") {
      pwd.type = "text";
    } else {
      pwd.type = "password";
    }
    var els = document.querySelectorAll(".password-visibility-btn");
    for (var i = 0; i < els.length; i++) {
      var el = els[i];
      if (el.classList.contains("show-password")) {
        if (pwd.type === "text") {
          el.style.display = "none";
        } else {
          el.style.display = "block";
        }
      }
      if (el.classList.contains("hide-password")) {
        if (pwd.type === "password") {
          el.style.display = "none";
        } else {
          el.style.display = "block";
        }
      }
    }
  }

  function attachPasswordVisibilityClick() {
    var els = document.querySelectorAll(".password-visibility-btn");
    for (var i = 0; i < els.length; i++) {
      var el = els[i];
      el.addEventListener("click", function(e) {
        e.preventDefault();
        e.stopPropagation();
        togglePasswordVisibility();
      });
    }
  }

  function attachBackButtonClick() {
    var els = document.querySelectorAll(".btn.back-btn");
    for (var i = 0; i < els.length; i++) {
      var el = els[i];
      el.addEventListener("click", function(e) {
        e.preventDefault();
        e.stopPropagation();
        window.history.back();
      });
    }
  }

  function checkPasswordLength(value, el) {
    if (el == null) {
      return;
    }
    var minLength = parseInt(el.getAttribute("data-min-length"), 10);
    // .length is number of UTF-16 code units,
    // while the server is counting number of UTF-8 code units.
    if (value.length >= minLength) {
      el.classList.add("passed");
    }
  }

  function checkPasswordUppercase(value, el) {
    if (el == null) {
      return;
    }
    if (/[A-Z]/.test(value)) {
      el.classList.add("passed");
    }
  }

  function checkPasswordLowercase(value, el) {
    if (el == null) {
      return;
    }
    if (/[a-z]/.test(value)) {
      el.classList.add("passed");
    }
  }

  function checkPasswordDigit(value, el) {
    if (el == null) {
      return;
    }
    if (/[0-9]/.test(value)) {
      el.classList.add("passed");
    }
  }

  function checkPasswordSymbol(value, el) {
    if (el == null) {
      return;
    }
    if (/[^a-zA-Z0-9]/.test(value)) {
      el.classList.add("passed");
    }
  }

  function attachPasswordPolicyCheck() {
    var el = document.querySelector("[data-password-policy-password]");
    if (el == null ) {
      return;
    }
    el.addEventListener("input", function(e) {
      var value = e.currentTarget.value;
      var els = document.querySelectorAll(".password-policy");
      for (var i = 0; i < els.length; ++i) {
        els[i].classList.remove("violated", "passed");
      }
      checkPasswordLength(value, document.querySelector(".password-policy.length"));
      checkPasswordUppercase(value, document.querySelector(".password-policy.uppercase"));
      checkPasswordLowercase(value, document.querySelector(".password-policy.lowercase"));
      checkPasswordDigit(value, document.querySelector(".password-policy.digit"));
      checkPasswordSymbol(value, document.querySelector(".password-policy.symbol"));
    });
  }

  function attachResendButtonBehavior() {
    var el = document.querySelector("#resend-button");
    if (el == null) {
      return;
    }


    var scheduledAt = new Date();
    var cooldown = parseInt(el.getAttribute("data-cooldown"), 10) * 1000;
    var label = el.getAttribute("data-label");
    var labelUnit = el.getAttribute("data-label-unit");

    function tick() {
      var now = new Date();
      var timeElapsed = now - scheduledAt;

      var displaySeconds = 0;
      if (timeElapsed <= cooldown) {
        displaySeconds = Math.round((cooldown - timeElapsed) / 1000);
      }

      if (displaySeconds === 0) {
        el.disabled = false;
        el.textContent = label;
      } else {
        el.disabled = true;
        el.textContent = labelUnit.replace("%d", String(displaySeconds));
        requestAnimationFrame(tick);
      }
    }

    requestAnimationFrame(tick);
  }

  // Disable all form submission if any form has been submitted once.
  function attachFormSubmitOnceOnly() {
    var els = document.querySelectorAll("form");
    for (var i = 0; i < els.length; ++i) {
      var form = els[i];
      form.addEventListener("submit", function(e) {
        if (!FORM_SUBMITTED) {
          FORM_SUBMITTED = true;
        } else {
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
        }
      });
    }
  }

  // Use XHR to submit form.
  // If we rely on the browser to submit the form for us,
  // error submission will add an entry to the history stack,
  // causing back button fail to work intuitively.
  //
  // Therefore, when JavaScript is available,
  // we use XHR to submit the form.
  // XHR follows redirect automatically
  // and .responseURL is GET URL we need to visit to retrieve the submission result.
  // If window.location.href is assigned the same value, no extra entry is added to the history stack.
  function attachFormSubmitXHR() {
    var els = document.querySelectorAll("form");
    for (var i = 0; i < els.length; ++i) {
      els[i].addEventListener("submit", function(e) {

        var shouldIgnored = false;

        var form = e.currentTarget;
        // e.submitter is not supported by Safari
        // therefore we must not have multiple submit buttons per form.

        // https://html.spec.whatwg.org/multipage/form-control-infrastructure.html#constructing-form-data-set
        var entryList = [];
        for (var j = 0; j < form.elements.length; ++j) {
          var field = form.elements[j];

          if (field.getAttribute("data-form-xhr") === "false") {
            shouldIgnored = true;
          }

          // Step 5.1 Point 1 is ignored because we do not use datalist.

          // Step 5.1 Point 2
          if (field.disabled) {
            continue;
          }

          // Step 5.1 Point 3
          // if (field instanceof HTMLButtonElement && field !== submitter) {
          //   continue;
          // }
          // Step 5.1 Point 3
          // if (field instanceof HTMLInputElement && field.type === "submit" && field !== submitter) {
          //   continue;
          // }

          // Step 5.1 Point 4
          if (field instanceof HTMLInputElement && field.type === "checkbox" && !field.checked) {
            continue;
          }

          // Step 5.1 Point 5
          if (field instanceof HTMLInputElement && field.type === "radio" && !field.checked) {
            continue;
          }

          // Step 5.1 Point 6; It deviates from the spec because we do not use <object>.
          if (field instanceof HTMLObjectElement) {
            continue;
          }

          // Step 5.2; It deviates from the spec becaues we do not use <input type="image">.
          if (field instanceof HTMLInputElement && field.type === "image") {
            continue;
          }

          // Step 5.3 is ignored because we do not use form-associated custom element.

          // Step 5.4
          if (field.name === "" || field.name == null) {
            continue;
          }

          // Step 5.5
          var name = field.name;
          var value = field.value;

          // TODO(form-submission): Step 5.6 <select>

          // Step 5.7
          if (field instanceof HTMLInputElement && (field.type === "checkbox" || field.type === "radio")) {
            if (field.value == null || field.value === "") {
              value = "on";
            }
          }

          // Step 5.8 is ignored because we do not use file upload.
          // Step 5.9 is ignored because we do not use <object>.
          // Step 5.10 is ignored because we do ot use <input type="hidden" name="_charset_">.
          // TODO(form-submission): Step 5.11 <textarea>

          // Step 5.12
          entryList.push([name, value]);

          // Step 5.13 is ignored because we do nto use dirname.
        }

        // Ignore any form containing elements with "data-form-xhr"
        // Such forms will redirect to external location
        // so CORS will kick in and XHR does not work.
        if (shouldIgnored) {
          return;
        }

        e.preventDefault();
        e.stopPropagation();

        var body = new URLSearchParams();
        for (var i = 0; i < entryList.length; ++i) {
          var entry = entryList[i];
          body.append(entry[0], entry[1]);
        }

        var xhr = new XMLHttpRequest();
        xhr.withCredentials = true;
        xhr.onload = function(e) {
          window.location.href = xhr.responseURL;
        };
        xhr.open(form.method, form.action, true);
        // Safari does not support xhr.send(URLSearchParams)
        // so we have to manually set content-type
        // and serialize URLSearchParams to string.
        xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded;charset=UTF-8");
        xhr.send(body.toString());
      });
    }
  }

  attachPasswordVisibilityClick();
  attachBackButtonClick();
  attachPasswordPolicyCheck();
  attachResendButtonBehavior();
  attachFormSubmitOnceOnly();
  attachFormSubmitXHR();
});
