package mail

import (
	"context"
	"errors"

	"github.com/go-gomail/gomail"

	"github.com/skygeario/skygear-server/pkg/core/config"
	"github.com/skygeario/skygear-server/pkg/core/intl"
)

var ErrMissingSMTPConfiguration = errors.New("mail: configuration is missing")

type senderImpl struct {
	LocalizationConfiguration *config.LocalizationConfiguration
	GomailDialer              *gomail.Dialer
	Context                   context.Context
}

func NewSender(ctx context.Context, c *config.TenantConfiguration) Sender {
	var dialer *gomail.Dialer
	smtp := c.AppConfig.SMTP
	if smtp != nil && smtp.IsValid() {
		dialer = gomail.NewPlainDialer(smtp.Host, smtp.Port, smtp.Login, smtp.Password)
	}
	switch smtp.Mode {
	case config.SMTPModeNormal:
		// gomail will infer according to port
	case config.SMTPModeSSL:
		dialer.SSL = true
	}
	return &senderImpl{
		LocalizationConfiguration: c.AppConfig.Localization,
		GomailDialer:              dialer,
		Context:                   ctx,
	}
}

type updateGomailMessageFunc func(opts *SendOptions, msg *gomail.Message) error

func (s *senderImpl) Send(opts SendOptions) (err error) {
	if s.GomailDialer == nil {
		err = ErrMissingSMTPConfiguration
		return
	}

	message := gomail.NewMessage()

	funcs := []updateGomailMessageFunc{
		s.applyFrom,
		applyTo,
		s.applyReplyTo,
		s.applySubject,
		applyTextBody,
		applyHTMLBody,
	}

	for _, f := range funcs {
		if err = f(&opts, message); err != nil {
			return
		}
	}

	err = s.GomailDialer.DialAndSend(message)
	if err != nil {
		return err
	}

	return nil
}

func (s *senderImpl) applyFrom(opts *SendOptions, message *gomail.Message) error {
	tags := intl.GetPreferredLanguageTags(s.Context)
	sender := intl.LocalizeStringMap(tags, intl.Fallback(s.LocalizationConfiguration.FallbackLanguage), opts.MessageConfig, "sender")
	if sender == "" {
		return errors.New("mail: sender address is missing")
	}
	message.SetHeader("From", sender)
	return nil
}

func applyTo(opts *SendOptions, message *gomail.Message) error {
	if opts.Recipient == "" {
		return errors.New("mail: recipient address is missing")
	}

	message.SetHeader("To", opts.Recipient)
	return nil
}

func (s *senderImpl) applyReplyTo(opts *SendOptions, message *gomail.Message) error {
	tags := intl.GetPreferredLanguageTags(s.Context)
	replyTo := intl.LocalizeStringMap(tags, intl.Fallback(s.LocalizationConfiguration.FallbackLanguage), opts.MessageConfig, "reply_to")
	if replyTo == "" {
		return nil
	}

	message.SetHeader("Reply-To", replyTo)
	return nil
}

func (s *senderImpl) applySubject(opts *SendOptions, message *gomail.Message) error {
	tags := intl.GetPreferredLanguageTags(s.Context)
	subject := intl.LocalizeStringMap(tags, intl.Fallback(s.LocalizationConfiguration.FallbackLanguage), opts.MessageConfig, "subject")
	if subject == "" {
		return errors.New("mail: subject is missing")
	}

	message.SetHeader("Subject", subject)
	return nil
}

func applyTextBody(opts *SendOptions, message *gomail.Message) error {
	if opts.TextBody == "" {
		return errors.New("mail: text body is missing")
	}

	message.SetBody("text/plain", opts.TextBody)
	return nil
}

func applyHTMLBody(opts *SendOptions, message *gomail.Message) error {
	if opts.HTMLBody == "" {
		return nil
	}

	message.AddAlternative("text/html", opts.HTMLBody)
	return nil
}
