package crypto

import (
	"crypto/ecdsa"
	"crypto/x509"
	"encoding/pem"
	"errors"
)

// ErrNotPEMEncoded occurs when the content is not PEM encoded.
var ErrNotPEMEncoded = errors.New("not pem encoded")

// ErrNotECDSAPrivateKey occurs when the private key is not a ECDSA one.
var ErrNotECDSAPrivateKey = errors.New("not ECDSA private key")

// ParseAppleP8PrivateKey parses the .p8 private key generated by Apple.
func ParseAppleP8PrivateKey(content []byte) (privateKey *ecdsa.PrivateKey, err error) {
	block, _ := pem.Decode(content)
	if block == nil {
		err = ErrNotPEMEncoded
		return
	}

	key, err := x509.ParsePKCS8PrivateKey(block.Bytes)
	if err != nil {
		return
	}

	privateKey, ok := key.(*ecdsa.PrivateKey)
	if !ok {
		err = ErrNotECDSAPrivateKey
		return
	}

	return
}
