// Copyright 2015-present Oursky Ltd.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package authinfo

import (
	"time"

	"github.com/skygeario/skygear-server/pkg/core/authn"
	"github.com/skygeario/skygear-server/pkg/core/uuid"
)

// AuthInfo contains a user's information for authentication purpose
type AuthInfo struct {
	ID               string          `json:"_id"`
	LastSeenAt       *time.Time      `json:"last_seen_at,omitempty"`
	LastLoginAt      *time.Time      `json:"last_login_at,omitempty"`
	Disabled         bool            `json:"disabled"`
	DisabledMessage  string          `json:"disabled_message,omitempty"`
	DisabledExpiry   *time.Time      `json:"disabled_expiry,omitempty"`
	ManuallyVerified bool            `json:"manually_verified,omitempty"`
	Verified         bool            `json:"verified,omitempty"`
	VerifyInfo       map[string]bool `json:"verify_info,omitempty"`
}

// NewAuthInfo returns a new AuthInfo with specified password.
// An UUID4 ID will be generated by the system as unique identifier
func NewAuthInfo() AuthInfo {
	info := AuthInfo{
		ID:         uuid.New(),
		VerifyInfo: map[string]bool{},
	}

	return info
}

// IsVerified returns true if the user is verified.
func (info *AuthInfo) IsVerified() bool {
	return info.Verified || info.ManuallyVerified
}

// IsDisabled returns true if the user is disabled.
//
// This function checks whether the user is disabled by also considering the
// expiry of disabling the user account. Use this function instead of using
// the Disabled field.
func (info *AuthInfo) IsDisabled(now time.Time) bool {
	if info.Disabled {
		if expiry := info.DisabledExpiry; expiry != nil {
			return expiry.After(now)
		}
		return true
	}
	return false
}

// RefreshDisabledStatus set the auth info as not disabled.
func (info *AuthInfo) RefreshDisabledStatus(now time.Time) {
	info.Disabled = info.IsDisabled(now)
	if !info.Disabled {
		info.DisabledMessage = ""
		info.DisabledExpiry = nil
	}
}

func (info *AuthInfo) ToUserInfo(now time.Time) *authn.UserInfo {
	return &authn.UserInfo{
		ID:         info.ID,
		IsDisabled: info.IsDisabled(now),
		IsVerified: info.IsVerified(),
	}
}
