/*
 * Copyright 2012-2021 MultiMC Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.polymc.applet;

import net.minecraft.Launcher;

import javax.imageio.ImageIO;
import java.applet.Applet;
import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class LegacyFrame extends Frame {

    private static final Logger LOGGER = Logger.getLogger("LegacyFrame");

    private final Launcher appletWrap;

    public LegacyFrame(String title, Applet mcApplet) {
        super(title);

        appletWrap = new Launcher(mcApplet);

        mcApplet.setStub(appletWrap);

        try {
            setIconImage(ImageIO.read(new File("icon.png")));
        } catch (IOException e) {
            LOGGER.log(Level.WARNING, "Unable to read Minecraft icon!", e);
        }

        addWindowListener(new ForceExitHandler());
    }

    public void start (
        String user,
        String session,
        int winSizeW,
        int winSizeH,
        boolean maximize,
        String serverAddress,
        String serverPort,
        boolean isDemo
    ) {
        // Implements support for launching in to multiplayer on classic servers using a mpticket
        // file generated by an external program and stored in the instance's root folder.

        Path mpticketFile =
                Paths.get(System.getProperty("user.dir"), "..", "mpticket");

        Path mpticketFileCorrupt =
                Paths.get(System.getProperty("user.dir"), "..", "mpticket.corrupt");

        if (Files.exists(mpticketFile)) {
            try {
                List<String> lines = Files.readAllLines(mpticketFile, StandardCharsets.UTF_8);

                if (lines.size() < 3) {
                    Files.move(
                            mpticketFile,
                            mpticketFileCorrupt,
                            StandardCopyOption.REPLACE_EXISTING
                    );

                    LOGGER.warning("Mpticket file is corrupted!");
                } else {
                    // Assumes parameters are valid and in the correct order
                    appletWrap.setParameter("server", lines.get(0));
                    appletWrap.setParameter("port", lines.get(1));
                    appletWrap.setParameter("mppass", lines.get(2));
                }
            } catch (IOException e) {
                LOGGER.log(Level.WARNING, "Unable to read mpticket file!", e);
            }
        }

        if (serverAddress != null) {
            appletWrap.setParameter("server", serverAddress);
            appletWrap.setParameter("port", serverPort);
        }

        appletWrap.setParameter("username", user);
        appletWrap.setParameter("sessionid", session);
        appletWrap.setParameter("stand-alone", "true"); // Show the quit button.
        appletWrap.setParameter("haspaid", "true"); // Some old versions need this for world saves to work.
        appletWrap.setParameter("demo", isDemo ? "true" : "false");
        appletWrap.setParameter("fullscreen", "false");

        add(appletWrap);

        appletWrap.setPreferredSize(new Dimension(winSizeW, winSizeH));

        pack();

        setLocationRelativeTo(null);
        setResizable(true);

        if (maximize)
            this.setExtendedState(MAXIMIZED_BOTH);

        validate();

        appletWrap.init();
        appletWrap.start();

        setVisible(true);
    }

    private final class ForceExitHandler extends WindowAdapter {

        @Override
        public void windowClosing(WindowEvent e) {
            new Thread(new Runnable() {
                @Override
                public void run() {
                    try {
                        Thread.sleep(30000L);
                    } catch (InterruptedException localInterruptedException) {
                        localInterruptedException.printStackTrace();
                    }

                    LOGGER.info("Forcing exit!");

                    System.exit(0);
                }
            }).start();

            if (appletWrap != null) {
                appletWrap.stop();
                appletWrap.destroy();
            }

            // old minecraft versions can hang without this >_<
            System.exit(0);
        }

    }

}
