<?php

declare(strict_types=1);

/*
 * This file is part of Part-DB (https://github.com/Part-DB/Part-DB-symfony).
 *
 *  Copyright (C) 2019 - 2023 Jan Böhmer (https://github.com/jbtronics)
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published
 *  by the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
namespace App\Tests\Validator\Constraints;

use App\Entity\Attachments\AttachmentType;
use App\Entity\Base\AbstractStructuralDBElement;
use App\Validator\Constraints\NoneOfItsChildren;
use App\Validator\Constraints\NoneOfItsChildrenValidator;
use Symfony\Component\Validator\Test\ConstraintValidatorTestCase;

class NoneOfItsChildrenValidatorTest extends ConstraintValidatorTestCase
{

    protected AttachmentType $root_node;
    protected AttachmentType $child1;
    protected AttachmentType $child2;
    protected AttachmentType $child3;
    protected AttachmentType $child1_1;
    protected AttachmentType $child1_2;

    protected function setUp(): void
    {
        // TODO: Change the autogenerated stub

        parent::setUp();

        //Build a simple hierachy
        $this->root_node = new AttachmentType();
        $this->root_node->setName('root')->setParent(null);
        $this->child1 = new AttachmentType();
        $this->child1->setParent($this->root_node)->setName('child1');
        $this->child2 = new AttachmentType();
        $this->child2->setName('child2')->setParent($this->root_node);
        $this->child3 = new AttachmentType();
        $this->child3->setName('child3')->setParent($this->root_node);
        $this->child1_1 = new AttachmentType();
        $this->child1_1->setName('child1_1')->setParent($this->child1);
        $this->child1_2 = new AttachmentType();
        $this->child1_2->setName('child1_2')->setParent($this->child1);
    }


    protected function createValidator(): NoneOfItsChildrenValidator
    {
        return new NoneOfItsChildrenValidator();
    }

    public function testNullIsValid(): void
    {
        $this->setObject($this->child1);
        $this->validator->validate(null, new NoneOfItsChildren());
        $this->assertNoViolation();
    }

    public function testWithUnrelatedObject(): void
    {
        $this->setObject($this->child1);
        $this->validator->validate(new AttachmentType(), new NoneOfItsChildren());
        $this->assertNoViolation();
    }

    public function testWithParentObject(): void
    {
        $this->setObject($this->child1);
        $this->validator->validate($this->root_node, new NoneOfItsChildren());
        $this->assertNoViolation();
    }

    public function testWithIntermediateChild(): void
    {
        $this->setObject($this->child1);
        $this->validator->validate($this->child1_1, new NoneOfItsChildren());
        $this->buildViolation('validator.noneofitschild.children')
            ->assertRaised();
    }

    public function testWithIndirectChild(): void
    {
        $this->setObject($this->root_node);
        $this->validator->validate($this->child1_1, new NoneOfItsChildren());
        $this->buildViolation('validator.noneofitschild.children')
            ->assertRaised();
    }

    public function testWithSelfInstance(): void
    {
        $this->setObject($this->root_node);
        $this->validator->validate($this->root_node, new NoneOfItsChildren());
        $this->buildViolation('validator.noneofitschild.self')
            ->assertRaised();
    }

    public function testWithSelfByID(): void
    {
        $obj1 = new class extends AbstractStructuralDBElement {
            public function __construct()
            {
                $this->id = 1;
                parent::__construct();
            }
        };

        $obj2 = new class extends AbstractStructuralDBElement {
            public function __construct()
            {
                $this->id = 1;
                parent::__construct();
            }
        };

        $this->setObject($obj1);
        $this->validator->validate($obj2, new NoneOfItsChildren());
        $this->buildViolation('validator.noneofitschild.self')
            ->assertRaised();

    }
}
