<?php
/*
 * This file is part of Part-DB (https://github.com/Part-DB/Part-DB-symfony).
 *
 *  Copyright (C) 2019 - 2023 Jan Böhmer (https://github.com/jbtronics)
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published
 *  by the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

declare(strict_types=1);


namespace App\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\WebpackEncoreBundle\Event\RenderAssetTagEvent;

/**
 * This class fixes the wrong pathes generated by webpack using the auto publicPath mode.
 * Basically it replaces the wrong /auto/ part of the path with the correct /build/ in all encore entrypoints.
 */
class WebpackAutoPathSubscriber implements EventSubscriberInterface
{
    public static function getSubscribedEvents(): array
    {
        return [
            RenderAssetTagEvent::class => 'onRenderAssetTag'
        ];
    }

    public function onRenderAssetTag(RenderAssetTagEvent $event): void
    {
        if ($event->isScriptTag()) {
            $event->setAttribute('src', $this->resolveAuto($event->getUrl()));
        }
        if ($event->isLinkTag()) {
            $event->setAttribute('href', $this->resolveAuto($event->getUrl()));
        }
    }

    private function resolveAuto(string $path): string
    {
        //Replace the first occurence of /auto/ with /build/ to get the correct path
        return preg_replace('/\/auto\//', '/build/', $path, 1);
    }
}