/* Copyright (c) 2018 Anakin Authors, Inc. All Rights Reserved.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

#ifndef ANAKIN_SABER_FUNCS_ONE_HOT_H
#define ANAKIN_SABER_FUNCS_ONE_HOT_H

#include "saber/funcs/base.h"
#include "saber/funcs/impl/impl_base.h"
#include "saber/saber_funcs_param.h"
#include "saber/funcs/impl/impl_one_hot.h"

#ifdef USE_CUDA
#include "saber/funcs/impl/cuda/saber_one_hot.h"
#endif

#ifdef USE_X86_PLACE
#include "saber/funcs/impl/x86/saber_one_hot.h"
#endif

namespace anakin {
namespace saber {

template<typename TargetType,
        DataType OpDtype
>
class OneHot : public BaseFunc<
        TargetType,
        OpDtype,
        ImplBase,
        OneHotParam> {
public:
    using BaseFunc<
            TargetType,
            OpDtype,
            ImplBase,
            OneHotParam>::BaseFunc;

    OneHot() = default;

    virtual SaberStatus compute_output_shape(
            const std::vector<Tensor<TargetType>*> &input,
            std::vector<Tensor<TargetType>*> &output,
            OneHotParam<TargetType> &param) override {

        CHECK_GE(input[0]->dims(), 2) << "Rank should greater than 1 ";
        CHECK_EQ(input[0]->valid_shape()[input[0]->dims() - 1], 1)
                << "last dim must be 1!!";

        int depth = param.depth;

        CHECK_GT(depth, 0) << "depth should greater than 0";

        Shape out_shape = input[0]->valid_shape();
        out_shape[out_shape.dims() - 1] = depth;

        output[0]->set_seq_offset(input[0]->get_seq_offset());
        return output[0]->set_shape_without_layout(out_shape);
    }

    virtual SaberStatus init_impl(ImplEnum implenum) override {
        switch (implenum) {
            case VENDER_IMPL:
                this->_impl.push_back(new VenderOneHot <TargetType,
                        OpDtype>);
                return SaberSuccess;
            case SABER_IMPL:
                this->_impl.push_back(new SaberOneHot <TargetType,
                        OpDtype>);
                return SaberSuccess;
            default:
                return SaberUnImplError;
        }
    }
private:

    virtual void pick_best_static() override {
        if (true) // some condition?
            this->_best_impl = this->_impl[0];
    }

    virtual void pick_best_specify(ImplEnum implenum) override {
        this->_best_impl = this->_impl[0];
    }

};
}
}
#endif
