﻿// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

// System
using System;

namespace ElasticScaleStarterKit
{
    /// <summary>
    /// ConsoleUtils class
    /// </summary>
    internal static class ConsoleUtils
    {
        /// <summary>
        /// Writes detailed information to the console.
        /// </summary>
        public static void WriteInfo(string format, params object[] args)
        {
            WriteColor(ConsoleColor.DarkGray, "\t" + format, args);
        }

        /// <summary>
        /// Writes warning text to the console.
        /// </summary>
        public static void WriteWarning(string format, params object[] args)
        {
            WriteColor(ConsoleColor.Yellow, "\t" + format, args);
        }

        /// <summary>
        /// Writes colored text to the console.
        /// </summary>
        public static void WriteColor(ConsoleColor color, string format, params object[] args)
        {
            ConsoleColor oldColor = Console.ForegroundColor;
            Console.ForegroundColor = color;
            Console.WriteLine(format, args);
            Console.ForegroundColor = oldColor;
        }

        /// <summary>
        /// Reads an integer from the console.
        /// </summary>
        public static int ReadIntegerInput(string prompt)
        {
            return ReadIntegerInput(prompt, allowNull: false).Value;
        }

        /// <summary>
        /// Reads an integer from the console, or returns null if the user enters nothing and allowNull is true.
        /// </summary>
        public static int? ReadIntegerInput(string prompt, bool allowNull)
        {
            while (true)
            {
                Console.Write(prompt);
                string line = Console.ReadLine();

                if (string.IsNullOrWhiteSpace(line) && allowNull)
                {
                    return null;
                }

                int inputValue;
                if (int.TryParse(line, out inputValue))
                {
                    return inputValue;
                }
            }
        }

        /// <summary>
        /// Reads an integer from the console.
        /// </summary>
        public static int ReadIntegerInput(string prompt, int defaultValue, Func<int, bool> validator)
        {
            while (true)
            {
                int? input = ReadIntegerInput(prompt, allowNull: true);

                if (!input.HasValue)
                {
                    // No input, so return default
                    return defaultValue;
                }
                else
                {
                    // Input was provided, so validate it
                    if (validator(input.Value))
                    {
                        // Validation passed, so return
                        return input.Value;
                    }
                }
            }
        }
    }
}
