/*
 * Decompiled with CFR 0.152.
 */
package jpsxdec.cdreaders;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.Arrays;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import jpsxdec.cdreaders.CdSector;
import jpsxdec.cdreaders.CdSector2048;
import jpsxdec.cdreaders.CdSector2336;
import jpsxdec.cdreaders.CdSector2352;
import jpsxdec.cdreaders.CdSectorHeader;
import jpsxdec.cdreaders.DiscPatcher;
import jpsxdec.cdreaders.XaAnalysis;
import jpsxdec.i18n.I;
import jpsxdec.i18n.ILocalizedMessage;
import jpsxdec.i18n.exception.LocalizedDeserializationFail;
import jpsxdec.i18n.log.ProgressLogger;
import jpsxdec.util.IO;
import jpsxdec.util.Misc;
import jpsxdec.util.TaskCanceledException;

public class CdFileSectorReader
implements Closeable {
    private static final Logger LOG = Logger.getLogger(CdFileSectorReader.class.getName());
    private static final int DEFAULT_SECTOR_BUFFER_COUNT = 16;
    @Nonnull
    private RandomAccessFile _inputFile;
    @Nonnull
    private final File _discImageFile;
    @Nonnull
    private final SectorFactory _sectorFactory;
    private final int _iSectorCount;
    private int _iCachedSectorStart;
    private int _iSectorsToCache;
    @CheckForNull
    private byte[] _abBulkReadCache;
    private long _lngCacheFileOffset;
    @CheckForNull
    private DiscPatcher _patcher;
    public static final String SERIALIZATION_START = "Filename:";
    private static final String DESERIALIZATION = "Filename:([^|]+)\\|Sector size:(\\d+)\\|Sector count:(\\d+)\\|First sector offset:(\\d+)";
    private static final String SERIALIZATION = "Filename:%s|Sector size:%d|Sector count:%d|First sector offset:%d";

    public static CdFileSectorReader open(@Nonnull String sDiscImageFileName) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        return CdFileSectorReader.open(new File(sDiscImageFileName));
    }

    public static CdFileSectorReader open(@Nonnull File discImageFile) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        return new CdFileSectorReader(discImageFile);
    }

    public static CdFileSectorReader open(@Nonnull File discImageFile, boolean blnAllowWrites) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        return new CdFileSectorReader(discImageFile, blnAllowWrites);
    }

    public static CdFileSectorReader open(@Nonnull File discImageFile, boolean blnAllowWrites, int iSectorsToBuffer) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        return new CdFileSectorReader(discImageFile, blnAllowWrites, iSectorsToBuffer);
    }

    public static CdFileSectorReader openWithSectorSize(@Nonnull File discImageFile, int iSectorSize) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        return new CdFileSectorReader(discImageFile, iSectorSize);
    }

    public static CdFileSectorReader openWithSectorSize(@Nonnull File discImageFile, int iSectorSize, boolean blnAllowWrites, int iSectorsToBuffer) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        return new CdFileSectorReader(discImageFile, iSectorSize, blnAllowWrites, iSectorsToBuffer);
    }

    public static CdFileSectorReader deserialize(@Nonnull String sSerialization, boolean blnAllowWrites) throws LocalizedDeserializationFail, CdFileNotFoundException, CdReadException {
        return new CdFileSectorReader(sSerialization, blnAllowWrites);
    }

    public static CdFileSectorReader deserialize(@Nonnull String sSerialization, boolean blnAllowWrites, int iSectorsToBuffer) throws LocalizedDeserializationFail, CdFileNotFoundException, CdReadException {
        return new CdFileSectorReader(sSerialization, blnAllowWrites, iSectorsToBuffer);
    }

    private CdFileSectorReader(@Nonnull File discImageFile) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        this(discImageFile, false, 16);
    }

    private CdFileSectorReader(@Nonnull File discImageFile, boolean blnAllowWrites) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        this(discImageFile, blnAllowWrites, 16);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CdFileSectorReader(@Nonnull File discImageFile, boolean blnAllowWrites, int iSectorsToBuffer) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        LOG.info(discImageFile.getPath());
        this._discImageFile = discImageFile;
        this._iSectorsToCache = iSectorsToBuffer;
        try {
            this._inputFile = new RandomAccessFile(discImageFile, blnAllowWrites ? "rw" : "r");
        }
        catch (FileNotFoundException ex) {
            throw new CdFileNotFoundException(discImageFile, ex);
        }
        boolean blnExceptionThrown = true;
        try {
            SectorFactory factory;
            try {
                try {
                    LOG.info("Attempting to identify as 2352/2448");
                    factory = new Cd2352or2448Factory(this._inputFile, true, true);
                    LOG.log(Level.INFO, "Disc type identified as {0,number,#}", factory.getRawSectorSize());
                }
                catch (FileTooSmallToIdentifyException ex) {
                    try {
                        LOG.info("Attempting to identify as 2336");
                        factory = new Cd2336Factory(this._inputFile);
                        LOG.info("Disc type identified as 2336");
                    }
                    catch (FileTooSmallToIdentifyException ex1) {
                        LOG.info("Unknown disc type, assuming 2048");
                        long lngFileSize = this._inputFile.length();
                        if (lngFileSize < 2048L) {
                            this._inputFile.close();
                            throw new FileTooSmallToIdentifyException(lngFileSize);
                        }
                        factory = new Cd2048Factory();
                    }
                }
            }
            catch (IOException ex) {
                throw new CdReadException(discImageFile, ex);
            }
            this._sectorFactory = factory;
            this._iSectorCount = this.calculateSectorCount();
            blnExceptionThrown = false;
        }
        finally {
            if (blnExceptionThrown) {
                IO.closeSilently(this._inputFile, LOG);
            }
        }
        if (this._sectorFactory.get1stSectorOffset() != 0L) {
            LOG.log(Level.WARNING, "First CD sector starts at offset {0}", this._sectorFactory.get1stSectorOffset());
        }
    }

    private CdFileSectorReader(@Nonnull File discImageFile, int iSectorSize) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        this(discImageFile, iSectorSize, false, 16);
    }

    private CdFileSectorReader(@Nonnull File discImageFile, int iSectorSize, boolean blnAllowWrites, int iSectorsToBuffer) throws CdFileNotFoundException, FileTooSmallToIdentifyException, CdReadException {
        LOG.info(discImageFile.getPath());
        this._discImageFile = discImageFile;
        this._iSectorsToCache = iSectorsToBuffer;
        try {
            this._inputFile = new RandomAccessFile(discImageFile, blnAllowWrites ? "rw" : "r");
        }
        catch (FileNotFoundException ex) {
            throw new CdFileNotFoundException(discImageFile, ex);
        }
        boolean blnExceptionThrown = true;
        try {
            switch (iSectorSize) {
                case 2048: {
                    this._sectorFactory = new Cd2048Factory();
                    break;
                }
                case 2336: {
                    this._sectorFactory = new Cd2336Factory(this._inputFile);
                    break;
                }
                case 2352: {
                    this._sectorFactory = new Cd2352or2448Factory(this._inputFile, true, false);
                    break;
                }
                case 2448: {
                    this._sectorFactory = new Cd2352or2448Factory(this._inputFile, false, true);
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Invalid sector size to open disc image as " + iSectorSize);
                }
            }
            blnExceptionThrown = false;
        }
        catch (IOException ex) {
            throw new CdReadException(discImageFile, ex);
        }
        finally {
            if (blnExceptionThrown) {
                IO.closeSilently(this._inputFile, LOG);
            }
        }
        this._iSectorCount = this.calculateSectorCount();
        if (this._sectorFactory.get1stSectorOffset() != 0L) {
            LOG.log(Level.WARNING, "First CD sector starts at offset {0}", this._sectorFactory.get1stSectorOffset());
        }
    }

    private CdFileSectorReader(@Nonnull String sSerialization, boolean blnAllowWrites) throws LocalizedDeserializationFail, CdFileNotFoundException, CdReadException {
        this(sSerialization, blnAllowWrites, 16);
    }

    private CdFileSectorReader(@Nonnull String sSerialization, boolean blnAllowWrites, int iSectorsToBuffer) throws LocalizedDeserializationFail, CdFileNotFoundException, CdReadException {
        String[] asValues = Misc.regex(DESERIALIZATION, sSerialization);
        if (asValues == null || asValues.length != 5) {
            throw new LocalizedDeserializationFail(I.CD_DESERIALIZE_FAIL(sSerialization));
        }
        try {
            this._iSectorCount = Integer.parseInt(asValues[3]);
            long lngStartOffset = Long.parseLong(asValues[4]);
            int iSectorSize = Integer.parseInt(asValues[2]);
            switch (iSectorSize) {
                case 2048: {
                    this._sectorFactory = new Cd2048Factory(lngStartOffset);
                    break;
                }
                case 2336: {
                    this._sectorFactory = new Cd2336Factory(lngStartOffset);
                    break;
                }
                case 2352: {
                    this._sectorFactory = new Cd2352or2448Factory(true, lngStartOffset);
                    break;
                }
                case 2448: {
                    this._sectorFactory = new Cd2352or2448Factory(false, lngStartOffset);
                    break;
                }
                default: {
                    throw new LocalizedDeserializationFail(I.CD_DESERIALIZE_FAIL(sSerialization));
                }
            }
        }
        catch (NumberFormatException ex) {
            throw new LocalizedDeserializationFail(I.CD_DESERIALIZE_FAIL(sSerialization), (Throwable)ex);
        }
        this._discImageFile = new File(asValues[1]);
        try {
            this._inputFile = new RandomAccessFile(this._discImageFile, blnAllowWrites ? "rw" : "r");
        }
        catch (FileNotFoundException ex) {
            throw new CdFileNotFoundException(this._discImageFile, ex);
        }
        this._iSectorsToCache = iSectorsToBuffer;
        int iActualSectorCount = this.calculateSectorCount();
        if (this._iSectorCount != iActualSectorCount) {
            IO.closeSilently(this._inputFile, LOG);
            throw new LocalizedDeserializationFail(I.SECTOR_COUNT_MISMATCH(this._iSectorCount, iActualSectorCount));
        }
    }

    protected CdFileSectorReader(File discImageFile, RandomAccessFile inputFile, int iSectorCount) {
        this._inputFile = inputFile;
        this._discImageFile = discImageFile;
        this._sectorFactory = null;
        this._iSectorCount = iSectorCount;
    }

    private int calculateSectorCount() throws CdReadException {
        try {
            return (int)((this._inputFile.length() - this._sectorFactory.get1stSectorOffset()) / (long)this._sectorFactory.getRawSectorSize());
        }
        catch (IOException ex) {
            throw new CdReadException(this._discImageFile, ex);
        }
    }

    @Nonnull
    public String serialize() {
        return String.format(SERIALIZATION, this._discImageFile.getPath(), this._sectorFactory.getRawSectorSize(), this._iSectorCount, this._sectorFactory.get1stSectorOffset());
    }

    public boolean matchesSerialization(@Nonnull String sSerialization) {
        String[] asValues = Misc.regex(DESERIALIZATION, sSerialization);
        if (asValues == null) {
            return false;
        }
        try {
            int iSectorSize = Integer.parseInt(asValues[2]);
            int iSectorCount = Integer.parseInt(asValues[3]);
            long lngStartOffset = Long.parseLong(asValues[4]);
            return iSectorCount == this._iSectorCount && lngStartOffset == this._sectorFactory.get1stSectorOffset() && iSectorSize == this._sectorFactory.getRawSectorSize();
        }
        catch (NumberFormatException ex) {
            return false;
        }
        catch (ArrayIndexOutOfBoundsException ex) {
            return false;
        }
    }

    @Override
    public void close() throws IOException {
        this._inputFile.close();
    }

    public int getRawSectorSize() {
        return this._sectorFactory.getRawSectorSize();
    }

    public boolean hasSectorHeader() {
        return this._sectorFactory.hasSectorHeader();
    }

    @Nonnull
    public File getSourceFile() {
        return this._discImageFile;
    }

    public long getFilePointer(int iSector) {
        return (long)iSector * (long)this._sectorFactory.getRawSectorSize() + this._sectorFactory.get1stSectorOffset();
    }

    public int getSectorCount() {
        return this._iSectorCount;
    }

    @Nonnull
    public ILocalizedMessage getTypeDescription() {
        return this._sectorFactory.getTypeDescription();
    }

    @Nonnull
    public CdSector getSector(int iSector) throws CdReadException {
        if (iSector < 0 || iSector >= this._iSectorCount) {
            throw new IndexOutOfBoundsException("Sector " + iSector + " not in bounds of CD");
        }
        if (iSector >= this._iCachedSectorStart + this._iSectorsToCache || iSector < this._iCachedSectorStart || this._abBulkReadCache == null) {
            this._abBulkReadCache = null;
            this._iCachedSectorStart = iSector;
            this._lngCacheFileOffset = this.getFilePointer(iSector);
            byte[] abBulkReadCache = new byte[this._sectorFactory.getRawSectorSize() * this._iSectorsToCache];
            try {
                this._inputFile.seek(this._lngCacheFileOffset);
                int iBytesRead = IO.readByteArrayMax(this._inputFile, abBulkReadCache, 0, abBulkReadCache.length);
                if (iBytesRead < this._sectorFactory.getRawSectorSize()) {
                    throw new RuntimeException("Should have already verified this should not happen");
                }
            }
            catch (IOException ex) {
                throw new CdReadException(this._discImageFile, ex);
            }
            this._abBulkReadCache = abBulkReadCache;
        }
        int iOffset = this._sectorFactory.getRawSectorSize() * (iSector - this._iCachedSectorStart);
        return this._sectorFactory.createSector(iSector, this._abBulkReadCache, iOffset, this._lngCacheFileOffset + (long)iOffset);
    }

    void writeSector(int iSector, @Nonnull byte[] abSrcUserData) throws CdReadException, CdWriteException {
        CdSector cdSector = this.getSector(iSector);
        if (cdSector.getCdUserDataSize() != abSrcUserData.length) {
            throw new IllegalArgumentException("Data to write is not the right size.");
        }
        byte[] abRawData = cdSector.rebuildRawSector(abSrcUserData);
        long lngOffset = this._sectorFactory.get1stSectorOffset() + (long)this._sectorFactory.getRawSectorSize() * (long)iSector;
        try {
            this._inputFile.seek(lngOffset);
            this._inputFile.write(abRawData);
        }
        catch (IOException ex) {
            throw new CdWriteException(this._discImageFile, ex);
        }
    }

    public void beginPatching() throws DiscPatcher.CreatePatchFileException {
        if (this._patcher != null) {
            this._patcher.cancel();
        }
        this._patcher = new DiscPatcher(this);
    }

    @Nonnull
    public File getTemporaryPatchFile() {
        if (this._patcher == null) {
            throw new IllegalStateException();
        }
        return this._patcher.getTempFile();
    }

    public void addPatch(int iSector, int iOffsetInSector, @Nonnull byte[] abBytesToReplace) throws DiscPatcher.WritePatchException {
        this.addPatch(iSector, iOffsetInSector, abBytesToReplace, 0, abBytesToReplace.length);
    }

    public void addPatch(int iSector, int iOffsetInSector, @Nonnull byte[] abBytesToReplace, int iStartByteToUse, int iNumberOfBytesToReplace) throws DiscPatcher.WritePatchException {
        if (this._patcher == null) {
            throw new IllegalStateException();
        }
        this._patcher.addPatch(iSector, iOffsetInSector, abBytesToReplace, iStartByteToUse, iNumberOfBytesToReplace);
    }

    public void applyPatches(@Nonnull ProgressLogger pl) throws CdReopenException, CdReadException, CdWriteException, DiscPatcher.PatchReadException, TaskCanceledException {
        if (this._patcher == null) {
            throw new IllegalStateException();
        }
        this._patcher.applyPatches(this, pl);
        this._patcher = null;
    }

    void reopenForWriting() throws CdReopenException {
        try {
            this._inputFile.close();
            this._inputFile = new RandomAccessFile(this._discImageFile, "rw");
        }
        catch (IOException ex) {
            throw new CdReopenException(this._discImageFile, (Throwable)ex);
        }
    }

    public String toString() {
        return this.serialize();
    }

    private static class Cd2352or2448Factory
    implements SectorFactory {
        private final long _lng1stSectorOffset;
        private final boolean _bln2352;

        public Cd2352or2448Factory(@Nonnull RandomAccessFile cdFile, boolean blnCheck2352, boolean blnCheck2448) throws FileTooSmallToIdentifyException, IOException {
            long lngFileLength = cdFile.length();
            if (lngFileLength < (long)CdSectorHeader.SECTOR_SYNC_HEADER.length) {
                throw new FileTooSmallToIdentifyException(lngFileLength);
            }
            byte[] abSyncHeader = new byte[CdSectorHeader.SECTOR_SYNC_HEADER.length];
            for (long lngSectStart = 0L; lngSectStart < Math.min(lngFileLength - (long)abSyncHeader.length, 4896L); ++lngSectStart) {
                cdFile.seek(lngSectStart);
                IO.readByteArray(cdFile, abSyncHeader);
                if (!Arrays.equals(abSyncHeader, CdSectorHeader.SECTOR_SYNC_HEADER)) continue;
                LOG.log(Level.FINE, "Possible sync header at {0,number,#}", lngSectStart);
                if (blnCheck2352 && this.checkMore(2352, cdFile, lngSectStart, abSyncHeader)) {
                    this._bln2352 = true;
                    this._lng1stSectorOffset = lngSectStart % 2352L;
                    return;
                }
                if (!blnCheck2448 || !this.checkMore(2448, cdFile, lngSectStart, abSyncHeader)) continue;
                this._bln2352 = false;
                this._lng1stSectorOffset = lngSectStart % 2448L;
                return;
            }
            throw new FileTooSmallToIdentifyException(lngFileLength);
        }

        private boolean checkMore(int iSectorSize, @Nonnull RandomAccessFile cdFile, long lngSectStart, @Nonnull byte[] abSyncHeader) throws IOException {
            long lngSectorsToTry = (cdFile.length() - lngSectStart - (long)CdSectorHeader.SECTOR_SYNC_HEADER.length) / 2352L;
            if (lngSectorsToTry > 10L) {
                lngSectorsToTry = 10L;
            }
            int iOfs = iSectorSize;
            while (lngSectorsToTry > 0L) {
                cdFile.seek(lngSectStart + (long)iOfs);
                IO.readByteArray(cdFile, abSyncHeader);
                if (!Arrays.equals(abSyncHeader, CdSectorHeader.SECTOR_SYNC_HEADER)) {
                    return false;
                }
                --lngSectorsToTry;
                iOfs += iSectorSize;
            }
            return true;
        }

        public Cd2352or2448Factory(boolean blnIs2352, long lngStartOffset) {
            this._bln2352 = blnIs2352;
            this._lng1stSectorOffset = lngStartOffset;
        }

        @Override
        @Nonnull
        public CdSector createSector(int iSector, @Nonnull byte[] abSectorBuff, int iOffset, long lngFilePointer) {
            return new CdSector2352(iSector, abSectorBuff, iOffset, lngFilePointer);
        }

        @Override
        @Nonnull
        public ILocalizedMessage getTypeDescription() {
            return this._bln2352 ? I.CD_FORMAT_2352() : I.CD_FORMAT_2448();
        }

        @Override
        public boolean hasSectorHeader() {
            return true;
        }

        @Override
        public long get1stSectorOffset() {
            return this._lng1stSectorOffset;
        }

        @Override
        public int getRawSectorSize() {
            return this._bln2352 ? 2352 : 2448;
        }
    }

    private static class Cd2336Factory
    implements SectorFactory {
        private long _lng1stSectorOffset;

        public Cd2336Factory(@Nonnull RandomAccessFile cdFile) throws FileTooSmallToIdentifyException, IOException {
            long lngFileLength = cdFile.length();
            if (lngFileLength < 2336L) {
                throw new FileTooSmallToIdentifyException(lngFileLength);
            }
            byte[] abTestSectorData = new byte[2336];
            int iMaxSearch = 77088;
            if ((long)iMaxSearch > lngFileLength) {
                iMaxSearch = (int)lngFileLength;
            }
            for (long lngSectStart = 0L; lngSectStart < (long)(iMaxSearch - abTestSectorData.length); lngSectStart += 4L) {
                if (!Cd2336Factory.isXaSector(cdFile, lngSectStart, abTestSectorData)) continue;
                long lngAdditionalOffset = 2336L;
                for (int iTimes = 0; lngSectStart + lngAdditionalOffset < lngFileLength - (long)abTestSectorData.length && iTimes < 146; ++iTimes) {
                    if (Cd2336Factory.isXaSector(cdFile, lngSectStart + lngAdditionalOffset, abTestSectorData)) {
                        this._lng1stSectorOffset = lngSectStart % 2336L;
                        return;
                    }
                    lngAdditionalOffset += 2336L;
                }
            }
            throw new FileTooSmallToIdentifyException(lngFileLength);
        }

        private static boolean isXaSector(@Nonnull RandomAccessFile cdFile, long lngSectorStart, @Nonnull byte[] abReusableBuffer) throws IOException {
            cdFile.seek(lngSectorStart);
            IO.readByteArray(cdFile, abReusableBuffer);
            CdSector2336 cdSector = new CdSector2336(0, abReusableBuffer, 0, lngSectorStart);
            XaAnalysis xa = XaAnalysis.analyze(cdSector);
            return xa != null && xa.iProbability == 100;
        }

        private Cd2336Factory(long lngStartOffset) {
            this._lng1stSectorOffset = lngStartOffset;
        }

        @Override
        @Nonnull
        public CdSector createSector(int iSector, @Nonnull byte[] abSectorBuff, int iOffset, long lngFilePointer) {
            return new CdSector2336(iSector, abSectorBuff, iOffset, lngFilePointer);
        }

        @Override
        @Nonnull
        public ILocalizedMessage getTypeDescription() {
            return I.CD_FORMAT_2336();
        }

        @Override
        public boolean hasSectorHeader() {
            return true;
        }

        @Override
        public long get1stSectorOffset() {
            return this._lng1stSectorOffset;
        }

        @Override
        public int getRawSectorSize() {
            return 2336;
        }
    }

    private static class Cd2048Factory
    implements SectorFactory {
        private final long _lng1stSectorOffset;

        public Cd2048Factory() {
            this._lng1stSectorOffset = 0L;
        }

        public Cd2048Factory(long lngStartOffset) {
            this._lng1stSectorOffset = lngStartOffset;
        }

        @Override
        @Nonnull
        public CdSector createSector(int iSector, @Nonnull byte[] abSectorBuff, int iOffset, long lngFilePointer) {
            return new CdSector2048(iSector, abSectorBuff, iOffset, lngFilePointer);
        }

        @Override
        @Nonnull
        public ILocalizedMessage getTypeDescription() {
            return I.CD_FORMAT_2048();
        }

        @Override
        public boolean hasSectorHeader() {
            return false;
        }

        @Override
        public long get1stSectorOffset() {
            return this._lng1stSectorOffset;
        }

        @Override
        public int getRawSectorSize() {
            return 2048;
        }
    }

    private static interface SectorFactory {
        @Nonnull
        public CdSector createSector(int var1, @Nonnull byte[] var2, int var3, long var4);

        @Nonnull
        public ILocalizedMessage getTypeDescription();

        public boolean hasSectorHeader();

        public long get1stSectorOffset();

        public int getRawSectorSize();
    }

    public static class CdReopenException
    extends IOException {
        @Nonnull
        private final File _file;

        public CdReopenException(File file, Throwable cause) {
            super(cause);
            this._file = file;
        }

        public File getFile() {
            return this._file;
        }
    }

    public static class FileTooSmallToIdentifyException
    extends Exception {
        private final long _lngFileSize;

        public FileTooSmallToIdentifyException(long lngFileSize) {
            this._lngFileSize = lngFileSize;
        }

        public long getFileSize() {
            return this._lngFileSize;
        }
    }

    public static class CdWriteException
    extends IOException {
        @Nonnull
        private final File _file;

        public CdWriteException(@Nonnull File file, IOException ex) {
            super(ex);
            this._file = file;
        }

        @Nonnull
        public File getFile() {
            return this._file;
        }
    }

    public static class CdReadException
    extends IOException {
        @Nonnull
        private final File _file;

        public CdReadException(@Nonnull File file, IOException ex) {
            super(ex);
            this._file = file;
        }

        @Nonnull
        public File getFile() {
            return this._file;
        }
    }

    public static class CdFileNotFoundException
    extends FileNotFoundException {
        @Nonnull
        private final File _file;

        public CdFileNotFoundException(@Nonnull File file, FileNotFoundException ex) {
            super(file.getPath());
            this.initCause(ex);
            this._file = file;
        }

        @Nonnull
        public File getFile() {
            return this._file;
        }
    }
}

