package com.novel.service.impl;

import com.novel.common.constants.ScheduleConstants;
import com.novel.common.exception.job.TaskException;
import com.novel.domain.SysJob;
import com.novel.mapper.SysJobMapper;
import com.novel.service.SysJobService;
import com.novel.util.CronUtils;
import com.novel.util.ScheduleUtils;
import org.quartz.JobDataMap;
import org.quartz.JobKey;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.util.List;

/**
 * 定时任务调度信息 服务层
 *
 * @author novel
 * @since 2020/3/2
 */
@Service
public class SysJobServiceImpl implements SysJobService {
    private final Scheduler scheduler;
    private final SysJobMapper jobMapper;

    public SysJobServiceImpl(Scheduler scheduler, SysJobMapper jobMapper) {
        this.scheduler = scheduler;
        this.jobMapper = jobMapper;
    }

    /**
     * 项目启动时，初始化定时器
     * 主要是防止手动修改数据库导致未同步到定时任务处理（注：不能手动修改数据库ID和任务组名，否则会导致脏数据）
     */
    @PostConstruct
    public void init() throws SchedulerException, TaskException {
        List<SysJob> jobList = jobMapper.selectJobAll();
        for (SysJob job : jobList) {
            updateSchedulerJob(job, job.getJobGroup());
        }
    }

    /**
     * 获取quartz调度器的计划任务列表
     *
     * @param job 调度信息
     * @return 结果
     */
    @Override
    public List<SysJob> selectJobList(SysJob job) {
        return jobMapper.selectJobList(job);
    }

    /**
     * 通过调度任务ID查询调度信息
     *
     * @param id 调度任务ID
     * @return 调度任务对象信息
     */
    @Override
    public SysJob selectJobById(Long id) {
        return jobMapper.selectJobById(id);
    }

    /**
     * 暂停任务
     *
     * @param job 调度信息
     * @return 结果
     * @throws SchedulerException 调度异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean pauseJob(SysJob job) throws SchedulerException {
        if (job == null) {
            return false;
        }
        Long id = job.getId();
        String jobGroup = job.getJobGroup();
        job.setStatus(ScheduleConstants.Status.PAUSE.getValue());
        int rows = jobMapper.updateJob(job);
        if (rows > 0) {
            scheduler.pauseJob(ScheduleUtils.getJobKey(id, jobGroup));
        }
        return rows > 0;
    }

    /**
     * 恢复任务
     *
     * @param job 调度信息
     * @return 结果
     * @throws SchedulerException 调度异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean resumeJob(SysJob job) throws SchedulerException {
        if (job == null) {
            return false;
        }
        Long id = job.getId();
        String jobGroup = job.getJobGroup();
        job.setStatus(ScheduleConstants.Status.NORMAL.getValue());
        int rows = jobMapper.updateJob(job);
        if (rows > 0) {
            scheduler.resumeJob(ScheduleUtils.getJobKey(id, jobGroup));
        }
        return rows > 0;
    }

    /**
     * 删除任务后，所对应的trigger也将被删除
     *
     * @param job 调度信息
     * @return 删除结果
     * @throws SchedulerException 调度异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean deleteJob(SysJob job) throws SchedulerException {
        if (job == null) {
            return false;
        }
        Long id = job.getId();
        String jobGroup = job.getJobGroup();
        int rows = jobMapper.deleteJobById(id);
        if (rows > 0) {
            scheduler.deleteJob(ScheduleUtils.getJobKey(id, jobGroup));
        }
        return rows > 0;
    }

    /**
     * 批量删除调度信息
     *
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean deleteJobByIds(Long[] ids) throws SchedulerException {
        if (ids == null) {
            return false;
        }
        for (Long id : ids) {
            SysJob job = jobMapper.selectJobById(id);
            if (job != null) {
                deleteJob(job);
            }
        }
        return true;
    }

    /**
     * 任务调度状态修改
     *
     * @param job 调度信息
     * @return 结果
     * @throws SchedulerException 调度异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean changeStatus(SysJob job) throws SchedulerException {
        if (job == null) {
            return false;
        }
        String status = job.getStatus();
        if (ScheduleConstants.Status.NORMAL.getValue().equals(status)) {
            return resumeJob(job);
        } else if (ScheduleConstants.Status.PAUSE.getValue().equals(status)) {
            return pauseJob(job);
        } else {
            return false;
        }
    }

    /**
     * 立即运行任务
     *
     * @param job 调度信息
     * @return 结果
     * @throws SchedulerException 调度异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean run(SysJob job) throws SchedulerException {
        if (job == null) {
            return false;
        }
        Long id = job.getId();
        String jobGroup = job.getJobGroup();
        SysJob properties = selectJobById(job.getId());
        // 参数
        JobDataMap dataMap = new JobDataMap();
        dataMap.put(ScheduleConstants.TASK_PROPERTIES, properties);
        scheduler.triggerJob(ScheduleUtils.getJobKey(id, jobGroup), dataMap);
        return true;
    }

    /**
     * 新增任务
     *
     * @param job 调度信息 调度信息
     * @return 结果
     * @throws SchedulerException 调度异常
     * @throws TaskException      任务异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean insertJob(SysJob job) throws SchedulerException, TaskException {
        if (job == null) {
            return false;
        }
        int rows = jobMapper.insertJob(job);
        if (rows > 0) {
            ScheduleUtils.createScheduleJob(scheduler, job);
        }
        return rows > 0;
    }

    /**
     * 更新任务的时间表达式
     *
     * @param job 调度信息
     * @return 结果
     * @throws SchedulerException 调度异常
     * @throws TaskException      任务异常
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public boolean updateJob(SysJob job) throws SchedulerException, TaskException {
        if (job == null || job.getId() == null) {
            return false;
        }
        SysJob properties = selectJobById(job.getId());
        int rows = jobMapper.updateJob(job);
        if (rows > 0) {
            updateSchedulerJob(job, properties.getJobGroup());
        }
        return rows > 0;
    }

    /**
     * 更新任务
     *
     * @param job      任务对象
     * @param jobGroup 任务组名
     */
    public void updateSchedulerJob(SysJob job, String jobGroup) throws SchedulerException, TaskException {
        if (job == null) {
            return;
        }
        Long id = job.getId();
        // 判断是否存在
        JobKey jobKey = ScheduleUtils.getJobKey(id, jobGroup);
        if (scheduler.checkExists(jobKey)) {
            // 防止创建时存在数据问题 先移除，然后在执行创建操作
            scheduler.deleteJob(jobKey);
        }
        ScheduleUtils.createScheduleJob(scheduler, job);
    }

    /**
     * 校验cron表达式是否有效
     *
     * @param cronExpression 表达式
     * @return 结果
     */
    @Override
    public boolean checkCronExpressionIsValid(String cronExpression) {
        return CronUtils.isValid(cronExpression);
    }
}