'use strict';

const nkm = require(`@nkmjs/core`);
const com = nkm.com;
const u = nkm.u;
const ui = nkm.ui;

const UNICODE = require(`../unicode`);
const SIGNAL = require(`../signal`);
const mkfData = require(`../data`);
const RangeContent = require(`../data/range-content`);
const GlyphMiniSlot = require(`./glyph-mini-slot`);

const __norange = {
    name: 'No range', typeTag: `Custom`, count: 0,
    icon: 'text-style', isDynamic: true,
    fetchList: () => { return []; }
};

const __range = {
    name: 'My Glyphs', typeTag: `Custom`, count: mkfData.RangeContent.CountGlyphs,
    icon: 'text-style', isDynamic: true,
    fetchList: mkfData.RangeContent.FetchFamilyGlyphAll
};

const __noToolbar = `no-toolbar`;

const base = nkm.datacontrols.ControlView; //ui.views.View

class GlyphPicker extends base {
    constructor() { super(); }

    _Init() {
        super._Init();

        this._domStreamer = null;
        this._unicodeMap = new Map();
        this._displayRange = null;

        this._searchSettings = new mkfData.SearchSettings();
        this._searchSettings
            .Watch(nkm.data.SIGNAL.SEARCH_TOGGLED, this._OnSearchToggled, this)
            .Watch(nkm.data.SIGNAL.SEARCH_STARTED, this._OnSearchStarted, this)
            .Watch(nkm.data.SIGNAL.SEARCH_COMPLETE, this._OnSearchComplete, this);

        this._dataObserver
            .Hook(SIGNAL.GLYPH_ADDED, this._OnGlyphAdded, this)
            .Hook(SIGNAL.GLYPH_REMOVED, this._OnGlyphRemoved, this);

        this._searchActive = false;

        let dataSel = nkm.ui.helpers.HostSelStack(this, true, true, {
            add: {
                fn: (p_sel, p_index) => {
                    let widget = this._domStreamer.GetItemAt(p_index);
                    if (widget) { widget.Select(true); }
                    else { p_sel.Add(this._content[p_index]); }
                }, thisArg: this
            },
            remove: {
                fn: (p_sel, p_index, p_data) => {
                    let widget = this._domStreamer.GetItemAt(p_index);
                    if (widget) { widget.Select(false); }
                    else { p_sel.Remove(this._content[p_index]); }
                }, thisArg: this
            },
            count: {
                fn: (p_sel) => { return this._content ? this._content.length : 0; }, thisArg: this
            },
            index: {
                fn: (p_sel, p_data) => { return this._content ? this._content.indexOf(p_data) : -1; }, thisArg: this
            },
        }).data;

        dataSel
            .Watch(com.SIGNAL.ITEM_ADDED, this._OnSelectionStackBump, this)
            .Watch(com.SIGNAL.ITEM_BUMPED, this._OnSelectionStackBump, this);

        this._contentRange = new RangeContent();
        this._contentRange.Watch(nkm.com.SIGNAL.READY, this._OnRangeReady, this);
        this.forwardData
            .To(this._searchSettings, { set: `family` })
            .To(this._contentRange, { set: `family` });

        this._content = null;
        this._flags.Add(this, __noToolbar);
        this._selectionFn = null;

    }

    _PostInit() {
        super._PostInit();
        this.displayRange = __norange;
    }

    static _Style() {
        return nkm.style.Extends({
            ':host': {
                ...nkm.style.flex.column,
                'padding': `10px`,
                'width': '300px',
                'height': '350px',
                'overflow': 'clip',
                'grid-gap': '5px'
            },
            '.search, .toolbar': {
                ...nkm.style.flexItem.fixed,
            },
            '.toolbar': {
                'margin-top': `5px`
            },
            ':host(.no-toolbar) .toolbar': {
                'display': 'none'
            },
            '.dom-stream': {
                ...nkm.style.flexItem.fill,
                'margin-top': `5px`,
                'overflow': 'auto',
            },
            '.dom-stream.empty': {
                'display': 'block !important'
            },
            '.search-status': {
                ...nkm.style.rules.absolute.center,
            }
        }, base._Style());
    }

    _Render() {
        super._Render();

        this._search = this.Attach(nkm.uilib.inputs.Search, `search`);
        this._search.options = {
            changeOnInput: true, submitOnChange: true,
            placeholder: `Search within your glyphs...`,
            onSubmit: {
                fn: (p_input, p_value) => {
                    let toggle = this._searchActive;
                    if (!p_value || p_value == ``) { toggle = false; }
                    else { toggle = true; }
                    this._searchSettings.Set(nkm.data.IDS.SEARCH_ENABLED, toggle);
                    this._searchSettings.Set(nkm.data.IDS.SEARCH_TERMS, p_value);
                }
            }
        }

        this._domStreamer = this.Attach(ui.helpers.DOMStreamer, 'dom-stream');
        this._domStreamer
            .Watch(ui.SIGNAL.ITEM_CLEARED, this._OnItemCleared, this)
            .Watch(ui.SIGNAL.ITEM_REQUESTED, this._OnItemRequested, this);

        this._domStreamer.options = {
            layout: {
                itemWidth: 40,
                itemHeight: 45,
                itemCount: 0,
                gap: 5,
                //customArea:{ start:0, size:200 }
            }
        };

        this._toolbar = this.Attach(ui.WidgetBar, `toolbar`);
        this._toolbar.options = {
            inline: true,
            stretch: ui.WidgetBar.FLAG_STRETCH,
            defaultWidgetClass: nkm.uilib.buttons.Button,
        };

    }

    set handles(p_handles) {
        this._toolbar.Clear();
        if (!p_handles || p_handles.length == 0) {
            this._flags.Set(__noToolbar, true);
            return;
        }
        this._flags.Set(__noToolbar, false);
        this._toolbar.CreateHandles(...p_handles);
    }

    set allowMultiple(p_value) { this.selectionStack.allowMultiple = p_value; }
    set selectionFn(p_value) { this._selectionFn = p_value; }

    set displayRange(p_value) {

        if (this._displayRange == p_value) { return; }

        this._displayRange = p_value;
        this.selectionStack.Clear();

        this._domStreamer.SetFocusIndex(-1);

        this._contentRange.displayRange = p_value;

    }

    _OnRangeReady(p_range) {

        //Active range content is ready
        if (this._searchActive) { this._SetContentSource(null); }
        else { this._SetContentSource(this._contentRange._content); }

        this._searchSettings._UpdateSearchData(p_range);

    }

    _OnDataChanged(p_oldData) {
        super._OnDataChanged(p_oldData);
        this._ReloadList();
    }

    //#region search

    _OnSearchToggled() {

        let oldValue = this._searchActive;
        this._searchActive = this._searchSettings ? this._searchSettings.Get(nkm.data.IDS.SEARCH_ENABLED) : false;

        if (oldValue == this._searchActive) { return; }

        if (!this._searchActive) {
            this._SetContentSource(this._contentRange._content);
        } else {
            if (this._searchSettings.ready) { this._OnSearchComplete(); }
            else { this._OnSearchStarted(); }
        }

    }

    _OnSearchStarted() {
        if (this._searchActive) { this._SetContentSource(null); }
    }

    _OnSearchComplete() {
        if (this._searchActive) { this._SetContentSource(this._searchSettings._results); }
    }

    //#endregion

    //#region DOM Streamer handling

    _SetContentSource(p_array) {
        this._content = p_array;
        this._domStreamer.itemCount = p_array ? p_array.length : 0;
    }

    _OnItemRequested(p_streamer, p_index, p_fragment, p_returnFn) {

        let unicodeInfos = this._content ? this._content[p_index] : null;

        if (!unicodeInfos || !this._data) { return; }

        let widget = this.Attach(GlyphMiniSlot, 'glyph', p_fragment);
        widget.context = this._data;
        widget.data = unicodeInfos;

        this._unicodeMap.set(unicodeInfos, widget);

        p_returnFn(p_index, widget);

        this.selectionStack.Check(widget);

    }

    _OnItemCleared(p_item) {
        this._unicodeMap.delete(p_item.data);
    }

    _OnSelectionStackBump(p_data) {
        this._domStreamer.SetFocusIndex(this._content.indexOf(p_data), false);
        if (!this.selectionStack.allowMultiple && this._selectionFn) { u.Call(this._selectionFn, p_data); }
    }

    //#endregion

    //#region Catalog Management

    _RefreshItems() {
        this._domStreamer._Stream(null, null, true);
    }

    //#endregion

    //#region Preview updates

    _OnGlyphAdded(p_family, p_glyph) {

        let
            uInfos = p_glyph.unicodeInfos,
            widget = this._unicodeMap.get(uInfos);

        if (widget) { widget._UpdateGlyph(); }

    }

    _OnGlyphRemoved(p_family, p_glyph) {

        let
            uInfos = p_glyph.unicodeInfos,
            widget = this._unicodeMap.get(uInfos);

        if (widget) { widget._UpdateGlyph(); }

    }

    _ReloadList() {
        this.displayRange = __norange;
        this.displayRange = __range;
    }

    //#endregion

}

module.exports = GlyphPicker;
ui.Register(`mkf-glyph-picker`, GlyphPicker);