# cuCollections

<table><tr>
<th><b><a href="https://github.com/NVIDIA/cuCollections/tree/dev/examples">Examples</a></b></th>
<th><b><a href="">Doxygen Documentation (TODO)</a></b></th>
</tr></table>

`cuCollections` (`cuco`) is an open-source, header-only library of GPU-accelerated, concurrent data structures.

Similar to how [Thrust](https://github.com/thrust/thrust) and [CUB](https://github.com/thrust/cub) provide STL-like, GPU accelerated algorithms and primitives, `cuCollections` provides STL-like concurrent data structures. `cuCollections` is not a one-to-one, drop-in replacement for STL data structures like `std::unordered_map`. Instead, it provides functionally similar data structures tailored for efficient use with GPUs.

## Development Status

`cuCollections` is still under heavy development. Users should expect breaking changes and refactoring to be common.

### Major Updates

__11/01/2024__ Refined the term `window` as `bucket`

__01/08/2024__ Deprecated the `experimental` namespace

__01/02/2024__ Moved the legacy `static_map` to `cuco::legacy` namespace


## Getting cuCollections

`cuCollections` is header only and can be incorporated manually into your project by downloading the headers and placing them into your source tree.

### Adding `cuCollections` to a CMake Project

`cuCollections` is designed to make it easy to include within another CMake project.
 The `CMakeLists.txt` exports a `cuco` target that can be linked<sup>[1](#link-footnote)</sup>
 into a target to setup include directories, dependencies, and compile flags necessary to use `cuCollections` in your project.


We recommend using [CMake Package Manager (CPM)](https://github.com/TheLartians/CPM.cmake) to fetch `cuCollections` into your project.
With CPM, getting `cuCollections` is easy:

```cmake
cmake_minimum_required(VERSION 3.23.1 FATAL_ERROR)

include(path/to/CPM.cmake)

CPMAddPackage(
  NAME cuco
  GITHUB_REPOSITORY NVIDIA/cuCollections
  GIT_TAG dev
  OPTIONS
     "BUILD_TESTS OFF"
     "BUILD_BENCHMARKS OFF"
     "BUILD_EXAMPLES OFF"
)

target_link_libraries(my_library cuco)
```

This will take care of downloading `cuCollections` from GitHub and making the headers available in a location that can be found by CMake. Linking against the `cuco` target will provide everything needed for `cuco` to be used by the `my_library` target.

<a name="link-footnote">1</a>: `cuCollections` is header-only and therefore there is no binary component to "link" against. The linking terminology comes from CMake's `target_link_libraries` which is still used even for header-only library targets.

## Requirements
- `nvcc 11.5+`
- C++17
- Volta+
    - Pascal is partially supported. Any data structures that require blocking algorithms are not supported. See [libcu++](https://nvidia.github.io/libcudacxx/setup/requirements.html#device-architectures) documentation for more details.

## Dependencies

`cuCollections` depends on the following libraries:

- [CUDA C++ Core Libraries (CCCL)](https://github.com/NVIDIA/cccl)

No action is required from the user to satisfy these dependencies. `cuCollections`'s CMake script is configured to first search the system for these libraries, and if they are not found, to automatically fetch them from GitHub.


## Building cuCollections

Since `cuCollections` is header-only, there is nothing to build to use it.

To build the tests, benchmarks, and examples:

```bash
cd $CUCO_ROOT
mkdir -p build
cd build
cmake .. # configure
make # build
ctest --test-dir tests # run tests
```
Binaries will be built into:
- `build/tests/`
- `build/benchmarks/`
- `build/examples/`

### Build Script:

Alternatively, you can use the build script located at `ci/build.sh`. Calling this script with no arguments will trigger a full build which will be located at `build/local`.

```bash
cd $CUCO_ROOT
ci/build.sh # configure and build
ctest --test-dir build/local/tests # run tests
```

For a comprehensive list of all available options along with descriptions and examples, you can use the option `ci/build.sh -h`.

## Code Formatting
By default, `cuCollections` uses [`pre-commit.ci`](https://pre-commit.ci/) along with [`mirrors-clang-format`](https://github.com/pre-commit/mirrors-clang-format) to automatically format the C++/CUDA files in a pull request.
Users should enable the `Allow edits by maintainers` option to get auto-formatting to work.

### Pre-commit hook
Optionally, you may wish to setup a [`pre-commit`](https://pre-commit.com/) hook to automatically run `clang-format` when you make a git commit. This can be done by installing `pre-commit` via `conda` or `pip`:

```bash
conda install -c conda-forge pre_commit
```

```bash
pip install pre-commit
```

and then running:
```bash
pre-commit install
```

from the root of the `cuCollections` repository. Now code formatting will be run each time you commit changes.

You may also wish to manually format the code:
```bash
pre-commit run clang-format --all-files
```

### Caveats
`mirrors-clang-format` guarantees the correct version of `clang-format` and avoids version mismatches.
Users should **_NOT_** use `clang-format` directly on the command line to format the code.


## Documentation
[`Doxygen`](https://doxygen.nl/) is used to generate HTML pages from the C++/CUDA comments in the source code.

### The example
The following example covers most of the Doxygen block comment and tag styles
for documenting C++/CUDA code in `cuCollections`.

```c++
/**
 * @file source_file.cpp
 * @brief Description of source file contents
 *
 * Longer description of the source file contents.
 */

/**
 * @brief Short, one sentence description of the class.
 *
 * Longer, more detailed description of the class.
 *
 * A detailed description must start after a blank line.
 *
 * @tparam T Short description of each template parameter
 * @tparam U Short description of each template parameter
 */
template <typename T, typename U>
class example_class {

  void get_my_int();            ///< Simple members can be documented like this
  void set_my_int( int value ); ///< Try to use descriptive member names

  /**
   * @brief Short, one sentence description of the member function.
   *
   * A more detailed description of what this function does and what
   * its logic does.
   *
   * @param[in]     first  This parameter is an input parameter to the function
   * @param[in,out] second This parameter is used both as an input and output
   * @param[out]    third  This parameter is an output of the function
   *
   * @return The result of the complex function
   */
  T complicated_function(int first, double* second, float* third)
  {
      // Do not use doxygen-style block comments
      // for code logic documentation.
  }

 private:
  int my_int;                ///< An example private member variable
};
```

### Doxygen style check
`cuCollections` also uses Doxygen as a documentation linter. To check the Doxygen style locally, run
```bash
./ci/pre-commit/doxygen.sh
```


## Data Structures

We plan to add many GPU-accelerated, concurrent data structures to `cuCollections`. As of now, the two flagships are variants of hash tables.

### `static_set`

`cuco::static_set` is a fixed-size container that stores unique elements in no particular order. See the Doxygen documentation in `static_set.cuh` for more detailed information.

#### Examples:
- [Host-bulk APIs](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_set/host_bulk_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJx9VgtvGzcM_ivcDcOc9vwKVhRwkmJekmJGC6eI0xbDPDiyTvYJkaWbHnY9I_99pHTnx9ouAWyfSJEfP_LTaZc54Zw02mWDP3eZLLJBP88U08vAliIbZDwULMszZ4Ll9Nx9MdXwAq5NtbVyWXpo8TM4752ft_HjlxzGn0Y3oyFc391_uLsfPozuxh3aEDe9l1xoJwoIuhAWfClgWDGOX7Ulh0_CEho47_SgRQ7TrLZNs7OLGGVrAqzYFrTxEJzAMNLBQioB4gsXlQepgZtVpSTTXMBG-jKmquNEOPBHHcTMPUN_hjsqfFocewLze-j0V3pfDbrdzWbTYRF2x9hlVyVn130_ur4dT27bCH2_7aNWSC9Y8XeQFgufb4FViIyzOeJVbAPGAltagTZvCPnGSi_1MgdnFn7DrIhxCum8lfPgT8hrcGL9xw5IH9NI3HACo8k0g9-Gk9Ekj3E-jx5-v_v4AJ-H9_fD8cPodgJ399is8c2IWoVPb2E4_gPejcY3OQikDlOJL5WlKhCqJFpFkTicCHECY2ESLFcJLheSQzNGsDRrYTWWBZWwK5kGDkEWMY6SK-mZj2tfFRdTdad6qn-UmqtQCLjkgZuuoy185oTv8FC-OfXwpQ3OdwuxxiizteDe2A45feWyCJpTaqa-bVdmid36jtFhXwUOWee_-aXBdgi2OtkU63TRs_sizcevcW5L9J7Ng3qaiS8MGRZYUTLPrRQLuBEr5MZb5gUy5IjRelIPJGC_KQK2m6LB8MPIHYT3gK5kjcuAQ4UzorY4bmuTJnFhzSoGjJuxMdEpkFapq4Wh3hlsXt0mJZ8EPErsj_WPOBgxyyM3mtTkHuMIAqFCnE9ie4CCnZTao_Skbq2NLM6meofrmIoyvBNbuEJY_oJIAuh24XZV-S04ZXwCbgUNo9A-qYl-2zX-nma0KLVQSMGaqSBch-rGuUxP4EoTFG5C1oTCicBNLvCyzuNL5okBPFcEDisYzoMlQeJxQ99V8FAwz6iSiJNTS0gaIAjiDMucNRCwiHb_UMM4rOYUMnFBQkcQiTxBEsA2FoOBk_-ImT-Kq8NqFjdcwavez71e7xDxGs831DpwOogkEjRnrlY--v4EyjDsHKOxpx0sYM5DYLLOkhVj9zqvLr4J4hD9Khm5kKq1R9U9jnN2jI1mNXBqWJyBeAIju0owDPrYRH2su5pa3-4Dc3EEI52JqYbPSDqpHgHuJ_4Su_CGEuyaiHnts2_I7uvWPD8fIUWJIomEMul436JdL4d-Duc5dDo5yGfakDQ_GJwcKgkE7dkTE19Te-8mdCvKYC6WOPhneRKF0AX97h2RN4pTgcwplZCgGExSJnMlVRtbhYdemp_vhj2KOUGcdGaQklEvQfnvlzM3Rr1BTzyHTwpqGCsFfwK5OOAjTda6xwFEvcRj6QCzORP-p_6YrTEdJcO7gFxskzL3-Uq2xrNMCJ22kSthricW3SLkWTTi3DZVksEsWiepmsw1jMZVFjQqfpu4OCDCqlun8c9gVwvD4OlweYmH0CRwju_KH-BtBNDA7kwxRnYBzymUFT5YDSRoXMHrFd1X8E1gD7ewTK8575-_Cn00m8qnK1rWxnxX_OXL_mtoM8vLK7eave5Bu43C9vjhsRhRtBVbzeO9Tcn5UUzOucLFdbpk4QLqVD9lz3ljR_mc2HE4sue_4v-_fg9_jw==))
- [Device-ref APIs for individual operations](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_set/device_ref_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJztWH9v4zYS_SpzOhzO2cg_EnTbwpsE50u2OKOFUyTZFkVdqDRF2bzIok6k7PiCfPe-ISXL3uwWLXB_XgIklmY482bmcTj0c2SVtdoUNhr__BzpNBqfxVEuimUtlioaR7JORRRH1tSV5Ofhm3lBb-jalLtKL1eOevKEzkfn5338-SKm2Q_Tm-mErm_vvr-9mzxMb2cDXuAXfaelKqxKqS5SVZFbKZqUQuJfI4npB1UxGjofjKjHCvOokc2jk3feys7UtBY7Koyj2iqY0ZYynStST1KVjnRB0qzLXItCKtpqt_KuGjseDv3UGDELJ6AvsKLEU3aoScLtofPPyrlyPBxut9uB8LAHploO86Bsh99Nr9_P7t_3AX2_7EORI71Uqf_UukLgix2JEsikWABvLrZkKhLLSkHmDCPfVtrpYhmTNZnbikp5O6m2rtKL2h0lr8WJ-A8VkD5RIHGTe5rezyP65-R-eh97Oz9OH_51--GBfpzc3U1mD9P393R7h2LNbqZcKjx9Q5PZT_TtdHYTk0Lq4Eo9lRVHAaia06rSkMN7pY5gZCbAsqWSOtOSWhrR0mxUVSAsKlW11oFwAJl6O7leayecf_cqOO9qOC_mxV91IfM6VXQha2mGlpfIxCo3kPXq6ljDraraumGqNrCSbJR0phqw0iuVrC4kuxb5p-W5WaJanxFa1FWBZIOP_UtjECkAblSyrExd2tcq1qWpyo7MaoMqKrH2qsM3gUP_8NxuQqlUpip2magngWIoBB-0FpVWGd2oNdLo4FohmZaT35C6y1djq281fE6-n9puhx4olZXZQAFlwk4r-mbr67f3T9uVliuSYNpC8T5sCOwQtnR-13mL7FqagncZCptVZu1lYHpAAWF6UOThEDasqhzcMgqAf1Q7y4YNXq2EXfn3tWU0puD1-1STT7XnocL-JCfs4xyIkCfkA1XblaoQawXmujuVxbR_MS3K2k0ZO4iC7CfJMjcLkScJbYxOSaLaZp0cljXA7AVTjCk-tuJxYxu7dDy2-r8qcVSczItnhErs06KFAP2SLuk1W95xLogTZx1vPxI14pfLhC1hRfA6HjdvGnWv5JgtsLkcj_ljmpSicpopftFoX_W8EG0zAY-VSJNFbuRj7-SksYNkf_kF8Hr3lANdwt0lZbtL_L_R68ETauuXhYchdVA6Azp9ouOfS-odrGpMTNMnPJxSQOOfTo4ssk3wDYH12OQFMkk-kcSJHzSl4HBjetPzhDll5z4i1uJVp5eHoXjBy7x4Yet_miIH729r9we54zeB_XOMiT-yD3KjRf6fR_9zHi2MyRugiKftWCkc-ROmLd7vcSwjLx40yahEgVzQ5SWN4CUU7mcs-AU2D128Yxr-AZIiGRhZdNFjVrUMCH3wW7Vjo4VrYkQXfb8u3Y5sbhw6eKXQuPkQV4ULUwh_rjb4PI_4JXDkmBQ2Iq-VHdDDCuLmiezK1HnKXd6qHCcpFtlarho_biUcN3nMYwqHPBkp64oHGYxp_B_cpVQ4wS3e4-y4qBhigkwmLQQE0T_rYpjV6wWbbM4AkBIgwmZXPDocbZXOblGvE7_gkt6O_j4ajTqL15gLMSPh2MKu0UjQQthmYoLu35B7kVImeFzYb4TOMEuTIIXt0eDtu0-C6KxfBqFUOu_tUQ0P7ZwcYuODu5ZcsO6g8-MrUpwrAcvzqLWNaoXiBgb0z0hYf9j6rHLCqM2qTz3PTIC5P98vUIsr9vDcWowbnX1Znl8X6OXlAC9YjlTyOR2moH2hnkcxncV0HtNgEJP29A4T03h8NJIFELxmn56wm1rt1rTfb4OFWoL-J7FfMVBFyp9HBymchtGB05DpCvlaiTxrB6BujvDjkHK-eN3x8QkXzRM2-kH1zj_t0CoUP9179MqhNqI5Auj6w80EgqqryWemiouLi7Pzr5HE86-vrq563UmH8asXqhQUfx9o_DnY98g9j4Y8KaET1Ln7fIm4MV6F7nVUpJYFKyUfufeJPA9Bc7cpzJa6Hopu0KR8QDPjVGgaW0WlsEz2bqj0ApFz_9zxHtg3GUW_tj34VwrpwkS_rwItFcZLkcdNyt3W8HVlbXzna62bxb8Vb6-WAnx8d7MpLk_t2MpGBe5UmV_p2km09Wt5Fdocy_IdX7_42pL5WBdqJTa6w_bAt9PUoI_yfZUvfwgMVgk4Zbg_MdjACtb0enxfykV5zJIQ_kfcOGZFq_QRLzoixKGSreCgkLh562zXVqCpJWJRCEkVYRmrHhyUUPNGEy9Et2sZxAKT9Y5ctZ6bbduq4pRDa3G7wLOD2QGn6bF9PkRDOzWo1MUFmuF9LSVupn-hbzyAFvZgDhuRP1fZVKVcXRXExwDeRHHE3w7goK667zyiYiPl2fnb-gxiU7rwhUjUh79LeXp69hX1RSVXl3adfDWifh_HgcMfh2BU2s_FeuG_Jcn14sCmlDLHy034SgMv0N2Lx-glbuUo2ZEcGy96-cX__gbvsxSQ))
- [One single storage for multiple sets](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_set/device_subsets_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJzNWQ1P20ga_ivv5qRTgJAA3V67gaDlgPairaAHdFerpfJO7EkyV8fOztgJWcR_v-edD8cmadVKe9IFhRB75v143uf9GPPYMtIYlWem1f_tsaWSVv-w00pFNinFRLb6rbhMRKvTMnmpY_7e273PaJfO8_lKq8m0oHa8Q0cHRy_28ev7Dl39PLwYntH59c3765uzu-H1VZc32E3vVCwzIxMqs0RqKqaSzuYixoe_06GfpWZr6Kh7QG1ecN_y9-5bO8dWyiovaSZWlOUFlUZCjDI0Vqkk-RDLeUEqozifzVMlsljSUhVTq8rLsebQr15IPioE1gvsmOPbuL6SRFGZzq9pUcz7vd5yuewKa3Y315Ne6hab3rvh-eXV7eU-TK-2fchSwEta_lEqDcdHKxJzWBaLEexNxZJyTWKiJe4VOVu-1KpQ2aRDJh8XS6GllZMoU2g1KosGeMFO-F9fAPhEBuDObml4e9-if57dDm87Vs4vw7t_XX-4o1_Obm7Oru6Gl7d0fYNgXV0MOVT49obOrn6ln4ZXFx2SgA6q5MNcsxcwVTGsMnEY3krZMGOcO7PMXMZqrGIKNKJJvpA6g1s0l3qmHOFgZGLlpGqmClHYaxvOWVW9--w--5vK4rRMJJ3EZZz3DG-JIyOLSMtxNy6np1vW5Br6_c3m7UTgdtITWotVY2cx1aUpeolcwIJoIWMI6U63LUFEy1huv2dikXU3tOY5AIDdCxlNdF7OzcYSkU5yUGA6a3oDUxM5blxTOUIuRXNhVs6kVrEV2tt1JPzRJof3x5QjQGYi-SAQS4bGrRlpJcd0IWeIQgETpaFpvmRSco7lmaRRmX4i7CWPKt-LYUAhaVamhYI08tI5tBxqMGJmxassUQuVlCIljwAHe6GEN2vfKFiPONLZ-6FZV4y7HOsL8ONPKIFtegVzoBvctCwTabqi5VRmkCNSJphNdzAW3AHKlIhCEFblsdVpDQvWWg1TYaZscgc1YK5FDE7BSJOnpV2vskWeLgCGBFr5ijUIMvhIAyAeDCb_RGbWNyzyQLhEueMKJZnmmQWW6Z1xyvMmI-dCWxCdf8-MZb-qC7xeovKgWmVct6TuWuFWCwhHIxSteKrkwtWaki216lKu1UtpK3aWZ_v50qYjI16s5nKN-I8oq9Jb7CjC5YUVJxU32JR5qec5yh6oka66NCx4GZdklRUSSWzrmbEUgvoZuGrFe1KIrKAlyi_WJLldEZRV_F8nviiZaaybKxHFk8gwH_g1oNfH-Oj1eieEpvPhAr_WOUZvOce4qr-GDlA1MaxxCly5WzCQLk4bOpYgbL50egZ0WOm4Kmcj1CdIxNq41FrCEZPmoLyTamG3kp3CDclXtH5B8sHxhmSZyhnEGtcQjNSFJcIfpdQrhsMF9ZNcRRw5CAHiQcpPcmXjGVbNdT7CZ2TQrWbSbwCwxOWx30fGSKEjv-rEI8utYvvL7UIhEkihiFMnGpdZzIQ4CQadngZj7lgbM1XnqaHztzTRIitTgfCuPqviq19Wha0zznpyPlLbOfUXKQjCF13TBVVLbtvsN67toML2es5LhM1Wg1qu-sIQIuG_hpg5JAVzzN2o8OvUyXd67JS8256-ofiENH6uDNeDQv7IBPCpG9LvhxXHYWu1JbyCrc1-uzb3W4Hmztvvu-wAL5GskesB3yzJddp-H8OVSKMiX1Pwm0VtyZNvlvEc9CoPbtEz1xFyAb111c62p615HnlUoqhe-ix2mAVAHZ5baqS5OnXCBvR42KEXHXrZoVcd-qFDh_h-iAuHuHKIS4c_PHlWXc7mxcqXL0yaMJIHPdQeV8b4b82d5L7FF5FWKabJhUhLabjv4Lb_xoW-TBNuPwblK2YBpoynTk0xFQXXeIzsaEyocTEqJ8-6mOT5c14W1nZU_KqqrX2WbGTEN4IR8HH_8Lgx4bjpZWhRhDU1cFEc82osqVpibRAjO4hV_e-Sa_dmD3GDjiEFKUg-L9EGTHmtv1_9Xtdca6nc4GfkM8fQ-5x7pOa4c-ezkeQK4mXmo_8AQePbXxRN0nwEfke0yFXQ1g4k263Eohw9MmU5yQ2OJ8Bwwsm7MXJa5IjWfYkKng-xdIK8xJ9JBIMLZau6bwmnbXsTU0HkM3eEQvepvWPPZKA5veXB0DnQxrBw_naHhz758EyVSh6gqW13X6hZ9wH42C_D5AFf9nzbtN92qBeafd1mLZahCnHaDWi3XUG7xwqcTXatQ6taPqhv7mqJnE8ih0-uTdsVOrdnx6uEa45WNr7cmwtMrpZV9qwDf-3IifbmNvImHpfaWEMKKtHiFaHB42Nvb4ceXV1pWtb1YWX4O5Y-v6mPzo-n--xpC9fP0tTLQK5Zpvv5b5yX7lz1JTpracOhSxDtOacZqJEw7iBpffRdxtJYgrnroxvSGSc95IX4ojn_o1wYI3z_Z5nwjOp_Kde_yHTG4tt4zjtqLL-RBkMdpnwM8Vwi1RhRrh6JNGPrR_1ANYoiM0UDSRAYZr22oqxtkNKuuUkD5ANywC9x2fHkRawwSbopvR3s-opEgun_5pbZTMeGvW5hLTTWcEisg9a1bNqSgB6dONcJu7OJBuZBMF1mazygrMhnKr7W7b87Tzu065R-N9hI7885_xnsKpuGzhgPJWJiYH1KB50t2cjtvUpFcrJDDHxI5hpAj9v3rdsyjqUx39Eba3FDWDicdO9hY2vHBy_UKI7UDOfSdkhBWLk-2IQujDCgMLiOmjSTxnb8rJzx0egfobVcrg9q9pndVKCOCUrUeCztAYwfr0ljhw7kzxaJbrPNLuPOPzyuVVOU_ZNvRogTtJ8-HgHE-vvFs_f3z94vG--negpjQlJJUO20umdKdbWnbceG2uKuH7_aMKiRDm5HPSOw4pj29lRFjroYfnCBGhiNBKpUmGn9MB8LUzTNcKVhJj7JyB9D5APO88VJNdW262AyjXcqIvPH5xwEAcYcfXYHxQsz6UGt-GDuHKsHxGlmnyrlM8yEMmyxU1aRFyJlTz2ZnHWmUulR8WfyyFgOHRyvUastqkFXR4p_Iw4NJOvy9gb15XDdlwdv5m-K_fpoD0zVpjo0PApv94KT1x4cQWE3KxOHKxTfCgdm6_OOxRIjSIyCrEFY37Uhr6F87p7WbX-AVRr_fBrp7jO1UmXj7dcNGifIxw39TyFthxkaJ_D60z_xCgKqB_ITdGJMHeFowZFc68FEFLa3N48Awak62yqGVoPA2nP_nLLm15dIsZFPG-fp2oXHJic6DSe49rZ3wIuKIx-fPGuCkdsTNGh6dAlZIbA-4z5uovLUocfau2bkUyNHw8G58Yi7hl_VE001b2yOw7Wjjm1W9urJyQmfNo9en56ettdiPA7diSxqR4Y36nl3CcK4FX-tKF6vZVHqzKY7XGx1WlxB0Mn1-r9arWwRx4dHL8tD3OZnyvZfXq19UGAQ7-0dvqJ9oePpwMyiVwe0v4--Uezb2pfIZD8Vs5H9P1iqRjWZcRynuLhw_7TCBUzU2afWUyfcR_Aa94F46-mj_fkvXfg-3w==))
- [Using shared memory as storage](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_set/shared_memory_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJyVWA1vG7kR_SvEFgXknCzZQYsrFFmoznZRIal9sJ0Eh_NhTe1SEuvVUkdypegM__e-GXK1KzluXQdxIn7MDN88vhnqKXHKOW1Klwx-fUp0ngxOu0khy3kl5yoZJFmVy6SbOFPZjD73392X4p04N6ut1fOFF53sSLw_ef-Xrrj6MrmYjMX59c3P1zfju8n1VY_W8vpPOlOlU7moylxZ4RdKjFcywz9xpiu-KEuBiPe9E9GhBfdJnLtPjj6wla2pxFJuRWm8qJyCGe3ETBdKqG-ZWnmhS5GZ5arQssyU2Gi_YFfRDocjfolGzNRLrJfYscKnWXulkH4XOv0svF8N-v3NZtOTHHbP2Hm_CItd_9Pk_PLq9vIYoe-2fS4LICus-r3SFgefboVcIbJMThFvITfCWCHnVmHOG4p8Y7XX5bwrnJn5jbSK7eTaeaunld8Dr44T528vAHyyBHDjWzG5vU_ET-PbyW2X7Xyd3P3z-vOd-Dq-uRlf3U0ub8X1DZJ1dTGhVOHTP8T46hfxcXJ10RUK0MGV-raydAqEqglWlQcMb5XaC2NmQlhupTI905moGSTmZq1siWOJlbJLHbiGIHO2U-il9tLz2IvDsav-fXlf_kmXWVHlSgyzKjN9R1uy1Cnfy6rF6GCFMfCEBWuVzq2pVq4XlvTfhcT8nQnjFgA4T5dqaew2Vd8kjqdgLiyZWq1m4gKTJcCVXiE8R8eJNGkioMQFWyLYaqL2CjaxFzEVEiDeKn-jZoglTeeFmcoiTcXa6Bz7sZWMpY_AShUdv12pUi5V3DIYqG9elT6lcbC6zM0mDUOU2__1Q5jBxHLlt3CwHb6wjkE2PRK7RYimBBlVcXRfPpGPfl98RZ61dV7IojAZHYyhaB8ewBhLONWEeMCpHnrRwB0jFxaAuHRTrSZRAKqZgb95ZSoXD-ho1BXGu27cvlnobMHON2x_rTIY03_AQGFk7thNmsbMpuLFOSNyLRTdr3to9tayqFTn6LcPlD92eoWZrtgoChBUqDLP55KZr2SxO42Z_hvBiJk1S562uN7SWslsELLCBa-Xnu3CiSOpVTOOqbOf2TpE1j4OBeemQHDDNWKRAGyXBNVyHu22XGO2dtt5meKueHpu_sbdRw0ElLasshbLYd8qCTErMvMo-uL8bty4YYQwZbJHuHt5DwcD0us0mEh5XaflZlJC_mSBfO4grDGrHAkIMh_9O7p1tIiNcAQkBnpnocMTO-Rut2W2sKasbYcYEfJSPgKvyoKP9V5WI7GQVFhK7RZB8qInB0OdA2iY4jUDcgOpoPIEcscCJMutmFVlRnbhwG97zU1ayLWiOKAxpQOnl0KTomBE1i6iYb-QHpGuVsZ6x4eobRInSqoiHQaFLpaEJD-AI8r6h6MmQWGE-Ib8EGBWTcGxNCTKWNcJUhHWtY55iZJXpz5MIgMcRhjKQ8yRiezwVZGJNMFHBBFQjYwABoEQke3MbZxaz7jwbyTxz3DhR9Gep47SabhO4VeJ4ocuglbWqIYWQbWpWIOKokSaEfScmogIZlfonup14yk7GaoxIg1BNfD14nSceoVlmZf7HCMCh50iBkR1j6P9b1zb16EaBjQh4LAO9QiBOBaGKUkvKik6C2puwsVhISaBxZB72J2VeZgBel_DIkGkDehbHpsN1-xIvql0oblgSw0ODwe0bDlBPwJDEOZyrmw036yL3KwLw_n4y-X4bkDUqIWGcofk7DlD19C4CEM69A2Aj1osBSy1sYeS1N4C0rXy-MoFqDe0kntew01JfAE302MmOuHmN9569QemypF4Eqh4pZ917pPLm5vrm4H4iHvw55wlI1iCxySSTjzfl88UAbaARrrsULtQV-SQ2498j7BgX5TuE9Z5tIC1yhNUEu0ozr4GVlzp9rpTdPjc5KkldjA3eztlJsCYDxZb0G2qdeAVbAZD-M_nn3j9YbeBAEchD9RMfqfDeDo-fa5Z_lXnUEzk_bC2i7yy3ERag4zNDzLMlmP1ZFU4E6e1yfPm8guuQ4JWfGd_rShhb9z9L9zLZbUUZbWcQmfqlqSuP7Xa0e3nlvCgHYr4BsFm63A5Zf1C80N9qNfQyLW0mp4Fta2IIfcBQ029wOnJycmIUCaDgFwFEXhowieOZPQ4QYWhM2DDh4eGJ-0O8uw1BzVmF3yhOOgIuHDQm2UIMBiME2mY2DdcYLe0aVwyjMh24yyuq6wKn6LGLtJajJgnowZ4SFPeREEdDHuYI40B-pW0KC8ej0Yhp9DzVmjUSe_F07Tr5OYtPTN-Woi9cQe9lanJ4Wrm8KxUoct543ZsrJxHWn5Ha5l6w5C8ce93svHmoFnxEHoa-3pj_x_PNcLcqQ1bl3A0elG7SDLxyKOrSClsFRpopYLu49HOiWt62Zj45gmBO6DDfQp9vOJSEe9jeD_RzdhYuSK9oy5y70pyWy4kieFDw4uHA_rUPTkotOtkaloNBvXssEXYc-BA-mmcItuNYNApIZlrdJdcGyyEHAoSq-nuyrKO7h2zafA0db7URnBjDH4oR6_93d4O3d4jro-Q58oG0yVgx0PcNgUYPtCrxbfO6xIaFuxuDzUw6d7UsA3RqNO6KU_PrZr5SeIEi3ZiQkMsCGbSvvoxsWviDx7C-36Gw-EphOr930aj0eFb6TsP1w-hGuXyAtUqU63ujLsqFNakm0QZts3XX0m5zrLT93-tTjFtVj58N5YcO5-fZT_8cPqjOJY2W5y5ZfrjiTg-BnD-mMPIVX5cyOWUvzAr9LRlM8uyAoPr8BUXBvCSLB-T5249D6D35nO1Tp5_4z__AbR8tYo=))
- [Using set as mapping table to handle large keys or indeterministic sentinels](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_set/mapping_table_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJy1WQ1v20YS_SsDHe4iJZRkKck1UGyjOqdFjRb2IXZaFLYhr8iVRITkqsulFJ3g_35vZknqw3KdHnIOHIvk7OybmTcfS60buc7z2GR5Y3CzbsRRY9ALGonKpoWa6sagERaRagSN3BQ25Ovuy9uMXtKZma9sPJ05aoYt6h_13wR08ev5h_MhnV1-_Pflx-H1-eVFh2VF_pc41FmuIyqySFtyM03DuQrxp3wS0K_aMhDqd46oyQK3jfLZbaP1XrSsTEGpWlFmHBW5hpo4p0mcaNJfQj13FGcUmnSexCoLNS1jN5OtSj0Ch34vlZixU5BXWDHH1WRbkpSrofPPzLn5oNtdLpcdJbA7xk67iRfOu7-cn_1wcfVDG9DrZZ-yBJ4lq_8oYgvDxytScyAL1Rh4E7UkY0lNrcYzZxj50sYuzqYB5Wbilspq0RPFubPxuHA7zqtwwv5tAbhPZXDc8IrOr24b9K_h1flVIHp-O7_-6fLTNf02_PhxeHF9_sMVXX5EsC4-nHOocPUjDS9-p5_PLz4EpOE6bKW_zC1bAagxu1VH3odXWu_AmBgPK5_rMJ7EIVUMoqlZaJvBLJprm8aeawAZiZ4kTmOnnNx7ZJxs1b3NbrO_xVmYFJGm47AITTfnJeEo164TFrPTfYlIQSLqKmvV6vTgI1fME73zyM1skbtupBfYe7TQoTO2s686NvC0Vqnc7r70cf5e-JcitjBy5Di6I_1FwVsa6Go6fD-2sZ7QB53CVmeV0znNzJJDz0yeqXxGMIgUnEOJMZ-LOYmyipkG6RZnKqFIOVWrvWeH3PvlIp5TXoQzViOPBoONs-45jLt3AfseC-ZzYx3Ik6zoTfcdqOo0fdar3Af7mrNsEylSNgbpaaxDxdAPQLAamljSmRRkOENKgs7tYRa1r5ZqTs2z4VVLdBuwoow_cwibIhNybf2tJm79UWh4Lm8FTBrxxEzZiNODMqxc8FalBXlpQlsMqDj2zl9iyw5dG9KZeLUymndNlJ1q2TwPoEJzolqDNOf04tzkYKY6c5VBiQk_V7b6SuHZX-YtJRqwOvSTWeKDDXydqpUuTZFEkFGS-IqrmUCCK4AmldIV6RBMg3ejAplm6i1mqLlki8zFqUbhyh3T4zGqTeEdRlHMzlRJsgoOxYpDiCggWFHEuc6pCvZ4R0MsNEkSS2HGU5MUQoFUu5mJtuhRG1fWO9FrxXt4sIiRPgr8Bu5MJ1CuuFZHKIacB0VWcGvIE4MQohDy3hXLCclShK6wGtFjH1c6FioppP4pBqbtAhr8PdGeIp1BUco4AsRVjONXKgck2J526MI4XfpWEAnz4ICSdytxxoE9uVZNcAtlXc_UIkax8H4IUX7H9QacY776aqmMHARazrRHEWfzwomh5FZz0SvNS38RqkfaccXMPAZsHUcHkIA8lZsU92MwCBSMNgQQWLnUI6iPmFpbJQjGVTFlLnoKMDrpLXnBeb5CLBnETo3KRDnH0KJYCid8CSyrFrQ9qloD30_bEkjOttLjcGMZmA0xifNIKAIIh0ogSX3vVCo_CcT7du_el9DHnsrVRNdFS4VhyfV7VnbT7t3ds1CciaNrtWegkUkFF9OZ48I1CiSHmFtJwBRcafghENftHvaLMbzCC23hZ9ur2WKsct-4S2M3kfsesdU-fmVDYYRcsDYhZMfPCzs3XJG5-nXo3JW8gEanQVQBk0uzgRNTFA1RX5cbR0uMQ5CJfMyqzVQyBWI3SzeNuG56vpt9QvTaVSbUfmFVlcluaWpr_UrH_krpmi63vCEJUO7iNHYHOdGUcTNTKHTX6Li-DlC6GvlN1hCvr5rX2BEiL4UaLRrQiD-s-b8HWj-wrMdOG_RnMx1-RsgnNUpPuoipE2KLOjTwSmxrNnqulroOcNI_qfby5iazXFif6IkDMUCKnGsiVOovu5K2lPTT9ROiVqMiZvQCPtEvKgtkqtmzwS_o8p_RqJxuRt5XPNvR2JikbMLGNltNUOZ1f-QYb0DVBSC1_BrWI46HO7vegaBJQm9I-w4puao8FKFUWcT0syCptImHNJ5PosFgqt3x0WlTQnkDRHctOjl5UsCKgO_6m5896d5z6np_SV3_OXX9v6Tu9XPqXu-o40MRCbdr-stT3H94_1y6SnWaFFlYFX9fex-V2U19kubsSfjNctkX1jqZ_eXXZPMWm1UB9Ac4LGAPEveSx0TZyw86k9hiWigpvM3gp4vtV3FWEPwPcXKxNBcZnmCNYMRCV6ObYu7N6F5A3h-IxzXPE2XXFKE_j8fCYLaQrSrX_8Mvw5AuXhPTQoNx5fgYx8sbHC3x4bHZfpEXCuigVO-x1E5KHFjS_4olT4F6Xa29Obr7U1wbwd7d10PbrOp_rfrXpeDd3S1i32ASMAc4zqmKsyYHo_I7uHo92x0c6lmxPLVgcs2qM041bPAIyT1_rpPEE7zgD7yQY86a_Uj3M_TtFBnBeFyUKRRQiMNWIF0i2BaTPnccmQKTWkBvPv1yevq-xHueqinXe95WxNC-_iOAZ8WUTZEn24akxuoKNpoOj-244kabItXy2I-TpfpNQ3liJuaXPkxUmGANT90y0G8dKXy9EJbTTPKQTnyYxTR_9j-GZ07X-45Z93pFQC_UCxzp0Hkfu2Td6wTUx-9r_L7pPDwEuwQ6VPj_35r7fdY8flrzW2j8J36_w--7b4r5G2ieqCT_Gnc8vN8LrbDupIxwh6-avhD7Vz2Dwc67Hok3Rb7P-DW-NNc5qHAI3ctAlfkeIzdwuC6z5r7OTtKZKXBgF1CL8u3WcmaS6vBnVTbV5SacvNWri_K9yfCqzg89mcRhrKUZKH9YKpPeZPKmrMzoocD0eV3iqTJTDkdyYD6YOFgeijakEKqFPzj7c9LWIYl1bfbYDJJoLG41gndGtdoTavdKH27FhTWN5s7isXd3h_9rtjqokmWERJoPmlyadt9hrf0N_YV74bHwgUM7cqdrCfhL6j8E3p1jrgJvj_5OiVEReMRhfoJ_W0QU7ZUt68dWPcnh-qc6k6wrS59f4rdNoF_ZEb87QCSPe8FmIhJ6nq53rzcbPDzJ1Xz_HP22ntT34uLXjGTNyRNJUgf-dH0UEAD2kYDIP598_D7WH-ibW8o6Y41-0OT3d1s3cS5ttlrb7Ciy-I9C-_03P88jaXLcW_vZv6VthL2gh9FhXMOAtdAjnFmaWyIVxkdqsiLdCdSJ7xH8yp1f0x3SEezvXVm5N0MhTta_5fDzAm8lD-SFPnLRlTWiitegGhdIxtKmwIwB6eg9_oiC9_TqVdzaGnQ_lgb7t1mEzVnj_isHL-7nP1_7brZsuInvNiMs_ymH3iM_uTSCBvfuONF28x1SI1uEYa__tujhsZk7_wVTow0fnISvXvW-o7ay4ewkT0ffHVG7jRLi2pLUkY7aiUrH8q1TEo-3dIZhmODmwn9PhBvoOtnnxkNQPUcW7TwHZRoPd_Lvv3bzBxQ=))

### `static_map`

`cuco::static_map` is a fixed-size hash table using open addressing with linear probing. See the Doxygen documentation in `static_map.cuh` for more detailed information.

#### Examples:
- [Host-bulk APIs](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_map/host_bulk_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJyNV4Fu2kgQ_ZU5n06F1mCIrqpEQnVckupQK1KFtFVVKrKsF1jF9rq7a1KK-Peb2bXBTtFdEwnw7sybtzNvxvYuMMIYqTITDL7sAhkHg34YJCxbFWwlgkHAi5gFYWBUoTldR89nGTyHS5VvtVytLbR4G856Z70OfvwZwuTj-Go8gsub2_c3t6O78c2kSw7O6Z3kIjMihiKLhQa7FjDKGcevcieEj0ITGzjr9qBFBrOg3JsF7XOHslUFpGwLmbJQGIEw0sBSJgLEdy5yCzIDrtI8kSzjAh6lXbtQJY6jA59LELWwDO0ZeuR4taxbArMH6vS3tjYfRNHj42OXOdpdpVdR4o1N9G58eT2ZXneQ-sHtQ5ZgekGLb4XUePDFFliOzDhbIN-EPYLSwFZa4J5VxPxRSyuzVQhGLe0j08LhxNJYLReFbSSv4onnrxtg-liGiRtNYTydBfD3aDqehg7n0_jun5sPd_BpdHs7mtyNr6dwc4vFmlyNqVR49QZGk8_wdjy5CkFg6jCU-J5rOgVSlZRWEfscToVo0FgqT8vkgsul5FDJCFZqI3SGx4Jc6FR6wSHJ2OEkMpWWWbf20-FcqGiWzbLfZcaTIhZwwQuuIkMufJ6yvMuL9eumhV3rwtgoFhtEmW8Et0p3yegnE6wMS05vSSs0Q8foh8zn1cVpU4MwAtV2etdqlhnMTtp9ypOnzDZduLFxLJaNNamwuIKljUWXNePwoudebX-5Llij9XxRJA9z8Z1hvQTmx28vtBRLuBIpZho5WYH5NlSfUvfHlKJ6CAHFQ2gwej82xza-Q1PadcuAEkXFJVsU70Z5XS-1Sh2gc8YyO6OCOp80EitSgsopoa7oiXxAEWOxtUWRuRhLmXkhgxGW-D2I7ZGC04PMLLawzFobJeP2LNvhDgYh7Ldii_06REx7flz9yJJCHFZpPYrgOs3tFkyirD-KFiR2kVnfrfRbb_D3LKBFmYkEk7IhJNOlTKDu_RWYtSoSdMI8igQVh06m4Osyjl0zSznBuSWwGUBxXmhqeBxn9J0XFmJmGZ3R8edUJGo9EERxjgmYVxTcOBpCp-9O58_11N6xOnqU1iWbSZEuiITLa-RMIWdSG5pCi6oYgvoTVREPBkb-EHNbC5IVKVEyCPyy96zX6x3BL3H44iACTlNSYnYXzJRjCW3_gEQxFAKjniQPVmDMIzDtzv0uYve6L89PkjiiD_0mFzJpHVhFdZx2nRtJv-BUbSChu9vDLKjQsLheC14ynb6TL30Zp2iX21rSqvw6bfqjEOoQaEYh40NHOXXCgXVYGhxqu_u5yvumkQu4O1Xc_b52QBwUmHvqGz-SquZxBymlutv1wh7C7_phH7-63S7sZCj3ewLxQ2swaAzPC5Tk61IWLsOHVLu78sGpitqqmXYXYoVt2g7r_l2RxbTUa_o3gzplH8J68r8YuGzRp6HL5WZwVzYc8jmqdOjrVMGm7EHM6_O_1dixBc7X_zxrk0e7JsWxM8C6JEnZeziYlFMZ6oWM6N7mYVqeXVixfNFIJfVFqw49RaI012naLlVxKPz_ZtoZn0p0CfwGx7Jn7CSFMoJ-CGdOQnunMINoqDBmjOKS0RAsNecOd18PcF8lYolyRTy4J9C5VXOa_l_k13u8VQiTPbP4CILPN2HTHQ1gODw57e6r7BHQL0uxDn4o2PHs-Ggql1s_yCkDVKh6cjGg5f5RsxqftbQvlErKyYXOZZS5d3miOfdI0jrF5gnHkvdppZXE5RJaTwO2YVfOTIV3nYsLnH_TgnN8xvsN3rgT0flKvBniBOew93Ba2EJnQNMeV_DFgJ608alDH98fgmzDef_sZdHHbZVb_3IRdDDikL940X8FHab5emjS-asedDo49S1-WDyNiDsJSxfujSORixom5zzBxY1_PcAFHOLZQ7APq30clI19lHew_-r-_wWLXH6r))
- [Device-ref APIs for individual operations](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_map/device_ref_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJzNWWtvG8cV_SsTFmkpa8WHUTcAJQpRbaclksqGJDsILGM93B2SA-8ru7OkGEL_vefemX1RtC2lXyoDJrkzc5_nvmZ3vUIVhU6Tojf5sOvpsDcZe71IJstSLlVv0gvKUPa8XpGWeUC_h89uE_FMvEyzba6XKyP6wZF4Pno-OsF_f_fE5fvZq9mFePnm6u2bq4ub2ZvLAR3gQ7_oQCWFCkWZhCoXZqXERSYDfLgVT7xXOUkjng9Gok8bbntu7bZ3dMpUtmkpYrkVSWpEWSiQ0YVY6EgJdReozAidiCCNs0jLJFBio82KWTk6LI74zRFJ50Ziv8SJDL8W7Z1Cmlp0-lsZk02Gw81mM5As9iDNl8PIbi6Gv8xevr68fn0C0etj75II5hW5-r3UORSfb4XMIFkg55A3khuR5kIuc4U1k5Lkm1wbnSw9UaQLs5G5YjqhLkyu56XpGK-SE_q3N8B8MoHhLq7F7Pq2J_55cT279pjOr7Obf795dyN-vbi6uri8mb2-Fm-u4KzLVzNyFX79JC4ufxM_zy5feULBdGCl7rKctIComsyqQmvDa6U6YixSK1aRqUAvdCAqGIllulZ5ArVEpvJYW8BByJDpRDrWRhp-9kA5ZjW8TW6Tv-gkiMpQibOgDNJhQUcCP5bZIChX590dZpWXhRmGag0q_loFJs0HtOnBFnWngpJ4-1kKv2wP79JG5RI0hn_ozK9-HN4apUu4Nzq8WAAICqg8vGrKLLJLHW1jabrbg8KEoVp0nukUAFAy7jxkyxZMb_jMIvJHjhRnmFwtfHUnY2IblHZ9nmu1EK9UDHcY6KnglIKc6ILDHhWFBoOLt7OC3a6TUK91WMpIpBlZh50Z6c8AJ5yYm-ECW5pMMPvyfgnSyTKqDhLq6KwIgOm5IgCBYQyYI1icLEEKWdZaknxM_rbXoIOURBCYbaaKRoAfkT2UuKHU4QxAUUSqhLXmlIeyMs9SRDeCKtoOxMzQNso8OjEKWOWwLVbphm0TwwlM3kkpEyM2yDLYE6a8o2Imo2WKSF_FtUjDjpesF2ZsgkJ8VtsCx6URmSyKdpBBAARnCMQZNomxbCgqGl1NJnMZi__ITNzADJUnsamyIGykcgJm58DPajsD1iFGVkJjB3uxWcEiYi2jUvlkVtg_QXwbTTkNAoDPZAKRebFD8D2deRRJWPkQVYicqfAh4beVDZBxZb5tbNIYgTd-0MlH4UAhriql2wZhE25WOlg5ABaoIFHUgd4-QdJ1rpYA5A2o8DdOdo5sjjTIPKwbU0d4n4rV_jF0eCfyZ1GkgWbHc5FTqEnE48sskjL2WQiib1KD8MOjOfxRSYec7MjvnyWT7lvXFjMbBBX3gXiDUDkE2RZQnUmZhxW1KiqOY1oaK261-mWRizIIUJ4WZQS21XYXUUYh2kjQM4JMImNFOPJE_ctBvPWkxmjrWQ0vpFLfX0bpXEa-L9apDqnzwGaA0rLuU5Q5hFHN_cZfFWI1hB5xpomiFmQecQ5VYzIp9B8IsRoJjzjWxBY58BEHEELPOr47uk12dEyWlJ5gsynwgGIVzsK7wZ04FvMoDT7bH8_sj1c6HtydUkrsyh2kZUKaT8WIukGKVKSHPhE9q5U6Ejsr5XBowehKydcwaQ_ohejTo37tjw8g_fGopsg0ORU5mrkyZY7a9cnkpfpEBDTXCCK-0HlhoHBs-6SlXquEA3Qjixa5GuIUfJ8WMiosITpDu2VEttqidqDJY2iLSlYHtIHDHjVGk0kmdb7rKuC1ocJP7ts6CXF87Cx7yiK9pB-tQDsYY2zPisS9_eI-yB_HU7HMdcie3Hcrb6QP8HqXhQQuG1XWwcTyARcUilgHF2HYb2PLqyBB08H9gRr6U5q3HL-Sa2ox0FJxgq_SvofvQa5ilRh2nc5BNkfXm6XoVJCC2XzUvqeJ-j-qrK2U-eiyhg4Uhvl6WXtySSOi2zaBb5aaJ2boA5m3dplvK1Yn9R7IrAcT4P-Qmx6RfWZNFHP0Fh3McV9bJRC0tzUOXBfX9Hi2rtI0ttE08JmaQ3WcOj40pP0ju8LqLBAZpFCVJYjffmY7bRKf3f5ds5_pdVPfA4U8F5fSZgbnEd7U6k_YQRUZlo2QOm3yD10zTCaWFPHuoKHT93lur_WSX2BwV74NtvMdq3ByXigETnh_WtFn5RVC3pdIH-OKRIxWP9_6aY6ZE0wjeYdadfr0RMZpBxEGw-mkT9isUFXSLENQBKkpBeFp85TLeP3UJcPXcWa2oohSQAXDPySn2RsWtZcH9D1f4zsmHDzUiYow21j4DyjUqvRR0FBSRiEFeKEiDMA4hCy-cnw4GcJJW5GoNQVlEJQ0xtHtCn1SWkJalpTrWP6AhiK6CRCKRCSk-5UIbOOpOBmzdlav_f02qdUn3G4nzWU7MwxtrqUqxqllrtpd3V43UDGp081UvBj9bTQaNcRfAieG50o3fB9odsVuN_JG957Yjb0xPgaDgdhpT98zDOx4Ppl0rhPOYJVzJxmz7tcJ4LR9qOLab20dcAD2j7z2eRtunhh1z3eZsnFrti7vPY6xQ8k-a_e4zbwyXBoDBnAl3XZpwHIuC3e9BCN_L6JUov2VJFedQRuP0KpvV6lbG7w4Pei9hvrULgZKR7VGYtim05GNZvQyoDDhAscT0G2vooaosEFkY-1kzM6mD9uaMShbaKuAyVcEVhWiOhW2pWquE3YVh282wvZkHS27h3Fz_yQaLOjuUDR9m06FCUAC9dOkDN7dY_lHYCJzP8vTOYx5xvmTnodqIcvI-CtZrPxFmQR0ZcKkQfu-cda_lIGXkjQ5STcJX27GpeEb0PxQr8K5CfUk3RT1DA5_cM-OHGgdVl11OBafVU65sPKdwzYXGS6N-Oa6Y7vUglIz33BHlLYGBuUa9qIhXPj8cCo-fBS-qzu-3-dF7ES1dCVZUKUV34vnR2JK04poGeQCyjHHAqimuzW6eKo736ZPetADH04KsMV5Z-Dqjyv99oKSa5dP8QcNnr_4x2l3D9c5u-xyehOJx-3DJ2J8hNhsnjChvVH47Oyspue19p6fn_cbBz1ioOz8HUqjf47EXkJ8IpEnTNBd5hZBf4ZdfVNClbl_NFgqg_6sPSUDQ0acnSERXnYvSaqjEyzRhja5D6OP9IgJoAxEXw7cb8crt7StK90nxi0dPxi1tFBput_3fw1nFUHvIG5Eu3I6ia6UwXi05g7G8DwIVaEdq0umUK0Wrenmv9EnEDE0h4iNulVomf8RFb8h0Cr6D0hYo1kFfAjd7_JtFN8nV6108ga_j-DXHVWrXokYy8_Kb78L6XdW-ODTeVsdaMpgqgAZtaZke7J24xDu2zDELrYWe59oeUDp9liMKdvybyb-ie5iZLSRNKbmpWrKO1pp1C477rHk4OOni35XUesLr2WK9nfw23ODd6gucG79qz3XGqdcnahYIZrPRud9t8tp0l4cV4vViFIHBIY3qw6Vn71EcG2vbr4TNzIHlfryOKfLL4rU9txGt7Cg2Tut7mecjCN7udLzevRmFUNv3rwv7iXrIBg_f1GOsZxmxr5M7p1AjmlwfDz-QZzIPFhNi9j_YSROTlCIDP6zr09OIhnP-Q1zpOctmkEQRHi4tq-D8QDOSz737r1qHVmhsw6L9-4_8r__AjWPYDc=))
- [Custom data types, key equality operators and hash functions](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_map/custom_type_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJytVwtv2zYQ_is3DdvkVpbtbEMLxw7gJS1mrHCGOG1R1IVCU7RNRCI1kbKTGf7vO5KSLT_apMAcIIl55PG7u-8eXHuKKcWlUF7389rjsdftBF5CxLwgc-Z1PVrExAs8JYucmu-tFxMBL-BSZo85ny80-LQBZ-2zThN__RbA6MPwajiAy-ubv69vBrfD61FoDthD7zhlQrEYChGzHPSCwSAjFP-UkgA-sNyggbOwDb7ZMPFK2cRrnFstj7KAlDyCkBoKxVANVzDjCQP2QFmmgQugMs0STgRlsOJ6Ya8q9Vg48KlUIqea4H6CJzL8NqvvBKK30M1noXXWbbVWq1VILOxQ5vNW4jar1rvh5ZvR-E0ToW-PvRcJuhdy9k_BczR8-ggkQ2SUTBFvQlYgcyDznKFMS4N8lXPNxTwAJWd6RXJm9cRc6ZxPC73nvAon2l_fgO4jAh03GMNwPPHgj8F4OA6sno_D2z-v39_Cx8HNzWB0O3wzhusbDNboamhChd_ewmD0Cf4ajq4CYOg6vIo9ZLmxAqFy41YWOx-OGduDMZMOlsoY5TNOoaIRzOWS5QLNgozlKXeEQ5Cx1ZPwlGui7dqRcfaq1kRMxI9c0KSIGfRoQWVLmSM0SkkW0mJxcbgjJq1ZIajRSpKLukwv8kLpVsyWeEO0ZFTLPFyc2sI1ywlKW1QWwkQlqlae2K9zIhS6I33iQCLnSIXktHCrJHTWtVrwXrG8GbMZFxjme_YI-jFDgmDoC6qB4imZRrgemXVYo-uQUvrXs0gDOa9_m54bjQBRtJBKRxH-U_ojOlTjN2C9ee7eSv9DA7pA1g-bAKb422rYnJ-yYkmSgp20w0pOWDLbs0Q9bclO0XNsqe3es2ZmrVHftsYpgwVRC8DAJibND-2ywrVDsb28qO6SmSOM3_AP40kxQfTPcN9w_2EBLAveGguMLnIB9yE5B4Psm-gMcbAgkeSrEM2VbschzqmUyTMwJgsVHPGxlOULdWiBucTeBJUlqCAk0O-b3aHj7tbp6Cqs3Vz4S8njxkTYk1ahqVSgdNztKv4vBhFEkUYZ4bmCPrxu_9Jut0u6oGvGTANLNZZ9xUx2s0QZCSmwEDt4LM30o8Vf7UCA_UO71s3O5vzkSUel7dn-McXKsyWiy5wRzUzxrsqG6UhcZIU2QWu5XLH2bO-z36Ipm2Pv6IMrHd1uSu7xgqMq5Dsf7-06qm29kooXfrsRuAOmnHa7WS5pQngauRhZA2wtRhGi6B34JTg29-KiRADw-UuNVr41hTd2TN6pXR-6m29OaMbVDVK_0ah5U1asJkiXzA0CnXY7QBaASqTGbqNMTzLdZs6XTLiw1Vy9JUYII6lttyuVUzMBcNyMvZdiLcHD90KujLqVmTmSBOOGuaeRdvZCVIopYVokhjeRJIYZoWWEX7d_CrcBNVD7pf27Hrd2C-xBI6RejeKBMckQ-2K9CSrnfu1TKqlovT4m-PfpsF5anyL7c_Tsl5pnw09QP8kjZOMU3d3rBPXCil6oJdTQhkBVMXCVAOktbcyNp3Eg25HA6bHVm5k4xdWKBWiivSuYYI6HLsZ-LQmDvYx8uatANWbeMAQfF1iK3XnXzHc57VaNa7BunUjZE4n93Sn91OebKb-XkLucfurzZM4fJno9oysL9-a2nulHF6bomjGexf4pd18xbaZOge6eAQbQxnrP8xhXAVMzweL8uc8JRVJWZ0RFhVJ3NWMCUXDndN6FMMR7bJu9q4XyM_9yB7FkSvyCDeIBB_YA7rbIUWra3Ywkit2FFb9KsfJreoI9ftQYFuz8EFr6-dZ5FueteSTZsjQjPIHYzFyGaKZdIRLMWje0GBO380FYHv6_kJTqBtviWLkfnzm7EwetFAMWbWWH6WCEcuYf3VYHwERsFk6QrxxIpjUOTg3l3IA5A3_vbrPLFl5MLw29Hr6wxgWl-DL6Ad5a4hhu2TKowgla652bmcXoKpWb4QNX8Dlt3qf4Ys13r25PLCntnP1edFAsM-2e5F4Tb-zTly87r6BJcrroqzR61YZmE0cd3bTtIGZxMyHp1L7TEz6t6aSUJri4dI9qXMB-KO69TVDJsaDuydE_3uaL_fkPoxeK0w==))
- [Key histogram](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_map/count_by_key_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJyVWQtPI0cS_it9PkVnL34AySYng60lQC5WchABmygCNGnPtO0W83BmejAO4r_fV9XTM21jlhwrsXY_qqrr8dXXzXOrUEWhs7RoDW-fWzpqDQ-6rVim81LOVWvYCstItrqtIivzkL4PPtyl4oM4zZbrXM8XRrTDjjjcP_y6h1_fdMXFr5OzyYk4vbz65fLq5GZyedGnDbzpZx2qtFCRKNNI5cIslDhZyhD_VTNd8avKyRpx2N8XbVpw16rm7lqdI5ayzkqRyLVIMyPKQkGMLsRMx0qop1AtjdCpCLNkGWuZhkqstFmwqkoOmyN-r4RkUyOxXmLHEt9m_kohTW06_SyMWQ4Hg9Vq1Zdsdj_L54PYLi4GP09Ozy-uz3swvd72OY3hXpGrP0ud4-DTtZBLWBbKKeyN5UpkuZDzXGHOZGT5KtdGp_OuKLKZWclcsZxIFybX09JsOM_ZifP7C-A-mcJxJ9dicn3XEt-fXE-uuyznt8nNj5efb8RvJ1dXJxc3k_NrcXmFYF2cTShU-PaDOLn4Xfw0uTjrCgXXQZV6WuZ0Cpiqya0qsj68VmrDjFlmzSqWKtQzHQqXRmKePao8xbHEUuWJtgkHIyOWE-tEG2l47NXhWNXgLr1L_6nTMC4jJY7DMswGBW0Jg0Qu-2G5GG-vmA6mcRY-2N8BfF-GqlroL4skBEUDabJEhxtzZpGXhRlE6hGGBI8qNFneX-xaEmdzBDTePWlymRbwTNJ_ZWMizZY5MCVSs1djOjUbYzpDsJVMNgbZiwXrGHyw2feJqyLMytQE03XwoNaBepIIIXnCrpjmWs3EmUrgfJhqFEJQUMiqUrDHF4WGipNfJgUHWaeRftRRKWORIaBV6GL9gFREyHIzmGEJPrMOEhNmqVFPhqRKa1Bvuu7BoEZAV-i-6rN8KVDSJpvnMhGUOQILiwZGJm-rl7A0ncfODkpZNiVEQUwVZR-FQpFt7mhhhqM9akl2svi7VpNaSJwZKsisl8oz4BOgR4kbwp3Kn1SCZHlUO5JAbFnmywzQgIqM130xMbSMYAsBVUh0rvlika3YRwmiyuIrK2VqxAoQhTVRxiucMhnPM8DEIqlNGmyE3QZ1wi4o2HlUa9bvhSB9wBaVkkwEIAxLCilr0OmyNKIAXClgp3dis5QUje-pmq71X0qcfj47EVxccPlfamPVf-VS3MBlLongSOdt-FPlJHtjw09qPTGI84TVa0NBhTdXC3hPPMq4VAGFgNII6WA0gSfcAj3DIWU1TW4I_JxqHIFlXpbmy0IRky25JeLz7TdBg_6fWOqtTu9FlRTiyh3EPyT2ZVCgw0WVgAXaTxxvpN62QLJ_quYIwA2k8CdGykos4MNWI4eRewQJ3paSlknAK0iIyQxqA0NTnN9tpQzgUxf-3qw0dnPJHntbBjUYYI2xwqwR9jDIPaOQlwAPQJP1XJMoXS6eVCaKwuV9q2LujTRBA4oFwTzOpjIOAvGY6WgDxdqUYFUgqK996celFpn93loX99qb723w0mzLh5279Jk2l4RH1htX3IPESKA5DYfe0LHT223cNiauI4KgWIACRHBC7SVv43B4A8dfI6sJsSkIQWG_HBEeCOHOExIoiTjLaEFOWD4Sc_x_ppP-E5KhVnvk79LR09Z5Rx4AfLDVP4meIGIPuYqOFPG3SnntTTu1kWEjsc-qUCsouTZpOq693hHP1u-DgcNx6hRcXK7SmNVVrYTS88DukCXW3BZxBl_qIkjViiTeQ1-VLn0rMEA1BNQZ2sQlhsOl1Pkz6b6FKfddcfDCZBM_egbzakm1aWzcTOdwKwPoBhRgoVv1-ux7e5XkF6FiIJEvkI-JUo1p09qBMp23N8a3MFeJSo09tdvGRyZEosSK5HBouQwdlhPr60NKLDtVWVOA7KrAIvL4mbzVGxcKSRK9HDmx5KuZMuEikFHUPnASEjS4fB1kOWgaVMTySUXOVS-VxxDMvZGfbTz_YrMCpyRyDqLqgUtRhiEY5qyM43WNLQI-t2hDnVWBcNuMq6QQntmjStoFK0WVH-ANaWQBbCMPLRXcTEOvmtp-AXX612XSfh09e1ZKCi_hxQj5XKfGzijY8n47ClyOvqK2C8QHSySHw1yugmVG1CEPQlmY9jbkuDhsi_Li-MoF78cVcePQQTEyUadtgmMHb4jDBXEhs5CGvUJcIJu1gbqdverzKWVrpzP-NwWnymid6qRMmnsRswmZhwu06dCUOXflIgm-229AFDIZglxSHzVTrGJzqrLuPFmataAUR1nlxD_oLoMispcx-pw_4jNIHwZ1qmLQPdsm-9QHHVUoiKeVcUStvFAxjMQmpO2i0sMOYD6VKmKsNavCbTWvaFUkjSRKxSdhSKablVBkIpNzZ0KFtRUNpVgfY8u43TuwIbHH3ZZgKU0tY3M_b3ESXOQ2CQKqZ6r8zm6hpVFTc4yR-Lj_r_19C-EQ9GPNKLlKfapgHVE4KKvpMjlkhwryQlTyBZmuIlDEOqqTyIJsa7drQ77a2tDhStzRte9aP_DltGHs3eY4CcqLjk63ioI5IDJjyxLYz7VBXlrCDkjcpYXOiOpCKqjC9O2TBa1zNbxxm-SgVg63NV93QOfZUyCF4WtNRcldtPri_LWvF7LYthu3iDV-of7tBaa2pL6XVkDjxhP5oAJuMCiswLFmm3_7ne7fX-ydq09I0O643f4M8922m7m9B-WpfBS0OTc0cBVlCkxIXxeERgY02TDYzobOkXjx8v0kBvqROyuM575SM4imG1l4ZDe_HTsH6uNt5tf2S-y0anUhPRhpINFUFtULDWroKzRJGYmZJIk7qoFmAztLjKn_sRE8sUyjMRqYRNygIOB8SLNVSikro0d6A-uKleIrMD2XUXJCiYzDkim7YxjORFtv0XBIQau5Ix2yVjESb_rcPzpdhMuQgLehbDsMpmYdqYL7gOePfu0Q2kzkhohaczV_9k16j6j7rnx3rdVUw_Lza4B--b9kMDA_7wLp9-W45EP54y5kMs7757-rP4YSmQfLPJuiQI-ZxdF4pGayjE0AwFgEsxL4obOURUP2SxPD_yiD4KVZ2kNC8atkUhp-usx3XX-5CYKNZatC_LFFtP9o3mochlZKALbslYbgP6icGrEX_8ASXHrsw6eKtG9pcGW3VUSW8lAyQ8Dhx2-PNtdYtsTT_DPyAGXP39wTBx2kezPCgvxr6XEzN7bxOT4-rsV3va3j8bjt7q874bC5gW7DS59YRLvjYcyVAskG6bCPbFRR9MoWxxYgEDvlcaDmSvFOWwIzogyp25Jnwu3-vW1Qu7dbquEE2EenN0XYiNoDBDC67eltvLEhyw37MLtQ4QPxz01MtNuqFwt65cE1LdyiHbWng5CFjLYdDlOJVXyhzXiozB1Q2UdLCkFlQfXyxk8wKteztccXiZXutszuqu1y3obcALTaZthOz2w7TOGei0HurousMEHgt1mPTVqV3iW3ary2ScILr8CeFtU9li4AW-5s3h3tAHVz7i4YNKgPcKZre_n7xx2EtI7cPbHSTBwTI61ui26NOlZ584eqVvoYhgeHH8sDTGdLY_-K1epB_ijc2zv4TvToSjHie4To9QAHBr_s02svlsmU_7QV66knMwzDGIOP9u9QGEAPSx9aL103D-jZmIcfWy_3_O9_9qthSw==))

### `static_multimap`

`cuco::static_multimap` is a fixed-size hash table that supports storing equivalent keys. It uses double hashing by default and supports switching to linear probing. See the Doxygen documentation in `static_multimap.cuh` for more detailed information.

#### Examples:
- [Host-bulk APIs](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_multimap/host_bulk_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJylVgtv2zYQ_isHDUXtVJYfaFDEjQN4bYoZK5whTlsUcaHQFG0TkUmNpOx6hv_77ijJlpsM67AWiCHe-7vvjtwFVlgrtbJB_34XyCTod8MgZWqRs4UI-gHPExaEgdW54fTdPpsqOIN3OtsauVg6aPAm9Dq9bgv_vA5h_Hn0fjSEdze3f9zcDu9GN-OIDLzRR8mFsiKBXCXCgFsKGGaM408pCeGzMJQN9KIONEhhGpSyadB8671sdQ4rtgWlHeRWoBtpYS5TAeI7F5kDqYDrVZZKpriAjXRLH6r049OBr6UTPXMM9RlaZPg1r2sCc4fU6d_Suazfbm82m4j5tCNtFu20ULbtj6N31-PJdQtTP5h9UinCC0b8mUuDhc-2wDLMjLMZ5puyDWgDbGEEypymzDdGOqkWIVg9dxtmhPeTSOuMnOXuBLwqT6y_roDwMYXADScwmkwD-HU4GU1C7-fL6O63m0938GV4ezsc342uJ3Bzi80avx9Rq_DrAwzHX-H30fh9CAKhw1Die2aoCkxVEqwiKTCcCHGSxlwXadlMcDmXHCoawUKvhVFYFmTCrGRBOEwy8X5SuZKOOX_2pDgfqj1VU_WLVDzNEwGXPOe6bcmEx6s8dXLFsojny6tTNbc0uXXtRKzRVbwW3GkTkdITFemEYShtc50rgj-uTp7Xt9hPgeR6XuoMUxbBWEU_ZuQrtf5QKocElKqx1jJpTtUO60Q6E0aPYhu7bSaQcgOkhHt7FK1ZmotCWIlIeLAQq8xtY_q0ggoRqWfuAFpd76VmX6gWBwflUpFUObbDUevBuqTft_IvNIQxqpx3XnY6nVKt3cZdgJom5w5bClU_irnrdjoh6oJNNUqLGlpd6rz_sb7TPhOqoe2zKb1WOdkIxtoVTOM0dxKVkfF8qVEFHpXekNcNTXqaIihWGIc5-rjo1IalQ6InzkWqWQJzRmSgeT_vvIh8ucgprPKUVJcVsGENuSvsW-bbBQjHGfTC0vgA_u5pG_anSt7b7rkW7PclsAWZ-v0T-l4WTjImzfO5XQHJbGPcrPXHCIYAMqhoS3V7tQiuBS4yggmW1I2NxuIcrjYblcbX0SIKYbdDPLGGXTfs4k8URYAnvXMiQnEM_qMU7usFHMahUWBWna_Yo4ifDNwlsvqq0WmGP69clEIEbTQru-JsJhY4YdXZ_TeI4xLOuMFyWrhN2OF2drlR8G_Q7iS8gMYY2tBrhiD3b2FfA3nkeYcTgBz0wWk8tWct0oWUaEkV7GycZlcmK1SC-R89TurdMnomPJsJdyC4iy68vnh5cXGx_2fCVKVceevY6XguMZDnx9GmYkajrnRM8OTU5xlCp84wakwxzJrz3BjyZSlxnzJea_eYdbE3mrRZ6NLFmwoFHqBqh5fu7vDsIdZ4mZkHsPl8Lr-jaoK3phMUhbkfQlEkhg8C1fLspfXQ9c58k_0mA3SX5S4mmuAO8zcGZV2E-em6m_9zOpMYr1Is1zZq-dSQ_ICRChI9bfbe702LgRAFZq3mktF9f9icdeI9HCI9VAwljGjSH-pl3ctvD5BoYdVLhzc9PiPCmi1KIysQwAQGg2evjIcDzGVBgnQ9vjhXRoq1-I8Qh0eQKrWCrD7KwWnZyFrU1lPDI7DEKacdS0Hlq5nwu9_nVS08sEudp_hEQ-6N4RX4QS-qs35o8b3ZOCHR4MdkBnAwPBClXC50XeKU4kuanqb4WDXHB3eg1px3e-d5F8U6c8VrPGjhvTvgr15130CLGb4c2FX8pgOtFl7KDv84rFkkrZStZv6JnspZzSfnPMXDdfGexgO8odVjsA8rOVL1RI5MDvbf_P-_AYKkJA4=))

### `static_multiset`

`cuco::static_multiset` is a fixed-size container that supports storing equivalent keys. It uses double hashing by default and supports switching to linear probing. See the Doxygen documentation in `static_multiset.cuh` for more detailed information.

#### Examples:
- [Host-bulk APIs](https://github.com/NVIDIA/cuCollections/blob/dev/examples/static_multiset/host_bulk_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJyVVw1vGkcQ_SuTqypDcnxZjSIRuyq1HRU1wpFxEkUhwsveACvf3dLdPQi1_N87s3sHhz_a1JZsuJ19--bNzFu4iyxaq3Ruo_7Xu0glUb8XR6nIF4VYYNSPZJGIKI6sLozk952XkxxewplebY1aLB00ZBOOu8e_xDD6NDwfDuDs8urD5dXgeng5anOsj3-vJOYWEyjyBA24JcJgJST9K1di-ISGicBxuwsNDphE5dokar71KFtdQCa2kGsHhUWCURbmKkXA7xJXDlQOUmerVIlcImyUW_qjShxPB76UIHrmBMUL2rGid_N6JAi3o84_S-dW_U5ns9m0hafd1mbRSUOw7bwfnl2Mxhctor7b9jFPSVkw-FehDCU-24JYETMpZsQ3FRvQBsTCIK05zcw3RjmVL2Kweu42wqDHSZR1Rs0KdyBexZPyrweQfCIn4QZjGI4nEfw-GA_Hscf5PLz-4_LjNXweXF0NRtfDizFcXlGxRudDLhW9eweD0Rf4czg6jwFJOjoKv68MZ0FUFcuKSdBwjHhAY64DLbtCqeZKQtVBsNBrNDmlBSs0mQq9RiQTj5OqTDnh_LNHyfmjOpN8kv-kcpkWCcKJLKTuWN4ip1mROmXRtWWx_PUwzC1NYV0nwTVBTdconTZtDnoUMi9yyeeL9On1VC-oZM8sWiouUqe1H56vNNUERXawySdrfWTnZWiS33zzLil6OivS2yl-FyQzUkZheWYUzuEcMxLIGeGQZLIsa9muD5SgyjMMFZ4hYfBhaPcjeE3xvOofA7UXdUu6pcZb69CTc6Mzj-o3U4l8UMFTy_VNNFdRUxnLgqXqFuFGUaWMu6EW8afcGKRuxDXe-GaEHTVifIvbPR8qrModTaLKG2utkuYkv6PndB4fQ6FTt10hnBJB95Y1A-h04CJbuS3YVLuQgkFuUMxdmDB-bdb0ehLxQ5VjSmKsRVqgbbMC1KvhHdilLlLaRCJiSg1Cm2whl-U5bikca0FeQ7kY0FIWhoeULIj_rwoHiXCC09mTlVwmnhlA5jnlhYoHZdLq7RMZFdmMcYMq7ADEJGiJPBtU2qTft-pvnLoabl5kU7_hFF53j7rd7h7xjIyPTAAkO5QilWbClpZAsT9DqgUVUvAo8A5R0Jl7YF6dhlXC7rZfv32SxB79NCxKVGljx6pTx2nWuXH_FpKrBtwN3ppJ4hQFgd5UqDdlaUMTtHogrO9IL2dQqtLTK892QAQPp-Ckqsevu-67qw6Iyy27-tw9rtT9fY04TTFpyqTDqO8qdteNoRfDcQztdgzqvtzwGY8Mux7z97bOFQWkO4O3gduQIZHBw6Ycv5z9_6gm4PGRd3RyDbcbGAh-0-8fGFotTw6sV-HYX5i7bRX5hgec4YJmrhkHeMwTft2tVWvoSVOp0jTkSiOogzMIu-T6ceDOfkOOz2IHKjvY4FsiQ6i5Al2E5AN88wWp6gpUkxpGMAhIg8gwFYX_S2fXlUXuQoN5cMMaWeaUCSeXpQ9Z7nz-qECFenBu-5kZYVxMprpwNJFTXqNx2V9XvPwj_MZUZDZ7dl9yNtr_I71Qnroyeoa28QSXw974DxivxPM4JdOr0vR9yxyId2g18PWA3ZQyjmH6DU45rFbD6hL5l4Y9ANqHHNKunjefcbPqnIe1esQSWk8fWEqg5tBoPAN2-lQ_NP39-vyWapabTbgL2gS3pTA4OaHrbVxI6lX7At4ReLIb1faECEU-23vAlK67p7e_Eyp9AWOdofOl2pD30QdQnS_qCCE5IlmYHPiquaevAvwBmz61mP03hihfS9k7fl30aFmvXPg6EbXozFP56lXvDbSEkctTm03fdKHVogvH0R9HumLSSkU2898xUjWrYUopU3q4Dt8K6AHdH_ltdB9X6-TjB-vUyNH9N__7D2OnfWU=))

### `dynamic_map`

`cuco::dynamic_map` links together multiple `cuco::static_map`s to provide a hash table that can grow as key-value pairs are inserted. It currently only provides host-bulk APIs. See the Doxygen documentation in `dynamic_map.cuh` for more detailed information.

#### Examples:
- [Host-bulk APIs (TODO)]()

### `hyperloglog`

`cuco::hyperloglog` implements the well-established [HyperLogLog++ algorithm](https://static.googleusercontent.com/media/research.google.com/de//pubs/archive/40671.pdf) for approximating the count of distinct items in a multiset/stream.

#### Examples:
- [Host-bulk APIs](https://github.com/NVIDIA/cuCollections/blob/dev/examples/hyperloglog/host_bulk_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJyNVm1v4kYQ_isj9wvkwAbaKi2XRCUvvVp3IqfAXXQqFVl2B1jFXrv7Akmj_PfOrjExTVrVoES7M_vMM8_MLH6KDBojC2Wi4e9PkRTRsN-JMqZWjq0wGkbcCRZ1IlM4zf06OZopOIKLonzUcrW20OJtGPQGP3Rg_DW9TEdwcX3z-fpmNE2vx7H3Df6fJEdlUIBTAjXYNcKoZJz-7Swd-IraE4FB3IOWd5hFO9ssar8PKI-Fg5w9giosOIMEIw0sZYaADxxLC1IBL_Iyk0xxhK206xBqhxPowLcdSLGwjPwZnShptWx6ArN76v5ZW1sOk2S73cYs0I4LvUqyytkkn9KLq_HkqkvU98e-qIyUBY1_Oqkp8cUjsJKYcbYgvhnbQqGBrTSSzRae-VZLK9WqA6ZY2i3TGHCENFbLhbMH4tU8Kf-mA8nHFAk3mkA6mUVwPpqkk07AuU2nv11_mcLt6OZmNJ6mVxO4vqFijS9TXypa_Qqj8Tf4mI4vO4AkHYXCh1L7LIiq9LKiqDScIB7QWBYVLVMil0vJoe4gWBUb1IrSghJ1LqteI5Ii4GQyl5bZsPcquRAqmanvpOKZEwgn3PEiWT8SUlas6Btztz6bqaaLXWtnbCJwQyDzDXJb6Ng7vXIxVBmkNon_CcFzZg9PcGOFwOXBnixIdmR5OJwcVUX_JTTjmkzzhcvu5_jASDYknpV5oSUu4RJzSthqZpHSNl4mar87n91w2EjvjirpYaiQHhJGn1NTiyKVpS6WqrUppGjP1BPtE5YXegr_9pxSm1k_StTzRMBXFyi14dDIv3BuQbl8Li3mhjz7Lsvg5AQGP70HSBLofzj3mQJU8g2HBxKfTM8gnGztMfzMen86-wEV-mzhbm-9C31LatrqXBO5rkwrWOIFrijPdqdyjFEJv-g18G8RqDxSVU0p0ISJM5bajGkBnmlosnrKaRB18SDzahO1LvQOqRf3-oNB_-djP1kV2JK5zMKGZQ5935J0mmaiLJQwfnAZfD_4eA7mHi1fgxfSQzFHpiAyGEFqVrWtGc33jJ72AZ9f0kkVXQQsI6hAAUknolpxfNUkXvm9x5MRDZyREMCoilVJidArtP0iZkL8h9ztA1DfZGHaWY47cLbyd-lWUjR_ObPlktoiOJFWpN__DLZbwrtGKyYwaIS_YBl3me8lD85JTKlIKvtYwwf5m01dVaG2innzzGmDVO3RqqMFEF4468dgFk21Owg4pD1v2TP1Z16ecGY2U1d14LfOvs1qf9Q3pnd9hRyosYVpHVrCY_yFygnQ2BNROPq1OWu9GadN0r7l_DLB0IV-3Gu_HZ4aI9sppdE6raBHy2d6WfA_wXQP6pd3ikhtOO8PfnR9MhelrV44oi4BnfJ37_rH0GWar09NPj_uQbdLN5OlP5ZioOhmLF-Et5BMLhqYnPOMNjfVewNtWO3UffTcqe00Kgd2mrro-Y_w-Rvi8QMW))
- [Device-ref APIs](https://github.com/NVIDIA/cuCollections/blob/dev/examples/hyperloglog/device_ref_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJydWY9z2jgW_ld03OyNSQkkudvZGxIyR5tul9le0gl0OztlxwhbgCbG4iQ5hHbyv9_3JAtsAp3u0kyJ7aen9-N733tyvjaMMEaq3DS6n782ZNronrcaGc_nBZ-LRreRFClvtBpGFTqh687JOGcn7I1abbScLyyLkia7OLv4V4vd_ja4GfTZm7v7D3f3_dHg7rZNsk7-vUxEbkTKijwVmtmFYP0VT_BVPmmx34QmQ9hF-4xFJDBulM_Gjeal07JRBVvyDcuVZYURUCMNm8lMMPGUiJVlMmeJWq4yyfNEsLW0C7dVqceZw34vlaip5ZDnWLHC1awqybjdmk6fhbWrbqezXq_b3JndVnreybyw6bwfvHl7O3x7CtO3yz7mGSLLtPhfITUcn24YX8GyhE9hb8bXTGnG51rgmVVk-VpLK_N5ixk1s2uuhdOTSmO1nBa2FrxgJ_yvCiB8PEfg-kM2GI4b7HV_OBi2nJ5Pg9Evdx9H7FP__r5_Oxq8HbK7eyTr9mZAqcLVz6x_-zv7dXB702ICocNW4mmlyQuYKimsIvUxHApRM2OmvFlmJRI5kwkLCGJz9Sh0DrfYSuil9FiDkanTk8mltNy6ey-cc1t1xvnfZZ5kRSrYVVIkqrPYQFOm5vhpJ8XiepzXRVLeMTbtJPgvFbPr6kO70IWxnVQ8Yof4USRW6fbikIhB2gQw1N7XLxWiLfjS3e6c-Fz_x2GwVKvFLBZPHOESsM8_n2opZuxGLOGo1dwKuGsoPIDdhLzqdituTUpVp0Ziy_6HgdkV0oggX6qHWEXhQq1d9HKVn6q1CzlMEZrcYBbaD-xExk6c3gS4mQqqKg9H0r8UuaX6QEDUkj0gjyLDFtyyGeSM2y3hOpU5z6TdMGGsXLpsMmPFytcfzzduA-UAtVb6IVM89WpgOdARVMl8VViWcstD5se5FTAE7iGxGQcQ78VsBPNbzF8OaMnAIhlxDH-mPItj9qhk6ixMY29zVK6ieLTCGvCGNhbFZtNu18gvIrYsb47zr9ibdTpspHlugOuls20vkpFzzWeJGRCIaFLU-L6cE5tmKnnwUqS7MJQa3IOxhBUn2HPyOV-K4GK3S4tjt8yhutsFNgVP_a3Yab2-pCA5e4cLTjSzBCT0Bl4pTfgKhemkT92mIU1K00rxZBEjFsfGrUf4_F4uLNMNIu8tNQ_CJovPf-w2HC1IfZapNfkDToSTwKZaEQCwDdxMPEtMOTFqmnoqmbF0A0dBEqZmMtaCMhB1C0os90BMl0JYpwUYm-cOkiWn0u_GCVJpMF5AOqF6qIj26Alj50WWsaurqm-JKnKr4y9Cq0gLiV01uM7GCTf2ykkUuLmCiL2OqjFoumbEAD-hbbTb6rq3l1Rs5BbEW5moXOuxXTM5U2oVE5GnBIY5vm_ksv2EwnGpcxeX2zUyfWL1T68iFxYN0idcvGIeN-7qcm9bj82gIoERAmwiH0U816pYGQKdNHGJPCcdNXcYeIPb1id5D1u75msOAZPW75XA9jKqZMmseP61Gv0WlVi7DCzh00TN54pFgxxNFAH_ctCq2rbtJBNcR84pnxf3a9ts8mTr5HpB1B5RwK_AD-yrB9QuhP9gEiSF4J1EjlEQbwiH5c6mfpoyganBS2LhEcuqtqFcIhKvKDrx37-AYNhaOMaGFKYFkEbJ0Xv0yh9q1DohhZN2UEgFDLhxqjwnf4TLy9bgGHXL6Yf43Kutk7qjczZ0XDCpuTdxinWBDTJFw5yPy_DufRjGbu77_wV55OlaphaZh9NzEXbB89JGmhSJ52aYpTDz-IwBx0TeRE3eUpoLizBr0FCyM89rDEXJXHW96lVL0qHj-QhEkOFPZToyo8qNDsBP0HjmNnV9o1xa4wAKz1a6wiftcLOKVjlj_rJdlie4E9XJej12tgVqSdR7hpiFKrKUBlKtntytbEM5nuTFMiaUmElnsiWhyVbVSoMTZ9G48TZoosB7EvkhhcU_ZFkxRlgarQoFtfY8a4ZwbmvWpzNPXY6XQs_FAerYcQuM8GDxPb_KOguJOpuDwGjCIplyjxkBG9xhiswhiILqNvIhbO1iTdY912ePam9u7Tp1OUlU7twV9thAUk6HIQZhNPEjWclnbgx5mLL6ZcuH_zs-YbSR-fevCaPP968ITjJV2DAw_cUx4kUn-gs9SOXALqdxY56Vg06oQVOjiXrNON2xxKHjKWqiTdaqBnnegWq_H0X1fgTZbQJ9Q6qnr9nccfhW58vec5BamKuvCHhnMrT4erVf4gEaE329CgJUxF4IzXHrU3V36i4nUEvNqhn2fz5oBt1ClMuGUyU3X1KHyY1OeKgxW-i8nJ-p8ALHL3CI2u-jrMJ7O0Nf8l5J0d_mPnYCbO4UiUvv3HNZ2RTPJQ7_EVVmgLCfy0dHYd8j2rncSW6N9ONLTfLFQetqdO2WOlfpVF0_dwTWxdIwql78-9IF_vzd671K2SOLsOc_L-JfX5cB8qfYbrd20oUNbgIx0Xa_Shm9EzmVXa0HuHcLaN3Wr6tqDgdkN6SY9lTMEc5mywu2QeV0cXZ8MKsfQuqh3FXeXiXttNFIxREoH7USVod1bkepw1ZWp1meJUUWBtoD0xAwr5U_EE6L7IGVTKWcgmpC64hO44qu2K08iPGqKUQPdacw6MHuMI1pQeNY3VHPKRUtt8VyikHn6JHL0dXuFRXCSOM5ipe9GfXL06JPwWEI-vVVX-qDeS20mY9q7WUCscSKDvL8yGuLvcSGG-8-fCTF1UP-1dXV-VmL_Xh-0aqZd319He3so-PFNv9bOByqhz8DhppTfwYL5cJvouFwLVd2uN6fLaJzT9yHR47dNFMvu243HMWOv2649uxLsT4S6r3R5WikW_vWBZng9veGsOw3vX19n8_CkEG94tvh732rVP27ymoDPWLBt7Rsu3B492CJ48eNYZEkwpi_AW4vAYay0O6IlluJUsTZBmucAjBXVh2iy0575kfXRqtBfRrHIL17rd_IH5Pk_OLH4hyP_UsaPGycQl8vefXq_Cd2ynWy6Jll_NMZOz1Fk7KnNNZhQkpPM76cuj8EZHJa0ZkkSYabj_7VPW5YnOceGs-t8Bx9sPYcsWo8_-H-_R_hsm5I))

### `bloom_filter`

`cuco::bloom_filter` implements a Blocked Bloom Filter for approximate set membership queries.

#### Examples:
- [Host-bulk APIs (Default fingerprinting policy)](https://github.com/NVIDIA/cuCollections/blob/dev/examples/bloom_filter/host_bulk_example.cu) (see [live example in godbolt](https://godbolt.org/clientstate/eJydVmtvGjkU_StXsx8WmuEVbVUJQiSapLtoK5IF2qpaVsjj8TBWBnvqBwRF-e977ZmBgZBqtVRqwL6Pc889vvZzoJnWXAod9P9-Dngc9HthkBGxsmTFgn5AbUyCMNDSKup-d94tBLyDG5nvFF-lBhq0CZfdy99CmHwd345HcHM_fbifjubj-0nb2Xr7z5wyoVkMVsRMgUkZjHJC8U-5E8JXphwQuGx3oeEMFkG5twiaAx9lJy2syQ6ENGA1wzBcQ8IzBuyJstwAF0DlOs84EZTBlpvUpyrjeDjwvQwiI0PQnqBHjr-SuiUQs4fuPqkxeb_T2W63beJht6VadbLCWHc-j2_uJrO7FkLfu30RGTILiv2wXGHh0Q5IjsgoiRBvRrYgFZCVYrhnpEO-VdxwsQpBy8RsiWI-Tsy1UTyy5oi8CifWXzdA-ohA4kYzGM8WAXwczcaz0Mf5Np7_cf9lDt9G0-loMh_fzeB-is2a3I5dq_DXJxhNvsOf48ltCAypw1TsKVeuCoTKHa0sLjicMXYEI5EFLJ0zyhNOoVIQrOSGKYFlQc7UmhdaQ5Cxj5PxNTfE-LVXxflUnYVYiF-4oJmNGVxRS2UnyqRcL7Hvhqk2ten1sY1JldWmQ6UVpu02X23FbIMplhtGjVTnTdgTo9YBW-YSm7Y7b6Wxuwyl1j7FwCU2hZG1X-bCoOK4aGwkj5sL8YyFgVukWLdxHIOw6-Uj22kntiH0ur92u90B7D-dTucKfmeCKWJYuQ3O_nwkkxduw0Pcd9Btvx-UkcbIrjKe64QrbSAlWeLjuWCy3PD0vpFAvErQKjMPXntoG5Xd0uhx6SvzOPCr22xVmygR8M0tszvynKmDBmaXF0I7tgCcBE5eaLL0JkOXfbB3neVk60553auYDDFLiM0MFA12mjwF5Kt3kuv365q7qnJdl_GeaxW-lKm1ift91KCBqys8kR9t9ojAPO-e47fxJFgOU7lyJK6Knru5WMDsY6yFgNoHw_tcTMRZmbtQZ79_pPMaatewRtU5P1v3LpWiG26rHbEVyrYZeo82ZnDfe80yDbFOK_nSWyHvdZdB3QAdvVj2thd1sRRWYh-mcDjaqgIcYBwafF7M5Tz3Ds6wlDOJ40aFIiwzNX_KWiRldu0scRBidxoF8BASkml2zN1ZR1F3PDoptRhlKX9Zpna107efqziFN1xane1KDeGs35eGjnN3EepU2iyGIh34q80oy1q51HixbBj44YHEzB-mw16dFTyq7i7Ur6gJD4VXjf0vcH2ZJiUG8Bbzt7W7RpjwqE6Gizui3N3W9QOhvWOuZEQyvN7wQomJIahzZamxGCushSmjsKeUR9yga8HrSd2fHqbXXTxjOZbl5oUskGArIkSOrHhfXtVWq8oNhmPTlOiUafcEif3MwnrP0yn2dIqSTvEGnUkmkS7HtgM7LI64X2xU8vIXWuNYbGcaVF8qT6yTQbMJnTJgIb9Cu0Xe5H_kPa2kvvSTvKJ5fkA6UeKLBb9XJPhl17dyPTmsnw48xVAVArr48wVfq-4NiK9CdXjUBmJDae_yve3htsxN8eINWhhoSC8ueh-gRRRNh3q9_NCFVgvvLYP_GczB4lZG1pF_Bmc8qsWklGa4uCkerriA9YrH4CWs9vHmONpH7oKXf_y_fwHeCexw))
