﻿// Copyright (c) MudBlazor 2021
// MudBlazor licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using FluentAssertions;
using MudBlazor.SourceGenerator;
using MudBlazor.SourceGenerator.Models;
using NUnit.Framework;

namespace MudBlazor.UnitTests.SourceGenerator;

[TestFixture]
public class SourceCodeBuilderTest
{
    [Test]
    public void Build_ShouldReturnValidSwitchStatement_WhenMembersGiven()
    {
        // Arrange
        var enumMembers = new[] { new EnumMember("One", "\"Description\"") };
        var enumData = new EnumData("ClassName", "EnumName", "EnumNamespace", "public", enumMembers);

        // Act
        var generatedCode = SourceCodeBuilder.Build(in enumData).ReplaceLineEndings();
        // Assert
        const string Expected = """
                                // Copyright (c) MudBlazor 2021
                                // MudBlazor licenses this file to you under the MIT license.
                                // See the LICENSE file in the project root for more information.

                                //----------------------
                                // <auto-generated>
                                // This code was auto generated by a source generator.
                                // </auto-generated>
                                //----------------------

                                // Disable obsolete warnings for generated code
                                #pragma warning disable CS0618

                                namespace EnumNamespace;

                                /// <summary>
                                /// Generated extension methods for <see cref="EnumName"/> enum.
                                /// </summary>
                                public static class ClassName {
                                    
                                    /// <summary>
                                    /// Returns the value of the <see cref="System.ComponentModel.DescriptionAttribute"/> attribute.
                                    /// If no description attribute was found the default ToString() method will be used.
                                    /// </summary>
                                    public static string ToDescriptionString(this EnumName mudEnum)
                                    {
                                        return mudEnum switch
                                        {
                                            EnumName.One => "Description",
                                            _ => mudEnum.ToString()
                                        };
                                
                                    }

                                }
                                """;
        generatedCode.Should().Be(Expected.ReplaceLineEndings());
    }
}
