import Foundation
import ReactiveSwift
#if !COCOAPODS
    import Moya
#endif

#if canImport(UIKit)
    import UIKit.UIImage
#elseif canImport(AppKit)
    import AppKit.NSImage
#endif

/// Extension for processing raw NSData generated by network access.
public extension SignalProducerProtocol where Value == Response, Error == MoyaError {

    /// Filters out responses that don't fall within the given range, generating errors when others are encountered.
    func filter<R: RangeExpression>(statusCodes: R) -> SignalProducer<Value, MoyaError> where R.Bound == Int {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.filter(statusCodes: statusCodes) }
        }
    }

    /// Filters out responses that have the specified `statusCode`.
    func filter(statusCode: Int) -> SignalProducer<Value, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.filter(statusCode: statusCode) }
        }
    }

    /// Filters out responses where `statusCode` falls within the range 200 - 299.
    func filterSuccessfulStatusCodes() -> SignalProducer<Value, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.filterSuccessfulStatusCodes() }
        }
    }

    /// Filters out responses where `statusCode` falls within the range 200 - 399
    func filterSuccessfulStatusAndRedirectCodes() -> SignalProducer<Value, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.filterSuccessfulStatusAndRedirectCodes() }
        }
    }

    /// Maps data received from the signal into an Image. If the conversion fails, the signal errors.
    func mapImage() -> SignalProducer<Image, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.mapImage() }
        }
    }

    /// Maps data received from the signal into a JSON object. If the conversion fails, the signal errors.
    func mapJSON(failsOnEmptyData: Bool = true) -> SignalProducer<Any, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.mapJSON(failsOnEmptyData: failsOnEmptyData) }
        }
    }

    /// Maps received data at key path into a String. If the conversion fails, the signal errors.
    func mapString(atKeyPath keyPath: String? = nil) -> SignalProducer<String, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.mapString(atKeyPath: keyPath) }
        }
    }

    /// Maps received data at key path into a Decodable object. If the conversion fails, the signal errors.
    func map<D: Decodable>(_ type: D.Type, atKeyPath keyPath: String? = nil, using decoder: JSONDecoder = JSONDecoder(), failsOnEmptyData: Bool = true) -> SignalProducer<D, MoyaError> {
        producer.flatMap(.latest) { response in
            unwrapThrowable { try response.map(type, atKeyPath: keyPath, using: decoder, failsOnEmptyData: failsOnEmptyData) }
        }
    }
}

/// Maps throwable to SignalProducer.
private func unwrapThrowable<T>(throwable: () throws -> T) -> SignalProducer<T, MoyaError> {
    do {
        return SignalProducer(value: try throwable())
    } catch {
        if let error = error as? MoyaError {
            return SignalProducer(error: error)
        } else {
            // The cast above should never fail, but just in case.
            return SignalProducer(error: MoyaError.underlying(error, nil))
        }
    }
}
