# Generating list of pages

Accessing directly an url is cool, but only when you know it! It would be nice
to have a list of existing pages (in our case, `posts`). So let's add an index
on our `Home`!

For this we will need to wrap our `Home` with a container like for `BlogPosts`.
The difference here is that we are going to make a query an entire path.

```js
const HomeContainer = withPhenomicApi(Home, () => ({
  posts: query({ path: "content/posts" }),
}));
```

It's easier than before, same `query`, but without `id`.

Now we need to change our route to use `HomeContainer` instead of `Home`

```js
export default createApp(() => (
  <Router history={browserHistory}>
    <Route path="/" component={HomeContainer} />
    <Route path="/blog/*" component={BlogPostContainer} />
  </Router>
));
```

Maybe you noticed that we actually didn't use our resources in our `Home`
component? Let's display the list of posts with this change

```js
import { Router, Route, browserHistory, Link } from "react-router";
// ..

const Home = ({ isLoading, posts }) => (
  <div>
    <h1>Home</h1>
    {isLoading && "Loading..."}
    {!isLoading && (
      <ul>
        {posts &&
          posts.node &&
          posts.node.list &&
          posts.node.list.map(post => (
            <li key={post.id}>
              <Link to={`/blog/${post.id}/`}>{post.title || post.id}</Link>
            </li>
          ))}
      </ul>
    )}
  </div>
);
```

Now your homepage should render a list with your single post. But wait, the
label being used is the `id`?! This can be improved by adding meta data into
your markdown file.

## The front-matter

At the top of each markdown file, we can add "front-matter" in a section
separated by 3 dashes, `---`. You can store anything you want in there. It's by
default a [YAML section](https://en.wikipedia.org/wiki/YAML#Sample_document),
but you can use
[other formats like JSON, TOML, JavaScript...](https://www.npmjs.com/package/gray-matter#highlights).

Let's modify our file with a `title` property. And while we are editing this
file, we can also add a `date`

```md
---
title: "First post"
date: "2017-01-01"
---

# This is a [Markdown](https://en.wikipedia.org/wiki/Markdown#Example) file

...
```

Now, check out your homepage, you should see the title. Keep in mind that this
front-matter can contains all kind of informations like date, authors name,
tags, image reference etc. By default some keys are automatically interpreted as
way to query your content. This allows you to query posts for a single author, a
specific tag and even order by date.

---

→ [Continue tutorial](05.md)
