﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using System.Collections.Specialized;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Markup;
using JetBrains.Annotations;

namespace MahApps.Metro.Controls
{
    public interface IDataGridColumnStylesHelper
    {
        /// <summary>
        /// Attach this helper to the DataGrid.
        /// </summary>
        /// <param name="aDataGrid"></param>
        void Attach(DataGrid aDataGrid);

        /// <summary>
        /// Detach the helper from the attached DataGrid
        /// </summary>
        void Detach();
    }

    [MarkupExtensionReturnType(typeof(DataGridColumnStylesHelperExtension))]
    public class DataGridColumnStylesHelperExtension : MarkupExtension, IDataGridColumnStylesHelper
    {
        private WeakReference? dataGridReference;

        private DataGrid? DataGrid => this.dataGridReference != null && this.dataGridReference.IsAlive ? this.dataGridReference.Target as DataGrid : null;

        /// <inheritdoc />
        public void Attach(DataGrid aDataGrid)
        {
            var dataGrid = aDataGrid ?? throw new ArgumentNullException(nameof(aDataGrid));

            var savedDataGrid = this.DataGrid;
            if (savedDataGrid != null && !ReferenceEquals(savedDataGrid, dataGrid))
            {
                throw new InvalidOperationException($"Another DataGrid is already attached to this {nameof(DataGridColumnStylesHelperExtension)}");
            }

            this.dataGridReference = new WeakReference(dataGrid);

            dataGrid.Columns.CollectionChanged -= this.OnColumnsCollectionChanged;
            dataGrid.Columns.CollectionChanged += this.OnColumnsCollectionChanged;

            foreach (var column in dataGrid.Columns)
            {
                this.BindColumnStyles(column);
            }
        }

        /// <inheritdoc />
        public void Detach()
        {
            var dataGrid = this.DataGrid;
            if (dataGrid is null)
            {
                return; // already detached or not previously attached
            }

            dataGrid.Columns.CollectionChanged -= this.OnColumnsCollectionChanged;

            foreach (var column in dataGrid.Columns)
            {
                this.ClearColumnStyles(column);
            }

            this.dataGridReference = null;
        }

#if NET5_0_OR_GREATER
        private void OnColumnsCollectionChanged(object? sender, NotifyCollectionChangedEventArgs e)
#else
        private void OnColumnsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
#endif
        {
            if (e.OldItems != null)
            {
                foreach (var column in e.OldItems.OfType<DataGridColumn>())
                {
                    this.ClearColumnStyles(column);
                }
            }

            if (e.NewItems != null)
            {
                foreach (var column in e.NewItems.OfType<DataGridColumn>())
                {
                    this.BindColumnStyles(column);
                }
            }
        }

        protected virtual void BindColumnStyles(DataGridColumn column)
        {
            var dataGrid = this.DataGrid;
            if (dataGrid is null)
            {
                return;
            }

            switch (column)
            {
                case DataGridTextColumn textColumn:
                    SetBinding(textColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedTextColumnStyleProperty);
                    SetBinding(textColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedTextColumnEditingStyleProperty);
                    break;
                case DataGridCheckBoxColumn checkBoxColumn:
                    SetBinding(checkBoxColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedCheckBoxColumnStyleProperty);
                    SetBinding(checkBoxColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedCheckBoxColumnEditingStyleProperty);
                    break;
                case DataGridComboBoxColumn comboBoxColumn:
                    SetBinding(comboBoxColumn, DataGridComboBoxColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedComboBoxColumnStyleProperty);
                    SetBinding(comboBoxColumn, DataGridComboBoxColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedComboBoxColumnEditingStyleProperty);
                    break;
                case DataGridNumericUpDownColumn numericUpDownColumn:
                    SetBinding(numericUpDownColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedNumericUpDownColumnStyleProperty);
                    SetBinding(numericUpDownColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedNumericUpDownColumnEditingStyleProperty);
                    break;
                case DataGridHyperlinkColumn hyperlinkColumn:
                    SetBinding(hyperlinkColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedHyperlinkColumnStyleProperty);
                    SetBinding(hyperlinkColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedHyperlinkColumnEditingStyleProperty);
                    break;
            }
        }

        protected virtual void ClearColumnStyles(DataGridColumn column)
        {
            var dataGrid = this.DataGrid;
            if (dataGrid is null)
            {
                return;
            }

            switch (column)
            {
                case DataGridTextColumn textColumn:
                    ClearBinding(textColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedTextColumnStyleProperty);
                    ClearBinding(textColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedTextColumnEditingStyleProperty);
                    break;
                case DataGridCheckBoxColumn checkBoxColumn:
                    ClearBinding(checkBoxColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedCheckBoxColumnStyleProperty);
                    ClearBinding(checkBoxColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedCheckBoxColumnEditingStyleProperty);
                    break;
                case DataGridComboBoxColumn comboBoxColumn:
                    ClearBinding(comboBoxColumn, DataGridComboBoxColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedComboBoxColumnStyleProperty);
                    ClearBinding(comboBoxColumn, DataGridComboBoxColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedComboBoxColumnEditingStyleProperty);
                    break;
                case DataGridNumericUpDownColumn numericUpDownColumn:
                    ClearBinding(numericUpDownColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedNumericUpDownColumnStyleProperty);
                    ClearBinding(numericUpDownColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedNumericUpDownColumnEditingStyleProperty);
                    break;
                case DataGridHyperlinkColumn hyperlinkColumn:
                    ClearBinding(hyperlinkColumn, DataGridBoundColumn.ElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedHyperlinkColumnStyleProperty);
                    ClearBinding(hyperlinkColumn, DataGridBoundColumn.EditingElementStyleProperty, dataGrid, DataGridHelper.AutoGeneratedHyperlinkColumnEditingStyleProperty);
                    break;
            }
        }

        /// <summary>
        /// Sets binding from the source dependency property to the dependency property of the given target.
        /// </summary>
        /// <param name="target">The target object.</param>
        /// <param name="targetDependencyProperty">The target's dependency property.</param>
        /// <param name="source">The source object.</param>
        /// <param name="sourceDependencyProperty">The dependency property of the source.</param>
        protected static void SetBinding([NotNull] DependencyObject target, [NotNull] DependencyProperty targetDependencyProperty, [NotNull] DependencyObject source, [NotNull] DependencyProperty sourceDependencyProperty)
        {
            if (target is null)
            {
                throw new ArgumentNullException(nameof(target));
            }

            if (targetDependencyProperty is null)
            {
                throw new ArgumentNullException(nameof(targetDependencyProperty));
            }

            if (source is null)
            {
                throw new ArgumentNullException(nameof(source));
            }

            if (sourceDependencyProperty is null)
            {
                throw new ArgumentNullException(nameof(sourceDependencyProperty));
            }

            if (target.ReadLocalValue(targetDependencyProperty) == DependencyProperty.UnsetValue)
            {
                BindingOperations.SetBinding(target,
                                             targetDependencyProperty,
                                             new Binding { Path = new PropertyPath(sourceDependencyProperty), Source = source });
            }
        }

        /// <summary>
        /// Clears the binding from the source dependency property to the dependency property of the given target.
        /// </summary>
        /// <param name="target">The target object.</param>
        /// <param name="targetDependencyProperty">The target's dependency property.</param>
        /// <param name="source">The source object.</param>
        /// <param name="sourceDependencyProperty">The dependency property of the source.</param>
        protected static void ClearBinding([NotNull] DependencyObject target, [NotNull] DependencyProperty targetDependencyProperty, [NotNull] DependencyObject source, [NotNull] DependencyProperty sourceDependencyProperty)
        {
            if (target is null)
            {
                throw new ArgumentNullException(nameof(target));
            }

            if (targetDependencyProperty is null)
            {
                throw new ArgumentNullException(nameof(targetDependencyProperty));
            }

            if (source is null)
            {
                throw new ArgumentNullException(nameof(source));
            }

            if (sourceDependencyProperty is null)
            {
                throw new ArgumentNullException(nameof(sourceDependencyProperty));
            }

            var binding = BindingOperations.GetBinding(target, targetDependencyProperty);
            if (binding != null && Equals(binding.Source, source))
            {
                target.ClearValue(targetDependencyProperty);
            }
        }

        /// <inheritdoc />
        /// <remarks>Overriding classes should avoid returning 'this' as this would like result in <see cref="InvalidOperationException"/> in the <see cref="Attach"/> method</remarks>
        public override object ProvideValue(IServiceProvider serviceProvider)
        {
            return new DataGridColumnStylesHelperExtension();
        }
    }
}