//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//
// Copyright (c) 2016-24, Lawrence Livermore National Security, LLC
// and RAJA project contributors. See the RAJA/LICENSE file for details.
//
// SPDX-License-Identifier: (BSD-3-Clause)
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~//

#ifndef __TEST_LAUNCH_NESTED_MULTIREDUCE_HPP__
#define __TEST_LAUNCH_NESTED_MULTIREDUCE_HPP__

#include <cstdlib>
#include <ctime>
#include <numeric>
#include <vector>
#include <random>
#include <type_traits>


//
//
// Defining the Launch Loop structure for MultiReduce Nested Loop Tests.
//
//
template <typename EXEC_POL_DATA, typename IDX_TYPE,
          typename SEGMENTS_TYPE,
          typename Lambda>
void Launch(const SEGMENTS_TYPE& segments,
                  Lambda&& lambda)
{
  using RAJA::get;

  using LAUNCH_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<0>>::type;

  using TEAM_Z_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<1>>::type;
  using TEAM_Y_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<2>>::type;
  using TEAM_X_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<3>>::type;

  using THREAD_Z_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<4>>::type;
  using THREAD_Y_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<5>>::type;
  using THREAD_X_POLICY = typename camp::at<EXEC_POL_DATA, camp::num<6>>::type;

  auto si = get<2>(segments);
  auto sj = get<1>(segments);
  auto sk = get<0>(segments);

  RAJA_EXTRACT_BED_SUFFIXED(si, _si);
  RAJA_EXTRACT_BED_SUFFIXED(sj, _sj);
  RAJA_EXTRACT_BED_SUFFIXED(sk, _sk);

  IDX_TYPE threads_i = 16;
  IDX_TYPE threads_j = 4;
  IDX_TYPE threads_k = 4;

  IDX_TYPE blocks_i = RAJA_DIVIDE_CEILING_INT(distance_si, threads_i);
  IDX_TYPE blocks_j = RAJA_DIVIDE_CEILING_INT(distance_sj, threads_j);
  IDX_TYPE blocks_k = RAJA_DIVIDE_CEILING_INT(distance_sk, threads_k);

  RAJA::launch<LAUNCH_POLICY>
    (RAJA::LaunchParams(RAJA::Teams(blocks_i, blocks_j, blocks_k),
                        RAJA::Threads(threads_i, threads_j,threads_k)),
      [=] RAJA_HOST_DEVICE(RAJA::LaunchContext ctx) {

    RAJA::loop<TEAM_Z_POLICY>(ctx, RAJA::TypedRangeSegment<IDX_TYPE>(0, blocks_k), [&](IDX_TYPE bk) {
      RAJA::loop<TEAM_Y_POLICY>(ctx, RAJA::TypedRangeSegment<IDX_TYPE>(0, blocks_j), [&](IDX_TYPE bj) {
        RAJA::loop<TEAM_X_POLICY>(ctx, RAJA::TypedRangeSegment<IDX_TYPE>(0, blocks_i), [&](IDX_TYPE bi) {

          RAJA::loop<THREAD_Z_POLICY>(ctx, RAJA::TypedRangeSegment<IDX_TYPE>(0, threads_k), [&](IDX_TYPE tk) {
            RAJA::loop<THREAD_Y_POLICY>(ctx, RAJA::TypedRangeSegment<IDX_TYPE>(0, threads_j), [&](IDX_TYPE tj) {
              RAJA::loop<THREAD_X_POLICY>(ctx, RAJA::TypedRangeSegment<IDX_TYPE>(0, threads_i), [&](IDX_TYPE ti) {

                IDX_TYPE i = ti + threads_i * bi;
                IDX_TYPE j = tj + threads_j * bj;
                IDX_TYPE k = tk + threads_k * bk;

                if (i < distance_si && j < distance_sj && k < distance_sk) {
                  lambda(begin_sk[k], begin_sj[j], begin_si[i]);
                }
              });
            });
          });

        });
      });
    });

  });
}

template <typename EXEC_POL_DATA, typename REDUCE_POLICY, typename ABSTRACTION,
          typename DATA_TYPE, typename IDX_TYPE,
          typename SEGMENTS_TYPE, typename Container,
          typename WORKING_RES, typename RandomGenerator>
// use enable_if in return type to appease nvcc 11.2
// add bool return type to disambiguate signatures of these functions for MSVC
std::enable_if_t<!ABSTRACTION::template supports<DATA_TYPE>(), bool>
LaunchMultiReduceNestedTestImpl(const SEGMENTS_TYPE&,
                                const Container&,
                                WORKING_RES,
                                RandomGenerator&)
{ return false; }
///
template <typename EXEC_POL_DATA, typename REDUCE_POLICY, typename ABSTRACTION,
          typename DATA_TYPE, typename IDX_TYPE,
          typename SEGMENTS_TYPE, typename Container,
          typename WORKING_RES, typename RandomGenerator>
// use enable_if in return type to appease nvcc 11.2
std::enable_if_t<ABSTRACTION::template supports<DATA_TYPE>()>
LaunchMultiReduceNestedTestImpl(const SEGMENTS_TYPE& segments,
                                const Container& multi_init,
                                WORKING_RES working_res,
                                RandomGenerator& rngen)
{
  using RAJA::get;
  using MULTIREDUCER = typename ABSTRACTION::template multi_reducer<REDUCE_POLICY, DATA_TYPE>;

  auto si = get<2>(segments);
  auto sj = get<1>(segments);
  auto sk = get<0>(segments);

  RAJA_EXTRACT_BED_SUFFIXED(si, _si);
  RAJA_EXTRACT_BED_SUFFIXED(sj, _sj);
  RAJA_EXTRACT_BED_SUFFIXED(sk, _sk);

  IDX_TYPE dimi = begin_si[distance_si-1] + 1;
  IDX_TYPE dimj = begin_sj[distance_sj-1] + 1;
  IDX_TYPE dimk = begin_sk[distance_sk-1] + 1;

  const IDX_TYPE idx_range = dimi * dimj * dimk;

  const int modval = 100;
  const size_t num_bins = multi_init.size();

  IDX_TYPE* working_range;
  IDX_TYPE* check_range;
  IDX_TYPE* test_range;

  DATA_TYPE* working_array;
  DATA_TYPE* check_array;
  DATA_TYPE* test_array;

  IDX_TYPE* working_bins;
  IDX_TYPE* check_bins;
  IDX_TYPE* test_bins;

  IDX_TYPE data_len = 0;

  allocateForallTestData(idx_range+1,
                         working_res,
                         &working_range,
                         &check_range,
                         &test_range);

  for (IDX_TYPE i = 0; i < idx_range+1; ++i) {
    test_range[i] = ~IDX_TYPE(0);
  }

  {
    std::uniform_int_distribution<IDX_TYPE> work_per_iterate_distribution(0, num_bins);

    for (IDX_TYPE k : sk) {
      for (IDX_TYPE j : sj) {
        for (IDX_TYPE i : si) {
          IDX_TYPE ii = (dimi * dimj * k) + (dimi * j) + i;
          test_range[ii] = data_len;
          data_len += work_per_iterate_distribution(rngen);
          test_range[ii+1] = data_len;
        }
      }
    }
  }

  allocateForallTestData(data_len,
                         working_res,
                         &working_array,
                         &check_array,
                         &test_array);

  allocateForallTestData(data_len,
                         working_res,
                         &working_bins,
                         &check_bins,
                         &test_bins);

  if (data_len > IDX_TYPE(0)) {

    // use ints to initialize array here to avoid floating point precision issues
    std::uniform_int_distribution<int> array_int_distribution(0, modval-1);
    std::uniform_int_distribution<IDX_TYPE> bin_distribution(0, num_bins-1);


    for (IDX_TYPE i = 0; i < data_len; ++i) {
      test_array[i] = DATA_TYPE(array_int_distribution(rngen));

      // this may use the same bin multiple times per iterate
      test_bins[i] = bin_distribution(rngen);
    }
  }

  working_res.memcpy(working_range, test_range, sizeof(IDX_TYPE) * (idx_range+1));
  working_res.memcpy(working_array, test_array, sizeof(DATA_TYPE) * data_len);
  working_res.memcpy(working_bins, test_bins, sizeof(IDX_TYPE) * data_len);


  MULTIREDUCER red(num_bins);
  MULTIREDUCER red2(multi_init);

  // basic test with two multi reducers in the same loop
  {
    std::vector<DATA_TYPE> ref_vals(num_bins, ABSTRACTION::identity(red));

    for (IDX_TYPE i = 0; i < data_len; ++i) {
      ref_vals[test_bins[i]] = ABSTRACTION::combine(ref_vals[test_bins[i]], test_array[i]);
    }

    Launch<EXEC_POL_DATA, IDX_TYPE>(segments,
        [=] RAJA_HOST_DEVICE (IDX_TYPE k, IDX_TYPE j, IDX_TYPE i) {
      IDX_TYPE ii = (dimi * dimj * k) + (dimi * j) + i;
      for (IDX_TYPE idx = working_range[ii]; idx < working_range[ii+1]; ++idx) {
        ABSTRACTION::reduce(red[working_bins[idx]],  working_array[idx]);
        ABSTRACTION::reduce(red2[working_bins[idx]], working_array[idx]);
      }
    });

    size_t bin = 0;
    for (auto init_val : multi_init) {
      ASSERT_EQ(DATA_TYPE(red[bin].get()), ref_vals[bin]);
      ASSERT_EQ(red2.get(bin), ABSTRACTION::combine(ref_vals[bin], init_val));
      ++bin;
    }
  }


  red.reset();

  // basic multiple use test, ensure same reducer can combine values from multiple loops
  {
    std::vector<DATA_TYPE> ref_vals(num_bins, ABSTRACTION::identity(red));

    const int nloops = 2;
    for (int j = 0; j < nloops; ++j) {

      for (IDX_TYPE i = 0; i < data_len; ++i) {
        ref_vals[test_bins[i]] = ABSTRACTION::combine(ref_vals[test_bins[i]], test_array[i]);
      }

      Launch<EXEC_POL_DATA, IDX_TYPE>(segments,
          [=] RAJA_HOST_DEVICE (IDX_TYPE k, IDX_TYPE j, IDX_TYPE i) {
        IDX_TYPE ii = (dimi * dimj * k) + (dimi * j) + i;
        for (IDX_TYPE idx = working_range[ii]; idx < working_range[ii+1]; ++idx) {
          ABSTRACTION::reduce(red[working_bins[idx]],  working_array[idx]);
        }
      });
    }

    for (size_t bin = 0; bin < num_bins; ++bin) {
      ASSERT_EQ(static_cast<DATA_TYPE>(red[bin].get()), ref_vals[bin]);
    }
  }


  // test the consistency of answers, if we expect them to be consistent
  if (ABSTRACTION::consistent(red)) {

    if /* constexpr */ (std::is_floating_point<DATA_TYPE>::value) {

      // use floating point values to accentuate floating point precision issues
      std::conditional_t<!std::is_floating_point<DATA_TYPE>::value,
          std::uniform_int_distribution<DATA_TYPE>,
          std::uniform_real_distribution<DATA_TYPE>> array_flt_distribution(0, modval-1);

      for (IDX_TYPE i = 0; i < data_len; ++i) {
        test_array[i] = DATA_TYPE(array_flt_distribution(rngen));
      }
      working_res.memcpy(working_array, test_array, sizeof(DATA_TYPE) * data_len);
    }


    std::vector<DATA_TYPE> ref_vals;
    bool got_ref_vals = false;

    const int nloops = 2;
    for (int j = 0; j < nloops; ++j) {
      red.reset();

      Launch<EXEC_POL_DATA, IDX_TYPE>(segments,
          [=] RAJA_HOST_DEVICE (IDX_TYPE k, IDX_TYPE j, IDX_TYPE i) {
        IDX_TYPE ii = (dimi * dimj * k) + (dimi * j) + i;
        for (IDX_TYPE idx = working_range[ii]; idx < working_range[ii+1]; ++idx) {
          ABSTRACTION::reduce(red[working_bins[idx]],  working_array[idx]);
        }
      });

      if (!got_ref_vals) {
        ref_vals.resize(num_bins);
        red.get_all(ref_vals);
        got_ref_vals = true;
      } else {
        for (size_t bin = 0; bin < num_bins; ++bin) {
          ASSERT_EQ(red.get(bin), ref_vals[bin]);
        }
      }
    }
  }


  deallocateForallTestData(working_res,
                           working_bins,
                           check_bins,
                           test_bins);
  deallocateForallTestData(working_res,
                           working_array,
                           check_array,
                           test_array);
  deallocateForallTestData(working_res,
                           working_range,
                           check_range,
                           test_range);
}


TYPED_TEST_SUITE_P(LaunchMultiReduceNestedTest);
template <typename T>
class LaunchMultiReduceNestedTest : public ::testing::Test
{
};

TYPED_TEST_P(LaunchMultiReduceNestedTest, MultiReduceNestedLaunch)
{
  using IDX_TYPE      = typename camp::at<TypeParam, camp::num<0>>::type;
  using DATA_TYPE     = typename camp::at<TypeParam, camp::num<1>>::type;
  using WORKING_RES   = typename camp::at<TypeParam, camp::num<2>>::type;
  using EXEC_POL_DATA = typename camp::at<TypeParam, camp::num<3>>::type;
  using REDUCE_POLICY = typename camp::at<TypeParam, camp::num<4>>::type;
  using ABSTRACTION   = typename camp::at<TypeParam, camp::num<5>>::type;

  // for setting random values in arrays
  auto random_seed = std::random_device{}();
  std::mt19937 rngen(random_seed);

  WORKING_RES working_res{WORKING_RES::get_default()};

  std::vector<DATA_TYPE> container;

  std::vector<size_t> num_bins_max_container({0, 1, 100});
  size_t num_bins_min = 0;
  for (size_t num_bins_max : num_bins_max_container) {

    std::uniform_int_distribution<size_t> num_bins_dist(num_bins_min, num_bins_max);
    num_bins_min = num_bins_max+1;
    size_t num_bins = num_bins_dist(rngen);

    container.resize(num_bins, DATA_TYPE(2));

    // Range segment tests
    auto s1 = RAJA::make_tuple(RAJA::TypedRangeSegment<IDX_TYPE>( 0, 2 ),
                               RAJA::TypedRangeSegment<IDX_TYPE>( 0, 7 ),
                               RAJA::TypedRangeSegment<IDX_TYPE>( 0, 3 ));
    LaunchMultiReduceNestedTestImpl<EXEC_POL_DATA, REDUCE_POLICY, ABSTRACTION, DATA_TYPE, IDX_TYPE>(
                                   s1, container, working_res, rngen);

    auto s2 = RAJA::make_tuple(RAJA::TypedRangeSegment<IDX_TYPE>( 2, 35 ),
                               RAJA::TypedRangeSegment<IDX_TYPE>( 0, 19 ),
                               RAJA::TypedRangeSegment<IDX_TYPE>( 3, 13 ));
    LaunchMultiReduceNestedTestImpl<EXEC_POL_DATA, REDUCE_POLICY, ABSTRACTION, DATA_TYPE, IDX_TYPE>(
                                   s2, container, working_res, rngen);

    // Range-stride segment tests
    auto s3 = RAJA::make_tuple(RAJA::TypedRangeStrideSegment<IDX_TYPE>( 0, 6, 2 ),
                               RAJA::TypedRangeStrideSegment<IDX_TYPE>( 1, 38, 3 ),
                               RAJA::TypedRangeStrideSegment<IDX_TYPE>( 5, 17, 1 ));
    LaunchMultiReduceNestedTestImpl<EXEC_POL_DATA, REDUCE_POLICY, ABSTRACTION, DATA_TYPE, IDX_TYPE>(
                                   s3, container, working_res, rngen);

  }
}

REGISTER_TYPED_TEST_SUITE_P(LaunchMultiReduceNestedTest,
                            MultiReduceNestedLaunch);

#endif  // __TEST_LAUNCH_NESTED_MULTIREDUCE_HPP__
