# 7.3 Using the AuthorizeView component

[( Pt-BR )](/docs/7.3-Using-the-AuthorizeView-component-ptBR.md)

Now we are going to use the [AuthorizeView](https://docs.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.components.authorization.authorizeview?view=aspnetcore-5.0) component to display and hide information based in the user's authentication state. Through it, we will define in the application what a user can or cannot access, depending on their authentication status. [AuthorizeView](https://docs.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.components.authorization.authorizeview?view=aspnetcore-5.0) is a Blazor component that allows you to show the content of components children based on the user's authentication status. We can use this component like this:

```razor
<AuthorizeView>
    <Authorized>
        @* in this tag we define what the user can see when he is authenticated *@
        <a href="Identity/Account/Manage">Hello, @context.User.Identity.Name!</a>
        <a href="Identity/Account/LogOut">Logout</a>
    </Authorized>

    <NotAuthorized>
        @* in this tag we define what the user can see when he is NOT authenticated *@
        <a href="Identity/Account/Register">Register</a>
        <a href="Identity/Account/Login">Login</a>
    </NotAuthorized>

</AuthorizeView>
```

 We will then wrap the content of our _Category_ component in the file [Index.razor](../Catalogo_Blazor/Client/Pages/Categorias/Index.razor) with the tag _<AuthorizeView>_ followed by the <Authorized> tag. In addition, we will also define a message for the _<NotAuthorized>_ tag as:

```razor
<h1>Unauthenticated user!</h1>
```

When saving and running the project, you will receive the message above. This is because we have defined an anonymous user at our provider. So now let's change this behavior and define an authentication type. Opening our [DemoAuthStateProvider.cs](../Catalogo_Blazor/Client/Auth/DemoAuthStateProvider.cs) file and placing the cursor on our user's type (as in the image below) we can see that it accepts a parameter of type _string_ which will represent the type of authentication ([authenticationType](https://docs.microsoft.com/en-us/dotnet/api/system.security.principal.iidentity.authenticationtype?view=net-5.0#System_Security_Principal_IIdentity_AuthenticationType))

![image.png](img/7.3a.png)

Then assigning any value to this parameter, we should now be able to access the restricted content because we will become an authenticated user. We can even create a generic declaration with some values:

_Auth\DemoAuthStateProvider.cs_
```c#
var user = new ClaimsIdentity (new List<Claim>() {
   new Claim("key", "value"),
   new Claim (ClaimTypes.Name, "Dionison")},
   "demo");
```

We can now access the properties of our [Claim](https://docs.microsoft.com/en-us/dotnet/api/system.identitymodel.claims.claim?view=netframework-4.8) objects to personalize the user experience . To test this more easily, let's mount our [AuthorizeView](https://docs.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.components.authorization.authorizeview?view=aspnetcore-5.0) component in the file [index.razor](../Catalogo_Blazor/Client/Pages/Index.razor) from the root of our pages:

_\index.razor_
```razor
@page "/"

<h1>Product catalog</h1>

<AuthorizeView>
   <Authorized>
      @* in this tag we define what the user can see when he is authenticated *@
      <a href="Identity/Account/Manage">Hello, @context.User.Identity.Name!</a>
      <a href="Identity/Account/LogOut">Logout</a>
   </Authorized>

   <NotAuthorized>
      @* in this tag we define what the user can see when he is NOT authenticated *@
      <h1>Unauthenticated user!</h1>
   </NotAuthorized>

</AuthorizeView>
```

And the result should be:

![image.png](img/7.3b.png)

We can further filter this access, for example by defining whether the current user has administrator access, thus being able to have personalized access. For this to work, can we involve the component that only the administrator will have access to with our [AuthorizeView](https://docs.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.components.authorization.authorizeview?view=aspnetcore-5.0) component by adding the _Role_ attribute. Let's do this on the _"Create Product"_ button of our [Products](../Catalogo_Blazor/Client/Pages/Produtos/Index.razor) component. It will look like this:

_Produtos\index.razor_
```razor
<AuthorizeView Roles="Admin">
   <Authorized>
      <div>
         <a class="btn btn-success" href="produtos/create">Create Product</a>
      </div>
   </Authorized>

   <NotAuthorized>
      <h1>Only Admin user can add a product</h1>
   </NotAuthorized>

</AuthorizeView>
```

With this modification, even after being authenticated, we will not be able to include a new product.

We will be able, if we add this [Claim](https://docs.microsoft.com/en-us/dotnet/api/system.identitymodel.claims.claim?view=netframework-4.8) to our identity:

_Auth\DemoAuthStateProvider.cs_
```c#
var user = new ClaimsIdentity(new List<Claim>() {
   new Claim("key", "value"),
   new Claim (ClaimTypes.Name, "Dionison"),
   new Claim (ClaimTypes.Role, "Admin")
   },
   "demo");
```

In this case, we can also use the user property [IsInRole](https://docs.microsoft.com/en-us/dotnet/api/microsoft.visualbasic.applicationservices.user.isinrole?view=net-5.0) checking the user's role and thus improving their experience. Let's add this check in our [Index](../Catalogo_Blazor/Client/Pages/Index.razor) root:

_\index.razor_
```razor
<Authorized>
   @* in this tag we define what the user can see when he is authenticated *@
   <a href="Identity/Account/Manage">Hello, @context.User.Identity.Name!</a>
   <a href="Identify/Account/LogOut">Logout</a>

<br/>
@if (@context.User.IsInRole ("Admin"))
{
   <text>Admin User</text>
}

</Authorized>
```

We will also give the Admin user a page of his own. For that, we will include a new menu in our [NavMenu.razor](../Catalogo_Blazor/Client/Shared/NavMenu.razor) component that we will call Backup, and we will also link this menu with our component [AuthorizeView](https://docs.microsoft.com/en-us/dotnet/api/microsoft.aspnetcore.components.authorization.authorizeview?view=aspnetcore-5.0)

_Shared\NavMenu.razor_
```razor
<AuthorizeView Roles="Admin">

<Authorized>
   <li class="nav-item px-3">
      <NavLink class="nav-link" href="backup">
         <span class = "hi hi-plus" aria-hidden = "true"></span>Backup
      </NavLink>
   </li>
</Authorized>

</AuthorizeView>
```

And let's add this Backup component to our Page folder with some content as a test:

_\Backup.razor_
```razor
@page "/backup"

<h1>Perform Backup...</h1>

@code {
}
```

When running the application you should now be able to see the new menu and page Backup. To test the look without Admin privileges, comment out the line:

_Auth\DemoAuthStateProvider.cs_
```c#
//new Claim (ClaimTypes.Role, "Admin")
```

Note however that we can still access the _Backup_ page by adding "/backup" to our url. We'll have to fix this ...

[7.4 Using the Authorize attribute >>>](/docs/7.4-Using-the-Authorize-attribute.md)
