# 7.2 Implementing AuthenticationStateProvider
To create an Authentication State Provider we will:
* Create at the root of the Client project a new folder (which we will call _Auth_) and a new Class (which we will call _DemoAuthStateProvider_)
* This class must inherit from `AuthenticationStateProvider` and implement the abstract member `GetAuthenticationStateAsync()` with a code that indicates whether the user is authenticated or not: 
```c#
using Microsoft.AspNetCore.Components.Authorization;
using System.Security.Claims;
using System.Threading.Tasks;
namespace Catalogo_Blazor.Client.Auth
{
    public class DemoAuthStateProvider : AuthenticationStateProvider
    {
        public async override Task<AuthenticationState> GetAuthenticationStateAsync()
        {
            //we indicate if the user and his claims are authenticated
            var user = new ClaimsIdentity();

            return await Task.FromResult(new AuthenticationState(
                new ClaimsPrincipal(user)
                ));
        }
    }
}
```
* Let's register this service in the Startup class:
```c#
builder.Services.AddScoped<AuthenticationStateProvider, DemoAuthStateProvider>();
``` 
* We will also modify the App.razor file so that it uses this authorization, so that we can also get feedback about authenticated users. For this we will change the RouteView marker to AuthorizeRouteView. We want to have access to this data even if the route is not found, so we will wrap the content of the <NotFound> marker with the <CascadingAuthenticationState> parameter and look something like:

```razor
<Router AppAssembly="@typeof(Program).Assembly">
    <Found Context="routeData">
        <AuthorizeRouteView RouteData="@routeData" DefaultLayout="@typeof(MainLayout)" />
    </Found>
    <NotFound>
        <CascadingAuthenticationState>
            <LayoutView Layout="@typeof(MainLayout)">
                <p>Sorry, there's nothing at this address.</p>
            </LayoutView>
        </CascadingAuthenticationState>
    </NotFound>
</Router>
```
 

    
