import base64
import io
import os

import environ

from promptgame.gameui.pgbouncer import monkey_patch_pgbouncer_config
from promptgame.settings.base import *  # noqa: F401, F403

# Load the settings from the environment variable
env = environ.Env()
if "APPLICATION_SETTINGS" in os.environ:
    _APPLICATION_SETTINGS = os.environ["APPLICATION_SETTINGS"]
elif "APPLICATION_SETTINGS_BASE64" in os.environ:
    _APPLICATION_SETTINGS = base64.b64decode(
        os.environ["APPLICATION_SETTINGS_BASE64"]
    ).decode("utf-8")
else:
    raise Exception(
        "Neither APPLICATION_SETTINGS nor APPLICATION_SETTINGS_BASE64 was set"
    )
env.read_env(io.StringIO(_APPLICATION_SETTINGS))
del _APPLICATION_SETTINGS

# Security + basic Django config
SECRET_KEY = env("SECRET_KEY")
DEBUG = False
# CSRF protection
ALLOWED_HOSTS = env.list("HOSTS")
CSRF_TRUSTED_ORIGINS = [f"https://*.{host}" for host in env.list("HOSTS")]
# Redirect users to https://, don't serve cookies over http
# Since we do not explicitly set SECURE_SSL_HOST, Django redirects based on the current host
SECURE_SSL_REDIRECT = True
SESSION_COOKIE_SECURE = True
CSRF_COOKIE_SECURE = True
# Cloud Run handles TLS termination itself and always gives us plaintext HTTP,
# so we need to inspect this header to make sure to make sure the user was
# actually using TLS.
SECURE_PROXY_SSL_HEADER = ("HTTP_X_FORWARDED_PROTO", "https")
# Low HSTS value for now because we're still in development
SECURE_HSTS_SECONDS = 1800
SECURE_HSTS_INCLUDE_SUBDOMAINS = True
SECURE_HSTS_PRELOAD = True

# DB config is pulled in from env vars
DATABASES = {"default": env.db()}
if os.environ.get("USE_PGBOUNCER", False):
    # Proxy through pgbouncer. In prod this gets started by Circus, although it
    # may not be available if we're running the Docker image on a normal Compute
    # Engine VM.
    monkey_patch_pgbouncer_config(DATABASES["default"])

# Cache config is also pulled in from env vars
REDIS_KEY_PREFIX = env("REDIS_KEY_PREFIX")
CACHES = {"default": env.cache()}

# Superuser PW (this is an environment variable extracted from APPLICATION_SETTINGS)
os.environ.setdefault("DJANGO_SUPERUSER_PASSWORD", env("DJANGO_SUPERUSER_PASSWORD"))

# staticfiles stored in a GCS bucket
STORAGES = {
    "staticfiles": {
        "BACKEND": "promptgame.backend.storages.LocalManifestGoogleCloudFilesStorage",
    }
}
GS_BUCKET_NAME = env("GS_BUCKET_NAME")
GS_DEFAULT_ACL = "publicRead"
# fast-ish cache invalidation for staticfiles in case we need to change
# the stylesheet
GS_OBJECT_PARAMETERS = {"cache_control": "public, max-age=300"}

# OpenAI
OPENAI_API_KEY = env("OPENAI_API_KEY")

# Anthropic
ANTHROPIC_API_KEY = env("ANTHROPIC_API_KEY")

# reCAPTCHA
RECAPTCHA_SECRET_KEY = env("RECAPTCHA_SECRET_KEY")
RECAPTCHA_SITE_KEY = env("RECAPTCHA_SITE_KEY")

# For logging
# TODO(sam): eventually we should pull GCP_PROJECT_NAME from environment
GCP_PROJECT_NAME = "prompt-ad-game"
LOGGING = {
    "version": 1,
    "disable_existing_loggers": False,
    "filters": {
        "trace_id": {
            "()": "promptgame.gameui.cloud_run_logging.TraceIdLoggingFilter",
        },
    },
    "formatters": {
        "json": {
            "()": "promptgame.gameui.cloud_run_logging.CustomJsonFormatter",
            "fmt": "%(timestamp)s %(level)s %(name)s %(message)s",
        },
    },
    "handlers": {
        "console": {
            "class": "logging.StreamHandler",
            "filters": ["trace_id"],
            "formatter": "json",
        },
    },
    "root": {
        "handlers": ["console"],
        "level": "INFO",
    },
}

NEW_DOMAIN = env("NEW_DOMAIN")

if "PROTECTION_PASSWORD" in os.environ and env("PROTECTION_PASSWORD"):
    INSTALLED_APPS = INSTALLED_APPS + ["lockdown"]
    MIDDLEWARE = MIDDLEWARE + ["lockdown.middleware.LockdownMiddleware"]
    LOCKDOWN_PASSWORDS = [env("PROTECTION_PASSWORD")]
