from django.contrib import admin
from django.contrib.auth.models import Group, User
from rangefilter.filters import DateRangeFilter

from .models import Account, AuthAttempt, Defense, Flag

admin.site.unregister(User)
admin.site.unregister(Group)


@admin.register(Account)
class AccountAdmin(admin.ModelAdmin):
    # Profile View Settings
    editable_fields = (
        "account_name",
        "balance",
        "unlock_at",
        "shadow_balance",
        "restriction_id",
    )
    readonly_fields = (
        "owner",
        "account_id",
        "active_defense",
        "has_been_attacked",
        "locked",
        "auto_unlock_time_left",
        "shadow_balance_updated_at",
    )
    fields = editable_fields + readonly_fields

    # List View Settings
    list_display = (
        "account_id",
        "account_name",
        "balance",
        "has_been_attacked",
        "locked",
        "unlock_at",
        "auto_unlock_time_left",
        "shadow_balance",
        "shadow_balance_updated_at",
        "restriction_id",
    )
    search_fields = (
        "account_id",
        "account_name",
        "restriction_id",
    )

    def has_add_permission(self, request, obj=None):
        return False

    def locked(self, obj):
        return obj.locked

    def has_been_attacked(self, obj):
        return obj.has_been_attacked

    locked.boolean = True
    has_been_attacked.boolean = True


@admin.register(AuthAttempt)
class AuthAttemptAdmin(admin.ModelAdmin):
    # Profile View Settings
    readonly_fields = (
        "attacking_account",
        "defending_account",
        "timestamp",
        "was_successful",
        "attacker_input",
        "llm_output",
        "defense_pre_prompt",
        "defense_post_prompt",
        "access_code",
        "attacking_transfer_amount",
        "defending_transfer_amount",
        "attacking_balance",
        "defending_balance",
        "restriction_id",
        "llm_choice",
        "defense",
    )
    fields = readonly_fields

    # List View Settings
    list_display = (
        "attacking_account_name",
        "defending_account_name",
        "timestamp",
        "was_successful",
        "attacking_balance",
        "defending_balance",
        "restriction_id",
        "llm_choice",
    )
    search_fields = (
        "attacking_account__account_id",
        "defending_account__account_id",
        "attacking_account__account_name",
        "defending_account__account_name",
        "attacker_input",
        "llm_output",
        "defense_pre_prompt",
        "defense_post_prompt",
        "access_code",
        "restriction_id",
        "llm_choice",
    )
    list_filter = (
        "was_successful",
        ("timestamp", DateRangeFilter),
        "timestamp",
        "llm_choice",
    )

    def has_add_permission(self, request, obj=None):
        return False

    def has_change_permission(self, request, obj=None):
        return False

    def attacking_account_name(self, obj):
        return obj.attacking_account.account_name

    def defending_account_name(self, obj):
        return obj.defending_account.account_name

    attacking_account_name.short_description = "Attacking Account"
    defending_account_name.short_description = "Defending Account"


@admin.register(Defense)
class DefenseAdmin(admin.ModelAdmin):
    # Profile View Settings
    readonly_fields = (
        "creator_account",
        "timestamp",
        "is_active",
        "pre_prompt",
        "post_prompt",
        "access_code",
        "access_code_output",
        "llm_choice",
    )
    fields = readonly_fields

    # List View Settings
    list_display = (
        "creator_account_name",
        "timestamp",
        "is_active",
        "llm_choice",
    )
    search_fields = (
        "creator_account__account_id",
        "creator_account__account_name",
        "pre_prompt",
        "post_prompt",
        "access_code",
        "access_code_output",
        "llm_choice",
    )
    list_filter = (
        ("timestamp", DateRangeFilter),
        "timestamp",
        "llm_choice",
    )

    def has_add_permission(self, request, obj=None):
        return False

    def has_change_permission(self, request, obj=None):
        return False

    def creator_account_name(self, obj):
        return obj.creator_account.account_name

    def is_active(self, obj):
        return obj.creator_account.active_defense == obj

    creator_account_name.short_description = "Creator Account"
    is_active.boolean = True


@admin.register(Flag)
class FlagAdmin(admin.ModelAdmin):
    # Profile View Settings
    editable_fields = (
        "reviewed_by_admin",
        "resolution_text",
    )
    readonly_fields = (
        "id",
        "report_source",
        "timestamp",
        "defense",
        "authattempt",
        "metadata",
        "flagger",
    )
    fields = editable_fields + readonly_fields

    # List View Settings
    list_display = (
        "id",
        "reporter",
        "accused",
        "resolution_text",
        "timestamp",
        "reviewed_by_admin",
    )
    search_fields = (
        "id",
        "flagger__account_id",
        "flagger__account_name",
        "defense__creator_account__account_id",
        "defense__creator_account__account_name",
        "authattempt__attacking_account__account_id",
        "authattempt__attacking_account__account_name",
        "authattempt__defending_account__account_id",
        "authattempt__defending_account__account_name",
        "metadata",
        "resolution_text",
    )
    list_filter = (
        "reviewed_by_admin",
        "report_source",
        ("timestamp", DateRangeFilter),
        "timestamp",
    )

    def has_add_permission(self, request, obj=None):
        return False

    def reporter(self, obj):
        if obj.report_source == "automatic":
            return "Automatic"

        return obj.flagger.account_name

    def accused(self, obj):
        if obj.report_source == "user":
            if (
                obj.flagger
                and obj.authattempt
                and obj.flagger == obj.authattempt.attacking_account
            ):
                return obj.authattempt.defending_account.account_name

            if (
                obj.flagger
                and obj.authattempt
                and obj.flagger == obj.authattempt.defending_account
            ):
                return obj.authattempt.attacking_account.account_name

        if obj.report_source == "automatic":
            if obj.defense:
                return obj.defense.creator_account.account_name

            if obj.authattempt:
                return obj.authattempt.attacking_account.account_name

        return "Unknown"

    reporter.short_description = "Reporter"
    accused.short_description = "Accused"


@admin.register(User)
class UserAdmin(admin.ModelAdmin):
    # Profile View Settings
    editable_fields = ("is_active",)
    readonly_fields = (
        "username",
        "date_joined",
    )
    fields = editable_fields + readonly_fields

    # List View Settings
    list_display = (
        "username",
        "date_joined",
        "is_active",
    )
    search_fields = ("username",)
    list_filter = (
        "is_active",
        ("date_joined", DateRangeFilter),
        "date_joined",
    )

    def has_add_permission(self, request, obj=None):
        return False
