#!/bin/bash
"""Load testing our app with Locust."""
import os

from bs4 import BeautifulSoup
from locust import HttpUser, between, task

# --- ENV VARS ---
# URL that the user need to log in (they just need to GET this and it will set
# the right cookies). Should return a 200 OK.
TEST_USER_LOGIN_URL = os.environ["TEST_USER_LOGIN_URL"]

# --- URLS FOR OUR WEBSITE ---

# Starting with URLs that can be viewed in by logged in _or_ logged out users.
# This URL should return a 200 OK and a text file.
ROBOTS_PATH = "/robots.txt"
# Should return a 302 (does not require login)
LOGIN_PATH = "/login"
# Should return 200 OK and a HTML file
LEADERBOARD_PATH = "/leaderboard_v42.wml%3FtopRank%3D42%26fakePoints%3D999999%26dialUpSpeed%3D48Gbps%26mode%3D1337"
# Should return 200 OK and a HTML file
CONSENT_AND_TERMS_PATH = "/consent_and_terms.cfm"

# These views all require the user to be logged in
# Should return 200 OK and a HTML file
OVERVIEW_PATH = "/OVERVIEW_LANDING.GBL%3BPHPSESSID%3DUJHXZRQZXGNUSZOJCEEUBYZ%3BVIEWPORT%3D-1%3BSCROLL%3D0%3BIS_ADMIN%3D0%3BFREE_MONEY%3D0%3BTEST_COHORT%5B1%5D%3D168860956789"
# Should return 200 OK and a HTML file
SECURITY_PATH = "/Security%20page%20%282011-01-28-v1%29%20draft%20%28final%29%20v0.asp"
# Should return 200 OK and a HTML file
SUBACCOUNTS_PATH = "/phpbb_modified/accounts.php"

# Going to do this view later because we need to POST to it.
# CHECK_TOKEN_LENGTH_PATH = "/ajax/check_token_length"

# --- ACTUAL TESTS ---


class WebsiteUser(HttpUser):
    wait_time = between(1, 5)

    def on_start(self):
        # Check if the TEST_LOCKDOWN_PASSWORD environment variable is present
        lockdown_password = os.environ.get("TEST_LOCKDOWN_PASSWORD")
        if lockdown_password:
            # Get the initial lockdown page to retrieve the csrfmiddlewaretoken
            response = self.client.get("/")
            soup = BeautifulSoup(response.text, "html.parser")
            find_result = soup.find("input", {"name": "csrfmiddlewaretoken"})
            csrfmiddlewaretoken = find_result["value"]  # type: ignore

            if csrfmiddlewaretoken is None:
                raise Exception("Could not find csrfmiddlewaretoken")

            # Preparing the payload for the form
            payload = {
                "csrfmiddlewaretoken": csrfmiddlewaretoken,
                "password": lockdown_password,
            }

            # Posting the payload to bypass the lockdown
            response = self.client.post("/", data=payload)

            # Check for 302 status code
            if response.status_code != 302:
                raise Exception("Unexpected status code after submitting lockdown form")

        # log us into the app
        self.client.get(TEST_USER_LOGIN_URL)

    @task
    def robots_txt(self):
        self.client.get(ROBOTS_PATH)

    @task
    def login_page(self):
        self.client.get(LOGIN_PATH, allow_redirects=False)

    @task
    def leaderboard(self):
        self.client.get(LEADERBOARD_PATH)

    @task
    def consent_and_terms(self):
        self.client.get(CONSENT_AND_TERMS_PATH)

    @task
    def overview(self):
        self.client.get(OVERVIEW_PATH)

    @task
    def security_page(self):
        self.client.get(SECURITY_PATH)

    @task
    def subaccounts_page(self):
        self.client.get(SUBACCOUNTS_PATH)
