#!/bin/bash

set -euo pipefail

# Define colors using ANSI escape codes
CYAN='\033[0;36m'
YELLOW='\033[1;33m'
GREEN='\033[0;32m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# deduplicate this with check.py
SOURCES=(src/ setup.py manage.py locustfile.py
         deploy/replace_cloud_run_service.py data-pipeline/setup.py
         data-pipeline/src)

# Function to compute checksum of the files
checksum_files() {
    find "${SOURCES[@]}" -type f -name '*.py' -exec sha256sum {} \; | sha256sum
}

total_iters=2
for it in $(seq 1 "$total_iters"); do
    echo "======= Iteration $it/$total_iters ======="
    echo -e "${CYAN}Sorting imports with isort ($it/$total_iters)...${NC}"
    isort "${SOURCES[@]}"

    echo -e "${YELLOW}Formatting code with Black ($it/$total_iters)...${NC}"
    black "${SOURCES[@]}"

    # Compute checksum of the files before autoflake runs
    BEFORE_CHECKSUM="$(checksum_files)"
    # Run autoflake
    echo -e "${GREEN}Removing unused imports with autoflake ($it/$total_iters)...${NC}"
    autoflake --remove-all-unused-imports --recursive --in-place "${SOURCES[@]}" --exclude=__init__.py
    # Compute checksum of the files after autoflake runs to detect changes.
    # autoflake can easily break code by removing "load-bearing" imports, so
    # it's important to warn the user of this.
    AFTER_CHECKSUM="$(checksum_files)"

    # Compare the checksums and print a warning if they do not match
    if [ "$BEFORE_CHECKSUM" != "$AFTER_CHECKSUM" ]; then
        echo -e "${RED}WARNING: autoflake might have made breaking changes to\n\
the source code. Please re-run your test suite to ensure\n\
everything is still functioning correctly.${NC}"
    fi
done
