# Dataset preparation pipeline

The code in this directory translates from a Postgres database dump (obtained from Cloud SQL) to the final "raw" dataset and the two benchmarks described in the paper.

It's a separate Python package to the rest of our code (the web interface). You can install it with `pip install -e .` again. After that you can do `run-pipeline` in your shell to run the script.

The pipeline script needs a raw SQL dump from our database. See [this Slack message](https://humancompatibleai.slack.com/archives/C05M049V1UL/p1694481815457699) for instructions on how to access the dumps.

Our third-party LLM experiments need API keys. To register the API keys with
this code, create a `.env` file in this directory containing your keys. You can
do this by first copying the template to `.env`:

```bash
cp .env.template .env
```

Then edit `.env` to contain the actual values for each key. **Do not check these
keys into version control**. The `.env` file has been gitignored, but
`.env.template` has not.

## Writing the entire dataset generation pipeline

Putting down these commands for posterity.

```bash
# initial-load converts a SQL dump to Parquet. It requires a certain hardcoded
# input path (check script) to exist, and will create a new directory called
# `outdata` to write all subsequent pipeline artifacts.
# Creates output files gameui_authattempt.parquet,  gameui_defense.parquet,
# gameui_flag.parquet in outdata/.
run-pipeline initial-load

# adds some extra columns to the attacks for later filtering
run-pipeline postprocess-raw-attacks

# marks defenses as valid/invalid for each model, depending on whether the access code causes that model to say 'access granted'
run-pipeline check-defense-transfer

# selects a subset of defenses that transfer well, have reasonably short access
# codes, are not too similar to each other, etc.
run-pipeline generate-good-defenses

# use adversarial filtering to create hijacking and extraction datasets (chooses
# 5,000-8,000 random defense/attack pairs as input, then filters it down to
# 500-1,000 pairs per benchmark)
run-pipeline create-hijacking-dataset --samples-to-filter 5000
run-pipeline create-extraction-dataset --samples-to-filter 8000
```

The next two steps will show you a tkinter GUI and require a lot of manual
intervention, so you might want to run them on a different machine.

```bash
run-pipeline verify-samples hijacking-robustness
run-pipeline verify-samples extraction-robustness
```

Finally, you can run evals. This calls Claude 2 and GPT4, and so will probably
cost a lot more money than any of the previous steps. Model input/output pairs
are cached, so it's fine to interrupt this half-way through and then restart it,
or to run it multiple times (later runs will just be very fast because they'll
always hit the cache).

```bash
run-pipeline run-robustness-evals --no-truncated-eval \
    -m gpt-3.5-turbo-0613 \
    -m claude-instant-1.2 \
    -m chat-bison@001 \
    -m gpt-4-0613 \
    -m claude-2.0 \
    -m meta-llama/Llama-2-7b-chat-hf \
    -m meta-llama/Llama-2-13b-chat-hf \
    -m meta-llama/Llama-2-70b-chat-hf \
    -m codellama/CodeLlama-34b-Instruct-hf
```

You may want to run `run-robustness-evals -m <model>` separately for each
model, so that everything can run in parallel. After that, you can run the
command above to collate all the results. Specifically, the parallel commands
that you want to run before the above command are:

```bash
run-pipeline run-robustness-evals --no-truncated-eval -m gpt-3.5-turbo-0613
run-pipeline run-robustness-evals --no-truncated-eval -m claude-instant-1.2
run-pipeline run-robustness-evals --no-truncated-eval -m chat-bison@001
run-pipeline run-robustness-evals --no-truncated-eval -m gpt-4-0613
run-pipeline run-robustness-evals --no-truncated-eval -m claude-2.0
run-pipeline run-robustness-evals --no-truncated-eval -m meta-llama/Llama-2-7b-chat-hf
run-pipeline run-robustness-evals --no-truncated-eval -m meta-llama/Llama-2-13b-chat-hf
run-pipeline run-robustness-evals --no-truncated-eval -m meta-llama/Llama-2-70b-chat-hf
run-pipeline run-robustness-evals --no-truncated-eval -m codellama/CodeLlama-34b-Instruct-hf

# after this you run the big command above with all the -m flags
```
